/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdio.h>

#include "run.h"
#include "settings.h"

#ifdef ZVT

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <zvt/zvtterm.h>
#include <gdk/gdkkeysyms.h>
#include "properties.h"
#include "notebook.h"
#include "dialog.h"
#include "menu.h"
#include "misc.h"
#include "statusbar.h"

#define LENGTH 100

GList *runs;

typedef struct _run_t run_t;
struct _run_t {
    char *name;
    char *command;
};

void try_create_run_defaults(const char *filename) {
    FILE *file;
    if((file = fopen(filename, "w"))) {
        fputs("Bash\n/bin/bash\n"
              "Lynx\nlynx\n"
              "Pine\npine\n"
              "$EDITOR\n$EDITOR\n",
              file);
        fclose(file);
    }
}

void gm_run_init() {
    FILE *file;
    char *filename;
    char buf[LENGTH + 1];
    int len;
    run_t *r;
    int tried = FALSE;
#define chop() len = strlen(buf); if(len) buf[len - 1] = '\0'

    runs = NULL;
    filename = g_strconcat(gm_settings_get_config_dir(), "/",
                           RUN_MENU_FILE, NULL);
  try_again:
    if((file = fopen(filename, "r"))) {
        if(debug) printf("Loading run menu\n");
        while(fgets(buf, LENGTH, file)) {
            chop();
            r = g_malloc(sizeof(run_t));
            r->name = g_strdup(buf);
            fgets(buf, LENGTH, file);
            chop();
            r->command = g_strdup(buf);
            if(debug) printf("\tAdded \"%s\" \"%s\"\n", r->name, r->command);
            runs = g_list_append(runs, r);
        }
        fclose(file);
    } else if(!tried) {
        try_create_run_defaults(filename);
        tried = TRUE;
        goto try_again;
    }
    g_free(filename);
#undef chop
}

void gm_run_free() {
    GList *l;
    for(l = runs; l; l = g_list_next(l)) {
        g_free(((run_t *)(l->data))->name);
        g_free(((run_t *)(l->data))->command);
        g_free(l->data);
    }
    g_list_free(runs);
    runs = NULL;
}

void run_add_from_run_t(run_t *r, GtkCList *clist) {
    char *text[2];
    text[0] = r->name;
    text[1] = r->command;
    gtk_clist_append(clist, text);
}

void run_update_sensitivities(d_run_t *r) {
    int row;

    if(GTK_CLIST(r->clist)->selection) {
        row = GPOINTER_TO_UINT(GTK_CLIST(r->clist)->selection->data);

        gtk_widget_set_sensitive(r->update_but, TRUE);
        gtk_widget_set_sensitive(r->remove_but, TRUE);
        gtk_widget_set_sensitive(r->up_but,     (row > 0));
        gtk_widget_set_sensitive(r->down_but,
                                 row < GTK_CLIST(r->clist)->rows - 1);
    } else {
        gtk_widget_set_sensitive(r->update_but, FALSE);
        gtk_widget_set_sensitive(r->remove_but, FALSE);
        gtk_widget_set_sensitive(r->up_but,     FALSE);
        gtk_widget_set_sensitive(r->down_but,   FALSE);
    }
}

void run_update_all_sensitivities(GtkWidget *check, d_run_t *r) {
    if(GTK_TOGGLE_BUTTON(check)->active) {
        gtk_widget_set_sensitive(r->name_w, TRUE);;
        gtk_widget_set_sensitive(r->command_w, TRUE);;
        gtk_widget_set_sensitive(r->clist, TRUE);;
        gtk_widget_set_sensitive(r->add_but, TRUE);;
        run_update_sensitivities(r);
    } else {
        gtk_widget_set_sensitive(r->name_w, FALSE);
        gtk_widget_set_sensitive(r->command_w, FALSE);
        gtk_widget_set_sensitive(r->clist, FALSE);
        gtk_widget_set_sensitive(r->add_but, FALSE);
        gtk_widget_set_sensitive(r->update_but, FALSE);
        gtk_widget_set_sensitive(r->up_but, FALSE);
        gtk_widget_set_sensitive(r->down_but, FALSE);
        gtk_widget_set_sensitive(r->remove_but, FALSE);
    }
}

void run_select_row(GtkCList *clist, int row, int column, GdkEvent *event,
                    d_run_t *r) {
    char *name, *command;
    run_update_sensitivities(r);
    gtk_clist_get_text(clist, row, 0, &name);
    gtk_clist_get_text(clist, row, 1, &command);

    if(debug) printf("Selected: \"%s\" \"%s\"\n", name, command);

    gtk_entry_set_text(GTK_ENTRY(r->name_w), name);
    gtk_entry_set_text(GTK_ENTRY(r->command_w), command);
}

void run_unselect_row(GtkCList *clist, int row, int column, GdkEvent *event,
                      d_run_t *r) {
    run_update_sensitivities(r);
}

void run_row_move(GtkCList *clist, int from, int to, d_run_t *r) {
    run_update_sensitivities(r);
}

void run_up(GtkWidget *but, d_run_t *r) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(r->clist)->selection->data);
    gtk_clist_row_move(GTK_CLIST(r->clist), row, row - 1);
}

void run_down(GtkWidget *but, d_run_t *r) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(r->clist)->selection->data);
    gtk_clist_row_move(GTK_CLIST(r->clist), row, row + 1);
}

void run_remove(GtkWidget *but, d_run_t *r) {
    int row = GPOINTER_TO_UINT(GTK_CLIST(r->clist)->selection->data);
    gtk_clist_remove(GTK_CLIST(r->clist), row);
}

int run_check_if_valid(const char *name, char *command) {
    if(name && strcmp(name, "") == 0) {
        gm_dialog_popup(_("Error"), _("The name is empty"), B_OK, TRUE);
        return FALSE;
    } else if(strcmp(command, "") == 0) {
        gm_dialog_popup(_("Error"), _("The command is empty"), B_OK, TRUE);
        return FALSE;
    }
    return TRUE;
}

void run_changed(GtkWidget *w, d_run_t *r) {
    gtk_widget_set_sensitive(r->pc->ok,    TRUE);
    gtk_widget_set_sensitive(r->pc->apply, TRUE);
}

void run_update(GtkWidget *but, d_run_t *r) {
    int row;
    char *name = gtk_entry_get_text(GTK_ENTRY(r->name_w));
    char *command = gtk_entry_get_text(GTK_ENTRY(r->command_w));
    if(run_check_if_valid(name, command)) {
        row = GPOINTER_TO_UINT(GTK_CLIST(r->clist)->selection->data);
        gtk_clist_set_text(GTK_CLIST(r->clist), row, 0, name);
        gtk_clist_set_text(GTK_CLIST(r->clist), row, 1, command);
        run_changed(but, r);
    }
}

void run_add(GtkWidget *but, d_run_t *r) {
    char *text[2];
    char *name = gtk_entry_get_text(GTK_ENTRY(r->name_w));
    char *command = gtk_entry_get_text(GTK_ENTRY(r->command_w));
    if(run_check_if_valid(name, command)) {
        text[0] = name;
        text[1] = command;
        gtk_clist_append(GTK_CLIST(r->clist), text);
        run_changed(but, r);
    }
}

d_run_t *gm_run_create_properties(GtkTable *table, settings_t *s) {
    GtkWidget *tmp;
    GtkWidget *scrollw;
    GtkWidget *hbb;
    d_run_t *r = g_malloc(sizeof(d_run_t));

    r->enable = pc_check_new(table, _("Enable run menu"), &s->enable_run_menu);
    gtk_signal_connect(GTK_OBJECT(r->enable), "clicked",
                       GTK_SIGNAL_FUNC(run_update_all_sensitivities), r);

    gtk_table_resize(table, 7, 3);
    r->table = GTK_WIDGET(table);


    r->name_w = gtk_entry_new_with_max_length(LENGTH);
    gtk_widget_show(r->name_w);
    gtk_table_attach(GTK_TABLE(r->table), r->name_w, 1, 2, 2, 3,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);

    r->command_w = gtk_entry_new_with_max_length(LENGTH);
    gtk_widget_show(r->command_w);
    gtk_table_attach(GTK_TABLE(r->table), r->command_w, 1, 2, 3, 4,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);

    tmp = gtk_label_new(_("Name"));
    gtk_widget_show(tmp);
    gtk_table_attach(GTK_TABLE(r->table), tmp, 0, 1, 2, 3,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(tmp), 1, 0.5);

    tmp = gtk_label_new(_("Command"));
    gtk_widget_show(tmp);
    gtk_table_attach(GTK_TABLE(r->table), tmp, 0, 1, 3, 4,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(tmp), 1, 0.5);

    scrollw = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(scrollw);
    gtk_table_attach(GTK_TABLE(r->table), scrollw, 0, 2, 5, 6,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollw),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    r->clist = gtk_clist_new(2);
    gtk_widget_show(r->clist);
    gtk_widget_set_usize(r->clist, -1, 100);
    gtk_container_add(GTK_CONTAINER(scrollw), r->clist);
    gtk_clist_set_column_width(GTK_CLIST(r->clist), 0, 80);
    gtk_clist_set_column_width(GTK_CLIST(r->clist), 1, 80);
    gtk_clist_column_titles_show(GTK_CLIST(r->clist));
    gtk_signal_connect(GTK_OBJECT(r->clist), "select_row",
                       GTK_SIGNAL_FUNC(run_select_row), r);
    gtk_signal_connect(GTK_OBJECT(r->clist), "unselect_row",
                       GTK_SIGNAL_FUNC(run_select_row), r);
    gtk_signal_connect(GTK_OBJECT(r->clist), "row_move",
                       GTK_SIGNAL_FUNC(run_row_move), r);

    tmp = gtk_label_new(_("Name"));
    gtk_widget_show(tmp);
    gtk_clist_set_column_widget(GTK_CLIST(r->clist), 0, tmp);
    gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
    gtk_misc_set_padding(GTK_MISC(tmp), 2, 0);

    tmp = gtk_label_new(_("Command"));
    gtk_widget_show(tmp);
    gtk_clist_set_column_widget(GTK_CLIST(r->clist), 1, tmp);
    gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
    gtk_misc_set_padding(GTK_MISC(tmp), 2, 0);

    hbb = gtk_hbutton_box_new();
    gtk_widget_show(hbb);
    gtk_table_attach(GTK_TABLE(r->table), hbb, 0, 2, 4, 5,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbb), GTK_BUTTONBOX_SPREAD);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbb), 5);
    gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbb), 0, 0);

    r->add_but = gtk_button_new_with_label(_("Add"));
    gtk_widget_show(r->add_but);
    gtk_container_add(GTK_CONTAINER(hbb), r->add_but);
    gtk_signal_connect(GTK_OBJECT(r->add_but), "clicked",
                       GTK_SIGNAL_FUNC(run_add), r);

    r->update_but = gtk_button_new_with_label(_("Update"));
    gtk_widget_show(r->update_but);
    gtk_container_add(GTK_CONTAINER(hbb), r->update_but);
    gtk_signal_connect(GTK_OBJECT(r->update_but), "clicked",
                       GTK_SIGNAL_FUNC(run_update), r);

    hbb = gtk_hbutton_box_new();
    gtk_widget_show(hbb);
    gtk_table_attach(GTK_TABLE(r->table), hbb, 0, 2, 6, 7,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbb), GTK_BUTTONBOX_SPREAD);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbb), 5);
    gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbb), 0, 0);

    r->up_but = gtk_button_new_with_label(_("Move Up"));
    gtk_widget_show(r->up_but);
    gtk_container_add(GTK_CONTAINER(hbb), r->up_but);
    gtk_signal_connect(GTK_OBJECT(r->up_but), "clicked",
                       GTK_SIGNAL_FUNC(run_up), r);
    gtk_signal_connect(GTK_OBJECT(r->up_but), "clicked",
                       GTK_SIGNAL_FUNC(run_changed), r);

    r->down_but = gtk_button_new_with_label(_("Move Down"));
    gtk_widget_show(r->down_but);
    gtk_container_add(GTK_CONTAINER(hbb), r->down_but);
    gtk_signal_connect(GTK_OBJECT(r->down_but), "clicked",
                       GTK_SIGNAL_FUNC(run_down), r);
    gtk_signal_connect(GTK_OBJECT(r->down_but), "clicked",
                       GTK_SIGNAL_FUNC(run_changed), r);

    r->remove_but = gtk_button_new_with_label(_("Remove"));
    gtk_widget_show(r->remove_but);
    gtk_container_add(GTK_CONTAINER(hbb), r->remove_but);
    gtk_signal_connect(GTK_OBJECT(r->remove_but), "clicked",
                       GTK_SIGNAL_FUNC(run_remove), r);
    gtk_signal_connect(GTK_OBJECT(r->remove_but), "clicked",
                       GTK_SIGNAL_FUNC(run_changed), r);

    run_update_all_sensitivities(r->enable, r);

    g_list_foreach(runs, (GFunc) run_add_from_run_t, (void *) r->clist);

    return r;
}

void gm_run_free_dialog(d_run_t *r) {
    g_free(r);
}

void run_fput(run_t *r, FILE *file) {
    fputs(r->name, file);
    fputc('\n', file);
    fputs(r->command, file);
    fputc('\n', file);
}

void gm_run_apply_changes(d_run_t *r) {
    run_t *run;
    char *name;
    char *command;
    int row;
    char *filename;
    FILE *file;

    gm_run_free();

    for(row = 0; row < GTK_CLIST(r->clist)->rows; row++) {
        gtk_clist_get_text(GTK_CLIST(r->clist), row, 0, &name);
        gtk_clist_get_text(GTK_CLIST(r->clist), row, 1, &command);
        run = g_malloc(sizeof(run_t));
        run->name = g_strdup(name);
        run->command = g_strdup(command);
        runs = g_list_append(runs, run);
    }

    filename = g_strconcat(gm_settings_get_config_dir(), "/",
                           RUN_MENU_FILE, NULL);    
    if((file = fopen(filename, "w"))) {
        g_list_foreach(runs, (GFunc) run_fput, file);
        fclose(file);
    }
    g_free(filename);

    gm_menu_update_run_menu();
}

extern void copy_enviroment();
extern char **env_copy;

void run_start_running(t_run_t *r) {
    char *cmd[4];
    char *s;

    if(!env_copy) {
        copy_enviroment();
    }

    cmd[0] = "/bin/sh";
    cmd[1] = "-c";
    cmd[2] = r->command;
    cmd[3] = NULL;

    if(debug) printf("\tCreating new process\n");
    switch(zvt_term_forkpty(ZVT_TERM(r->term),
                            ZVT_TERM_DO_UTMP_LOG|ZVT_TERM_DO_WTMP_LOG)) {
    case -1:
        s = g_strconcat(_("Could not fork child process:\n"),
                        g_strerror(errno), NULL);
        gm_dialog_popup_only(_("Error"), s, TRUE);
        g_free(s);
        gm_notebook_close_run(r);
    case 0:
        execve(cmd[0], cmd, env_copy);
        s = g_strconcat(_("Could not fork child process:\n"),
                        g_strerror(errno), NULL);
        gm_dialog_popup_only(_("Error"), s, TRUE);
        g_free(s);
        _exit(127);
    }
    if(debug) printf("Done opening\n");
}

void run_died(ZvtTerm *term, t_run_t *r) {
    if(debug) printf("Run \"%s\" died, closing\n", r->name);
    gm_notebook_close_run(r);
}

int run_key_pressed(GtkWidget *input, GdkEventKey *event, t_run_t *r) {
    int handled = TRUE;
    /* Tab switching ... */
    if((event->keyval == GDK_Tab ||
        event->keyval == GDK_KP_Tab ||
        event->keyval == GDK_ISO_Left_Tab) &&
       event->state & settings->hotkey_mask) {
        if(event->state & GDK_SHIFT_MASK)
            gm_notebook_prev();
        else
            gm_notebook_next();
    } else if(event->keyval >= GDK_0 && event->keyval <= GDK_9 && 
              event->state & settings->hotkey_mask) {
        gm_notebook_set(event->keyval - GDK_0 > 0 ? event->keyval-GDK_0-1 : 9);
    } else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
              event->keyval == GDK_Left) {
        gm_notebook_prev();
    } else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
              event->keyval == GDK_Right) {
        gm_notebook_next();
        /* END switching ... */
 
    } else {
        /* Unhandled key combo */
        handled = FALSE;
    }
    if(handled) {
        gtk_signal_emit_stop_by_name(GTK_OBJECT(input), "key_press_event");
    }
    return handled;
}

void run_title_changed(ZvtTerm *term, VTTITLE_TYPE type, char *newtitle,
                       t_run_t *r) {
    if(type == VTTITLE_WINDOWICON || type == VTTITLE_WINDOW) {
        g_free(r->title);
        r->title = g_strconcat(r->name, ": ", newtitle, NULL);
        gm_statusbar_set(r->title);
    }
}

void gm_run_try_run(const char *name, const char *command) {
    properties_t *p;
    t_run_t *r = g_malloc(sizeof(t_run_t));

    r->name = g_strdup(name ? name : command);
    r->command = g_strdup(command);
    r->title = NULL;
    
    r->main = gtk_frame_new(NULL);
    gtk_widget_ref(r->main);
    gtk_widget_show(r->main);
    gtk_frame_set_shadow_type(GTK_FRAME(r->main), GTK_SHADOW_IN);
    
    r->term = zvt_term_new();
    gtk_widget_show(r->term);
    p = gm_world_get_settings_by_name(TEMPLATE);
    zvt_term_init(r->term, p);
    gm_properties_free(p);
    gtk_signal_connect(GTK_OBJECT(r->term), "child_died",
                       GTK_SIGNAL_FUNC(run_died), r);
    gtk_signal_connect(GTK_OBJECT(r->term), "title_changed",
                       GTK_SIGNAL_FUNC(run_title_changed), r);
    gtk_signal_connect(GTK_OBJECT(r->term), "key_press_event",
                       GTK_SIGNAL_FUNC(run_key_pressed), r);
    gtk_container_add(GTK_CONTAINER(r->main), r->term);
    
    r->tab_label = gtk_label_new("");
    gtk_widget_show(r->tab_label);

    gm_notebook_append_type(TK_RUN, r, r->main, r->tab_label);

    run_start_running(r);
}

void run_handle_undefined() {
    char *command, *old_command = NULL;
    while((command = gm_dialog_text(_("Run program"),
                                    _("Please specify a program"),
                                    old_command ? old_command : "",
                                    FALSE))) {
        if(run_check_if_valid(NULL, command)) {
            gm_run_try_run(NULL, command);
            break;
        }
        g_free(old_command);
        old_command = command;
    }
    g_free(command);
    g_free(old_command);
}

void run_menu_clicked(GtkWidget *menu_item, run_t *r) {
    if(r) {
        if(debug) {
            printf("Clicked run menu \"%s\"\n", r->name);
            printf("\t=> \"%s\"\n",r->command);
        }
        gm_run_try_run(r->name, r->command);
    } else {
        if(debug) printf("Clicked non predefined run menu\n");
        run_handle_undefined();
    }
}

void run_add_to_menu(run_t *r, GtkWidget *menu) {
    GtkWidget *item = gtk_menu_item_new_with_label(r->name);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(menu), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
                       GTK_SIGNAL_FUNC(run_menu_clicked), r);
}

GtkWidget *gm_run_create_menu() {
    GtkWidget *menu, *item;

    menu = gtk_menu_new();
    gtk_widget_show(menu);

    item = gtk_tearoff_menu_item_new();
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(menu), item);

    if(runs) {
        g_list_foreach(runs, (GFunc) run_add_to_menu, (void *) menu);

        item = gtk_menu_item_new();
        gtk_widget_show(item);
        gtk_container_add(GTK_CONTAINER(menu), item);
    }
  
    item = gtk_menu_item_new_with_label(_("Run..."));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(menu), item);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
                       GTK_SIGNAL_FUNC(run_menu_clicked), NULL);

    return menu;
}

void gm_run_set_position(t_run_t *r, int pos) {
    char *s, *name = settings->tab_limit_length ?
        limit_string_to_length(r->name, settings->tab_max_length) :
        r->name;

    r->position = pos < 10 ? (pos + 1) % 10 : -1;

    if(settings->tab_pos_on_tab && r->position > -1) {
        s = g_strdup_printf("(%d) %s", r->position, name);
        gtk_label_set_text(GTK_LABEL(r->tab_label), s);
        g_free(s);
    } else {
        gtk_label_set_text(GTK_LABEL(r->tab_label), name);
    }
    if(settings->tab_limit_length) {
        g_free(name);
    }
}

void gm_run_close(t_run_t *r) {
    gtk_widget_destroy(r->main);
    g_free(r->name);
    g_free(r->command);
    g_free(r->title);
    g_free(r);
}

void gm_run_got_focus(t_run_t *r) {
    gm_statusbar_set(r->title ? r->title : r->name);
}

#endif /* ZVT */
