/******************************************************************************\
 gnofin/ui-menubar.c   $Revision: 1.37.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtkmenushell.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtktearoffmenuitem.h>
#include <gtk/gtkhseparator.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnome/gnome-config.h>
#include "ui.h"
#include "ui-menubar.h"
#include "ui-statusbar.h"
#include "ui-record-select.h"
#include "data-if.h"
#include "dialogs.h"
#include "file-filter.h"
#include "file-mru.h"
#include "file-utils.h"
#include "preferences.h"
#include "gnofin-defaults.h"
#include "config-saver.h"
#include "merging.h"
#include "clipboard.h"
#include "tool.h"


static const gchar *file_filter_key = "ui-menubar-file-filter";

struct _UI_Menubar
{
  UI        *ui;

  GtkWidget *file_save;
  GtkWidget *file_save_as;

  GtkWidget *file_imports;
  GtkWidget *file_exports;
  GtkWidget *file_mru;
  GtkWidget *tools;

  GtkWidget *edit_undo;
  GtkWidget *edit_redo;
  GtkWidget *edit_cut;
  GtkWidget *edit_copy;
  GtkWidget *edit_paste;
  GtkWidget *edit_select_all;
  GtkWidget *edit_toggle_status;
  GtkWidget *edit_delete;

  GtkWidget *account_delete;
  GtkWidget *account_props;

#ifdef ENABLE_UI_RECORD_SELECT
  GtkWidget *account_records;
#endif

  guint      imp_stamp;
  guint      exp_stamp;
  guint      mru_stamp;
  guint      tools_stamp;

  GtkWidget *popup_menu;
  GtkWidget *popup_undo;
  GtkWidget *popup_redo;
  GtkWidget *popup_cut;
  GtkWidget *popup_copy;
  GtkWidget *popup_paste;
  GtkWidget *popup_select_all;
  GtkWidget *popup_delete;
  GtkWidget *popup_toggle_status;
};

/******************************************************************************
 * Configuration
 */

typedef struct {
  gboolean spawn_new_ui;
  gboolean new_acc_on_open;
  gboolean load_lastfile;
} Config;

#define CAT  "UI_Menubar"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  trace ("");
  config->spawn_new_ui = gnome_config_get_int (KEY "spawn_new_ui=1");
  config->new_acc_on_open = gnome_config_get_int (KEY "new_account_on_open=1");
  config->load_lastfile = gnome_config_get_int (KEY "load_lastfile=1");
}

static void
save_config (const Config *config)
{
  trace ("");
  gnome_config_set_int (KEY "spawn_new_ui", config->spawn_new_ui);
  gnome_config_set_int (KEY "new_account_on_open", config->new_acc_on_open);
  gnome_config_set_int (KEY "load_lastfile", config->load_lastfile);
}

static Config *
get_config ()
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    init = TRUE;
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
  }
  return &config;
}

/******************************************************************************
 * Menu handlers
 */

static void
on_nop (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  dialog_notice (ui_get_window (menubar->ui),
	       _("Sorry, but this feature is not yet implemented."));
}

static void
on_file_new (GtkWidget *w, UI_Menubar *menubar)
{
  Config *config = get_config ();
  UI *ui;

  trace ("");
  g_return_if_fail (menubar);

  if (config->spawn_new_ui)
    ui = ui_create ();
  else
    ui = menubar->ui;

  if (ui_load_bankbook (ui, NULL, NULL) && config->new_acc_on_open)
    ui_insert_account (ui);
}

static void
on_file_open (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  if (get_config ()->spawn_new_ui && !ui_is_empty (menubar->ui))
  {
    Bankbook *book = NULL;
    gchar *filename = NULL;

    if (ui_load_file_1 (menubar->ui, &filename, &book))
    {
      UI *ui = ui_create ();
      ui_load_bankbook (ui, book, filename);
      g_free (filename);
    }
  }
  else
    ui_load_file (menubar->ui, NULL);
}

static void
on_file_save (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_save_file (menubar->ui, ui_get_current_filename (menubar->ui));
}

static void
on_file_save_as (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_save_file (menubar->ui, NULL);
}

static void
on_file_exit (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  ui_exit (menubar->ui);
}

static void
on_edit_undo (GtkWidget *w, UI_Menubar *menubar)
{
  Bankbook *book;

  trace ("");
  g_return_if_fail (menubar);

  book = ui_get_current_bankbook (menubar->ui);
  if (book)
    if_bankbook_undo (book);
}

static void
on_edit_redo (GtkWidget *w, UI_Menubar *menubar)
{
  Bankbook *book;

  trace ("");
  g_return_if_fail (menubar);

  book = ui_get_current_bankbook (menubar->ui);
  if (book)
    if_bankbook_redo (book);
}

static void
on_edit_copy (GtkWidget *w, UI_Menubar *menubar)
{
  Record *record;

  trace ("");

  record = ui_get_current_record (menubar->ui);
  if (record)
  {
    GSList list = {0};
    list.data = (gpointer) record;

    clipboard_set_source (w, &list);
  }
}

static void
on_clipboard_paste (Bankbook *book, UI_Menubar *menubar)
{
  Account *acc, *dest;
  Record *rec;

  trace ("");
  
  acc = if_bankbook_get_account_by_index (book, 0);
  rec = if_account_get_record_by_index (acc, 0);

  dest = ui_get_current_account (menubar->ui);
  rec = merge_record_into_account (dest, rec, TRUE);

  ui_select_record (menubar->ui, rec);
}

static void
on_notify_paste (GtkWidget *widget, gboolean can_paste, UI_Menubar *menubar)
{
  trace ("");

  can_paste = (can_paste && (ui_get_current_account (menubar->ui) != NULL));
  gtk_widget_set_sensitive (widget, can_paste);
}

static void
on_edit_paste (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  clipboard_paste (w);
}

static void
on_edit_delete (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_remove_current_record (menubar->ui);
}

static void
on_edit_cut (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");

  on_edit_copy (w, menubar);
  on_edit_delete (w, menubar);
}

static void
on_edit_toggle_status (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_toggle_current_record_status (menubar->ui);
}

static void
on_edit_define_record_types (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_modify_current_record_types (menubar->ui);
}

static void
on_edit_preferences (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);
  g_return_if_fail (menubar->ui);

  preferences_dialog_run (ui_get_window (menubar->ui));
}

static void
on_account_insert (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_insert_account (menubar->ui);
}

static void
on_account_delete (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_remove_current_account (menubar->ui);
}

static void
on_account_props (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_modify_current_account (menubar->ui);
}

#ifdef ENABLE_UI_RECORD_SELECT
static void
on_account_select_records (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);

  ui_select_records_current_account (menubar->ui);
}
#endif

static void
on_help_about (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");
  g_return_if_fail (menubar);
  g_return_if_fail (menubar->ui);

  dialog_about (ui_get_window (menubar->ui));
}

#ifdef ENABLE_DUMP_STATE
static void
on_dump_state (GtkWidget *w, UI_Menubar *menubar)
{
  trace ("");

  if (menubar && menubar->ui)
  {
    Bankbook *book = ui_get_current_bankbook (menubar->ui);

    if (book)
      if_bankbook_dump (book);
  }
}
#endif

static void
on_file_import (GtkMenuItem *item, UI_Menubar *menubar)
{
  Bankbook *current_book;
  GtkWindow *win;
  FileFilter *filt;
  gchar *fname;

  static FileFormatSelection fmt = {0};
  static gchar *last_fname = NULL;

  trace ("");
  g_return_if_fail (menubar);
  g_return_if_fail (menubar->ui);

  filt = (FileFilter *) gtk_object_get_data (GTK_OBJECT (item), file_filter_key);
  g_assert (filt);

  win = ui_get_window (menubar->ui);
  current_book = ui_get_current_bankbook (menubar->ui);

  if (!fmt.label)
  {
    fmt.label = _("Import mode:");
    fmt.strings = g_slist_append (fmt.strings, _("As new file"));
    fmt.strings = g_slist_append (fmt.strings, _("As new account in this file"));

    if ((if_bankbook_get_accounts (current_book) != NULL))
      fmt.strings = g_slist_append (fmt.strings, _("Into current account"));
    else
      fmt.selection = MIN (fmt.selection, 1);
  }
  fname = dialog_query_filename (win, _("Specify input file"), last_fname, FALSE, &fmt);

  if (fname)
  {
    Bankbook *book = if_bankbook_new ();
    if (filt->import (win, fname, book))
    {
      switch (fmt.selection)
      {
      case 0:
	{
	  Config *config = get_config ();
	  UI *ui;
	  if (config->spawn_new_ui && !ui_is_empty (menubar->ui))
	    ui = ui_create ();
	  else
	    ui = menubar->ui;
	  install_default_categories (book);
	  if (!ui_load_bankbook (ui, book, NULL))
	    if (config->spawn_new_ui) ui_exit (ui);
	}
	break;
      case 1:
	merge_bankbook_into_bankbook (current_book, book, TRUE);
	break;
      case 2:
	{
	  const GList *list = if_bankbook_get_accounts (book);
	  Account *account = LIST_DEREF (Account, list);
	  merge_account_into_account (ui_get_current_account (menubar->ui), account, TRUE);
	}
	break;
      }
    }
    else
      if_bankbook_destroy (book);
    g_free (last_fname);
    last_fname = fname;
  }
}

static void
on_file_export (GtkMenuItem *item, UI_Menubar *menubar)
{
  GtkWindow *win;
  FileFilter *filt;
  gchar *fname;
  
  static gchar *start_dir = NULL;

  trace ("");

  filt = (FileFilter *) gtk_object_get_data (GTK_OBJECT (item), file_filter_key);
  g_assert (filt);

  win = ui_get_window (menubar->ui);

  fname = dialog_query_filename (win, _("Specify output file"), start_dir, TRUE, NULL);

  if (fname)
  {
    g_free (start_dir);
    start_dir = file_get_directory (fname);

    filt->export (win, fname, ui_get_current_bankbook (menubar->ui));
    g_free (fname);
  }
}

static void
on_file_open_recent (GtkMenuItem *item, UI_Menubar *menubar)
{
  GtkWidget *label;
  gchar *text;
  UI *ui;

  trace ("");

  label = GTK_BIN (item)->child;
  gtk_label_get (GTK_LABEL (label), &text);

  if (get_config ()->spawn_new_ui && !ui_is_empty (menubar->ui))
    ui = ui_create ();
  else
    ui = menubar->ui;

  if (!ui_load_file (ui, text))
    if (get_config ()->spawn_new_ui) ui_exit (ui);
}

static void
on_tool_run (GtkMenuItem *item, UI_Menubar *menubar)
{
  Tool *tool;

  trace ("");

  tool = (Tool *) gtk_object_get_data (GTK_OBJECT (item), "tool");
  if (tool && tool->funcs.run)
    tool_run (tool, menubar->ui);
}

static void
on_import_select (GtkMenuItem *item, UI_Menubar *menubar)
{
  FileFilter *filt;
  gchar *buf;
  trace ("");

  filt = (FileFilter *) gtk_object_get_data (GTK_OBJECT (item), file_filter_key);
  g_assert (filt);

  buf = g_strdup_printf ("Import \"%s\" file format", filt->label);
  gnome_appbar_push (ui_get_appbar (menubar->ui), buf);
  g_free (buf);
}

static void
on_export_select (GtkMenuItem *item, UI_Menubar *menubar)
{
  FileFilter *filt;
  gchar *buf;
  trace ("");

  filt = (FileFilter *) gtk_object_get_data (GTK_OBJECT (item), file_filter_key);
  g_assert (filt);

  buf = g_strdup_printf ("Export \"%s\" file format", filt->label);
  gnome_appbar_push (ui_get_appbar (menubar->ui), buf);
  g_free (buf);
}

static void
on_file_recent_select (GtkMenuItem *item, UI_Menubar *menubar)
{
  gchar *buf, *text;

  trace ("");

  gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &text);

  buf = g_strdup_printf ("Open \"%s\"", text);
  gnome_appbar_push (ui_get_appbar (menubar->ui), buf);
  g_free (buf);
}

static void
on_tool_select (GtkMenuItem *item, UI_Menubar *menubar)
{
  Tool *tool;

  trace ("");

  tool = (Tool *) gtk_object_get_data (GTK_OBJECT (item), "tool");
  if (tool->description)
    gnome_appbar_push (ui_get_appbar (menubar->ui), tool->description);
}

static void
on_menuitem_deselect (GtkMenuItem *item, UI_Menubar *menubar)
{
  trace ("");
  gnome_appbar_pop (ui_get_appbar (menubar->ui));
}

static void
submenu_clear (GtkMenu *submenu)
{
  /* This function removes any menu items from a submenu.  It is careful
   * however not to remove any tearoff menu items. */

  GList *children = gtk_container_children (GTK_CONTAINER (submenu));
  while (children)
  {
    GtkWidget *child = GTK_WIDGET (children->data);
    children = children->next;

    if (GTK_OBJECT (child)->klass->type != gtk_tearoff_menu_item_get_type ())
      gtk_container_remove (GTK_CONTAINER (submenu), child);
  }
}

static void
on_popup_imports (GtkMenuItem *item, UI_Menubar *menubar)
{
  trace ("");

  if (file_import_filters_changed (menubar->imp_stamp))
  {
    GtkMenu *submenu;
    FileFilter *filt;
    GSList *ff;

    trace ("rebuilding imports list");

    submenu = GTK_MENU (item->submenu);
    submenu_clear (submenu);

    ff = file_import_filters_get (&menubar->imp_stamp);
    for (; ff; ff=ff->next)
    {
      filt = LIST_DEREF (FileFilter, ff);
      trace ("filt=%p, filt->label=%s", filt, filt->label);

      item = GTK_MENU_ITEM (gtk_menu_item_new_with_label (filt->label));
      gtk_object_set_data (GTK_OBJECT (item), file_filter_key, filt);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
      			  GTK_SIGNAL_FUNC (on_file_import), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "select",
      			  GTK_SIGNAL_FUNC (on_import_select), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "deselect",
      			  GTK_SIGNAL_FUNC (on_menuitem_deselect), menubar);
      gtk_widget_show (GTK_WIDGET (item));
      gtk_menu_append (submenu, GTK_WIDGET (item));
    }
  }
}

static void
on_popup_exports (GtkMenuItem *item, UI_Menubar *menubar)
{
  trace ("");

  if (file_export_filters_changed (menubar->exp_stamp))
  {
    GtkMenu *submenu;
    FileFilter *filt;
    GSList *ff;

    trace ("rebuilding exports list");

    submenu = GTK_MENU (item->submenu);
    submenu_clear (submenu);

    ff = file_export_filters_get (&menubar->exp_stamp);
    for (; ff; ff=ff->next)
    {
      filt = LIST_DEREF (FileFilter, ff);
      trace ("filt=%p, filt->label=%s", filt, filt->label);

      item = GTK_MENU_ITEM (gtk_menu_item_new_with_label (filt->label));
      gtk_object_set_data (GTK_OBJECT (item), file_filter_key, filt);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
      			  GTK_SIGNAL_FUNC (on_file_export), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "select",
      			  GTK_SIGNAL_FUNC (on_export_select), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "deselect",
      			  GTK_SIGNAL_FUNC (on_menuitem_deselect), menubar);
      gtk_widget_show (GTK_WIDGET (item));
      gtk_menu_append (submenu, GTK_WIDGET (item));
    }
  }
}

static void
on_popup_file_mru (GtkMenuItem *item, UI_Menubar *menubar)
{
  trace ("");

  if (file_mru_changed (menubar->mru_stamp))
  {
    GtkMenu *submenu;
    const gchar *text;
    GSList *node;

    trace ("rebuilding recent files list");

    submenu = GTK_MENU (item->submenu);
    submenu_clear (submenu);

    node = file_mru_get (&menubar->mru_stamp);
    for (; node; node=node->next)
    {
      text = LIST_DEREF (const gchar, node);
      trace ("file=%s", text);

      item = GTK_MENU_ITEM (gtk_menu_item_new_with_label (text));
      gtk_signal_connect (GTK_OBJECT (item), "activate",
      			  GTK_SIGNAL_FUNC (on_file_open_recent), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "select",
      			  GTK_SIGNAL_FUNC (on_file_recent_select), menubar);
      gtk_signal_connect (GTK_OBJECT (item), "deselect",
      			  GTK_SIGNAL_FUNC (on_menuitem_deselect), menubar);
      gtk_widget_show (GTK_WIDGET (item));
      gtk_menu_append (submenu, GTK_WIDGET (item));
    }
  }
}

static void
on_popup_tools (GtkWidget *item, UI_Menubar *menubar)
{
  trace ("");

  if (tools_changed (menubar->tools_stamp))
  {
    GnomeUIInfo *info;
    GtkMenu *submenu;
    GSList *node = NULL, *tools = NULL;
    Tool *tool;
    guint i, len;

    trace ("rebuilding tools menu");

    submenu = GTK_MENU (GTK_MENU_ITEM (item)->submenu);
    submenu_clear (submenu);

    tools = tools_get (&menubar->tools_stamp);
    len = g_slist_length (tools);

    trace ("len = %d", len);

    info = g_new0 (GnomeUIInfo, len + 1);

    for (i=0, node=tools; node; node=node->next, ++i)
    {
      tool = LIST_DEREF (Tool, node);
      trace ("tool=%s", tool->name);

      info[i].type = GNOME_APP_UI_ITEM;
      info[i].label = (gchar *) tool->label;
      info[i].hint = (gchar *) tool->description;
      info[i].moreinfo = on_tool_run;
      info[i].user_data = menubar;
      info[i].pixmap_type = GNOME_APP_PIXMAP_STOCK;
      info[i].pixmap_info = GNOME_STOCK_MENU_EXEC;
    }

    gnome_app_fill_menu (GTK_MENU_SHELL (submenu), info, NULL, TRUE, 1);

    for (i=0, node=tools; node; node=node->next, ++i)
    {
      tool = LIST_DEREF (Tool, node);
      gtk_object_set_data (GTK_OBJECT (info[i].widget), "tool", tool);
      gtk_signal_connect (GTK_OBJECT (info[i].widget), "select",
      			  GTK_SIGNAL_FUNC (on_tool_select), menubar);
      gtk_signal_connect (GTK_OBJECT (info[i].widget), "deselect",
     			  GTK_SIGNAL_FUNC (on_menuitem_deselect), menubar);
    }
  }
}


/******************************************************************************
 * Helpers */

static void
register_paste_hook (GtkWidget *widget, UI_Menubar *menubar)
{
  ClipboardSignals sigs = { (ClipboardNotifyFunc) on_notify_paste,
			    (ClipboardPasteFunc)  on_clipboard_paste };
  clipboard_attach_sink (widget, &sigs, menubar);

  gtk_signal_connect (GTK_OBJECT (widget), "destroy",
		      GTK_SIGNAL_FUNC (clipboard_remove_sink), NULL);
}


/******************************************************************************
 * Menubar interface
 */

UI_Menubar *
ui_menubar_create (UI *ui)
{
  UI_Menubar * menubar;

  trace ("");
  g_return_val_if_fail (ui, NULL);
  g_return_val_if_fail (ui->statusbar, NULL);
  g_return_val_if_fail (ui->statusbar->appbar, NULL);

  menubar = g_new0 (UI_Menubar, 1);
  menubar->ui = ui;

  /* Create the menus */
  {
    GnomeUIInfo imports_menu[] =
    {
      GNOMEUIINFO_END
    };
    GnomeUIInfo exports_menu[] =
    {
      GNOMEUIINFO_END
    };
    GnomeUIInfo recent_files_menu[] =
    {
      GNOMEUIINFO_END
    };
    GnomeUIInfo file_menu[] =
    {
      GNOMEUIINFO_MENU_NEW_ITEM (N_("_New"), N_("Begin a new accounts file"), on_file_new, menubar),
      GNOMEUIINFO_MENU_OPEN_ITEM (on_file_open, menubar),
      GNOMEUIINFO_MENU_SAVE_ITEM (on_file_save, menubar),
      GNOMEUIINFO_MENU_SAVE_AS_ITEM (on_file_save_as, menubar),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_SUBTREE (N_("_Import"), imports_menu),
      GNOMEUIINFO_SUBTREE (N_("_Export"), exports_menu),
 //   GNOMEUIINFO_SEPARATOR,
 //   { GNOME_APP_UI_ITEM, N_("_Plugins..."), N_("Run plugin manager"),
 //     on_nop, menubar, NULL,
 //     GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_SUBTREE (N_("_Recent Files"), recent_files_menu),
      GNOMEUIINFO_SEPARATOR,
      //GNOMEUIINFO_MENU_EXIT_ITEM (on_file_exit, menubar),
      { GNOME_APP_UI_ITEM, N_("E_xit"), N_("Exit (close this window)"),
	on_file_exit, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, GDK_q, GDK_CONTROL_MASK, NULL },
      GNOMEUIINFO_END
    };
    enum file_indices
    {
      file_new_idx,
      file_open_idx,
      file_save_idx,
      file_save_as_idx,
      file_imports_idx=5,
      file_exports_idx,
      file_mru_idx=8 //10
    };
    GnomeUIInfo edit_menu[] = 
    {
      GNOMEUIINFO_MENU_UNDO_ITEM (on_edit_undo, menubar),
      GNOMEUIINFO_MENU_REDO_ITEM (on_edit_redo, menubar),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_CUT_ITEM (on_edit_cut, menubar),
      GNOMEUIINFO_MENU_COPY_ITEM (on_edit_copy, menubar),
      GNOMEUIINFO_MENU_PASTE_ITEM (on_edit_paste, menubar),
      GNOMEUIINFO_MENU_SELECT_ALL_ITEM (on_nop, menubar),
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("_Delete"), N_("Delete selected transaction(s)"),
	on_edit_delete, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, N_("T_oggle Cleared Status"), N_("Toggle cleared status of selected transaction(s)"),
	on_edit_toggle_status, menubar, NULL,
	GNOME_APP_PIXMAP_NONE, NULL, GDK_F5, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("Transaction _Types..."), N_("Edit transaction types"),
	on_edit_define_record_types, menubar, NULL,
	GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_PREFERENCES_ITEM (on_edit_preferences, menubar),
      GNOMEUIINFO_END
    };
    enum edit_indices
    {
      edit_undo_idx,
      edit_redo_idx,
      edit_cut_idx = 3,
      edit_copy_idx = 4,
      edit_paste_idx = 5,
      edit_select_all_idx = 6,
      edit_delete_idx = 8,
      edit_toggle_status_idx,
    };
    GnomeUIInfo account_menu[] =
    {
      { GNOME_APP_UI_ITEM, N_("_Insert..."), N_("Insert a new account"),
	on_account_insert, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, N_("_Delete"), N_("Delete this account"),
	on_account_delete, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },

#ifdef ENABLE_UI_RECORD_SELECT
	/* 
	 * FIXME: The record-select module has problems, and until sorted out
	 *        we should disable this feature by default -- DWF (9-25-00)
	 */
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("_Select Records..."), N_("Restraint view of record"),
	 on_account_select_records, menubar, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
#endif

      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("_Properties..."), N_("View/edit the properties of this account"),
	on_account_props, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP, 0, 0, NULL },
      GNOMEUIINFO_END
    };
    enum account_indices
    {
      account_insert_idx=0,
      account_delete_idx=1,
#ifdef ENABLE_UI_RECORD_SELECT
      account_records_idx=3,
      account_props_idx=5,
#else
      account_props_idx=3
#endif
    };
    GnomeUIInfo tools_menu[] =
    {
      /*
       * Tools are now dynamically loaded into the menubar
       */
      GNOMEUIINFO_END
    };
    GnomeUIInfo help_menu[] =
    {
      GNOMEUIINFO_MENU_ABOUT_ITEM (on_help_about, menubar),
#ifdef ENABLE_DUMP_STATE
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("Dump _State"), N_("Dump program state for help with debugging"),
	on_dump_state, menubar, NULL,
	GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
#endif
      GNOMEUIINFO_END
    };
    GnomeUIInfo main_menu[] =
    {
      GNOMEUIINFO_MENU_FILE_TREE (file_menu),
      GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
      GNOMEUIINFO_SUBTREE (N_("_Account"), account_menu),
      GNOMEUIINFO_SUBTREE (N_("_Tools"), tools_menu),
      GNOMEUIINFO_MENU_HELP_TREE (help_menu),
      GNOMEUIINFO_END
    };
    enum main_indicies
    {
      tools_idx = 3
    };

    const struct
    {
      GtkWidget **ptr;
      GnomeUIInfo *tree;
      gint index;
    }
    menus[] =
    {
      { &menubar->file_save,          file_menu,      file_save_idx },
      { &menubar->file_save_as,       file_menu,      file_save_as_idx },
      { &menubar->file_imports,       file_menu,      file_imports_idx },
      { &menubar->file_exports,       file_menu,      file_exports_idx },
      { &menubar->file_mru,           file_menu,      file_mru_idx },
      { &menubar->tools,              main_menu,      tools_idx },
      { &menubar->edit_undo,          edit_menu,      edit_undo_idx },
      { &menubar->edit_redo,          edit_menu,      edit_redo_idx },
      { &menubar->edit_cut,           edit_menu,      edit_cut_idx },
      { &menubar->edit_copy,          edit_menu,      edit_copy_idx },
      { &menubar->edit_paste,         edit_menu,      edit_paste_idx },
      { &menubar->edit_select_all,    edit_menu,      edit_select_all_idx },
      { &menubar->edit_toggle_status, edit_menu,      edit_toggle_status_idx },
      { &menubar->edit_delete,        edit_menu,      edit_delete_idx },
      { &menubar->account_delete,     account_menu,   account_delete_idx },
      { &menubar->account_props,      account_menu,   account_props_idx },
#ifdef ENABLE_UI_RECORD_SELECT
      { &menubar->account_records,    account_menu,   account_records_idx }
#endif
    };
    gint i;

    /* Remember and create the main menu */
    gnome_app_create_menus (GNOME_APP (ui->app), main_menu);
    gnome_app_install_appbar_menu_hints (ui->statusbar->appbar, main_menu);

    /* Remember some of the menu items, so we can enable/disable them later */
    for (i=0; i<sizeof_array(menus); ++i)
      *menus[i].ptr = menus[i].tree[menus[i].index].widget;

    register_paste_hook (menubar->edit_paste, menubar);

    gtk_signal_connect (GTK_OBJECT (menubar->file_imports), "select",
    			GTK_SIGNAL_FUNC (on_popup_imports), menubar);
    gtk_signal_connect (GTK_OBJECT (menubar->file_exports), "select",
    			GTK_SIGNAL_FUNC (on_popup_exports), menubar);
    gtk_signal_connect (GTK_OBJECT (menubar->file_mru), "select",
    			GTK_SIGNAL_FUNC (on_popup_file_mru), menubar);
    gtk_signal_connect (GTK_OBJECT (menubar->tools), "select",
    			GTK_SIGNAL_FUNC (on_popup_tools), menubar);

    /* Pre-get this menu so its items will be properly enabled/disabled */
    ui_menubar_get_popup_edit_menu (menubar);
  }
  return menubar;
}

void
ui_menubar_enable_save (UI_Menubar *menubar, gboolean enable)
{
  trace ("");
  g_return_if_fail (menubar);

  gtk_widget_set_sensitive (menubar->file_save, enable);
}

void
ui_menubar_enable_undo (UI_Menubar *menubar, gboolean enable)
{
  trace ("");
  g_return_if_fail (menubar);

  gtk_widget_set_sensitive (menubar->edit_undo, enable);
  gtk_widget_set_sensitive (menubar->popup_undo, enable);
}

void
ui_menubar_enable_redo (UI_Menubar *menubar, gboolean enable)
{
  trace ("");
  g_return_if_fail (menubar);

  gtk_widget_set_sensitive (menubar->edit_redo, enable);
  gtk_widget_set_sensitive (menubar->popup_redo, enable);
}

void
ui_menubar_enable_account_ops (UI_Menubar *menubar, gboolean enable)
{
  trace ("");
  g_return_if_fail (menubar);

  gtk_widget_set_sensitive (menubar->account_delete, enable);
  gtk_widget_set_sensitive (menubar->account_props, enable);

#ifdef ENABLE_UI_RECORD_SELECT
  gtk_widget_set_sensitive (menubar->account_records, enable);
#endif
}

void
ui_menubar_enable_record_ops (UI_Menubar *menubar, gboolean enable)
{
  trace ("");
  g_return_if_fail (menubar);

  gtk_widget_set_sensitive (menubar->edit_toggle_status, enable);
  gtk_widget_set_sensitive (menubar->edit_delete, enable);
  gtk_widget_set_sensitive (menubar->edit_copy, enable);
  gtk_widget_set_sensitive (menubar->edit_cut, enable);
  gtk_widget_set_sensitive (menubar->edit_paste, clipboard_can_paste ());
  gtk_widget_set_sensitive (menubar->edit_select_all, FALSE /*FIXME: enable*/);

  gtk_widget_set_sensitive (menubar->popup_toggle_status, enable);
  gtk_widget_set_sensitive (menubar->popup_delete, enable);
  gtk_widget_set_sensitive (menubar->popup_copy, enable);
  gtk_widget_set_sensitive (menubar->popup_cut, enable);
  gtk_widget_set_sensitive (menubar->popup_paste, clipboard_can_paste ());
  gtk_widget_set_sensitive (menubar->popup_select_all, FALSE /*FIXME: enable*/);
}


/******************************************************************************
 * Behavior
 */

void
ui_menubar_set_behavior (gboolean spawn_new_ui, gboolean new_acc_on_open, gboolean load_lastfile)
{
  Config *config = get_config ();

  trace ("");

  config->spawn_new_ui = spawn_new_ui;
  config->new_acc_on_open = new_acc_on_open;
  config->load_lastfile = load_lastfile;
}

void
ui_menubar_get_behavior (gboolean *spawn_new_ui, gboolean *new_acc_on_open, gboolean *load_lastfile)
{
  Config *config = get_config ();

  trace ("");

  if (spawn_new_ui) *spawn_new_ui = config->spawn_new_ui;
  if (new_acc_on_open) *new_acc_on_open = config->new_acc_on_open;
  if (load_lastfile) *load_lastfile = config->load_lastfile;
}


/*****************************************************************************/

GtkWidget *
ui_menubar_get_popup_edit_menu (UI_Menubar *menubar)
{
  trace ("");

  if (menubar->popup_menu == NULL)
  {
    GnomeUIInfo edit_menu[] = 
    {
      GNOMEUIINFO_MENU_UNDO_ITEM (on_edit_undo, menubar),
      GNOMEUIINFO_MENU_REDO_ITEM (on_edit_redo, menubar),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_CUT_ITEM (on_edit_cut, menubar),
      GNOMEUIINFO_MENU_COPY_ITEM (on_edit_copy, menubar),
      GNOMEUIINFO_MENU_PASTE_ITEM (on_edit_paste, menubar),
      GNOMEUIINFO_MENU_SELECT_ALL_ITEM (on_nop, menubar),
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("_Delete"), N_("Delete selected transaction(s)"),
	on_edit_delete, menubar, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, N_("T_oggle Cleared Status"), N_("Toggle cleared status of selected transaction(s)"),
	on_edit_toggle_status, menubar, NULL,
	GNOME_APP_PIXMAP_NONE, NULL, GDK_F5, 0, NULL },
      GNOMEUIINFO_END
    };
    enum edit_indices
    {
      edit_undo_idx,
      edit_redo_idx,
      edit_cut_idx = 3,
      edit_copy_idx = 4,
      edit_paste_idx = 5,
      edit_select_all_idx = 6,
      edit_delete_idx = 8,
      edit_toggle_status_idx,
    };
    const struct
    {
      GtkWidget **ptr;
      gint index;
    }
    menus[] =
    {
      { &menubar->popup_undo,          edit_undo_idx          },
      { &menubar->popup_redo,          edit_redo_idx          },
      { &menubar->popup_cut,           edit_cut_idx           },
      { &menubar->popup_copy,          edit_copy_idx          },
      { &menubar->popup_paste,         edit_paste_idx         },
      { &menubar->popup_select_all,    edit_select_all_idx    },
      { &menubar->popup_delete,        edit_delete_idx        },
      { &menubar->popup_toggle_status, edit_toggle_status_idx },
    };
    GtkWidget *menu;
    gint i;

    menu = gtk_menu_new ();

    gnome_app_fill_menu (GTK_MENU_SHELL (menu), edit_menu, NULL, TRUE, 0);

    /* Remember some of the menu items, so we can enable/disable them later */
    for (i=0; i<sizeof_array(menus); ++i)
      *menus[i].ptr = edit_menu[menus[i].index].widget;

    menubar->popup_menu = menu;
    register_paste_hook (menubar->popup_paste, menubar);
  }
  return menubar->popup_menu;
}
