/******************************************************************************\
 gnofin/ui.c   $Revision: 1.35.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <string.h>
#include <gtk/gtkmain.h>
#include <gnome.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <libgnomeui/gnome-app.h>
#include <libgnome/gnome-config.h>
#include "ui.h"
#include "ui-record-list.h"
#include "ui-record-editor.h"
#include "ui-menubar.h"
#include "ui-toolbar.h"
#include "ui-statusbar.h"
#include "data-if.h"
#include "dialogs.h"
#include "file-utils.h"
#include "file-mru.h"
#include "xml-io.h"
#include "fin-io.h"
#include "config-saver.h"
#include "gnofin-defaults.h"
#include "merging.h"

#ifdef ENABLE_UI_RECORD_SELECT
#include "ui-record-select.h"
#endif

static void on_bankbook_balance_changed (const Bankbook *, const Account *, UI *);


/******************************************************************************
 * Configuration
 */

typedef struct {
  guint width;
  guint height;
} Config;

#define CAT  "UI_AppWindow"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  trace ("");
  config->width = gnome_config_get_int (KEY "width=741");
  config->height = gnome_config_get_int (KEY "height=482");
}

static void
save_config (const Config *config)
{
  trace ("");
  gnome_config_set_int (KEY "width", config->width);
  gnome_config_set_int (KEY "height", config->height);
}

static Config *
get_config (void)
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
    init = TRUE;
  }
  return &config;
}


/******************************************************************************
 * Instances
 */

static GSList *ui_instances = NULL;

static inline void
ui_instance_register (UI *ui)
{
  ui_instances = g_slist_prepend (ui_instances, ui);
}

static inline void
ui_instance_unregister (UI *ui)
{
  ui_instances = g_slist_remove (ui_instances, ui);

  if (ui_instances == NULL)
  {
    config_save_all ();
    gtk_main_quit ();
  }
}


/******************************************************************************
 * Helper functions
 */

#define CURRENT_ACCOUNT(ui) ((Account *) \
				((ui)->current_list ? \
				 (ui)->current_list->account : NULL))
#define CURRENT_RECORD(ui)  ((Record *) \
				((ui)->current_list ? \
				 (ui)->current_list->selected_record : NULL))

static inline void
block_switch_page (UI *ui)
{
  trace ("");
  gtk_signal_handler_block (GTK_OBJECT (ui->notebook), ui->switch_page_id);
}

static inline void
unblock_switch_page (UI *ui)
{
  trace ("");
  gtk_signal_handler_unblock (GTK_OBJECT (ui->notebook), ui->switch_page_id);
}

static inline void
remove_notebook_page (UI *ui, guint position)
{
  trace ("");
  g_return_if_fail (ui);

  gtk_notebook_remove_page (ui->notebook, position);
}

static void
remove_all_notebook_pages (UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  while (gtk_notebook_get_current_page (ui->notebook) >= 0)
    gtk_notebook_remove_page (ui->notebook, 0);
}

static inline void
select_notebook_page (UI *ui, guint position)
{
  trace ("");
  g_return_if_fail (ui);

  gtk_notebook_set_page (ui->notebook, position);
}

static UI_RecordList *
get_record_list_by_account (UI *ui, const Account *account)
{
  GtkNotebookPage *page;
  UI_RecordList *list;
  GList *it;

  trace ("");
  g_return_val_if_fail (ui, NULL);
  g_return_val_if_fail (account, NULL);

  for (it = ui->notebook->children; it; it=it->next)
  {
    page = LIST_DEREF (GtkNotebookPage, it);
    list = UI_RECORD_LIST (page->child);

    if (list->account == account)
      return list;
  }
  return NULL;
}

static inline void
refresh_enable_record_ops (UI *ui)
{
  gboolean have_records;

  trace ("");
  g_return_if_fail (ui);

  have_records = (CURRENT_RECORD (ui) != NULL);
  ui_menubar_enable_record_ops (ui->menubar, have_records);
  ui_toolbar_enable_record_ops (ui->toolbar, have_records);
}

static void
refresh_title (UI *ui)
{
  gchar dirty_indicator = ' ';
  gchar *title;

  trace ("");
  g_return_if_fail (ui);

  if (ui->book && if_bankbook_is_dirty (ui->book))
    dirty_indicator = '*';
  
  if (ui->filename)
    title = g_strdup_printf ("%s - %s%c", APPNAME, ui->filename, dirty_indicator); 
  else
    title = g_strdup_printf ("%s%c", APPNAME, dirty_indicator);
  
  gtk_window_set_title (GTK_WINDOW (ui->app), title);
  g_free (title);
}

static void
merge_record_types (RecordTypeContext *context, Bankbook *book)
{
  trace ("");

  if (context->remove)
    if_record_type_delete (context->type);
  else if (context->type == NULL)
    if_bankbook_insert_record_type (book, &context->info);
  else if (context->mask)
    if_record_type_set_info (context->type, context->mask, &context->info);

  /* cleanup */
  if (context->mask)
    if_record_type_info_clear (&context->info, context->mask);
}


/******************************************************************************
 * GTK signals
 */

static gboolean
on_delete_event (GtkWidget *w, GdkEvent *event, UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, TRUE);

  return ! ui_exit (ui);
}

static void
on_configure_event (GtkWidget *w, GdkEventConfigure *event, UI *ui)
{
  Config *config = get_config ();

  config->width = event->width;
  config->height = event->height;
}

static gint
on_record_list_button_pressed (UI_RecordList *list, GdkEventButton *event, UI *ui)
{
  trace ("");

  if (event->type == GDK_BUTTON_PRESS)
  {
    if (event->button == 3)
    {
      GtkWidget *menu = ui_menubar_get_popup_edit_menu (ui->menubar);
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
      		      event->button, event->time);
      return TRUE;
    }
  }
  return FALSE;
}

static void
on_record_list_selection_changed (UI_RecordList *list, UI *ui)
{
  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (ui);

  /* Ignore record selection by background lists
   */
  if (list == ui->current_list)
  {
    ui_record_editor_set_record (ui->editor,
				 CURRENT_ACCOUNT (ui),
				 CURRENT_RECORD (ui));
    refresh_enable_record_ops (ui);
  }
}

static void
on_notebook_switch_page (GtkNotebook *notebook, GtkNotebookPage *page, guint page_num, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);
  g_return_if_fail (page);

  ui->current_list = UI_RECORD_LIST (page->child);

  /* Update record data shown in editor.  selected_record may be
   * NULL, implying that the editor should show previous settings
   * provided they are consistent with the selected account. */
  ui_record_editor_set_record (ui->editor,
			       CURRENT_ACCOUNT (ui),
			       CURRENT_RECORD (ui));

  on_bankbook_balance_changed (ui->book, CURRENT_ACCOUNT (ui), ui);
}

static void
on_notebook_button_press_event (GtkWidget *w, GdkEventButton *event, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  /* If the right button is choosen and the popup menu is created and an account exist */
  if (event->button == 3 && ui->notebook_menu && ui->notebook->children)
   { /* ... popup the menu */
     gtk_menu_popup (ui->notebook_menu, NULL, NULL, NULL, NULL, 3, event->time);
   }
}

static void
on_notebook_menus_reorder(GtkWidget *w, UI *ui)
{
  trace("");

  dialog_order_account (GTK_WINDOW (ui->app), ui);
}

/******************************************************************************
 * Initializers
 */

static void
create_app (UI *ui)
{
  Config *config = get_config ();
  GnomeApp *app;

  trace ("");

  app = GNOME_APP (gnome_app_new (PACKAGE, APPNAME));
  gtk_window_set_policy (GTK_WINDOW (app), 1, 1, 0);
  gtk_window_set_default_size (GTK_WINDOW (app), config->width, config->height);
  gtk_window_set_wmclass (GTK_WINDOW (app), PACKAGE, APPNAME);

  gtk_signal_connect (GTK_OBJECT (app), "delete_event",
  		      GTK_SIGNAL_FUNC (on_delete_event), ui);
  gtk_signal_connect (GTK_OBJECT (app), "configure_event",
  		      GTK_SIGNAL_FUNC (on_configure_event), ui);

  gtk_widget_show (GTK_WIDGET (app));  /* This has to be done */
  ui->app = app;
}

static void
create_notebook (UI *ui)
{
  GtkWidget *notebook;
  GtkWidget *w;

  trace ("");
  g_return_if_fail (ui);

  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_BOTTOM);
  gtk_notebook_set_scrollable (GTK_NOTEBOOK(notebook), TRUE);
  gtk_box_pack_start (ui->vbox, notebook, TRUE, TRUE, 0);

  ui->switch_page_id =
    gtk_signal_connect (GTK_OBJECT (notebook), "switch_page",
    			GTK_SIGNAL_FUNC (on_notebook_switch_page), ui);

  ui->notebook = GTK_NOTEBOOK (notebook);

  gtk_widget_show (notebook);

  /* Build the notebook menu when you click on the tab with the third button */
  ui->notebook_menu = (GtkMenu *)gtk_menu_new ();

  w = gnome_stock_menu_item (GNOME_STOCK_MENU_NEW, _("Insert a new account"));
  gtk_signal_connect_object (GTK_OBJECT (w), "activate",
      GTK_SIGNAL_FUNC(ui_insert_account),
      (gpointer)ui);
  gtk_widget_show ((GtkWidget *)w);
  gtk_menu_append (ui->notebook_menu, w);

  w = gnome_stock_menu_item (GNOME_STOCK_MENU_CLOSE, _("Delete this account"));
  gtk_signal_connect_object (GTK_OBJECT (w), "activate",
      GTK_SIGNAL_FUNC(ui_remove_current_account),
      (gpointer)ui);
  gtk_widget_show ((GtkWidget *)w);
  gtk_menu_append (ui->notebook_menu, w);

  w = gnome_stock_menu_item (GNOME_STOCK_MENU_PROP, _("Properties"));
  gtk_signal_connect_object (GTK_OBJECT (w), "activate",
      GTK_SIGNAL_FUNC(ui_modify_current_account),
      (gpointer)ui);
  gtk_widget_show ((GtkWidget *)w);
  gtk_menu_append (ui->notebook_menu, w);

  w = gnome_stock_menu_item (GNOME_STOCK_MENU_REFRESH,_("Reorder Accounts"));
  gtk_signal_connect (GTK_OBJECT (w), "activate",
      GTK_SIGNAL_FUNC(on_notebook_menus_reorder),
      (gpointer)ui);
  gtk_widget_show ((GtkWidget *)w);
  gtk_menu_append (ui->notebook_menu, w);

  gtk_signal_connect (GTK_OBJECT(notebook), "button-press-event",
      GTK_SIGNAL_FUNC(on_notebook_button_press_event),ui);
}

static void
insert_notebook_page (UI *ui, const Account *account)
{
  guint position;
  const gchar *name;
  GtkWidget *list;

  trace ("");
  g_return_if_fail (ui);
  g_return_if_fail (account);

  name = if_account_get_name (account);
  position = if_account_get_index (account);

  list = ui_record_list_new ();
  gtk_signal_connect (GTK_OBJECT (list), "event",
  		      GTK_SIGNAL_FUNC (on_record_list_button_pressed), ui);
  gtk_signal_connect (GTK_OBJECT (list), "selection_changed",
  		      GTK_SIGNAL_FUNC (on_record_list_selection_changed), ui);
  ui_record_list_set_account (UI_RECORD_LIST (list), (Account *) account);
  gtk_widget_show (list);

  gtk_notebook_insert_page (ui->notebook, list,
			    gtk_label_new (name),
			    position);
  
  ui_record_list_refresh (UI_RECORD_LIST (list), 0);
  ui_record_list_select_record (UI_RECORD_LIST (list), -1);
}


/******************************************************************************
 * Bankbook signals
 */

static void
on_bankbook_record_deleted (const Bankbook *book, const Account *account,
			    guint index, UI *ui)
{
  UI_RecordList *list;

  trace ("");
  g_return_if_fail (ui);

  list = get_record_list_by_account (ui, account);
  ui_record_list_remove_record (list, index);

  refresh_enable_record_ops (ui);
}

static void
on_bankbook_record_inserted (const Bankbook *book, const Account *account,
			     guint index, UI *ui)
{
  UI_RecordList *list;

  trace ("");
  g_return_if_fail (ui);

  list = get_record_list_by_account (ui, account);
  ui_record_list_insert_record (list, index, TRUE);

  refresh_enable_record_ops (ui);
}

static void
on_bankbook_account_deleted (const Bankbook *book, guint index, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  remove_notebook_page (ui, index);

  /* Unfortunately, if all pages are removed, we don't get a switch_page
   * signal from the notebook, so we have to check this condition manually. */
  if (ui->notebook->children == NULL)
  {
    ui->current_list = NULL;
    ui_record_editor_set_record (ui->editor, NULL, NULL);
  }
  ui_menubar_enable_account_ops (ui->menubar, ui->current_list != NULL);

  /* We have to refresh the type list to take into account the
   * fact that linked types only appear when there are at least
   * two accounts defined. */
  ui_record_editor_refresh_type_list (ui->editor);
}

static void
on_bankbook_account_inserted (const Bankbook *book, guint index, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  insert_notebook_page (ui, if_bankbook_get_account_by_index (book, index));
  ui_menubar_enable_account_ops (ui->menubar, TRUE);

  /* We have to refresh the type list to take into account the
   * fact that linked types only appear when there are at least
   * two accounts defined. */
  ui_record_editor_refresh_type_list (ui->editor);

  /* If the account has no records, then on_record_list_selection_changed
   * will not be called, so we must take care of configuring the editor. */
  if (CURRENT_RECORD (ui) == NULL)
    ui_record_editor_set_record (ui->editor, CURRENT_ACCOUNT (ui), NULL);
}

static void
on_bankbook_account_info_changed (const Bankbook *book, const Account *account,
				  guint mask, guint last_index, UI *ui)
{
  GtkWidget *list;
  guint index;

  trace ("");
  g_return_if_fail (ui);

  /* We can ignore changes to most fields */
  if (!((mask & ACCOUNT_FIELD_NAME) || (mask & ACCOUNT_FIELD_FOREIGN)))
    return;

  /* Update the name */
  list = GTK_WIDGET (get_record_list_by_account (ui, account));
  gtk_notebook_set_tab_label_text (ui->notebook, list,
				   if_account_get_name (account));

  index = if_account_get_index (account);
  if (index != last_index)
  {
    /* Reposition the notebook page */
    gtk_notebook_reorder_child (ui->notebook, list, index);
  }

  /* We must refresh the record editor */
  ui_record_editor_set_record (ui->editor, CURRENT_ACCOUNT (ui), CURRENT_RECORD (ui));
}

static void
on_bankbook_record_info_changed (const Bankbook *book, const Record *record,
				 guint mask, guint last_index, UI *ui)
{
  UI_RecordList *list;

  trace ("");
  g_return_if_fail (ui);

  list = get_record_list_by_account (ui, if_record_get_parent (record));
  ui_record_list_refresh_record (list, record, last_index, mask);
  
  refresh_enable_record_ops (ui);
}

static void
on_bankbook_record_list_invalidated (const Bankbook *book,
				     const Account *account, UI *ui)
{
  UI_RecordList *list;

  trace ("");
  g_return_if_fail (ui);

  list = get_record_list_by_account (ui, account);
  ui_record_list_refresh (list, 0);
}

static void
on_bankbook_type_list_invalidated (const Bankbook *book, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  ui_record_editor_refresh_type_list (ui->editor);
}

static void
on_bankbook_category_list_invalidated (const Bankbook *book, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  ui_record_editor_refresh_category_list (ui->editor);
}

static void
on_bankbook_payee_list_invalidated (const Bankbook *book, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  ui_record_editor_refresh_payee_list (ui->editor);
}

static void
on_bankbook_memo_list_invalidated (const Bankbook *book, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  ui_record_editor_refresh_memo_list (ui->editor);
}

static void
on_bankbook_linked_acc_list_invalidated (const Bankbook *book, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  ui_record_editor_refresh_linked_acc_list (ui->editor);
}

static void
on_bankbook_balance_changed (const Bankbook *book, const Account *account, UI *ui)
{
  trace ("");
  g_return_if_fail (ui);

  if (account == NULL)
  {
    ui_statusbar_set_cleared_bal (ui->statusbar, (money_t) 0);
    ui_statusbar_set_overall_bal (ui->statusbar, (money_t) 0);
  }
  else if (account == CURRENT_ACCOUNT (ui))
  {
    AccountInfo acc = {0};

    if_account_get_info (account,
			 ACCOUNT_FIELD_CLEARED_BAL |
			 ACCOUNT_FIELD_OVERALL_BAL,
			 &acc);
    
    ui_statusbar_set_cleared_bal (ui->statusbar, acc.cleared_bal);
    ui_statusbar_set_overall_bal (ui->statusbar, acc.overall_bal);
  }
}

static void
on_bankbook_history_changed (const Bankbook *book, UI *ui)
{
  gboolean can_undo, can_redo;

  trace ("");
  g_return_if_fail (ui);

  can_undo = (ui->book) ? if_bankbook_can_undo (ui->book) : FALSE;
  ui_menubar_enable_undo (ui->menubar, can_undo);
  ui_toolbar_enable_undo (ui->toolbar, can_undo);

  can_redo = (ui->book) ? if_bankbook_can_redo (ui->book) : FALSE;
  ui_menubar_enable_redo (ui->menubar, can_redo);
  ui_toolbar_enable_redo (ui->toolbar, can_redo);
}

static void
on_bankbook_dirty_flag_changed (const Bankbook *book, UI *ui)
{
  gboolean is_dirty;

  trace ("");
  g_return_if_fail (ui);

  is_dirty = if_bankbook_is_dirty (book);
  trace ("%d", is_dirty);

  /* Refresh the main window title here */
  refresh_title (ui);

  ui_menubar_enable_save (ui->menubar, is_dirty);
  ui_toolbar_enable_save (ui->toolbar, is_dirty);
}

static void
ui_refresh_all (UI *ui)
{
  const GList *ac;
  gint count;

  trace ("");
  g_return_if_fail (ui);
  g_return_if_fail (ui->editor);

  on_bankbook_type_list_invalidated (ui->book, ui);
  on_bankbook_category_list_invalidated (ui->book, ui);
  on_bankbook_payee_list_invalidated (ui->book, ui);
  on_bankbook_memo_list_invalidated (ui->book, ui);
  on_bankbook_linked_acc_list_invalidated (ui->book, ui);

  /* Remove all notebook pages and then insert one for each account.
   * We block the switch page handler and then manually select the
   * 1st notebook page if it exists.
   */
  block_switch_page (ui);
  remove_all_notebook_pages (ui);
  unblock_switch_page (ui);

  ac = if_bankbook_get_accounts (ui->book);
  for (count=0; ac; ac=ac->next, count++)
    insert_notebook_page (ui, LIST_DEREF (Account, ac)); 

  if (count > 0)
    select_notebook_page (ui, 0);
  else
  {
    ui->current_list = NULL;
    ui_record_editor_set_record (ui->editor, NULL, NULL);
  }
  ui_menubar_enable_account_ops (ui->menubar, (count > 0));
  refresh_enable_record_ops (ui);

  on_bankbook_history_changed (ui->book, ui);
  on_bankbook_dirty_flag_changed (ui->book, ui);
  on_bankbook_balance_changed (ui->book, CURRENT_ACCOUNT (ui), ui);
}


/******************************************************************************
 * Interface
 */

UI *
ui_create (void)
{
  UI *ui;

  trace ("");

  ui = g_new0 (UI, 1);
  ui_instance_register (ui);

  create_app (ui);
  ui->statusbar = ui_statusbar_create (ui);
  ui->menubar = ui_menubar_create (ui);
  ui->toolbar = ui_toolbar_create (ui);

  /* This vbox holds the notebook and editor */
  ui->vbox = GTK_BOX (gtk_vbox_new (FALSE, 0));
  gnome_app_set_contents (ui->app, GTK_WIDGET (ui->vbox));

  create_notebook (ui);

  ui->editor = UI_RECORD_EDITOR (ui_record_editor_new ());
  gtk_box_pack_start (ui->vbox, GTK_WIDGET (ui->editor), FALSE, FALSE, 0);

  gtk_widget_show (GTK_WIDGET (ui->editor));
  gtk_widget_show (GTK_WIDGET (ui->app));
  return ui;
}

gboolean
ui_exit (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, FALSE);

  if (!ui_close (ui))
    return FALSE;

  gtk_widget_destroy (GTK_WIDGET (ui->app));

  ui_instance_unregister (ui);
  return TRUE;
}

gboolean
ui_save_file (UI *ui, const gchar *filename)
{
  gchar *fname = NULL;

  trace ("file=%s", filename);
  g_return_val_if_fail (ui, FALSE);

  if (filename == NULL)
  {
    gchar *start_dir = NULL;

    if (ui->filename)
      start_dir = file_get_directory (ui->filename);

    fname = dialog_query_filename (GTK_WINDOW (ui->app),
    				 _("Save"), start_dir, TRUE, NULL);
    g_free (start_dir);
    if (fname == NULL)
      return FALSE;

    filename = fname;
  }

  if (file_exists (filename))
    file_backup (filename);

  if (xml_io_save (GTK_WINDOW (ui->app), filename, ui->book))
  {
    if (filename != ui->filename)
    {
      g_free (ui->filename);
      ui->filename = g_strdup (filename);
    }

    if_bankbook_reset_dirty (ui->book);
    file_mru_add (filename);
  }

  if (fname)
    g_free (fname);

  return TRUE;
}

gboolean
ui_close (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, FALSE);

  if (ui->book && if_bankbook_is_dirty (ui->book))
  {
    gint result;

    result = dialog_question (GTK_WINDOW (ui->app),
    			    _("Save modifications before closing?"));
    switch (result)
    {
    case DIALOG_YES:
      if (!ui_save_file (ui, ui->filename))
        return FALSE;
      break;
    case DIALOG_NO:
      g_print (_("Discarding modifications per user request...\n"));
      break;
    case -1:
    case DIALOG_CANCEL:
      return FALSE;
    }
  }

  block_switch_page (ui);
  remove_all_notebook_pages (ui);
  unblock_switch_page (ui);

  ui_record_editor_set_record (ui->editor, NULL, NULL);
  ui_record_editor_set_bankbook (ui->editor, NULL);

  if (ui->book)
  {
    if_bankbook_destroy (ui->book);
    ui->book = NULL;
  }
  return TRUE;
}

gboolean
ui_load_file_1 (UI *ui, gchar **fname, Bankbook **book)
{
  gboolean result = FALSE;
  static gchar *last_fname = NULL;

  trace ("");
  g_return_val_if_fail (ui, FALSE);
  g_return_val_if_fail (fname, FALSE);
  g_return_val_if_fail (book, FALSE);

  if (*fname == NULL)
  {
    *fname = dialog_query_filename (ui_get_window (ui),
    				  _("Open"), last_fname, FALSE, NULL);
    if (*fname == NULL)
      return FALSE;
  }

  if (!file_exists (*fname))
  {
    dialog_error (ui_get_window (ui),
    		_("Unable to load file: %s\n[File not found or permission denied]"), *fname);
    return FALSE;
  }

  *book = if_bankbook_new ();

  /* First try to load the file in the old format.  We do this _only_ because of
   * the junk libxml will spit out if the file is not XML.  We don't want the 
   * user of an old gnofin file seeing this, b/c it won't make any sense.
   */
  if (fin_io_probe (*fname))
    result = fin_io_load (ui_get_window (ui), *fname, *book);
  else if (xml_io_probe (*fname))
    result = xml_io_load (ui_get_window (ui), *fname, *book);
  else
    dialog_error (ui_get_window (ui),
    		_("Unable to load file: %s\n[Unknown file format]"), *fname);

  if (result)
  {
    GList *cats = if_bankbook_get_category_strings (*book);
    if (!cats || (cats->next == NULL))
      install_default_categories (*book);
    g_list_free (cats);

    g_free (last_fname);
    last_fname = g_strdup (*fname);
  }
  else
  {
    if_bankbook_destroy (*book);
    g_free (*fname);

    *book = NULL;
    *fname = NULL;
  }
  return result;
}

gboolean
ui_load_file (UI *ui, const gchar *filename)
{
  Bankbook *book = NULL;
  gchar *fname = NULL;
  gboolean result;

  trace ("file=%s", filename);
  g_return_val_if_fail (ui, FALSE);

  /* Discard existing data file if open */
  if (!ui_close (ui))
    return FALSE;

  if (filename)
    fname = g_strdup (filename);

  if (!ui_load_file_1 (ui, &fname, &book))
    return FALSE;

  if (!(result = ui_load_bankbook (ui, book, fname)))
    if_bankbook_destroy (book);

  g_free (fname);
  return result;
}

gboolean
ui_load_bankbook (UI *ui, Bankbook *book, const gchar *filename)
{
  trace ("book=%p, file=%s", book, filename);
  g_return_val_if_fail (ui, FALSE);

  /* Discard existing data file if open */
  if (!ui_close (ui))
    return FALSE;

  if (book == NULL)
  {
    book = if_bankbook_new ();
    install_default_record_types (book);
    install_default_categories (book);
  }

  /* Update stored filename */
  g_free (ui->filename);
  if (filename)
    ui->filename = g_strdup (filename);
  else
    ui->filename = NULL;

  /* Set book */
  ui->book = book;
  ui_record_editor_set_bankbook (ui->editor, book);

  ui_refresh_all (ui);

  /* Listen for various bankbook signals
   */
  gtk_signal_connect (GTK_OBJECT (book), "account_inserted",
  		      GTK_SIGNAL_FUNC (on_bankbook_account_inserted), ui);
  gtk_signal_connect (GTK_OBJECT (book), "account_deleted",
  		      GTK_SIGNAL_FUNC (on_bankbook_account_deleted), ui);
  gtk_signal_connect (GTK_OBJECT (book), "record_inserted",
  		      GTK_SIGNAL_FUNC (on_bankbook_record_inserted), ui);
  gtk_signal_connect (GTK_OBJECT (book), "record_deleted",
  		      GTK_SIGNAL_FUNC (on_bankbook_record_deleted), ui);
  gtk_signal_connect (GTK_OBJECT (book), "account_info_changed",
  		      GTK_SIGNAL_FUNC (on_bankbook_account_info_changed), ui);
  gtk_signal_connect (GTK_OBJECT (book), "record_info_changed",
  		      GTK_SIGNAL_FUNC (on_bankbook_record_info_changed), ui);
  gtk_signal_connect (GTK_OBJECT (book), "record_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_record_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "type_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_type_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "category_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_category_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "payee_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_payee_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "memo_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_memo_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "linked_account_list_invalidated",
  		      GTK_SIGNAL_FUNC (on_bankbook_linked_acc_list_invalidated), ui);
  gtk_signal_connect (GTK_OBJECT (book), "balance_changed",
  		      GTK_SIGNAL_FUNC (on_bankbook_balance_changed), ui);
  gtk_signal_connect (GTK_OBJECT (book), "history_changed",
  		      GTK_SIGNAL_FUNC (on_bankbook_history_changed), ui);
  gtk_signal_connect (GTK_OBJECT (book), "dirty_flag_changed",
  		      GTK_SIGNAL_FUNC (on_bankbook_dirty_flag_changed), ui);

  if_bankbook_enable_history (book, TRUE);

  refresh_title (ui);

  if (filename)
    file_mru_add (filename);

  return TRUE;
}

gboolean
ui_merge_account (UI *ui, const Account *account, gboolean as_new_account)
{
  gboolean ret;

  trace ("");
  g_return_val_if_fail (ui, FALSE);
  g_return_val_if_fail (account, FALSE);

  if (as_new_account)
    ret = merge_account_into_bankbook (ui->book, account, TRUE) != NULL;
  else
    ret = merge_account_into_account (CURRENT_ACCOUNT (ui), account, TRUE);

  return ret;
}

gboolean
ui_is_empty (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, TRUE);

  return (ui->filename == NULL) && (ui->current_list == NULL);
}

void
ui_remove_current_record (UI *ui)
{
  trace ("");
  g_return_if_fail (ui);
  g_return_if_fail (ui->current_list);

  if_record_delete (CURRENT_RECORD (ui));
}

void
ui_select_record (UI *ui, Record *record)
{
  Account *account;
  UI_RecordList *list;
  guint index;

  trace ("");
  g_return_if_fail (ui);
  g_return_if_fail (record);

  account = if_record_get_parent (record);
  index = if_record_get_index (record);
  list = get_record_list_by_account (ui, account);

  g_assert (list);

  ui_record_list_select_record (list, index);
}

void
ui_toggle_current_record_status (UI *ui)
{
  Record *record;

  trace ("");
  g_return_if_fail (ui);

  record = CURRENT_RECORD (ui);
  if (record)
  {
    RecordInfo rec;

    if_record_get_info (record, RECORD_FIELD_CLEARED, &rec);
    rec.cleared = !rec.cleared;
    if_record_set_info (record, RECORD_FIELD_CLEARED, &rec);
  }
}

void
ui_remove_current_account (UI *ui)
{
  Account *account;

  trace ("");
  g_return_if_fail (ui);

  account = CURRENT_ACCOUNT (ui);
  if (account)
  {
    switch (dialog_question_yes_no (GTK_WINDOW (ui->app),
    				  _("You are about to delete the current account?\n"
				    "Are you sure you wish to continue?")))
    {
    case DIALOG_YES:
      if_account_delete (account);
      break;
    case DIALOG_NO:
    }
  }
}

void
ui_insert_account (UI *ui)
{
  AccountInfo info = {0};
  Account *account = NULL;

  trace ("");
  g_return_if_fail (ui);

  while (account == NULL)
  {
    if (dialog_account_info (GTK_WINDOW (ui->app), &info))
    {
      account = if_bankbook_insert_account (ui->book, &info);

      if (account == NULL)
	dialog_error (GTK_WINDOW (ui->app),
		    _("Unable to create account.\n[%s]"),
		      if_bankbook_get_error (ui->book));
    }
    else
      break;
  }
  if_account_info_clear (&info, 0);
}

void
ui_modify_current_account (UI *ui)
{
  Account *acct;
  AccountInfo info;
  gboolean success = FALSE;
  guint mask;

  trace ("");
  g_return_if_fail (ui);

  acct = CURRENT_ACCOUNT (ui);
  g_return_if_fail (acct);

  if_account_get_info (acct, 0, &info);

  /* It is important to copy the info b/c get_info returns
   * references to many of the account fields. */
  if_account_info_copy (&info, &info, 0);

  while (!success)
  {
    if ((mask = dialog_account_info (GTK_WINDOW (ui->app), &info)) != 0)
    {
      success = if_account_set_info (acct, mask, &info);

      if (!success)
        dialog_error (GTK_WINDOW (ui->app),
		    _("Unable to modify account info.\n[%s]"),
		      if_bankbook_get_error (ui->book));
    }
    else
      break;
  }
  if_account_info_clear (&info, 0);
}

void
ui_modify_current_record_types (UI *ui)
{
  RecordTypeContext *c;
  RecordType *type;
  GSList *cts = NULL;
  const GList *node;

  trace ("");

  for (node=if_bankbook_get_record_types (ui->book); node; node=node->next)
  {
    type = LIST_DEREF (RecordType, node);

    c = g_new0 (RecordTypeContext, 1);
    c->type = type;
    if_record_type_get_info (type, 0, &c->info);
    
    cts = g_slist_prepend (cts, c);
  }
  
  if (cts)
  {
    cts = g_slist_reverse (cts);

    if (dialog_edit_record_types (GTK_WINDOW (ui->app), &cts))
      g_slist_foreach (cts, (GFunc) merge_record_types, ui->book);

    g_slist_foreach (cts, (GFunc) g_free, NULL);
    g_slist_free (cts);
  }
}

#ifdef ENABLE_UI_RECORD_SELECT
void
ui_select_records_current_account (UI *ui)
{
  Account *account;

  trace ("");
  g_return_if_fail (ui);

  account = CURRENT_ACCOUNT (ui);
  g_return_if_fail (account);

  ui_select_records_account(ui,account);
}
#endif

Account *
ui_get_current_account (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, NULL);

  return CURRENT_ACCOUNT (ui);
}

Record *
ui_get_current_record (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, NULL);

  return CURRENT_RECORD (ui);
}

GnomeAppBar *
ui_get_appbar (UI *ui)
{
  trace ("");
  g_return_val_if_fail (ui, NULL);
  g_return_val_if_fail (ui->statusbar, NULL);

  return GNOME_APPBAR (ui->statusbar->appbar);
}

// vim: ts=8 sw=2
