/* util.c
   General utility functions, eg for string formatting
   Copyright (C) 2001 Linus Walleij

This file is part of the GNOMAD package.

GNOMAD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

You should have received a copy of the GNU General Public License
along with GNOMAD; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. 

*/

#include "common.h"
#include <libgnomeui/libgnomeui.h>

/* Find the length of a string vector */
gint vectorlength(gchar **vector)
{
  gchar **tmp = vector;
  gint size = 0;

  if (!tmp)
    return 0;
  
  while (*tmp)
    {
      size ++;
      tmp ++;
    }
  return size;
}

/* Concatenate two dynamic strings */
gchar *stringcat(gchar *org, gchar *add)
{
  /* This can be improved under GTK+ 2.0 
   * new GLib functions -> 2.0 */
  gchar *tmpstring;

  if (!add)
    return org;
  if (!org)
    return g_strdup(add);
  tmpstring = (gchar *) g_malloc(strlen(org) + strlen(add) + 1);
  tmpstring = strcpy(tmpstring, org);
  tmpstring = strcat(tmpstring, add);
  g_free(org);
  return tmpstring;
}

void replacechar(gchar *string, gchar find, gchar replace)
{
  /* Replaces character find with character replace
   * in string */
  gchar *tmp = string;

  if (!string || !find || !replace)
    return;

  while (*tmp)
    {
      if (*tmp == find)
	*tmp = replace;
      tmp ++;
    }
}

gchar *replacestring(gchar *string, gchar *find, gchar *replace)
{
  /* Replaces the substring find with substring replace
   * in string */
  static gchar stringbuffer[512];
  gchar *returnstring;
  gchar *tmp = string;
  gchar **tmp2;
  gchar **tmp3;
  gint i;

  if (!string || !find || !replace)
    return string;
  
  tmp2 = g_strsplit(string, find, 0);

  /* This is the case when the replacement is in the
   * last characters of the string */
  if (vectorlength(tmp2) == 1 &&
      strlen(string) > strlen(*tmp2)) {
    stringbuffer[0] = '\0';
    strcat(stringbuffer, *tmp2);
    strcat(stringbuffer, replace);
    g_free(string);
    returnstring = g_strdup(stringbuffer);
  }
  /* If there were no matches, return original string */
  else if (vectorlength(tmp2) > 1) {
    stringbuffer[0] = '\0';
    tmp3 = tmp2;
    while (*tmp3)
      {
	strcat(stringbuffer, *tmp3);
	/* If there are more strings in the list, insert the replacement */
	if (*(tmp3+1))
	  strcat(stringbuffer, replace);
	tmp3 ++;
      }
    /* Free old string */
    g_free(string);
    returnstring = g_strdup(stringbuffer);
  } else {
    returnstring = string;
  }
  g_strfreev(tmp2);
  return returnstring;
}

/* Converts a string into guint representation (if possible) */
guint string_to_guint(gchar *string)
{
  gchar *dummy;

  if (!string)
    return 0;
  
  return (guint) strtoul(string, &dummy, 10);
}

/* Converts a figure representing a number of seconds to
 * a string in mm:ss notation */
gchar *seconds_to_mmss(guint seconds)
{
  gchar tmp2[4];
  gchar tmp[10];

  if (!seconds)
    return g_strdup("0:00");
  sprintf(tmp2, "0%u", seconds%60);
  while (strlen(tmp2)>2) {
    tmp2[0]=tmp2[1];
    tmp2[1]=tmp2[2];
    tmp2[2]='\0';
  }
  sprintf(tmp, "%lu:%s", seconds/60, tmp2);
  return g_strdup(tmp);
}

/* Converts a string in mm:ss notation to a figure
 * representing seconds */
guint mmss_to_seconds(gchar *mmss)
{
  gchar **tmp;
  guint seconds = 0;

  if (!mmss)
    return seconds;

  tmp = g_strsplit(mmss, ":", 0);
  if (vectorlength(tmp) == 2) {
    seconds = 60 * string_to_guint(tmp[0]);
    seconds += string_to_guint(tmp[1]);
  }
  if (tmp != NULL)
    g_strfreev(tmp);
  return seconds;
}

/* are there only numbers in this string? */
gboolean is_a_number(gchar *string)
{
  gchar *tmp;
  
  if (string == NULL)
    return FALSE;
  tmp = string;
  while (*tmp)
    {
      /* See if it is not a number, skip spaces */
      if ((*tmp < '0' || 
	   *tmp > '9') && 
	  *tmp != ' ' &&
	  *tmp != '.')
	return FALSE;
      tmp ++;
    }
  return TRUE;
}

/* Good for mocking up correctly adjusted dialogs */
void add_empty_hbox (GtkWidget *tobox)
{
  GtkWidget *thing = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (thing);
  gtk_box_pack_start (GTK_BOX (tobox), thing, TRUE, FALSE, 0);
}

/* Create an error dialog */
void create_error_dialog(gchar *errorstring)
{
  GtkWidget *dialog;
  GtkWidget *label, *button;

  dialog = gnome_message_box_new (errorstring,
				  GNOME_MESSAGE_BOX_ERROR,
				  GNOME_STOCK_BUTTON_OK,
				  NULL);
  gtk_widget_show(dialog);
}

void hexdump(unsigned char *data, guint len)
{
  guint i;

  for (i=0; i<len; i++) {
    if (i%16 == 0 && i != 0)
      g_print("\n");
    g_print("%02x ", (guchar) data[i]);
  }
  g_print("\n");
}

