/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

using namespace std;

#include "search.h"
#include "index.h"

#include <regex.h>
#include <algorithm>


SearchPattern::HitCompare SearchPattern::hit_compare;

void 
SearchPattern::search(GAptCacheFile* cachefile,
                      priority_queue<Hit,vector<Hit>,HitCompare> & hits)
{
  g_return_if_fail(cachefile != 0);

  if (regexps_.size() == 0) 
    return; // no matches for that!

  GAptCache* cache = cachefile->cache();

  g_return_if_fail(cache != 0);

  vector<regex_t*> res;

  vector<string>::const_iterator j = regexps_.begin();
  while (j != regexps_.end()) {
    int flags = 0x0;
    flags |= REG_EXTENDED | REG_NOSUB;
    if (case_sensitive_ == false) 
      {
        flags |= REG_ICASE;
      }

    regex_t* re = new regex_t;

    int err = regcomp(re, (*j).c_str(), flags);

    if (err != 0) {
      gchar buf[1024];
      regerror(err, re, buf, 1023);
      g_warning(buf);
      delete re;
      ++j;
      continue;
    }
    
    res.push_back(re);
    
    ++j;
  }
  
  if (res.empty()) return;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end()) {
    guint weight = 0;

    pkgRecords::Parser* parser = Util::pkg_parser(i, cache);

    if (parser == 0)
      {
        ++i;
        continue;
      }

    // Check each regexp in each field

    int q = 0;
    while (q < TextFieldTypeEnd) {
      TextFieldType tft = static_cast<TextFieldType>(q);

      if (!get_textfield(tft))
        {
          ++q;
          continue;
        }

      const gchar* field = 0;
      guint possible_weight = 0;

      switch (tft) {
      case Package:
        possible_weight = 5;     // name counts lots
        field = i.Name();
        break;
        
      case Maintainer:
        possible_weight = 5;     // ditto - not likely to get a match unless you meant it
        field = parser->Maintainer().c_str();
        break;
        
      case Conffiles:
        
        break;
        
      case ShortDescription:
        possible_weight = 3;
        field = parser->ShortDesc().c_str();
        break;
        
      case LongDescription:
        possible_weight = 2;
        field = parser->LongDesc().c_str();
        break;
        
      default:
        break;
      }

      if (field) {
        vector<regex_t*>::iterator r = res.begin();      
        while (r != res.end()) {
          int match = regexec(*r, field, 0, 0, 0);

          if (match == 0) {
            // match! 
            weight += possible_weight;
          }
          else if (match == REG_NOMATCH) {
            ;
          }
          else {
            gchar buf[1024];
            regerror(match, *r, buf, 1023);
            g_warning(buf);
          }
          ++r;
        }        
      }

      ++q;
    }
 
    if (weight > 0) {
      hits.push(Hit(const_cast<pkgCache::Package*>(&*i), weight));
    }

    ++i;
  }

  vector<regex_t*>::iterator x = res.begin();
  while (x != res.end()) {
    regfree(*x);
    delete *x;
    ++x;
  }
}

class Search {
  SearchPattern pattern_;

  GAptCacheFile* cf_;
  GAptPkgTree*   tree_;

  GtkWidget* dialog_;

  GtkWidget* entry_;

  GtkWidget* case_sensitive_toggle_;
  GtkWidget* package_toggle_;
  GtkWidget* maintainer_toggle_;
  GtkWidget* conffiles_toggle_;
  GtkWidget* short_toggle_;
  GtkWidget* long_toggle_;
  
  void results(priority_queue<SearchPattern::Hit,
               vector<SearchPattern::Hit>,
               SearchPattern::HitCompare> hits);

public:
  Search(GAptCacheFile* cf, GAptPkgTree* tree);
  ~Search();

  void run();
};

Search::Search(GAptCacheFile* cf, 
               GAptPkgTree* tree)
  : cf_(cf), tree_(tree), dialog_(0), entry_(0)
{
  dialog_ = gnome_dialog_new(_("Package Search"),
                             GNOME_STOCK_BUTTON_OK,
                             GNOME_STOCK_BUTTON_CANCEL,
                             NULL);
  
  gnome_apt_setup_dialog(dialog_);
  gnome_dialog_close_hides(GNOME_DIALOG(dialog_), TRUE);
  gnome_dialog_set_close(GNOME_DIALOG(dialog_), TRUE);
  gnome_dialog_set_default(GNOME_DIALOG(dialog_), GNOME_OK);
  gtk_window_set_modal(GTK_WINDOW(dialog_), TRUE);

  entry_ = gtk_entry_new();

  gnome_dialog_editable_enters(GNOME_DIALOG(dialog_), 
                               GTK_EDITABLE(entry_));  

  GtkWidget* frame = gtk_frame_new(_("Text search"));
  
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     frame,
                     TRUE, TRUE, GNOME_PAD_SMALL);

  GtkWidget* vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  GtkWidget* hbox   = gnome_apt_labelled_widget(_("Regular expression:"),
                                                entry_);

  gtk_box_pack_start(GTK_BOX(vbox),
                     hbox,
                     TRUE, TRUE, 0);

  case_sensitive_toggle_ = gtk_check_button_new_with_label(_("Case sensitive"));
  gtk_box_pack_start(GTK_BOX(vbox),
                     case_sensitive_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(case_sensitive_toggle_),
                               pattern_.get_case_sensitive());

  frame = gtk_frame_new(_("Fields to Search"));

  gtk_box_pack_start(GTK_BOX(vbox),
                     frame,
                     TRUE, TRUE, 0);
  
  gtk_container_set_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);

  vbox = gtk_vbox_new(FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);

  package_toggle_ = gtk_check_button_new_with_label(_("Package Name"));
  gtk_box_pack_start(GTK_BOX(vbox),
                     package_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(package_toggle_),
                               pattern_.get_textfield(SearchPattern::Package));

  maintainer_toggle_ = gtk_check_button_new_with_label(_("Maintainer"));
  gtk_box_pack_start(GTK_BOX(vbox),
                     maintainer_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(maintainer_toggle_),
                               pattern_.get_textfield(SearchPattern::Maintainer));

  conffiles_toggle_ = gtk_check_button_new_with_label(_("Configuration Files"));
  gtk_box_pack_start(GTK_BOX(vbox),
                     conffiles_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(conffiles_toggle_),
                               pattern_.get_textfield(SearchPattern::Conffiles));

  short_toggle_ = gtk_check_button_new_with_label(_("Short Desc."));
  gtk_box_pack_start(GTK_BOX(vbox),
                     short_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(short_toggle_),
                               pattern_.get_textfield(SearchPattern::ShortDescription));

  long_toggle_ = gtk_check_button_new_with_label(_("Long Desc."));
  gtk_box_pack_start(GTK_BOX(vbox),
                     long_toggle_,
                     TRUE, TRUE, 0);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(long_toggle_),
                               pattern_.get_textfield(SearchPattern::LongDescription));

  
  // FIXME rest of the fields.

}

Search::~Search()
{  
  if (dialog_) 
    gtk_widget_destroy(dialog_);
}

void
Search::run()
{
  gtk_widget_show_all(dialog_);

  int reply = gnome_dialog_run(GNOME_DIALOG(dialog_));

  // Extract user's choices from dialog

  const char* s = gtk_entry_get_text(GTK_ENTRY(entry_));

  string regexp = s ? s : "";

  pattern_.set_case_sensitive(GTK_TOGGLE_BUTTON(case_sensitive_toggle_)->active);

  pattern_.set_textfield(SearchPattern::Package,
                         GTK_TOGGLE_BUTTON(package_toggle_)->active);

  pattern_.set_textfield(SearchPattern::Maintainer,
                         GTK_TOGGLE_BUTTON(maintainer_toggle_)->active);

  pattern_.set_textfield(SearchPattern::Conffiles,
                         GTK_TOGGLE_BUTTON(conffiles_toggle_)->active);

  pattern_.set_textfield(SearchPattern::ShortDescription,
                         GTK_TOGGLE_BUTTON(short_toggle_)->active);

  pattern_.set_textfield(SearchPattern::LongDescription,
                         GTK_TOGGLE_BUTTON(long_toggle_)->active);


  gtk_widget_destroy(dialog_);
  dialog_ = 0;
  entry_  = 0;
  
  if (reply == GNOME_OK && !regexp.empty()) {
    pattern_.add_regexp(regexp);

    priority_queue<SearchPattern::Hit,
      vector<SearchPattern::Hit>,
      SearchPattern::HitCompare> hits;

    pattern_.search(cf_, hits);

    results(hits);
  }

  return;
}

void 
Search::results(priority_queue<SearchPattern::Hit,
                vector<SearchPattern::Hit>,
                SearchPattern::HitCompare> hits)
{
  if (hits.empty())
    {
      GtkWidget* dialog = 
        gnome_ok_dialog(_("Search failed: no matching packages found."));
      gnome_apt_setup_dialog(dialog);
    }
  else 
    {
      // gets deleted by the Indexes
      Index* index = new Index(gnome_apt_indexes(), tree_, tree_->filter());
      
      vector<pkgCache::Package*> to_add;
      
      while (!hits.empty()) {
        SearchPattern::Hit h = hits.top();
        hits.pop();
        
        to_add.push_back(h.package);
      }  
      
      index->add(to_add);
    }
}

void      
gnome_apt_search(GAptPkgTree* tree)
{
  g_return_if_fail(tree != 0);

  Search s(gnome_apt_cache_file(), tree);
  
  s.run();
}
