/*
  GNOME Commander - A GNOME based file manager 
  Copyright (C) 2001-2002 Marcus Bjurman

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Much of this DnD code has been "borrowed" from the Midnight Commander source
 */

#include <config.h>
#include "gnome-cmd-includes.h"

#define STRINGS_TO_URIS_CHUNK 1024
#define ELEMENTS(x) (sizeof (x) / sizeof (x[0]))

/* DnD target names */
#define TARGET_MC_DESKTOP_ICON_TYPE     "application/x-mc-desktop-icon"
#define TARGET_URI_LIST_TYPE            "text/uri-list"
#define TARGET_TEXT_PLAIN_TYPE          "text/plain"
#define TARGET_URL_TYPE                 "_NETSCAPE_URL"

/* Standard DnD types */
typedef enum {
        TARGET_MC_DESKTOP_ICON,
        TARGET_URI_LIST,
        TARGET_TEXT_PLAIN,
        TARGET_URL,
        TARGET_NTARGETS
} TargetType;


static GtkTargetEntry drag_types [] = {
	{ TARGET_URI_LIST_TYPE, 0, TARGET_URI_LIST },
	{ TARGET_TEXT_PLAIN_TYPE, 0, TARGET_TEXT_PLAIN },
	{ TARGET_URL_TYPE, 0, TARGET_URL }
};

static GtkTargetEntry drop_types [] = {
	{ TARGET_URI_LIST_TYPE, 0, TARGET_URI_LIST },
	{ TARGET_URL_TYPE, 0, TARGET_URL }
};


static char *
build_selected_file_list (GnomeCmdFileSelector *fs,
						  int *file_list_len);

static void
drag_data_get (GtkWidget        *widget,
			   GdkDragContext   *context,
			   GtkSelectionData *selection_data,
			   guint            info,
			   guint32          time,
			   GnomeCmdFileSelector *fs);

static void
drag_begin (GtkWidget *widget,
			GdkDragContext *context,
			GnomeCmdFileSelector *fs);

static void
drag_end (GtkWidget *widget,
		  GdkDragContext *context,
		  GnomeCmdFileSelector *fs);

static gboolean
drag_motion (GtkWidget *widget,
			 GdkDragContext *context,
			 gint x, gint y,
			 guint time,
			 GnomeCmdFileSelector *fs);

static void
drag_data_received (GtkWidget          *widget,
					GdkDragContext     *context,
					gint                x,
					gint                y,
					GtkSelectionData   *selection_data,
					guint               info,
					guint32             time,
					GnomeCmdFileSelector *fs);


void
gnome_cmd_file_selector_init_dnd (GnomeCmdFileSelector *fs)
{
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->list != NULL);


	/* Set up drag source */

	gtk_drag_source_set (GTK_WIDGET (fs->list), GDK_BUTTON1_MASK,
						 drag_types, ELEMENTS (drag_types),
						 (GDK_ACTION_LINK | GDK_ACTION_MOVE | GDK_ACTION_COPY
						  | GDK_ACTION_ASK | GDK_ACTION_DEFAULT));

	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_begin",
						GTK_SIGNAL_FUNC (drag_begin), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_end",
						GTK_SIGNAL_FUNC (drag_end), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_data_get",
						GTK_SIGNAL_FUNC (drag_data_get), fs);

	/* Set up drag destination */

	gtk_drag_dest_set (GTK_WIDGET (fs->list),
					   GTK_DEST_DEFAULT_DROP,
					   drop_types, ELEMENTS (drop_types),
					   GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK | GDK_ACTION_ASK);

	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_motion",
						GTK_SIGNAL_FUNC (drag_motion), fs);
//	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_leave",
//						GTK_SIGNAL_FUNC (drag_leave), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "drag_data_received",
						GTK_SIGNAL_FUNC (drag_data_received), fs);
}



/**
 * panel_drag_data_get:
 *
 * Invoked when a drag operation has been performed, this routine
 * provides the data to be transfered
 */
static void
drag_data_get (GtkWidget        *widget,
			   GdkDragContext   *context,
			   GtkSelectionData *selection_data,
			   guint            info,
			   guint32          time,
			   GnomeCmdFileSelector *fs)
{
	int len;
	char *data;
	GList *files;

	data = build_selected_file_list (fs, &len);

	switch (info){
		case TARGET_URI_LIST:
		case TARGET_TEXT_PLAIN:
			gtk_selection_data_set (selection_data, selection_data->target, 8, data, len);
			break;

		case TARGET_URL:
			files = gnome_uri_list_extract_uris (data);
			if (files) {
				gtk_selection_data_set (selection_data,
										selection_data->target,
										8,
										files->data,
										strlen (files->data));
			}
			gnome_uri_list_free_strings (files);
			break;

		default:
			g_assert_not_reached ();
	}

	gnome_cmd_file_selector_unselect_all (fs);
	g_free (data);
}


/**
 * Transform a "\r\n" separated string into a GList with GnomeVFSURI's
 */
static GList *
strings_to_uris (gchar *data)
{
	int i;
	GList *uri_list = NULL;
	gchar **filenames;

	filenames = g_strsplit (data, "\r\n", STRINGS_TO_URIS_CHUNK);
	
	for ( i=0 ; filenames[i] != NULL ; i++ ) {
		gchar *fn;
		
		if (i == STRINGS_TO_URIS_CHUNK) {
			uri_list = g_list_concat (uri_list, strings_to_uris (filenames[i]));
			break;
		}

		fn = g_strdup (filenames[i]);
		uri_list = g_list_append (uri_list, gnome_vfs_uri_new (fn));
		g_free (fn);
	}

	g_strfreev (filenames);
	return uri_list;
}


/**
 * drag_data_received:
 *
 * Invoked on the target side of a Drag and Drop operation when data has been
 * dropped.
 */
static void
drag_data_received (GtkWidget          *widget,
					GdkDragContext     *context,
					gint                x,
					gint                y,
					GtkSelectionData   *selection_data,
					guint               info,
					guint32             time,
					GnomeCmdFileSelector *fs)
{
	GnomeCmdFile *finfo;
	GtkCList *clist;
	GnomeCmdDir *to, *cwd;
	GList *uri_list = NULL;
	gchar *to_fn = NULL;
	int row;
	GnomeVFSXferOptions xferOptions;

	clist = GTK_CLIST (widget);

	
	/* Find out what operation to perform
	 *
	 */
	if (context->action == GDK_ACTION_MOVE)
		xferOptions = GNOME_VFS_XFER_REMOVESOURCE;
	else if (context->action == GDK_ACTION_COPY)
		xferOptions = GNOME_VFS_XFER_RECURSIVE;
	else {
		debug_print ("Unknown context->action in drag_data_received\n");
		return;
	}


	/* Find the row that the file was dropped on
	 *
	 */
	row = gnome_cmd_file_selector_get_row (fs, x, y);

	/* Transform the drag data to a list with uris
	 *
	 */
	uri_list = strings_to_uris (selection_data->data);

	if (g_list_length (uri_list) == 1) {
		GnomeVFSURI *uri = (GnomeVFSURI*)uri_list->data;
		to_fn = gnome_vfs_unescape_string (gnome_vfs_uri_get_basename (uri), 0);
	}

	finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, row);
	cwd = gnome_cmd_file_selector_get_directory (fs);
	
	if (finfo
		&& finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY
		&& strcmp (finfo->info->name, "..") != 0 ) {
		/* The drop was over a directory in the list, which means that the 
		 * xfer should be done to that directory instead of the current one in the list 
		 */
		const gchar *to_dir_str = gnome_cmd_file_get_uri_str (finfo);
		to = dir_pool_get (to_dir_str);
	}
	else
		to = cwd;

	/* Start the xfer
	 * 
	 */
	gnome_cmd_xfer_uris_start (uri_list, to, to_fn, 
							   xferOptions,
							   GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE);
}


/**
 * drag_begin:
 *
 * Invoked when a drag is starting in the List view or the Icon view
 */
static void
drag_begin (GtkWidget *widget, GdkDragContext *context, GnomeCmdFileSelector *fs)
{
}

/**
 * drag_end:
 *
 * Invoked when a drag has finished in the List view or the Icon view
 */
static void
drag_end (GtkWidget *widget, GdkDragContext *context, GnomeCmdFileSelector *fs)
{
}


/* Callback used for drag motion events over the clist.  We set up
 * auto-scrolling and validate the drop to present the user with the correct
 * feedback.
 */
static gboolean
drag_motion (GtkWidget *widget,
			 GdkDragContext *context,
			 gint x, gint y,
			 guint time,
			 GnomeCmdFileSelector *fs)
{
	gdk_drag_status (context, context->suggested_action, time);
	return TRUE;
}


static char *
build_selected_file_list (GnomeCmdFileSelector *fs, int *file_list_len)
{
	GList *sel_files = gnome_cmd_file_selector_get_selected_files (fs);
	int listlen = g_list_length (sel_files);
	
	if (listlen > 1) {
		char *data, *copy;
		int total_len = 0;
		GList *tmp = sel_files;
		GList *uri_str_list = NULL;

		/* create a list with the uri's of the selected files, and calculate the total_length needed */
		while (tmp) {
			GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
			char *uri_str;
			
			uri_str = g_strdup_printf (
				"%s\r\n",
				gnome_cmd_file_get_uri_str (finfo));
			g_printerr ("dnd-uri_str: %s\n", uri_str);
			uri_str_list = g_list_append (uri_str_list, uri_str);
			total_len += strlen (uri_str);
			
			tmp = tmp->next;
		}
		
		/* allocate memory */
		total_len++;
		data = copy = g_malloc (total_len+1);

		/* put the uri_str_list in the allocated memory */
		tmp = uri_str_list;
		while (tmp) {
			gchar *uri_str = (gchar*)tmp->data;
			
			strcpy (copy, uri_str);
			copy += strlen (uri_str);

			tmp = tmp->next;
		}

		g_list_foreach (uri_str_list, (GFunc)g_free, NULL);
		g_list_free (uri_str_list);
		
		data [total_len] = '\0';
		*file_list_len = total_len;
		return data;
	}
	else if (listlen == 1) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)sel_files->data;
		char *uri_str;

		uri_str = g_strdup (gnome_cmd_file_get_uri_str (finfo));
		
		*file_list_len = strlen (uri_str) + 1;
		return uri_str;
	}

	*file_list_len = 0;
	g_list_free (sel_files);
	return NULL;
}
