/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <errno.h>
#include "gnome-cmd-includes.h"

#define MAX_TYPE_LENGTH 2
#define MAX_NAME_LENGTH 128
#define MAX_OWNER_LENGTH 128
#define MAX_GROUP_LENGTH 128
#define MAX_PERM_LENGTH 10
#define MAX_DATE_LENGTH 64
#define MAX_SIZE_LENGTH 32

//#define DEBUG_FILE_REF 

int file_counter = 0;

typedef struct
{
	GnomeCmdFileUpdateCallbackFunc func;
	gpointer user_data;	
} FileUpdateData;

gchar *type_strings[] =
{
	N_("unknown file"),
	N_("regular file"),
	N_("directory"),
	N_("FIFO"),
	N_("UNIX Socket"),
	N_("character device"),
	N_("block device"),
	N_("symbolic link")
};

/*
static void update_observers (GnomeCmdFile *file)
{
	GList *tmp = file->observers;
	while (tmp)
	{
		FileUpdateData *data = (FileUpdateData*)tmp->data;
		if (data)
			data->func (file, data->user_data);
		tmp = tmp->next;
	}
}
*/

GnomeCmdFile *gnome_cmd_file_new (void)
{
	GnomeCmdFile *file = (GnomeCmdFile*)g_new (GnomeCmdFile, 1);
	file->ref_cnt = 0;
	file->observers = NULL;
	file->info = NULL;
	file->dir = NULL;
	file->uri = NULL;
	file->uri_str = NULL;
	file_counter++;
	return file;	
}


GnomeCmdFile *gnome_cmd_file_new_with_values (GnomeVFSFileInfo *info, GnomeCmdDir *dir)
{
	GnomeCmdFile *file = gnome_cmd_file_new ();

	g_return_val_if_fail (file != NULL, NULL);
	
	file->info = info;
	file->dir = dir;

	file->uri = gnome_cmd_dir_get_file_uri (dir, info->name);
	file->uri_str = gnome_vfs_uri_to_string (file->uri, 0);
	file->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (file->uri), 0);
	
	gnome_vfs_file_info_ref (file->info);

	return file;
}


void gnome_cmd_file_add_observer (GnomeCmdFile *file,
								  GnomeCmdFileUpdateCallbackFunc func,
								  gpointer user_data)
{
	FileUpdateData *data = g_new (FileUpdateData, 1);
	data->func = func;
	data->user_data = user_data;
	
	file->observers = g_list_append (file->observers, data); 
}


void gnome_cmd_file_remove_observer (GnomeCmdFile *file,
									 GnomeCmdFileUpdateCallbackFunc func,
									 gpointer user_data)
{
	GList *tmp;

	g_return_if_fail (file != NULL);

	tmp = file->observers;
	while (tmp)
	{
		FileUpdateData *data = (FileUpdateData*)tmp->data;
		if (data && data->func == func && data->user_data == user_data)
		{
			file->observers = g_list_remove (file->observers, data);
			g_free (data);
			return;
		}
		tmp = tmp->next;
	}
}


void gnome_cmd_file_ref (GnomeCmdFile *file)
{
	g_return_if_fail (file != NULL);

	file->ref_cnt++;
	
#ifdef DEBUG_FILE_REF
	g_printerr ("file refing: 0x%x %s to %d\n", file, file->info->name, file->ref_cnt);
#endif
}


void gnome_cmd_file_unref (GnomeCmdFile *file)
{
	g_return_if_fail (file != NULL);
	
	file->ref_cnt--;
#ifdef DEBUG_FILE_REF
	g_printerr ("file un-refing: 0x%x %s to %d\n", file, file->info->name, file->ref_cnt);
#endif
	
	if (file->ref_cnt < 1)
	{
#ifdef DEBUG_FILE_REF
		g_printerr ("file freeing 0x%x %s\n", file, file->info->name);
#endif
		gnome_vfs_file_info_unref (file->info);
		g_list_foreach (file->observers, (GFunc)g_free, NULL);
		gnome_vfs_uri_unref (file->uri);
		g_free (file->uri_str);
		g_free (file->path);
		g_free (file);
		file_counter--;
	}
}


GnomeVFSResult gnome_cmd_file_chmod (GnomeCmdFile *file, GnomeVFSFilePermissions perm)
{
	GnomeVFSResult ret;

	g_return_val_if_fail (file != NULL, 0);
	g_return_val_if_fail (file->info != NULL, 0);

	file->info->permissions = perm;
	ret = gnome_vfs_set_file_info_uri (
		gnome_cmd_file_get_uri (file),
		file->info,
		GNOME_VFS_SET_FILE_INFO_PERMISSIONS);

	return ret;
}


GnomeVFSResult gnome_cmd_file_chown (GnomeCmdFile *file, uid_t uid, gid_t gid)
{
	GnomeVFSResult ret;

	g_return_val_if_fail (file != NULL, 0);
	g_return_val_if_fail (file->info != NULL, 0);

	if (uid != -1)
		file->info->uid = uid;
	file->info->gid = gid;
	ret = gnome_vfs_set_file_info_uri (
		gnome_cmd_file_get_uri (file),
		file->info,
		GNOME_VFS_SET_FILE_INFO_OWNER);

	return ret;
}


const gchar *gnome_cmd_file_get_name (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	return file->info->name;
}


const gchar *gnome_cmd_file_get_owner (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	if (GNOME_VFS_FILE_INFO_LOCAL (file->info))
	{
		user_t *owner = OWNER_get_user_by_uid (file->info->uid);
		return owner->name;
	}
	else
	{
		static gchar owner_str[MAX_OWNER_LENGTH];
		g_snprintf (owner_str, MAX_OWNER_LENGTH, "%d", file->info->uid);
		return owner_str;
	}
}


const gchar *gnome_cmd_file_get_group (GnomeCmdFile *file)
{
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	if (GNOME_VFS_FILE_INFO_LOCAL (file->info))
	{
		group_t *group = OWNER_get_group_by_gid (file->info->gid);
		return group->name;
	}
	else
	{
		static gchar group_str[MAX_GROUP_LENGTH];
		g_snprintf (group_str, MAX_GROUP_LENGTH, "%d", file->info->gid);
		return group_str;
	}
}


const gchar *gnome_cmd_file_get_date (GnomeCmdFile *file)
{
	static gchar date_str[MAX_DATE_LENGTH];
	
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	time2string (file->info->mtime, date_str, MAX_DATE_LENGTH,
				 gnome_cmd_data_get_date_format ());
	return date_str;
}


const gchar *gnome_cmd_file_get_size (GnomeCmdFile *file)
{
	static gchar size_str[MAX_SIZE_LENGTH];
	
	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	size2string (file->info->size, size_str, MAX_SIZE_LENGTH,
				 gnome_cmd_data_get_size_disp_mode ());
	return size_str;
}


const gchar *gnome_cmd_file_get_perm (GnomeCmdFile *file)
{
	static gchar perm_str[MAX_PERM_LENGTH];

	g_return_val_if_fail (file != NULL, NULL);
	g_return_val_if_fail (file->info != NULL, NULL);
	
	perm2string (file->info->permissions, perm_str, MAX_PERM_LENGTH);
	return perm_str;
}


const gchar *gnome_cmd_file_get_type_string (GnomeCmdFile *finfo)
{
	static gchar type_str[MAX_TYPE_LENGTH];
	
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	type2string (finfo->info->type, type_str, MAX_TYPE_LENGTH);
	return type_str;
}


const gchar *gnome_cmd_file_get_type_desc (GnomeCmdFile *finfo)
{

	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	return type_strings[finfo->info->type];
}


GdkPixmap *gnome_cmd_file_get_type_pixmap_small (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	if (gnome_cmd_data_get_use_mime ())
		return mime2pixmap (finfo->info->mime_type, finfo->info->type, FALSE);

	return type2pixmap (finfo->info->type, FALSE);
}


GdkPixmap *gnome_cmd_file_get_type_pixmap_big (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	if (gnome_cmd_data_get_use_mime ())
		return mime2pixmap (finfo->info->mime_type, finfo->info->type, TRUE);
	
	return type2pixmap (finfo->info->type, TRUE);
}


GdkBitmap *gnome_cmd_file_get_type_mask_small (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	if (gnome_cmd_data_get_use_mime ())
		return mime2mask (finfo->info->mime_type, finfo->info->type, FALSE);
	
	return type2mask (finfo->info->type, FALSE);
}


GdkBitmap *gnome_cmd_file_get_type_mask_big (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	if (gnome_cmd_data_get_use_mime ())
		return mime2mask (finfo->info->mime_type, finfo->info->type, TRUE);
	
	return type2mask (finfo->info->type, TRUE);
}


const gchar *gnome_cmd_file_get_path (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);

	return finfo->path;
}


GnomeVFSURI *gnome_cmd_file_get_uri (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	
	return finfo->uri;	
}


const gchar *gnome_cmd_file_get_uri_str (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);

	return finfo->uri_str;
}


const gchar *gnome_cmd_file_get_mime_type (GnomeCmdFile *finfo)
{
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	return finfo->info->mime_type;
}


void gnome_cmd_file_show_properties (GnomeCmdFile *finfo)
{
	GtkWidget *dialog = gnome_cmd_file_props_dialog_create (finfo);	
	gtk_widget_ref (dialog);
	gtk_object_set_data_full (
		GTK_OBJECT (main_win), "gnome_cmd_file_props_dialog", dialog,
		(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog);
}


void gnome_cmd_file_show_chown_dialog (GList *files)
{
	GtkWidget *dialog;
	
	dialog = gnome_cmd_chown_dialog_new (files);
	gtk_widget_ref (dialog);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "gnome_cmd_chown_dialog", dialog,
							  (GtkDestroyNotify)gtk_widget_unref);
	gtk_widget_show (dialog);
}


void gnome_cmd_file_show_chmod_dialog (GList *files)
{
	GtkWidget *dialog;
	
	dialog = gnome_cmd_chmod_dialog_new (files);
	gtk_widget_ref (dialog);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "gnome_cmd_chmod_dialog", dialog,
							  (GtkDestroyNotify)gtk_widget_unref);
	gtk_widget_show (dialog);
}


void gnome_cmd_file_view (GnomeCmdFile *finfo)
{
	gchar *command;
	const gchar *viewer;
	const gchar *path;	

	g_return_if_fail (finfo != NULL);

	viewer = gnome_cmd_data_get_viewer ();
	path = gnome_cmd_file_get_path (finfo);
	
	command = g_strdup_printf (viewer, path);
	run_command (command);
	g_free (command);
}


void gnome_cmd_file_edit (GnomeCmdFile *finfo)
{
	gchar *command;
	
	g_return_if_fail (finfo != NULL);
	
	command = g_strdup_printf (gnome_cmd_data_get_editor (), gnome_cmd_file_get_path (finfo));
	run_command (command);
	g_free (command);
}


void gnome_cmd_file_delete (GList *files)
{
	gnome_cmd_delete_dialog_show (files);
}


void gnome_cmd_file_show_cap_cut (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_show_cap_copy (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_show_cap_paste (GnomeCmdFile *finfo)
{
}


void gnome_cmd_file_update_info (GnomeCmdFile *finfo, GnomeVFSFileInfo *info)
{
	g_return_if_fail (finfo != NULL);
	g_return_if_fail (info != NULL);
	
	gnome_vfs_file_info_unref (finfo->info);
	gnome_vfs_file_info_ref (info);
	finfo->info = info;
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_copy
*
*   Purpose: Refs all files in the passed list and return a copy of that list
*
*   Params: 
*
*   Returns: A copy of the passed list with all files ref'ed
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_file_list_copy (GList *files)
{
	g_return_val_if_fail (files != NULL, NULL);

	gnome_cmd_file_list_ref (files);
	return g_list_copy (files);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_free
*
*   Purpose: Unrefs all files in the passed list and then frees the list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_free (GList *files)
{
	if (!files) return;

	gnome_cmd_file_list_unref (files);
	g_list_free (files);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_ref
*
*   Purpose: Refs all files in the passed list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_ref (GList *files)
{
	g_return_if_fail (files != NULL);

	g_list_foreach (files, (GFunc)gnome_cmd_file_ref, NULL);
}


/******************************************************************************
*
*   Function: gnome_cmd_file_list_unref
*
*   Purpose: Unrefs all files in the passed list
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

void gnome_cmd_file_list_unref (GList *files)
{
	g_return_if_fail (files != NULL);

	g_list_foreach (files, (GFunc)gnome_cmd_file_unref, NULL);
}



