/* vi:set ts=8 sts=0 sw=8:
 * $Id: search.c,v 1.5 2001/01/27 17:32:23 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <config.h>
#include "win.h"
#include "file.h"
#include "msgbar.h"
#include "multiple.h"
#include "results.h"
#include "debug.h"
#include "misc.h"
#include "advopts.h"
#include "prefs.h"
#include "error.h"
#include "pathmax.h"
#include "search.h"

enum findopt_e {
	FO_NONE			= 0,
	FO_PARTIAL_FNAME	= (1 << 0),
	FO_CASE_SENS		= (1 << 1),
	FO_SEARCH_SUBDIR	= (1 << 2),
	FO_REGEX_FNAME		= (1 << 3),
	FO_LAST			= (1 << 31)
};
typedef enum findopt_e findopt_t;


/*** PRIVATE FUNCTION PROTOTYPES ***/
static void search_prev_cmd_dlg_create(win_t *w);
static void search_reset_pred_free(struct predicate *pred);
static void search_reset_uid_list_free(struct userid **uilist);
static void search_opts_set_clear(win_t *w, findopt_t opt, gboolean set);
static void search_exec(win_t *w, int argc, char *argv[]);
static void search_cleanup(int argc, char **argv, GSList *argvl);
static void search_exec_common(win_t *w, int argc, GSList *argvl);
static int search_dir_add(win_t *w, GSList *argvl);
static int search_file_add(win_t *w, GSList *argvl);
static void search_basic_begin(win_t *w);
static void search_adv_begin(win_t *w);
static void search_manual_ok_cb(GtkWidget *wgt, gpointer cbdata);
static void search_manual_cancel_cb(GtkWidget *wgt, gpointer cbdata);


enum search_pcmd_e {
	Ef_pcmd_dialog,		/* dialog */
	Ef_pcmd_list_wgt,	/* list */
	Ef_pcmd_apply,		/* button */
	Ef_pcmd_ok,		/* button */
	Ef_pcmd_MAX
};
typedef enum search_pcmd_e search_pcmd_t;

static ef_t search_pcmd_wgts[] = {
	{
		"List Dialog",
		Ef_pcmd_dialog,
		Ef_wgt_dialog
	},
	{
		"list_dlg_list",
		Ef_pcmd_list_wgt,
		Ef_wgt_list
	},
	{
		"list_dlg_apply",
		Ef_pcmd_apply,
		Ef_wgt_button
	},
	{
		"list_dlg_ok",
		Ef_pcmd_ok,
		Ef_wgt_button
	},

	{ NULL }
};

enum ef_manual_id_e {
	Ef_manual_dialog,
	Ef_manual_label,
	Ef_manual_entry,
	Ef_manual_ok,
	Ef_manual_cancel,
	Ef_manual_MAX
};
typedef enum ef_manual_id_e ef_manual_id_t;

static ef_t ef_manual_wgts[] = {
	{
		"Text Entry Dialog",
		Ef_manual_dialog,
		Ef_wgt_dialog,
	},
	{
		"entry_dlg_label",
		Ef_manual_label,
		Ef_wgt_label,
	},
	{
		"entry_dlg_entry",
		Ef_manual_entry,
		Ef_wgt_entry,
	},
	{
		"entry_dlg_ok",
		Ef_manual_ok,
		Ef_wgt_button,
	},
	{
		"entry_dlg_cancel",
		Ef_manual_cancel,
		Ef_wgt_button,
	},

	{ NULL }
}; /* ef_manual_wgts[] */


/*** PUBLIC FUNCTION DEFINITIONS ***/
/*
 * Callback for the "clear" button on the toolbar.  For a basic search, simply
 * clears the text entry fields for the filename and directory name.  For an
 * advanced search, also clears the list of multiple files and directories by
 * calling mult_files_clear_cb() and mult_dirs_clear_cb().
 */
void
search_clear_cb(GtkWidget *wgt, gpointer cbdata)
{
	ef_main_id_t id;
	GtkWidget *entry;
	int pagenum;
	char *txt;
	win_t *w = (win_t *)cbdata;

	pagenum = gtk_notebook_get_current_page(
				GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]));

	if (pagenum == 0) {
		for (id = Ef_main_find_text; ; id = Ef_main_dir_text) {
			entry = w->mainwgts[id];
			g_assert(GTK_IS_ENTRY(entry));
			if (GTK_WIDGET_IS_SENSITIVE(entry)) {
				txt = gtk_entry_get_text(GTK_ENTRY(entry));
				gtk_editable_delete_text(GTK_EDITABLE(entry),
							 0, strlen(txt));
			} else {
				mult_files_clear_cb(NULL, w);
				mult_dirs_clear_cb(NULL, w);
			}
			if (id == Ef_main_dir_text)
				break;
		}
	} else {
		entry = w->mainwgts[Ef_main_locate_text];
		g_assert(GTK_IS_ENTRY(entry));
		txt = gtk_entry_get_text(GTK_ENTRY(entry));
		gtk_editable_delete_text(GTK_EDITABLE(entry), 0, strlen(txt));
	}
} /* search_clear_cb */


/*
 * Data needed in the callbacks for manual search command.
 */
struct manual_cmd_data_s {
	win_t *w;
	GtkWidget *entry;
	GtkWidget *dlg;
};
typedef struct manual_cmd_data_s manual_cmd_data_t;


/*
 * "Manual" find
 */
void
search_manual_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget **wgts;
	int num;
	manual_cmd_data_t *mcdp;
	win_t *w = (win_t *)cbdata;

	num = (sizeof(ef_manual_wgts) / sizeof(ef_manual_wgts[0])) - 1;
	wgts = my_widgets_setup(w, NULL, num, ef_manual_wgts,
				"Text Entry Dialog",
				Ef_manual_MAX, Ef_manual_MAX, TRUE);
	gtk_window_set_title(GTK_WINDOW(wgts[Ef_manual_dialog]),
			     _("Manual Search..."));
	gnome_dialog_set_parent(GNOME_DIALOG(wgts[Ef_manual_dialog]),
				GTK_WINDOW(win_main_toplev(w)));

	gtk_label_set_text(GTK_LABEL(wgts[Ef_manual_label]),
	   _("Enter the exact \"find\" command\nto be executed by the shell"));

	mcdp = g_new(manual_cmd_data_t, 1);
	mcdp->w = w;
	mcdp->entry = wgts[Ef_manual_entry];
	mcdp->dlg = wgts[Ef_manual_dialog];

	gtk_signal_connect(GTK_OBJECT(wgts[Ef_manual_ok]), "clicked",
			   GTK_SIGNAL_FUNC(search_manual_ok_cb), mcdp);
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_manual_cancel]), "clicked",
			   GTK_SIGNAL_FUNC(search_manual_cancel_cb), mcdp);
	gtk_widget_show(wgts[Ef_manual_dialog]);
	g_free(wgts);
} /* search_manual_cb */


/*
 * Resets the window structure to reflect a "new" search.
 */
void
search_reset(win_t *w)
{
	GtkWidget *wgt;

	/* make sure widget is still there */
	if (w->progbar && GTK_IS_WIDGET(w->progbar)) {
		gtk_progress_set_activity_mode(w->progbar, FALSE);
		gtk_progress_set_value(w->progbar, 0);
	}

	/* keep (reuse) previously selection options on main window */
	if ((wgt = w->mainwgts[Ef_main_search_subdir]) &&
	    GTK_IS_WIDGET(wgt) && GTK_WIDGET_IS_SENSITIVE(wgt)) {
		search_opts_set_clear(w, FO_SEARCH_SUBDIR,
				      GTK_TOGGLE_BUTTON(wgt)->active);
	}
	if ((wgt = w->mainwgts[Ef_main_partial_fname]) && GTK_IS_WIDGET(wgt)) {
		search_opts_set_clear(w, FO_PARTIAL_FNAME,
				      GTK_TOGGLE_BUTTON(wgt)->active);
	}
	if ((wgt = w->mainwgts[Ef_main_regex_fname]) && GTK_IS_WIDGET(wgt)) {
		search_opts_set_clear(w, FO_REGEX_FNAME,
				      GTK_TOGGLE_BUTTON(wgt)->active);
	}
	if ((wgt = w->mainwgts[Ef_main_case_sens]) && GTK_IS_WIDGET(wgt)) {
		search_opts_set_clear(w, FO_CASE_SENS,
				      GTK_TOGGLE_BUTTON(wgt)->active);
	}

	w->stop_search = FALSE;
	w->is_searching = FALSE;
	w->num_found = 0;
	w->num_errs = 0;

	/* set default values for stuff that used to be static/global */
	search_reset_uid_list_free(&(w->user_alist));
	search_reset_uid_list_free(&(w->nouser_alist));
	search_reset_uid_list_free(&(w->group_alist));
	search_reset_uid_list_free(&(w->nogroup_alist));

	w->kilobytes = -1;

	w->launch_first_time = 1;
	w->fstype_known = FALSE;
	g_free(w->cur_fstype);
	w->cur_fstype = NULL;
	w->cur_dev = 0;

	search_reset_pred_free(w->predicates);
	w->predicates = NULL;
#if 0
	/* the contents pointed at by these should be freed in the previous
	 * call to search_reset_pred_free(), but i'm keeping these two lines
	 * here as a placeholder/reminder. */
	g_free(w->last_pred);
	g_free(w->eval_tree);
#endif
	w->last_pred = NULL;
	w->eval_tree = NULL;
	w->do_dir_first = TRUE;
	w->maxdepth = -1;
	w->mindepth = -1;
	w->cur_day_start = time ((time_t *) 0) - DAYSECS;
	w->full_days = FALSE;
	w->no_leaf_check = FALSE;
	w->stay_on_filesystem = FALSE;
#ifndef HAVE_FCHDIR
	g_free(w->starting_dir);
	w->starting_dir = NULL;
#else
	w->starting_desc = 0;
#endif
	w->have_stat = FALSE;
	w->exit_status = 0;
	w->dereference = FALSE;
	g_free(w->dir_ids);
	w->dir_ids = NULL;
	w->dir_alloc = 0;
	w->dir_curr = -1;
	w->root_dev = 0;
#ifdef CACHE_IDS
	g_free(uid_unused);
	w->uid_unused = NULL;
	w->uid_allocated = 0;
	g_free(gid_unused);
	w->gid_unused = NULL;
	w->gid_allocated = 0;
#endif
#if 0
	g_free(w->globfree);
	w->globfree = NULL;
	w->gfalloc = 0;

	g_free(w->path);
	w->path = NULL;
	w->final_colon = 0;
#endif

	msgbar_printf(w, " ");
} /* search_reset */


/*
 * Show results from last search
 */
void
search_prev_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	results_prev_show(w);
} /* search_prev_cb */


/*
 * Show actual find command from last search
 */
void
search_cmd_prev_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (w->pcmdwgts && w->pcmdwgts[Ef_pcmd_dialog] &&
	    GTK_IS_WIDGET(w->pcmdwgts[Ef_pcmd_dialog])) {

		gtk_widget_show(w->pcmdwgts[Ef_pcmd_dialog]);
	    	gdk_window_raise(w->pcmdwgts[Ef_pcmd_dialog]->window);
	} else
		msgbar_printf(w, _("No previous searches."));
} /* search_cmd_prev_cb */


int
search_fname_add(win_t *w, GSList *argvl, char *fname)
{
	/* indicate what to search for */
	if (w->findopts & FO_CASE_SENS) {
		if (w->findopts & FO_REGEX_FNAME)
			argvl = g_slist_append(argvl, g_strdup("-regex"));
		else
			argvl = g_slist_append(argvl, g_strdup("-name"));
	} else {
		if (w->findopts & FO_REGEX_FNAME)
			argvl = g_slist_append(argvl, g_strdup("-iregex"));
		else
			argvl = g_slist_append(argvl, g_strdup("-iname"));
	}

	/* exact match, or substring match */
	if (w->findopts & FO_PARTIAL_FNAME) {
		char *home = my_dir_expand_home(fname);
		char *buf = g_strdup_printf("*%s*", home);
		g_free(home);
		argvl = g_slist_append(argvl, buf);
	} else {
		argvl = g_slist_append(argvl, my_dir_expand_home(fname));
	}

	return 2;
} /* search_fname_add */


static void
locate_execute(win_t *w)
{
	char *tmpfile, *cmd, *locate_text, *locate_db;
	char *line;
	FILE *fp;
	int err;

	locate_text = gtk_entry_get_text(
				GTK_ENTRY(w->mainwgts[Ef_main_locate_text]));
	g_assert(locate_text && strlen(locate_text) > 0);

	locate_db = gtk_entry_get_text(
				GTK_ENTRY(w->mainwgts[Ef_main_locate_db]));

#if 0
	extern int locate_main(win_t *w, int argc, char *argv[]);
	char *argv[5];
	int argc;

	argv[0] = "locate";
	if (locate_db && strlen(locate_db) > 0) {
		argc = 4;
		argv[1] = "-d";
		argv[2] = locate_db;
		argv[3] = locate_text;
		argv[4] = NULL;
	} else {
		argc = 2;
		argv[1] = locate_text;
		argv[2] = NULL;
	}

	results_widgets_setup(w);
	(void)locate_main(w, argc, argv);
	results_dialog_show(w);
#endif
	tmpfile = my_tempname();
	if (locate_db && strlen(locate_db) > 0) {
		if (g_file_exists(locate_db)) {
			cmd = g_strdup_printf("locate -d %s %s > %s",
					      locate_db, locate_text, tmpfile);
		} else {
			cmd_err(locate_db, ENOENT);
			goto locate_exit;
		}
	} else {
		cmd = g_strdup_printf("locate %s > %s", locate_text, tmpfile);
	}
	APPDBG_SEARCH(("locate_exec: cmd = '%s'\n", cmd));
	err = cmd_exec(_("Executing locate..."), w, cmd, FALSE, TRUE);
	g_free(cmd);
	if (err)
		goto locate_exit;

	results_widgets_setup(w);
	if ((fp = file_open_fp(tmpfile, "r")) == NULL)
		goto locate_exit;

	line = g_new(char, PATH_MAX * 2);
	while ((fgets(line, (PATH_MAX * 2) - 1, fp))) {
		line[strlen(line) - 1] = '\0';	/* drop trailing \n */
		results_add(w, line, NULL, 0, NULL);
	}
	fclose(fp);
	g_free(line);
	results_dialog_show(w);

locate_exit:
	(void)unlink(tmpfile);
	g_free(tmpfile);
} /* locate_execute */


/*
 * Callback for "start search" button on toolbar.
 */
void
search_begin_cb(GtkWidget *wgt, gpointer cbdata)
{
	int pagenum;
	win_t *w = (win_t *)cbdata;

	g_assert(w->mainwgts[Ef_main_find_text]);
	g_assert(w->mainwgts[Ef_main_dir_text]);

	pagenum = gtk_notebook_get_current_page(
				GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]));
	if (pagenum == 0) {
		if (GTK_TOGGLE_BUTTON(
				w->mainwgts[Ef_main_adv_search])->active &&
		    w->advwgts)
			search_adv_begin(w);
		else
			search_basic_begin(w);
	} else {
		locate_execute(w);
	}
} /* search_begin_cb */


/*
 * Callback for "stop search" button on toolbar.
 */
void
search_stop_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	w->stop_search = TRUE;
} /* search_stop_cb */


/*
 * Callback for "changed" signal for either text entry field.
 */
void
search_entry_file_dir_cb(GtkWidget *wgt, gpointer cbdata)
{
	int pagenum;
	char *findtxt, *dirtxt;
	win_t *w = (win_t *)cbdata;
	int len1 = 0;
	gboolean clear_sens = FALSE, search_sens = FALSE;

	pagenum = gtk_notebook_get_current_page(
				GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]));

	if (pagenum == 0) {
		int len2 = 0;

		g_assert(GTK_IS_ENTRY(w->mainwgts[Ef_main_find_text]));
		findtxt = gtk_entry_get_text(
				GTK_ENTRY(w->mainwgts[Ef_main_find_text]));
		if (findtxt)
			len1 = strlen(findtxt);

		g_assert(GTK_IS_ENTRY(w->mainwgts[Ef_main_dir_text]));
		dirtxt = gtk_entry_get_text(
				GTK_ENTRY(w->mainwgts[Ef_main_dir_text]));
		if (dirtxt)
			len2 = strlen(dirtxt);


		if (len1 || len2)
			clear_sens = TRUE;

		if (len1 && len2)
			search_sens = TRUE;
	} else {
		findtxt = gtk_entry_get_text(
				GTK_ENTRY(w->mainwgts[Ef_main_locate_text]));
		if (findtxt && strlen(findtxt) > 1)
			clear_sens = search_sens = TRUE;
	}

	gtk_widget_set_sensitive(w->mainwgts[Ef_main_clear], clear_sens);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_file_new], clear_sens);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_search], search_sens);
} /* search_entry_file_dir_cb */


/*
 * Callback for "search subdirectories" checkbutton.
 */
void
search_subdir_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(GTK_IS_CHECK_BUTTON(wgt));
	search_opts_set_clear(w, FO_SEARCH_SUBDIR,
			      GTK_TOGGLE_BUTTON(wgt)->active);
} /* search_subdir_cb */


/*
 * Callback for "partial filenames" checkbutton.
 */
void
search_partial_fname_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(GTK_IS_CHECK_BUTTON(wgt));
	search_opts_set_clear(w, FO_PARTIAL_FNAME,
			      GTK_TOGGLE_BUTTON(wgt)->active);
} /* search_partial_fname_cb */


/*
 * Callback for "regular expr filenames" checkbutton.
 */
void
search_regex_fname_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(GTK_IS_CHECK_BUTTON(wgt));
	search_opts_set_clear(w, FO_REGEX_FNAME,
			      GTK_TOGGLE_BUTTON(wgt)->active);
} /* search_regex_fname_cb */


/*
 * Callback for "case sensitive" checkbutton.
 */
void
search_case_sens_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(GTK_IS_CHECK_BUTTON(wgt));
	search_opts_set_clear(w, FO_CASE_SENS, GTK_TOGGLE_BUTTON(wgt)->active);
} /* search_case_sens_cb */


/*
 * Callback for "Advanced search" checkbutton.
 */
void
search_adv_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(GTK_IS_CHECK_BUTTON(wgt));
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_show_adv_opts],
				 GTK_TOGGLE_BUTTON(wgt)->active);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_search_subdir],
				 !GTK_TOGGLE_BUTTON(wgt)->active);

#if 0
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_partial_fname],
				 !GTK_TOGGLE_BUTTON(wgt)->active);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_case_sens],
				 !GTK_TOGGLE_BUTTON(wgt)->active);
#endif
} /* search_adv_cb */


/*** PRIVATE FUNCTION DEFINITIONS ***/
/*
 * Frees the predicate linked list.  Called by search_reset().
 */
static void
search_reset_pred_free(struct predicate *pred)
{
	struct predicate *pp, *np;

	pp = pred;
	while (pp) {
		np = pp->pred_next;
		pp->pred_next = NULL;
		g_free(pp);
		pp = np;
	}
} /* search_reset_pred_free */


/*
 * Frees a uid linked list.  Called by search_reset().
 */
static void
search_reset_uid_list_free(struct userid **uilist)
{
	struct userid *uip, *next;

	uip = *uilist;
	while (uip) {
		next = uip->next;
		g_free(uip->name);
		g_free(uip);
		uip = next;
	}

	*uilist = NULL;

} /* search_reset_uid_list_free */


/*
 * Sets or clears an FO_* flag.
 */
static void
search_opts_set_clear(win_t *w, findopt_t opt, gboolean set)
{
	if (set)
		w->findopts |= opt;
	else
		w->findopts &= ~opt;
} /* search_opts_set_clear */


/*
 * Starts the search.
 */
static void
search_exec(win_t *w, int argc, char *argv[])
{
	extern int main2(win_t *w, int argc, char *argv[]);
	GtkWidget *litem;
	char *buf;
	int i, len, err;

	/*
	 * if it doesn't exist already, create the list dialog which shows the
	 * previously executed search commands.  append the current search
	 * command to it.
	 */
	if (!w->pcmdwgts)
		search_prev_cmd_dlg_create(w);
	APPDBG_SEARCH(("argc = %d\n", argc));
	for (len = 0, i = 0; i < argc; i++) {
		APPDBG_SEARCH(("argv[%d] = '%s'\n", i, argv[i]));
		len += (strlen(argv[i]) + 1);
	}
	buf = g_new0(char, len + 2);
	for (i = 0; i < argc; i++) {
		strcat(buf, argv[i]);
		if (i < argc - 1)
			strcat(buf, " ");
	}
	litem = gtk_list_item_new_with_label(buf);
	g_free(buf);
	gtk_container_add(GTK_CONTAINER(w->pcmdwgts[Ef_pcmd_list_wgt]), litem);
	gtk_widget_show(litem);

	gtk_widget_set_sensitive(w->mainwgts[Ef_main_stop], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_frame_params], FALSE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_search], FALSE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_clear], FALSE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_manual], FALSE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_file_new], FALSE);

	gtk_progress_set_activity_mode(w->progbar, TRUE);
	w->is_searching = TRUE;
	err = main2(w, argc, argv);
	w->is_searching = FALSE;

	if (w->close_window) {
		win_close_cb(NULL, w);
		return;
	}

	results_found_set(w);
	search_reset(w);

	gtk_widget_set_sensitive(w->mainwgts[Ef_main_file_new], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_manual], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_clear], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_search], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_frame_params], TRUE);
	gtk_widget_set_sensitive(w->mainwgts[Ef_main_stop], FALSE);

	if (err != -1)
		results_dialog_show(w);
} /* search_exec */


/*
 * Minor cleanup after a search has finished.
 */
static void
search_cleanup(int argc, char **argv, GSList *argvl)
{
	int i;

	for (i = 0; i < argc; i++)
		g_free(argv[i]);
	g_free(argv);
	g_slist_free(argvl);
} /* search_cleanup */


/*
 * Common code to start executing a search.
 */
static void
search_exec_common(win_t *w, int argc, GSList *argvl)
{
	GSList *avlp;
	char **argv;
	int i;

	/* Setup command line style argc and argv. */
	argv = g_new(char *, argc + 1);
	for (i = 0, avlp = argvl; i < argc && avlp; i++, avlp = avlp->next)
		argv[i] = avlp->data;
	argv[argc] = NULL;

	results_widgets_setup(w);
	search_exec(w, argc, argv);
	search_cleanup(argc, argv, argvl);
} /* search_exec_common */


/*
 * Adds director(ies) to the search query.
 */
static int
search_dir_add(win_t *w, GSList *argvl)
{
	char *buf, *txt;

	if (is_multi_dir_search(w))
		return multi_dir_add(w, argvl);

	txt = gtk_entry_get_text(GTK_ENTRY(w->mainwgts[Ef_main_dir_text]));
	g_assert(txt);
	buf = my_dir_expand_home(txt);
	argvl = g_slist_append(argvl, buf);

	return 1;
} /* search_dir_add */


/*
 * Adds file(s) to the search query.
 */
static int
search_file_add(win_t *w, GSList *argvl)
{
	char *txt;

	if (is_multi_file_search(w))
		return multi_file_add(w, argvl);

	txt = gtk_entry_get_text( GTK_ENTRY(w->mainwgts[Ef_main_find_text]));
	g_assert(txt);

	return search_fname_add(w, argvl, txt);
} /* search_file_add */


/*
 * Starts a "basic" search.
 */
static void
search_basic_begin(win_t *w)
{
	GSList *argvl;
	int argc;

	argvl = g_slist_append(NULL, g_strdup("find"));
	argc = 1;

	/* append directory to search */
	argc += search_dir_add(w, argvl);

	/* search subdirectories? */
	if ((w->findopts & FO_SEARCH_SUBDIR) == 0) {
		argvl = g_slist_append(argvl, g_strdup("-maxdepth"));
		argvl = g_slist_append(argvl, g_strdup("1"));
		argc += 2;
	}

	/* add file(s) to search for */
	argc += search_file_add(w, argvl);

	/* where to print */
	argvl = g_slist_append(argvl, g_strdup("-wprint"));
	argc++;

	g_assert(argc == g_slist_length(argvl));
	search_exec_common(w, argc, argvl);
} /* search_basic_begin */


/*
 * Starts an "advanced" search.
 */
static void
search_adv_begin(win_t *w)
{
	GSList *argvl;
	int argc;

	argvl = g_slist_append(NULL, g_strdup("find"));
	argc = 1;

	/* append directory to search */
	argc += search_dir_add(w, argvl);

	/* insert "Options" */
	argc += advopts_options_add(w, argvl);

	/* add file(s) to search for */
	argc += search_file_add(w, argvl);

	/* insert "Tests" */
	argc += advopts_tests_add(w, argvl);

	/* insert "Actions" */
	argc += advopts_actions_add(w, argvl);

	g_assert(argc == g_slist_length(argvl));
	search_exec_common(w, argc, argvl);
} /* search_adv_begin */


static void
search_prev_cmd_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_hide(w->pcmdwgts[Ef_pcmd_dialog]);
} /* search_prev_cmd_ok_cb */


static void
search_prev_cmd_apply_cb(GtkWidget *wgt, gpointer cbdata)
{
	int argc;
	GSList *argvl;
	char *txt, *buf;
	GtkList *list;
	GtkLabel *label;
	GtkListItem *litem;
	win_t *w = (win_t *)cbdata;

	list = GTK_LIST(w->pcmdwgts[Ef_pcmd_list_wgt]);
	if (!list->selection)
		return;
	g_assert(GTK_IS_LIST_ITEM(list->selection->data));
	litem = GTK_LIST_ITEM(list->selection->data);
	g_assert(GTK_IS_LABEL(GTK_BIN(litem)->child));
	label = GTK_LABEL(GTK_BIN(litem)->child);
	gtk_label_get(label, &txt);

	argc = 0;
	buf = g_strdup(txt);
	argvl = my_argvl_build(buf, &argc, NULL);
	g_free(buf);

	search_exec_common(w, argc, argvl);
} /* search_prev_cmd_apply_cb */


static void
search_prev_cmd_dlg_create(win_t *w)
{
	GtkWidget *dlg;
	int num;

	g_assert(!w->pcmdwgts);

	num = (sizeof(search_pcmd_wgts) / sizeof(search_pcmd_wgts[0])) - 1;
	w->pcmdwgts = my_widgets_setup(w, w->pcmdwgts, num, search_pcmd_wgts,
				       "List Dialog", Ef_pcmd_MAX, Ef_pcmd_MAX,
				       FALSE);
	g_assert(GTK_IS_LIST(w->pcmdwgts[Ef_pcmd_list_wgt]));

	dlg = w->pcmdwgts[Ef_pcmd_dialog];
	gnome_dialog_set_parent(GNOME_DIALOG(dlg),
				GTK_WINDOW(win_main_toplev(w)));
	gtk_window_set_title(GTK_WINDOW(dlg), _("Previous \"find\" Commands"));
	gtk_signal_connect(GTK_OBJECT(w->pcmdwgts[Ef_pcmd_ok]), "clicked",
			   GTK_SIGNAL_FUNC(search_prev_cmd_ok_cb), w);
	gtk_signal_connect(GTK_OBJECT(w->pcmdwgts[Ef_pcmd_apply]), "clicked",
			   GTK_SIGNAL_FUNC(search_prev_cmd_apply_cb), w);
	gtk_signal_connect(GTK_OBJECT(w->pcmdwgts[Ef_pcmd_dialog]),
			   "delete_event", GTK_SIGNAL_FUNC(gtk_true), NULL);
} /* search_prev_cmd_dlg_create */


static void
search_manual_cancel_cb(GtkWidget *wgt, gpointer cbdata)
{
	manual_cmd_data_t *mcdp = (manual_cmd_data_t *)cbdata;

	gtk_widget_destroy(mcdp->dlg);
	g_free(mcdp);
} /* search_manual_cancel_cb */


static void
search_manual_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	int argc;
	GSList *argvl;
	char *txt, *buf;
	manual_cmd_data_t *mcdp = (manual_cmd_data_t *)cbdata;

	gtk_widget_hide(mcdp->dlg);
	txt = gtk_entry_get_text(GTK_ENTRY(mcdp->entry));
	if (!txt || strlen(txt) < 1)
		goto search_manual_out;

	argc = 0;
	buf = g_strdup(txt);
	argvl = my_argvl_build(buf, &argc, NULL);
	g_free(buf);

	search_exec_common(mcdp->w, argc, argvl);

search_manual_out:
	search_manual_cancel_cb(NULL, mcdp);
} /* search_manual_ok_cb */


/* the end */
