// KeyringProxy.cs
// 
// Copyright (C) 2008, James P. Michels III <james.p.michels@gmail.com>
// Copyright © 2008, David Paleino <d.paleino@gmail.com>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

using System;
using Gnome.Keyring;
using Mono.Unix;

namespace GnomeRDP
{
    /// <summary>
    /// Generic class to manage the GNOME keyring.
    /// </summary>
	public class KeyringProxy
	{		
		private const string key = "Gnome-RDP";
		
		private KeyringProxy()
		{
		}
		
		/// <summary>
		/// Gets a password from the GNOME keyring.
		/// </summary>
		/// <returns>
		/// Returns the password from the GNOME keyring. If no password is found, returns null.
		/// </returns>
		public static string GetPassword(string user, string domain, string server, int protocol)
		{
			try
			{
				string defaultKeyring = Ring.GetDefaultKeyring();
				string obj = key;
				string authType = null;
				int port = 0;
				
				NetItemData[] items = Ring.FindNetworkPassword(user, domain, server, obj, ProtocolToString(protocol), authType, port);
				
				if(items == null) return null;
				
				string password = null;
				
				foreach(NetItemData item in items)
				{
					if(item.Keyring == defaultKeyring && password == null)
					{
						password = item.Secret;
					}
					else
					{
						Console.WriteLine(string.Format(Catalog.GetString("Duplicate entries found. Deleting item {0} from keyring {1}"), item.ItemID, item.Keyring));
						Ring.DeleteItem(item.Keyring, item.ItemID);
					}
				}
																	
				return password;
			}
			catch (Exception ex)
			{
				Console.WriteLine(ex.ToString());
				return null;
			}
		}
		
		/// <summary>
		/// Stores a password in the gnome keyring.
		/// </summary>
		public static void SetPassword(string user, string domain, string server, int protocol, string password)
		{
			try
			{
				string defaultKeyring = Ring.GetDefaultKeyring();
				string obj = key;
				string authType = null;
				int port = 0;

				Ring.CreateOrModifyNetworkPassword(defaultKeyring, user, domain, server, obj, ProtocolToString(protocol), authType, port, password);
			}
			catch (Exception ex)
			{
				Console.WriteLine(ex.ToString());
			}
		}
		
		private static string ProtocolToString(int protocol)
		{
			switch (protocol)
			{
			case 0: return "rdp";
			case 1: return "vnc";
			case 2: return "ssh";
			default: throw new ArgumentException(string.Format(Catalog.GetString("Protocol value {0} is not valid"), protocol)); 
			}	
		}
		
		private static int StringToProtocol(string protocol)
		{
			switch (protocol)
			{
			case "rdp": return 0;
			case "vnc": return 1;
			case "ssh": return 2;
			default: throw new ArgumentException(string.Format(Catalog.GetString("Protocol value {0} is not valid"), protocol)); 
			}
		}
	}
}
