#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001 Free Software Foundation
#
# FILE:
# GComm.py
#
# DESCRIPTION:
# Class that provides a basis for GNUe Comm clients & servers.
#
# NOTES:
#

import GDebug
from gnue.common import dyn_import, openResource


##############################################################################
#
# Attach to a client driver
#
def attach(interface, params):
  try:
    driver = dyn_import('gnue.common.commdrivers.%s' % interface)
  except ImportError, mesg:
    GDebug.printMesg(1, \
       "Unable to import GComm Adapter '%s': \n%s" % (interface,mesg))
    raise InvalidAdapter, mesg

  if not (hasattr(driver,'CLIENT') and driver.CLIENT):
    raise NoClientAdapter, \
           "GComm adapter '%s' does not support clients" % interface

  adapter = driver.ClientAdapter(params)

  return adapter



##############################################################################
#
# Bind/export our services
#
# Attributes:
#
# rpcdef     The RPC definition loaded from an XML file. This can be
#              1) a string that contains the location of the XML file
#                 (can be a URL or pathname)
#              2) a file-handle (or file-like handle; e.g. StringBuffer)
#                 referencing an XML definition
#              3) a GnuRpc object created from GComm.loadDefinition()
#
# drivers    A dictionary that defines all the transports to be used. The
#            dictionary is of the form: {driver:params} where:
#              1) driver: A string containing the name of the commdriver
#                 (e.g., 'xmlrpc' or 'corba')
#              2) params: A dictionary containing parameters specific to
#                 the commdriver (e.g., {'port':'8888'} )
#
# bindings   A dictionary containing binding definitions. The dictionary
#            is of the form: {server:handler} where:
#              1) service: Name of the service being bound/exposed
#              2) handler: Method that when called returns a class instance
#
def bind(rpcdef, drivers, bindings):

  if type(rpcdef) == type(""):

    # Application supplied the location of their rpcdef.. load it
    fin = openResource(rpcdef)
    mapping = loadDefinition(fin,1)
    fin.close()

  elif rpcdef.hasattr("read"):

    # Application provided us a file-like object
    mapping = loadDefinition(rpcdef)

  else:

    # Otherwise, they must have specified a GnuRpc object
    mapping = rpcdef


  servers = {}

  for interface in drivers.keys():
    params = drivers[interface]

    try:
      driver = dyn_import('gnue.common.commdrivers.%s' % interface)
    except ImportError, mesg:
      GDebug.printMesg(1, \
         "Unable to import GComm Adapter '%s': \n%s" % (interface,mesg))
      raise InvalidAdapter, mesg

    if not (hasattr(driver,'SERVER') and driver.SERVER):
      raise NoServerAdapter, \
             "GComm adapter '%s' does not support servers" % interface

    adapter = driver.ServerAdapter(mapping, bindings, params)
    servers[interface] = adapter

  return servers

#
# To keep everything that a client/server app will need in this
# file, any convenience functions from the commdrivers directory.
# (i.e., a user app should never need to access the commdrivers
# directory.)
#

#
# Load an XML-based interface definition
#
from gnue.common.commdrivers._parser.Parser import loadDefinition




##############################################################################
#
# Exception hierarchy:
#
# Error
# |__ ProgrammingError
# |   |__ InvalidService
# |__ CommunicationsError
# |__ UserError (base for user-defined errors)
#

#
# The basis of all GComm exceptions
#
class Error(StandardError):
  pass

#
# The requested adapter does not exist...
#
class InvalidAdapter(Error):
  pass

#
# The requested adapter could not initialize. Perhaps the
# supplied parameters do not point to a valid server, etc.
#
class AdapterInitializationError(Error):
  pass

#
# The parameters supplied to the adapter are not in the
# correct format, or all the needed parameters were not
# supplied.
#
class AdapterConfigurationError(AdapterInitializationError):
  pass

#
# The application requested a "client" adapter, but the
# adapter only supports "server" adapters.
#
class NoClientAdapter(Error):
  pass

#
# The application requested a "server" adapter, but the
# adapter only supports "client" adapters.
#
class NoServerAdapter(Error):
  pass

#
# TBD..
#
class ProgrammingError(Error):
  pass

#
# A client application requested a service that is not
# exposed by the server.
#
class InvalidService(ProgrammingError):
  pass

#
# A communications error interfered with the client and
# server's transaction.
#
class CommunicationsError(Error):
  pass

#
# Hmm... I had something in mind when I put this down...
#
class UserError(Error):
  pass


