#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000 Free Software Foundation
#
# FILE:
# GParser.py
#
# DESCRIPTION:
# Class that contains a sax based xml processor for GNUE
#
# NOTES:
#
# HISTORY:
#

import sys

try:
  from xml.sax import saxexts
  from xml.sax import saxlib
  from xml.sax import saxutils
except ImportError:
  print """
   This GNUe tool requires PyXML to be installed.
   Please download and install PyXML from:

      http://pyxml.sourceforge.net/

"""
  sys.exit()

import string
import GDebug
import GTypecast
from GParserHelpers import GContent


#######################################################
#
# loadXMLObject
#
# This method loads an object from an XML file and
# returns that object.  If initialize is 1 (default),
# then the object is initialized and ready to go.
# Setting initialize to 0 is useful for a design
# environment where the object is not actually
# being used, but simply loaded as a document.
#
# "attributes" is a dictionary containing extra
# attributes that should be attached to this object.
#
#  e.g., if attributes={myproperty:[0,1,2]}, then
#    before the object is initialized or returned,
#    object.myproperty == [0,1,2].
#
#######################################################

def loadXMLObject(URL, handler, rootType, xmlFileType,
  initialize=1, attributes={}):
  # Create a parser
  parser = saxexts.make_parser()

  # Create a stack for the parsing routine
  object = None

  # Create the handler
  dh = handler()

  # Tell the parser to use our handler
  parser.setDocumentHandler(dh)
  parser.parseFile(URL)
  parser.close()

  object = dh.getRoot()

  if not object:
    print "Error loading %s: empty definition file" % (xmlFileType)
    sys.exit()
  elif object._type != rootType:
    print "Error loading %s: not a valid %s definition (expected: %s, got: %s)" % (xmlFileType,
         xmlFileType, rootType, object._type)
    sys.exit()

  #object.showTree()

  for att in attributes.keys():
    object.__dict__[att] = attributes[att]

  if initialize:
    GDebug.printMesg(10,"Initializing the object tree starting at %s" %(object))
    object.phaseInit(dh._phaseInitCount)

  return object



#######################################################
#
# char
#
# This is for typecasting strings
#
# NOTE: This is a redefinition of GTypecast.text.
#   This redefinition will be removed as soon as all
#   references to it are changes.  You should be using
#   GTypecast from now on.
#
#######################################################
char  = GTypecast.text


#######################################################
#
# bool
#
# This is for typecasting booleans
#
# NOTE: This is a redefinition of GTypecast.boolean.
#   This redefinition will be removed as soon as all
#   references to it are changes.  You should be using
#   GTypecast from now on. 
#
#######################################################
bool = GTypecast.boolean


#######################################################
#
# normalise_whitespace
#
# Remove redundant whitespace from a string ( from xml-howto )
#
#######################################################
def normalise_whitespace(text):
  return string.join( string.split(text), ' ')


def default(attrs, key, default):
  if attrs.has_key(key):
    return attrs[key]
  else:
    return default


#######################################################
#
# xmlHandler
#
# This class is called by the XML parser to
# process the xml file.
#
#######################################################
class xmlHandler(saxlib.HandlerBase):
  def __init__(self):

    self.xmlElements = {}

    self.xmlStack = []
    self.nameStack = []
    self.bootstrapflag = 0
    self.uniqueIDs = {}
    self.root = None
    self._phaseInitCount = 0

  def getRoot(self):
    return self.root

  def startElement(self, name, saxattrs):

    GDebug.printMesg(50, "<%s>" % name)

    # Make sure the tag we are looking at is a valid tag
    if not self.xmlElements.has_key (name):
      print 'Error processing <%s> tag [I do not know what a <%s> tag does]' % (name, name)
      sys.exit()

    baseAttrs = default(self.xmlElements[name],'Attributes',{})
    attrs = {}
    loadedxmlattrs = {}

    for attr in saxattrs.keys():

      # Make sure the attribute we are looking at is a valid attribute
      if not baseAttrs.has_key(attr):
        print 'Error processing <%s> tag [I do not recognize the "%s" attribute' % (name, attr)
        sys.exit()

      # Typecasting, anyone?  If attribute should be int, make it an int
      try:
        attrs[attr] = default(baseAttrs[attr],'Typecast',char)(saxattrs[attr])
        loadedxmlattrs[attr] = attrs[attr]
      except:
        print 'Error processing <%s> tag [invalid type for "%s" attribute; value is "%s"]' % (name, attr, saxattrs[attr])
        sys.exit()

      # If this attribute must be unique, check for duplicates
      if default (baseAttrs[attr],'Unique',0):
        if self.uniqueIDs.has_key('%s' % (saxattrs[attr])):
          print 'Error processing <%s> tag ["%s" attribute should be unique; duplicate value is "%s"]' % (name, attr, saxattrs[attr])
          sys.exit()

    for attr in baseAttrs.keys():
      if not attrs.has_key(attr):

        # Pull default values for missing attributes
        if baseAttrs[attr].has_key ('Default'):
          attrs[attr] = default(baseAttrs[attr],'Typecast', char) (baseAttrs[attr]['Default'])

        # Check for missing required attributes
        elif default(baseAttrs[attr], 'Required', 0):
          print 'Error processing <%s> tag [required attribute "%s" not present]' % (name, attr)
          sys.exit()


    if self.bootstrapflag:
      if self.xmlStack[0] != None:
        object = self.xmlElements[name]['BaseClass'](self.xmlStack[0])
    else:
      object = self.xmlElements[name]['BaseClass']()
      self.root = object
      self.bootstrapflag = 1

    object._xmltag = name

    # Save the attributes loaded from XML file
    # (i.e., attributes that were not defaulted)
    object._loadedxmlattrs = loadedxmlattrs

    for attr in attrs.keys():
      object.__dict__[attr] = attrs[attr]

    self.xmlStack.insert(0, object)
    self.nameStack.insert(0, name)


  def characters(self, ch, start, length):

    text = ch[start:start+length]

    if self.xmlStack[0] != None:

      # Should we keep the text?
      if default(self.xmlElements[self.nameStack[0]],'MixedContent',0):

        if default(self.xmlElements[self.nameStack[0]],'KeepWhitespace',0):
          GContent(self.xmlStack[0], text)
        else:
          # Normalize
          if len(string.replace(string.replace(string.replace(text,' ',''),'\n',''),'\t','')):
            text = normalise_whitespace (text)
          else:
            text = ""
          if len(text):
            GContent(self.xmlStack[0], text)


  def endElement(self, name):
    self.nameStack.pop(0)
    child = self.xmlStack.pop(0)
    inits = child._buildObject()
    self._phaseInitCount = (inits != None and inits > self._phaseInitCount \
                            and inits or self._phaseInitCount)
    GDebug.printMesg(50, "</%s>" % name)


