#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GConnections.py
#
# DESCRIPTION:
"""
Class that loads connection definition files and maintains
database connections.
"""
#
# NOTES:
#
# If you pass GConnections an "eventHandler" instance,
# it will generate a Connections:Connect(name, base)
# when a new connection is created.
#
# HISTORY:
#
# $Id: $

from ConfigParser import *
import sys, string, copy, netrc
from gnue.common.apps import plugin, errors, i18n
from gnue.common.datasources import Exceptions
from gnue.common.datasources import GLoginHandler
from gnue.common.utils.FileUtils import openResource, dyn_import
import GLoginHandler

class Error(gException):
  # Base error
  pass

class NotFoundError (Error):
  # Raised if a requested connection name does not
  # exist in the Connections Definition File.
  pass

class AdapterNotInstalled (errors.AdminError):
  # Raised if a provider is requested for which
  # the python libraries are not installed.
  pass

class DependencyError (errors.AdminError):
  # Raised by the dbdrivers if a dependency module is missing
  def __init__ (self, modulename, url):
    self.modulename = modulename
    self.url = url
    message = u_("Module '%s' is not installed.") % self.modulename
    if self.url:
      message += u_("  You can download it from %s.") % self.url
    errors.AdminError.__init__ (self, message)

class InvalidFormatError (errors.AdminError):
  # Raised if the Connections Definition File is
  # in an unreadable format.
  pass

LoginError = Exceptions.LoginError


class GConnections:

  def __init__(self, location, loginHandler=None, loginOptions={},
               eventhandler=None):

    self._loginHandler = loginHandler
    self._loginOptions = loginOptions
    self._parser = ConfigParser()
    self._location = location
    self._authenticatedUsers = {}
    self._eventHandler=eventhandler

    gDebug (2, 'Conn File: "%s"' % location)

    if len(location):
      fileHandle = openResource(location)

      try:
        self._parser.readfp(fileHandle)

      except DuplicateSectionError:
        tmsg =  u_("The connections file has duplicate source definitions."
                   "\n\nFile: %s") % location
        raise InvalidFormatError, tmsg
      except MissingSectionHeaderError:
        tmsg = u_("The connections file has no source definitions."
                  "\n\nFile: %s") % location
        raise InvalidFormatError, tmsg
      except Exception:
        tmsg =  u_("The connections file cannot be parsed."
                   "\n\nFile: %s") % location
        raise InvalidFormatError, tmsg

    self._openConnections = {}

    self._primaries = {}
    self._aliases = {}
    self._definitions = {}


    # Read all the sections into a dict
    # and make a note of all alias names
    for section in self._parser.sections():
       self._primaries[section]={}
       for att in self._parser.options(section):
         if att == 'aliases':
           for alias in string.split(string.lower(self._parser.get(section,att))):
             self._aliases[alias] = section
         else:
           self._primaries[section][att] = self._parser.get(section, att)

    # Fill in any aliases with their parameters
    for alias in self._aliases.keys():
      section = self._aliases[alias]
      self._aliases[alias] = copy.copy(self._primaries[section])
      self._aliases[alias]['_alias_of'] = section


    self._definitions.update(self._aliases)
    self._definitions.update(self._primaries)



  def setLoginHandler(self, loginHandler):
    self._loginHandler = loginHandler
    try:
      loginHandler.defaults.update(self._loginOptions)
    except AttributeError:
      print "WARNING: Login handler doesn't support 'default' login info."


  def hasConnectionParameters(self, connection_name):
    return self._definitions.has_key(connection_name)


  def getConnectionParameter(self, connection_name, attribute, default=None):
    try:
      definition = self._definitions[connection_name]
      try:
        return definition[attribute]
      except:
        return default
    except KeyError:
      tmsg = u_("The connections file does not contain a definition "
                "for \"%(connection)s\".\n\nFile: %(file)s") \
             % {'connection': connection_name,
                'file'      : self._location}
      raise NotFoundError, tmsg


  #
  # Returns an dictionary of dictionaries describing all connections:
  #  {connection name: {att name: value}}
  #
  def getConnectionNames(self, includeAliases=True):
    if includeAliases:
      return self._definitions.keys()
    else:
      return self._primaries.keys()


  #
  # Returns an dictionary of dictionaries describing all connections:
  #  {connection name: {att name: value}}
  #
  def getAllConnectionParameters(self, includeAliases=1):
    if includeAliases:
      return copy.deepcopy(self._definitions)
    else:
      return copy.deepcopy(self._primaries)


  #
  # Returns a dictionary describing a connection:
  #  {att name: value}
  #
  def getConnectionParameters(self, connection_name):
    try:
      return copy.deepcopy(self._definitions[connection_name])
    except KeyError:
      tmsg = u_("The connections file does not contain a definition "
                "for \"%(connection)s\".\n\nFile: %(file)s") \
             % {'connection': connection_name,
                'file'      : self._location}
      raise NotFoundError, tmsg


  #
  # Add a connection entry (session specific; i.e., doesn't add
  # to the connections.conf file, but to the current instance's
  # list of available connections.
  #
  def addConnectionSpecification (self, name, parameters):
    self._definitions[string.lower(name)] = copy.copy(parameters)



  # ---------------------------------------------------------------------------
  # get a connection instance and optionally log into it
  # ---------------------------------------------------------------------------

  def getConnection (self, connection_name, login = False):
    """
    This function returns an instance of the requested connection and
    optionally logs into it. If there's an already opened connection for the
    requested connectionname this instance will be returned.

    @param connection_name: name of the connection to be returned
    @param login: if TRUE, this function automatically tries to open the
        connection.
    @return: connection instance
    @raise GConnection.NotFoundError: if connection_name does not exist
    """
    connection_name = connection_name.lower ()

    if self._openConnections.has_key (connection_name):
      return self._openConnections [connection_name]

    # Support for multiple open connections to the same database.
    # Specify as 'gnue:1', 'gnue:2', etc, to open two actual connections to
    # 'gnue', each with their own transactions, etc.
    connection_base = connection_name.split (':') [0]

    # This will throw a GConnections.NotFoundError if an unknown connection
    # name is specified. The calling method should catch this exception and
    # handle it properly (exit w/message)
    parameters = self.getConnectionParameters (connection_base)

    driver   = parameters ['provider'].lower ().replace ('/', '.')
    behavior = parameters.get ('behavior','').lower ().replace ('/', '.')
    dbdriver = plugin.find (driver, 'gnue.common.datasources.drivers',
                            'Connection')

    try:
      conn = dbdriver.Connection (self, connection_name, parameters)

    except TypeError:
      # Check for the case that a "Connection" is passed instead of a
      # "Connection" class. This is the case, when the procedure for deferal of
      # the loading of the connection object is overwriten by the "connection"
      # module. i.e. there is a Connection.py file
      # TODO: remove this extra check by cleaning up the whole loading
      # procedure
      conn = dbdriver.Connection.Connection (self, connection_name, parameters)

    self._openConnections [connection_name] = conn

    # Create the introspection instance
    # TODO: add support of the behavior parameter. Does it describe a path to
    #       the introspection module or is it the name of a provider whose
    #       introspector would be used?
    if hasattr (conn, 'behavior'):
      behavior = conn.behavior
    else:
      behavior = conn.defaultBehavior

    conn.introspector = behavior (conn)

    if login:
      self.loginToConnection (conn)

    return conn


  #
  # Return a database provider object
  #
  def getDataObject(self, connection_name, connection_type):

    # This will throw a GConnections.NotFoundError if an unknown
    # connection name is specified.  The calling method should
    # catch this exception and handle it properly (exit w/message)
    connection = self.getConnection(connection_name)

    try:
      dd = connection.supportedDataObjects[connection_type](connection)
      gDebug (7, 'Attaching to %s (%s)' \
          % (dd.__class__.__name__, connection_type))
      return dd
    except KeyError:
      tmsg = u_("DB Driver '%(connection)s' does not support source type "
                "'%(conType)s'") \
             % {'connection': connection,
                'conType': connection_type}
      raise Exceptions.ObjectTypeNotAvailableError, tmsg



  #
  # Has a connection been initialized/established?
  #
  # TODO: this was likely broken
  def isConnectionActive(self, connection):
    return self._openConnections.has_key(string.lower(connection))


  #
  # Get a data connection for a specified database
  #
  def requestConnection(self, dataObject):

#   print """TODO: once this branch makes it into CVS head,
#     eliminate the GConnections.requestConnection logic!"""

    # Support for multiple open connections
    # to same database.
    # Specify as 'gnue:1', 'gnue:2', etc, to open
    # two actual connections to 'gnue', each with
    # their own transactions, etc.

    self.loginToConnection(dataObject._connection)
    dataObject.connect()


  # ---------------------------------------------------------------------------
  # login to a connection
  # ---------------------------------------------------------------------------

  def loginToConnection (self, connection):

    connection_name = connection.name
    connection_base = connection_name.split (':') [0]

    if not self._loginHandler:
      self.setLoginHandler (GLoginHandler.BasicLoginHandler ())

    try:
      connected = connection.__connected

    except AttributeError:
      connected = 0

    if not connected:
      loginData = connection.parameters
      loginData ['_language'] = i18n.getuserlocale ()

      try:
        # load the user's netrc file:
        # a sample .netrc could look like:
        # <.netrc begin>
        # machine 'gnue://my_connection/'
        # login 'mylogin'
        # password 'mypassword'
        # EOF
        # (Remark: if .netrc should work under Win32 you have to
        #  set the HOME environement variable [SET HOME=...])

        netrcData = netrc.netrc ().authenticators ("'gnue://%s/'" \
                                                   % connection_base)
        if netrcData is not None:
          gDebug (7, 'Read the user\'s .netrc file')
          loginData ['_username'] = netrcData [0][1:-1]
          loginData ['_password'] = netrcData [2][1:-1]

          gDebug (7, "Found useful stuff for connection %s in "
                               "the user\'s .netrc file" % connection_name)

      except (IOError, netrc.NetrcParseError, KeyError):
        pass

      if (loginData.has_key ('username')):
        loginData ['_username'] = loginData ['username']
        del loginData ['username']

      if (loginData.has_key ('password')):
        loginData ['_password'] = loginData ['password']
        del loginData ['password']

      # Load
      if loginData.has_key ('custom_auth'):
        authenticator = dyn_import (loginData ['custom_auth']).Authenticator ()
        checkFields   = authenticator.getLoginFields ( \
                                             connection.getLoginFields ())
      else:
        checkFields   = connection.getLoginFields ()
        authenticator = None

      haveAllInformation = True
      for rf, dummy1, dummy2 in checkFields:
        if not (loginData.has_key (rf) and loginData [rf] is not None):
          haveAllInformation = 0
          break

      if haveAllInformation:
#        try:
#          self._authenticatedUsers[base] = loginData['_username']
#        except KeyError:
#          self._authenticatedUsers[base] = None

        if authenticator:
          connection.connect (authenticator.login (loginData))
        else:
          connection.connect (loginData)
          
      else:
        attempts = 4

        gDebug (7, 'Getting new data connection to %s' % connection_name)

        errortext = None
        while attempts:

          try:

            # Ask the UI to prompt for our login data
            loginData.update(self._loginHandler.getLogin(
              [connection_base,
               self.getConnectionParameter(connection_base,'comment',''),
               checkFields], errortext))

            # Ask the data object to connect to the database
            if authenticator:
              connection.connect(authenticator.login(loginData))
            else:
              connection.connect(loginData)

            # We're done!
            attempts = 0
            self._loginHandler.destroyLoginDialog()

          except Exceptions.LoginError:
            # Oops, they must have entered an invalid user/password.
            # Those silly users.
            # user: Hey! Who are you calling silly?!!!
            # Ok, then "those d@mn users"
            attempts  = attempts - 1
            errortext = errors.getException () [2]
            self._loginHandler.destroyLoginDialog ()

            if not attempts:
              # Four times is plenty...
              tmsg = u_("Unable to log in after 4 attempts.\n\nError: %s") \
                     % errortext
              raise Exceptions.LoginError, tmsg

          except GLoginHandler.UserCanceledLogin:
            # Guess they changed their minds. Treat as a login error.
            self._loginHandler.destroyLoginDialog()
            tmsg = u_("User canceled the login request.")
            raise Exceptions.LoginError, tmsg
            
      # Add to authenticated user list
      try:
        self._authenticatedUsers[connection] = loginData['_username']
      except KeyError:
        self._authenticatedUsers[connection] = None


      if self._eventHandler:
        self._eventHandler.dispatchEvent('Connections:Connect',
              name=connection_name, base=connection_base)

    # Done
    connection.__connected = True



  def getAuthenticatedUser(self, connection=None):
    try:
      if connection == None:
        return self._authenticatedUsers[self._authenticatedUsers.keys()[0]]
      else:
        return self._authenticatedUsers[connection]
    except (KeyError, IndexError):
      return None


  def commitAll(self):
    """
    This function commits all transactions
    """
    for connection in self._openConnections.values():
      connection.commit()


  def rollbackAll(self):
    """
    This function rolls back all transactions
    """
    for connection in self._openConnections.values():
      connection.rollback()


  def closeAll(self):
    """
    This function closes all open connections.
    """
    for connection in self._openConnections.values():
      connection.close ()

      if self._authenticatedUsers.has_key (connection):
        del self._authenticatedUsers [connection]

    for k in self._openConnections.keys ():
      del self._openConnections [k]
