# GNU Enterprise RPC interface - Py-XMLRPC type conversion rules
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: typeconv.py 7028 2005-02-15 13:50:52Z reinhard $

import time
import types
import mx.DateTime
import xmlrpclib

# -----------------------------------------------------------------------------
# Convert native Python type to xmlrpc's type
# -----------------------------------------------------------------------------

def python_to_rpc (value, exception):

  # None
  if value is None:
    return ''

  # String
  elif isinstance (value, types.StringType):
    return value
  elif isinstance (value, types.UnicodeType):
    return value

  # Boolean (has to be checked before IntType)
  elif hasattr (types, 'BooleanType') and \
       isinstance (value, types.BooleanType):
    return xmlrpclib.boolean (value)

  # Number
  elif isinstance (value, types.IntType):
    return value
  elif isinstance (value, types.LongType):
    return value
  elif isinstance (value, types.FloatType):
    return value

  # Date/Time
  elif isinstance (value, mx.DateTime.DateTimeType):
    return xmlrpclib.DateTime ((value.year, value.month, value.day, value.hour,
                                value.minute, int (value.second), 1, 1, 1))

  elif isinstance (value, mx.DateTime.DateTimeDeltaType):
    return xmlrpclib.DateTime ((1, 1, 1, value.hour, value.minute,
                               int (value.second), 1, 1, 1))
  # List
  elif isinstance (value, types.ListType):
    return [python_to_rpc (element, exception) for element in value]

  # Tuple
  elif isinstance (value, types.TupleType):
    result = ()
    for element in value:
      result += (python_to_rpc (element, exception), )
    return result

  # Dictionary
  elif isinstance (value, types.DictionaryType):
    result = {}
    for (key, val) in value.items ():
      result [python_to_rpc (key, exception)] = python_to_rpc (val, exception)
    return result

  else:
    raise exception, repr (value)

# -----------------------------------------------------------------------------
# Convert xmlrpc's type to native Python type
# -----------------------------------------------------------------------------

def rpc_to_python (value, exception):

  # None or String
  if isinstance (value, types.StringType):
    if value:
      # if xmlrpclib returns an 8-bit string, it is always in default encoding
      return unicode (value)
    else:
      return None
  elif isinstance (value, types.UnicodeType):
    if value:
      return value
    else:
      return None

  # Boolean (has to be checked before IntType)
  elif hasattr (types, 'BooleanType') and \
       isinstance (value, xmlrpclib.boolean):
    if value:
      return True
    else:
      return False

  # Number
  elif isinstance (value, types.IntType):
    return value
  elif isinstance (value, types.LongType):
    return value
  elif isinstance (value, types.FloatType):
    return value

  # Date/Time
  elif isinstance (value, xmlrpclib.DateTime):
    # mx.DateTime.strptime doesn't work on windows
    return mx.DateTime.mktime (time.strptime (value.value, '%Y%m%dT%H:%M:%S'))

  # List
  elif isinstance (value, types.ListType):
    return [rpc_to_python (element, exception) for element in value]

  # Tuple
  elif isinstance (value, types.TupleType):
    result = ()
    for element in value:
      result += (rpc_to_python (element, exception), )
    return result

  # Dictionary
  elif isinstance (value, types.DictionaryType):
    result = {}
    for (key, val) in value.items ():
      result [rpc_to_python (key, exception)] = rpc_to_python (val, exception)
    return result

  else:
    raise exception, repr (value)
