#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# MenuBar.py
#
# DESCRIPTION:
# A generic UIdriver base for common message-based GUI toolkits.
#
# NOTES:
#

import string
from types import StringType

from gnue.forms.uidrivers._base import UserActions

class MenuBar:


  ####################################################################
  #
  # These methods should be superclassed
  # by the toolkit-specific MenuBar
  #
  
  # Smile.  It's menu-building time!
  def init(self): 
    pass

  # Add a (sub)menu
  def addMenu(self, name, parent):
    pass
  
  # Add a menu item (action)
  def addAction(self, name, parent, userAction):
    pass
    
  # Add a separator
  def addSeparator(self, parent): 
    pass
    
  # Enable a menu item
  def enableItem(self, item):
    pass
  
  # Disable a menu item
  def disableItem(self, item): 
    pass


  ####################################################################
  #
  # These methods should not be touched
  # by the toolkit-specific MenuBar.
  # 
  # Or, in the words of the great poet, 
  # M.C. Hammer,  "Can't touch this!"
  #
      
  def __init__(self, uidriver, container, form): 
    self.driver = uidriver
    self.container = container
    self.form = form
    self.enabled = 1
    self.__actionMap = {}
    
    self.finalize()
    
  def disable(self): 
    self.enabled = 0
    
  def finalize(self): 
    if self.enabled: 

      # Let specific drivers create themselves
      m = self.init()

      self.__createmenu(_DEFAULT_MENU,m)


  def __createmenu(self, contents, parent):
    for item in contents:
      if item == None:
        self.addSeparator(parent)
      else:
        name, contents = item
        if type(contents) == StringType: 
          userAction = UserActions.userActionMap[contents]
          self.__actionMap[userAction.event] = self.addAction(name, parent, userAction)
          self.driver.registerEventListeners( 
              {'can%s' %  userAction.event: self.__canRequestEvent,
               'cannot%s' %  userAction.event: self.__cannotRequestEvent })

        else: 
          m = self.addMenu(name, parent)
          self.__createmenu(contents, m)
        
  def __canRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[3:]]
    self.enableItem(item)

  def __cannotRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[6:]]
    self.disableItem(item)




####################################################################
#
#

_DEFAULT_MENU =  (
   ( u_('&File'), (
       (u_('&Save Changes'), "COMMIT"),
       (u_('&Revert Changes'), "ROLLBACK"),
       None,
       (u_('&Print'), "PRINTOUT"),
       None,
       (u_('&Close'), "EXIT") ) ),

   ( u_('&Edit'), (
       (u_('C&ut'), "CUT"),
       (u_('&Copy'), "COPY"),
#TODO: POST 0.5.0#       ('C&opy Special', (
#TODO: POST 0.5.0#          ('&Record as Text', 'COPYRECORD'),
#TODO: POST 0.5.0#          ('&Screen as Text', 'COPYSCREEN' ) ) ),
       (u_('&Paste'), "PASTE"),
       (u_('Select &All'), "SELECTALL"),
       None,
       (u_('&Insert New Record'),"NEWRECORD"),
       None,
       (u_('&Delete Current Record'),'MARKFORDELETE') ) ),

   ( u_('&Navigation'), (
       (u_('&Next Entry'), "NEXTENTRY"),
       (u_('&Previous Entry'), "PREVENTRY"),
       None,
       (u_('Next &Record'), "NEXTRECORD"),
       (u_('Previous R&ecord'), "PREVRECORD"),
       (u_('&First Record'), 'FIRSTRECORD'),
       (u_('&Last Record'), 'LASTRECORD'),
       (u_('&Go to record...'), 'JUMPPROMPT'),
       None,
       (u_('Next &Block'), 'NEXTBLOCK'),
       (u_('Previous Bl&ock'), 'PREVBLOCK'),
       None,
       (u_('Next P&age'), 'NEXTPAGE'),
       (u_('Pre&vious Page'), 'PREVPAGE') ) ),

   ( u_('&Search'), (
       (u_('Enter Query Mode'), "ENTERQUERY"),
       (u_('Cancel Query Mode'), "CANCELQUERY"),       
       None, 
       (u_('Repeat Last Query'), "COPYQUERY"),
       None,
       (u_('Perform Query'), "EXECQUERY") ) ),

   ( u_('&Help'), (
#TODO: POST 0.5.0#       (_('Keystrokes...'), "HELPKEYS"),
       (u_('About...'), "ABOUT"), ) ),
  )

