# GNU Enterprise Forms - Curses UI Driver - Form Widget
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: form.py 6851 2005-01-03 20:59:28Z jcater $

import curses
import string

from gnue.forms.GFKeyMapper import KeyMapper

from gnue.forms.uidrivers._base import UIdriver as BaseDriver
from gnue.forms.uidrivers._base.widgets._base import UIWidget

# =============================================================================
# Form class
# =============================================================================

class UIForm (UIWidget):

  # ---------------------------------------------------------------------------
  # Initialization
  # ---------------------------------------------------------------------------

  def __init__ (self, event):

    UIWidget.__init__ (self, event)

    self.__pages = []
    self.__currentPage = None

    # Status bar values
    self.__tip = ''
    self.__status = ''
    self.__insert = ''
    self.__curRec = 0
    self.__maxRec = 0
    self.__curPage = 0
    self.__maxPage = 0

  # ---------------------------------------------------------------------------
  # Initialize form
  # ---------------------------------------------------------------------------

  def createWidget (self, event, spacer):

    (x, y) = event.interface.screenSize ()
    self.__window = curses.newpad (y, x)

    self.setTitle (event.object.title)

  # ---------------------------------------------------------------------------
  # Set form title
  # ---------------------------------------------------------------------------

  def setTitle (self, title):

    (y, x) = self.__window.getmaxyx ()

    text = ' ' * ((x - len (title)) / 2) + title

    self.__window.bkgdset (' ', self._uiDriver.attr ['title'])
    self.__window.addstr (0, 0, o(text))
    self.__window.clrtoeol ()
    self.__window.refresh (0, 0, 0, 0, 0, x)

  # ---------------------------------------------------------------------------
  # Set status bar
  # ---------------------------------------------------------------------------

  def _setStatusBar (self, tip, status, insert, curRec, maxRec, curPage,
                     maxPage):

    # Gets called with incomplete parameters, so we always have to remember the
    # old values for missing parameters
    if tip     is not None: self.__tip     = tip
    if status  is not None: self.__status  = status
    if insert  is not None: self.__insert  = insert
    if curRec  is not None: self.__curRec  = curRec
    if maxRec  is not None: self.__maxRec  = maxRec
    if curPage is not None: self.__curPage = curPage
    if maxPage is not None: self.__maxPage = maxPage

    self.__updateStatusBar ()

  # ---------------------------------------------------------------------------
  # Output a message on the status bar
  # ---------------------------------------------------------------------------

  def statusMessage (self, message):

    if message:
      (y, x) = self.__window.getmaxyx ()
      self.__window.bkgdset (' ', self._uiDriver.attr ['status'])
      self.__window.addstr (y - 2, 0, message)
      self.__window.clrtoeol ()
      self.__window.refresh (y - 2, 0, y - 2, 0, y - 2, x)
    else:
      self.__updateStatusBar ()

  # ---------------------------------------------------------------------------
  # Go to a specific page
  # ---------------------------------------------------------------------------

  def gotoPage (self, page):

    self.__currentPage = page
    self.__updatePageList ()

  # ---------------------------------------------------------------------------
  # Add a page to the form
  # ---------------------------------------------------------------------------

  def addPage (self, page, caption):
    self.__pages.append ((page, caption))
    if not self.__currentPage:
      self.__currentPage = page
    self.__updatePageList ()

  # ---------------------------------------------------------------------------
  # Update screen and wait for user input
  # ---------------------------------------------------------------------------

  def wait (self):
    return self.__currentPage.wait ()

  # ---------------------------------------------------------------------------
  # Get free area in the window
  # ---------------------------------------------------------------------------

  def getCanvas (self):

    (y, x) = self.__window.getmaxyx ()
    return (0, 2, x, y - 2)

  # ---------------------------------------------------------------------------
  # Update page list
  # ---------------------------------------------------------------------------

  def __updatePageList (self):

    self.__window.bkgdset (' ', self._uiDriver.attr ['page'])
    self.__window.move (1, 0)

    for (page, caption) in self.__pages:
      if page == self.__currentPage:
        self.__window.addstr ('[' + o(caption) + ']',
                              self._uiDriver.attr ['currentpage'])
      else:
        self.__window.addstr ('[' + o(caption) + ']')
      # self.__window.addstr (' ')

    self.__window.clrtoeol ()

    (y, x) = self.__window.getmaxyx ()
    self.__window.refresh (1, 0, 1, 0, 1, x)

  # ---------------------------------------------------------------------------
  # Update status bar
  # ---------------------------------------------------------------------------

  def __updateStatusBar (self):

    (y, x) = self.__window.getmaxyx ()
    self.__window.bkgdset (' ', self._uiDriver.attr ['status'])

    tip = ('%-' + str (x - 25) + 's') % self.__tip

    recstr = '%d/%d' % (self.__curRec, self.__maxRec)
    pagestr = '%d/%d' % (self.__curPage, self.__maxPage)

    self.__window.addstr (y - 2, 0,      '%s'   % o(tip))
    self.__window.addstr (y - 2, x - 24, '%-4s' % o(self.__status))
    self.__window.addstr (y - 2, x - 19, '%-3s' % o(self.__insert))
    self.__window.addstr (y - 2, x - 15, '%-9s' % recstr)
    self.__window.addstr (y - 2, x -  5, '%-5s' % pagestr)

    self.__window.addch (y - 2, x - 25, curses.ACS_VLINE)
    self.__window.addch (y - 2, x - 20, curses.ACS_VLINE)
    self.__window.addch (y - 2, x - 16, curses.ACS_VLINE)
    self.__window.addch (y - 2, x -  6, curses.ACS_VLINE)

    self.__window.bkgdset (' ', self._uiDriver.attr ['fkeys'])

    self.__window.addstr (y-1, 0, o(self._uiDriver.getFunctionKeyLine ()))
    self.__window.clrtoeol ()

    self.__window.refresh (y - 2, 0, y - 2, 0, y, x)

# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass'  : UIForm,
  'provides'   : 'GFForm',
  'container'  : 1,
  }
