#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# UIwxpython.py
#
# DESCRIPTION:
# A wxPython based user interface driver for GNUe forms.
#
# NOTES:
#

import string
import os.path

from wxPython.wx import *

from gnue.common.datasources import GLoginHandler
from gnue.common.apps import GConfig, i18n
from GFwxApp import *
from common import wxEncode, wxDecode

# Shortcut
images_dir = GConfig.getInstalledBase('forms_images','common_images') + '/'


#####################################################################
##
## Login Support
##
## Everything below this point is in support of the wx drivers
## UILoginHandler
##  
#####################################################################

#
# UILoginHandler
#
class UILoginHandler(GLoginHandler.LoginHandler):
  def __init__(self):
    self._wxapp = getWxApp()
    self.dlg = None

  # Hack for McMillan packaging on win32
  def getDummyLogin(self):
    self.dlg = wxDialog(NULL, -1,'dummy')
    self.destroyLoginDialog()
    
  # ---------------------------------------------------------------------------
  # Get input for all fields listed in loginData
  # ---------------------------------------------------------------------------

  def getLogin (self, loginData, errortext = None):
    """
    """

    if len (loginData [1]):
      loginMesg = wxEncode (u_('Login required for\n"%s"') % loginData [1])
    else:
      loginMesg = wxEncode (u_('Login required for %s') % loginData [0])

    self.dlg = wxDialog (NULL, -1,
        wxEncode (u_("GNU Enterprise: Login to %s") % loginData [0]))
    self.dlg.SetAutoLayout (True)

    if os.path.isabs (gConfigForms ('loginPNG')):
      imageFile = gConfigForms ('loginPNG')
    else:
      imageFile = images_dir + gConfigForms ('loginPNG')

    bmp = wxImage (imageFile, wxBITMAP_TYPE_PNG).ConvertToBitmap ()

    messageField = WrappedStaticText (self.dlg, -1, loginMesg, 300,
                                      style = wxALIGN_CENTER)

    self.textctrlList = []
    labelList = []

    dlgWidth  = max (bmp.GetWidth (), messageField.GetSize ().GetWidth () + 20)
    dlgHeight = bmp.GetHeight () + messageField.GetSize ().GetHeight () + 80

    xSpacing        = 0
    ySpacing        = 0
    fieldLabelWidth = 0

    for prompt in loginData [2]:
      s = wxStaticText (self.dlg, -1, wxEncode ('%s:' % prompt [1]))
      
      labelList.append (s)
      if prompt[2]:
        t = wxTextCtrl (self.dlg, -1, "", wxPoint (1, 1), wxSize (150, 20),
                       style = wxTE_PASSWORD | wxTE_PROCESS_ENTER)
      else:
        t = wxTextCtrl (self.dlg, -1, "", wxPoint (1, 1), wxSize (150, 20),
                       style = wxTE_PROCESS_ENTER)

      myID = len (self.textctrlList)
      self.textctrlList.append (t)
      EVT_CHAR (t, LoginFieldHandler (self, myID).loginFieldEventTrap)

      fieldLabelWidth = max (fieldLabelWidth,
                s.GetSize ().GetWidth () + t.GetSize ().GetWidth () + 10)

      dlgWidth = max (dlgWidth, \
                     s.GetSize ().GetWidth () + t.GetSize ().GetWidth () + 20)

      xSpacing = max (xSpacing, s.GetSize ().GetWidth ())
      ySpacing = max (ySpacing, s.GetSize ().GetHeight ())
      ySpacing = max (ySpacing, t.GetSize ().GetHeight ())

    loginId  = wxNewId ()
    cancelId = wxNewId ()

    loginButton  = wxButton (self.dlg, loginId, _('Login'))
    cancelButton = wxButton (self.dlg, cancelId, _('Cancel'))

    EVT_BUTTON (self.dlg, loginId,  self.loginButtonEventTrap)
    EVT_BUTTON (self.dlg, cancelId, self.loginCancelEventTrap)

    dlgWidth = max (dlgWidth, loginButton.GetSize ().GetWidth () +
                    cancelButton.GetSize ().GetWidth () + 6) + 20

    dlgHeight += max (loginButton.GetSize ().GetHeight (),
                      cancelButton.GetSize ().GetHeight ()) - 6

    if errortext:
      errorField = WrappedStaticText (self.dlg, -1, wxEncode (errortext),
                                      300, style = wxALIGN_CENTER)
      errorField.SetForegroundColour (wxColour (223, 0, 0))

      dlgWidth = max (dlgWidth, errorField.GetSize ().width + 10)
      dlgHeight += errorField.GetSize ().height + 6


    firstY = bmp.GetHeight () + messageField.GetSize ().GetHeight () + 50
    lastY = firstY
    xSpacing += 10	# Add whitespace between widgets
    ySpacing += 6	# Add whitespace between widgets
    xPos = dlgWidth / 2 - fieldLabelWidth / 2

    # Move the fields and labels into position
    for i in range (0, len (self.textctrlList)):
      dlgHeight = dlgHeight + ySpacing
      labelList [i].SetPosition (wxPoint (xPos, lastY))
      self.textctrlList[i].SetPosition (wxPoint (xPos + xSpacing, lastY))
      lastY = lastY + ySpacing

    if errortext:
      errorField.SetPosition (
        wxPoint (dlgWidth / 2 - errorField.GetSize ().width / 2, lastY + 3))

    # Set the focus to the first text entry field
    self.textctrlList [0].SetFocus ()

    # Create and position the logo
    wxStaticBitmap (self.dlg, -1, bmp,
                   wxPoint ( (dlgWidth - bmp.GetWidth ()) / 2, 12),
                   wxSize (bmp.GetWidth (), bmp.GetHeight ()))

    # Move the various widgets into position
    messageField.SetPosition (
      wxPoint (dlgWidth / 2 - messageField.GetSize ().GetWidth () / 2,
              30 + bmp.GetHeight ()))

    cancelButton.SetPosition (
      wxPoint (dlgWidth - 10 - cancelButton.GetSize ().GetWidth (),
              dlgHeight - 10 - max (loginButton.GetSize ().GetHeight (),
                                    cancelButton.GetSize ().GetHeight ())))
    loginButton.SetPosition (
      wxPoint (dlgWidth - 16 - cancelButton.GetSize ().GetWidth () - \
              loginButton.GetSize ().GetWidth (),
              dlgHeight - 10 - max (loginButton.GetSize ().GetHeight (),
                                    cancelButton.GetSize ().GetHeight ())))

    self.loginButton = loginButton

    self.dlg.SetSize (wxSize (dlgWidth, dlgHeight))

    self.dlg.Refresh ()
    self.dlg.Fit ()
    self.dlg.Raise ()
    self.dlg.CenterOnScreen ()

    # If user cancels, this will be set to 0
    self._completed = 0
    self.dlg.ShowModal ()

    if not self._completed:
      raise GLoginHandler.UserCanceledLogin

    rv = {}
    for i in range (0, len (loginData [2])):
      rv [loginData [2] [i] [0]] = wxDecode (self.textctrlList [i].GetValue ())

    return rv


  #
  # Login is completed, for whatever reason
  #
  def loginCompleted(self, successful):
    self._completed = successful
    self.dlg.EndModal(1)

  #
  # Called when user clicks "login"
  #
  def loginButtonEventTrap(self, event):
    self.loginCompleted(1)

  #
  # Called when user clicks "cancel"
  #
  def loginCancelEventTrap(self, event):
    self.loginCompleted(0)

  #
  # TODO: This is a hack required because windows
  # TODO: seems to have issues with wxWidgets dialogs
  #
  def destroyLoginDialog(self):
    self.dlg.Destroy()

#
# LoginFieldHandler
#
# Used by the login handler
# enables the user to press return and have it jump to the next box
#
class LoginFieldHandler:
  def __init__(self, app, seq):
    self.app = app
    self.seq = seq

  def loginFieldEventTrap(self, event):
     if event.KeyCode() in (WXK_RETURN, WXK_TAB):
       if self.seq < len(self.app.textctrlList) - 1:
         self.app.textctrlList[self.seq+1].SetFocus()
       else:
         if event.KeyCode() == WXK_TAB:
           if event.ShiftDown():
             self.app.textctrlList[self.seq-1].SetFocus()
           else:
             self.app.loginButton.SetFocus()
         else:
           self.app.loginCompleted(1)
     else:
      event.Skip()



class WrappedStaticText(wxStaticText):
  def __init__(self, parent, id, label, width, *args, **params):
    wxStaticText.__init__(self, parent, id, "bah!", *args, **params)


    textSoFar = ""
    thisLine = ""
    for part in string.split(label,'\n'):
      for word in string.split(part):
        self.SetLabel(thisLine + word)
        if self.GetSize().width > width:
          textSoFar += thisLine + " \n"
          thisLine = word + " "
        else:
          thisLine += word + " "

      textSoFar += thisLine + " \n"
      thisLine = ""

    if len(textSoFar):
      self.SetLabel(string.replace(textSoFar,' \n','\n')[:-1])
    else:
      self.SetLabel("")

