/*
   interpreter.c: file contains read-eval-print loop and
   supportive functions

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <guile/gh.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <ftplib.h>
#include <libgen.h>

#include <config.h>
#include "gnuyahoo.h"
#include "messenger.h"
#include "interpreter.h"
#include "yahoo-wrapper.h"
#include "gy-utils.h"


char *commands[] = { "?ftp-ls", "?ftp-delete", "?ftp-send", "?send", "?who", "?add", "?remove", "?help", "?status",
		     "?refresh", "?toggle", "?bell", "?eval", "?load", "?quit", GY_ROBOT, NULL
};

SCM dynamic_commands = SCM_UNSPECIFIED;

unsigned char toggle_bell_state = 0;	               // bell - OFF / ON
unsigned char toggle_who_state = 0;	               // who - SHOW-ALL / ONLINE-ONLY
unsigned char toggle_session_mode = 0;	               // session - VANILLA / AUTO-INSERT
unsigned char toggle_status_mode = 0;	               // show buddy status notifications HIDE / SHOW
int current_status = -1;	                       // -1 means not set
char *current_status_custom_message = (char *) NULL;   // NULL means not set

unsigned char
get_bell_state (void)
{
  return toggle_bell_state;
}

unsigned char
get_who_state (void)
{
  return toggle_who_state;
}

unsigned char
get_session_mode (void)
{
  return toggle_session_mode;
}

unsigned char
get_status_mode (void)
{
  return toggle_status_mode;
}

void
toggle_bell (void)
{
  toggle_bell_state = ~toggle_bell_state;
}

void
toggle_who (void)
{
  toggle_who_state = ~toggle_who_state;
}

void
toggle_session (void)
{
  toggle_session_mode = ~toggle_session_mode;
}

void
toggle_status (void)
{
  toggle_status_mode = ~toggle_status_mode;
}

int
get_current_status (void)
{
  return current_status;
}

void
set_current_status (int current_status_value)
{
  current_status = current_status_value;
}

char *
get_current_status_custom_message (void)
{
  return current_status_custom_message;
}

void
set_current_status_custom_message (char *current_status_custom_message_value)
{
  current_status_custom_message = current_status_custom_message_value;
}

void
register_command (SCM command)
{
  if (gh_list_p (dynamic_commands) != 1)
    {// not a list
      dynamic_commands = gh_list (command, SCM_UNDEFINED);
      gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);

    }
  else
    {
      dynamic_commands =
        gh_append2 (dynamic_commands,
                    gh_list (command, SCM_UNDEFINED));
      gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);
    }
}

void
unregister_command (SCM command)
{
  SCM tmp_scm;
  char *dynamic_cmd = NULL;
  char *dynamic_command = NULL;
  int dynamic_command_index = 0;
  int dynamic_commands_count = 0;

  if (gh_list_p (dynamic_commands) != 1)
    return;
  
  dynamic_command = gh_scm2newstr (command, NULL);

  if (dynamic_command == NULL)
    return;

  dynamic_commands_count = gh_length (dynamic_commands);

  while (dynamic_command_index < dynamic_commands_count)
    {
      tmp_scm = gh_list_ref (dynamic_commands, gh_ulong2scm (dynamic_command_index++));
      dynamic_cmd = gh_scm2newstr (gh_list_ref (tmp_scm,
						 gh_ulong2scm (0)),
				    NULL);
      if (dynamic_cmd && strcasecmp (dynamic_cmd, dynamic_command) == 0)
	{
	  dynamic_commands = scm_delete (tmp_scm, dynamic_commands);
	  gh_define (EX_DYNAMIC_COMMANDS, dynamic_commands);
	  return;
	}
    }
}

char *
command_generator (const char *text, int stat)
{
  static int i, len;
  static int buddy_index, toggle_index;
  static unsigned long dynamic_cmd_index;
  unsigned long dynamic_commands_count;
  char *dynamic_command;
  char *name;
  struct buddy *temp_buddy;

  if (!stat)
    {
      len = strlen (text);
      i = 0;
      buddy_index   = 0;
      toggle_index  = 0;
      dynamic_cmd_index = 0;
    }

  // if your command takes buddy_name as argument then you
  // will have to add ur command in this check list
  if ((strncmp (rl_line_buffer, "?remove ", 8) == 0)
      || (strncmp (rl_line_buffer, "?send ", 6) == 0))
    {
      while ((temp_buddy = (struct buddy *) get_buddy_by_index (i)))
	{
	  if (!temp_buddy)
	    return NULL;

	  name = temp_buddy->buddy_id;
	  i++;

	  if (strncasecmp (name, text, len) == 0)
	    return strdup (name);
	}
      return NULL;
    }

  // ?toggle argument tabs
  if (strncmp (rl_line_buffer, "?toggle ", 8) == 0)
    {
      switch (toggle_index)
	{
	case 0:
	  toggle_index++;
	  if (strncasecmp ("who", text, len) == 0)
	    return strdup ("who");
	case 1:
	  toggle_index++;
	  if (strncasecmp ("bell", text, len) == 0)
	    return strdup ("bell");
	case 2:
	  toggle_index++;
	  if (strncasecmp ("session", text, len) == 0)
	    return strdup ("session");
	case 3:
	  toggle_index++;
	  if (strncasecmp ("status", text, len) == 0)
	    return strdup ("status");
	default:
	  return NULL;
	}
    }

  // for dynamic commands
  if (gh_list_p (dynamic_commands))
    {
      dynamic_commands_count = gh_length (dynamic_commands);
      while (dynamic_cmd_index < dynamic_commands_count)
	{
	  dynamic_command = 
	    gh_scm2newstr (gh_list_ref (gh_list_ref (dynamic_commands, 
					gh_ulong2scm (dynamic_cmd_index++)),
					gh_ulong2scm (0)),
			   NULL);
	  if (dynamic_command && strncasecmp (dynamic_command, text, len) == 0)
	    return strdup (dynamic_command);
	}
    }
  
  // for normal command
  while ((name = commands[i]))
    {
      i++;
      if (strncasecmp (name, text, len) == 0)
	return (char *) strdup (name);
    }

  while ((temp_buddy = (struct buddy *) get_buddy_by_index (buddy_index)))
    {
      if (!temp_buddy)
	return (char *) NULL;

      if (!temp_buddy->buddy_id)
	return (char *) NULL;

      name = temp_buddy->buddy_id;
      buddy_index++;

      if (strncasecmp (name, text, len) == 0)
	return (char *) strdup (name);
    }
  return (char *) NULL;
}

char **
complete_text (char *text, int start, int end)
{
  char **matches;
  matches = (char **) NULL;

#if defined (HAVE_RL_COMPLETION_MATCHES)
  matches = (char **) rl_completion_matches (text, command_generator);
#else
#if defined (HAVE_COMPLETION_MATCHES)
  matches = (char **) completion_matches (text, command_generator);
#else
  /* don't worry, 'else' will never happen. configure script exits if both
     functions are missing */
  assert (0);
#endif
#endif
  return matches;
}

void
syntax_error (void)
{
  printf ("syntax error: type \"?help [command]\" for more info\n");
}

void
interpreter (char *line)
{
  char *command, *dynamic_command;
  int i=0;

  set_auto_insert_mode (1);

  if (!line)
    {
      printf ("\n");
      return;
    }

  line = stripwhite (line);

  if (strlen (line) == 0)
    {
      return;
    }

  add_history (line);

  command = get_token (&line);

  if (line)
    line = stripwhite (line);

  if (strcasecmp (command, "?quit") == 0)
    {
      set_quit_flag (1);
      return;
    }

  if (strcasecmp (command, "?help") == 0)
    {
      // help arguments are optional
      command_show_help (line);
      return;
    }

  if (strcasecmp (command, "?status") == 0)
    {
      command_status (line);
      return;
    }

  if (strcasecmp (command, "?refresh") == 0)
    {
      reset_all_status ();
      yahoo_cmd_user_status (get_yahoo_context ());
      return;
    }

  if (strcasecmp (command, "?add") == 0)
    {
      if (line && *line)
	command_buddy_add (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?remove") == 0)
    {
      if (line && *line)
	command_buddy_remove (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?who") == 0)
    {
      display_buddy_list ();
      return;
    }

  if (strcasecmp (command, "?send") == 0)
    {
      if (line && *line)
	command_send_message (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?eval") == 0)
    {
      if (line && *line)
	command_eval_scheme_str (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?load") == 0)
    {
      if (line && *line)
	command_load_scheme_file (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?toggle") == 0)
    {
      if (line && *line)
	command_toggle (line);
      else
	syntax_error ();
      return;
    }

  if (strcasecmp (command, "?bell") == 0)
    {
      command_bell ();
      return;
    }

  if (gh_list_p (dynamic_commands) == 1)
    {
      for (i=0; i < gh_length (dynamic_commands); i++)
	{
	  dynamic_command = gh_scm2newstr (gh_list_ref (gh_list_ref (dynamic_commands, 
								     gh_ulong2scm (i)), 
							gh_ulong2scm (0)),
					   NULL);
	  if (dynamic_command && strcasecmp (dynamic_command, command) == 0)
	    {
	      command_dynamic_commands (dynamic_command, line);
	      return;
	    }
	}
    }
  
  if (strcasecmp (command, "?ftp-send") == 0)
    {
      if (line && *line)
	{
	  
	  command_ftp_send(line);
	}
      else
	syntax_error();
      return;
    }
  
  
  if (strcasecmp (command, "?ftp-delete") == 0)
    {
      if (line && *line)
	{
	  
	  command_ftp_delete(line);
	}
      else
	syntax_error();
      return;
    }
 
  if (strcasecmp (command, "?ftp-ls") == 0)
    {
      command_ftp_list(line);
      return;
    }
  
  /* you can add your new command like this
     if (strcasecmp (command, "?my_command") == 0)
     {
     if (line && *line)
     {
     // implement your command and also add a prototype declaration
     // in the header file
     command_my_command(line);
     }
     else
     syntax_error();
     return;
     }
  */
  
  //default handler
  if (line && *line)
    command_default_handler (command, line);
  //else
  //  syntax_error ();

  return;
}

void
command_show_help (char *line)
{
  char *command;
  char *dynamic_command;
  int all = 0;

  command = get_token (&line);
  if (command == (char *) NULL)
    all = 1;
  
  if (all || (strcasecmp (command, "?ftp-ls") == 0))
    {
      printf ("?ftp-ls \n"
	      "\t- list the files in your geocities ftp account\n");
    }
  if (all || (strcasecmp (command, "?ftp-delete") == 0))
    {
      printf ("?ftp-delete filename\n"
	      "\t- delete a file with \"filename\" from your geocities ftp account\n");
    }
  if (all || (strcasecmp (command, "?ftp-send") == 0))
    {
      printf ("?ftp-send username filepath\n"
	      "\t- send a \"file\" to \"username\"\n");
    }
  if (all || (strcasecmp (command, "?send") == 0))
    {
      printf ("?send username message\n"
	      "\t- send \"message\" to \"username\"\n");
    }
  if (all || (strcasecmp (command, "?who") == 0))
    {
      printf ("?who\n"
	      "\t- display buddy list as well as online or offline\n");
    }
  if (all || (strcasecmp (command, "?add") == 0))
    {
      printf ("?add buddy_name [group_name] [message to buddy]\n"
	      "\t- add buddy to friends list\n");
    }
  if (all || (strcasecmp (command, "?remove") == 0))
    {
      printf ("?remove buddy_name [message]\n"
	      "\t- remove buddy from friends list\n");
    }
  if (all || (strcasecmp (command, "?status") == 0))
    {
      printf ("?status [status] [message]\n"
	      "\t- manipulte status information\n");
      if (!all)
	{
	  // show complete status list
	  int i;
	  printf ("status number list\n");
	  for (i = 0; i <= YAHOO_STATUS_IDLE; i++)
	    {
	      if (yahoo_get_status_string (i))
		printf ("%5d: %s\n", i, yahoo_get_status_string (i));
	    }
	}
    }
  if (all || (strcasecmp (command, "?refresh") == 0))
    {
      printf ("?refresh\n" "\t- refresh userlist and status information\n");
    }
  if (all || (strcasecmp (command, "?eval") == 0))
    {
      printf ("?eval scheme-code\n" "\t- evaluates the scheme code\n");
    }
  if (all || (strcasecmp (command, "?load") == 0))
    {
      printf ("?load scheme-file\n"
	      "\t- loads and evaluates the scheme file\n");
    }
  if (all || (strcasecmp (command, "?toggle") == 0))
    {
      printf ("?toggle variable\n"
	      "\t- toggles the stage of variable to ON and OFF\n"
	      "\tVariables:\n"
	      "\t\tbell    - OFF / ON\n"
	      "\t\twho     - ONLINE-ONLY / SHOW-ALL\n"
	      "\t\tsession - VANILLA / AUTO-INSERT\n"
	      "\t\tstatus  - Buddy status notifications HIDE / SHOW\n");
    }
  if (all || (strcasecmp (command, "?bell") == 0))
    {
      printf ("?bell\n" "\t- toggles the bell ON and OFF\n");
    }
  if (all || (strcasecmp (command, "?quit") == 0))
    {
      printf ("?quit\n" "\t- logout of messenger and quit\n");
    }
  if (all || (strcasecmp (command, "?help") == 0))
    {
      printf ("?help [command]\n" "\t- help on commands\n");
    }
  /* adds help facility to your new command
     if (all || (strcasecmp (command, "?my_command") == 0))
     {
     printf("?my_command [optional argument]\n"
     "\t- my_command really freaks ;-)\n");
     }
  */
  if (gh_list_p (dynamic_commands) == 1)
    {
      int i;
      SCM tmp_scm;
      char *dynamic_syntax = NULL;

      for (i=0; i < gh_length (dynamic_commands); i++)
	{
	  dynamic_syntax = NULL;
	  tmp_scm = gh_list_ref (dynamic_commands, gh_ulong2scm (i));
	  dynamic_command = gh_scm2newstr (gh_list_ref (tmp_scm, gh_ulong2scm (0)),
					   NULL);
	  if (all || (strcasecmp (command, dynamic_command) == 0))
	    {
	      dynamic_syntax = gh_scm2newstr (gh_list_ref (tmp_scm, gh_ulong2scm (1)),
					      NULL);
	      if (dynamic_syntax != NULL)
		printf ("%s\n", dynamic_syntax);
	    }
	}
    }
  

}

void
command_status (char *line)
{
  char *current_status_str;
  int current_status;

  current_status_str = get_token (&line);
  if (current_status_str == (char *) NULL)
    {
      printf ("[%s] is currently [%s]\n",
	      get_default_login_id (),
	      (get_current_status () == YAHOO_STATUS_CUSTOM
	       && get_current_status_custom_message () != NULL
	      ? get_current_status_custom_message ()
	      : yahoo_get_status_string (get_current_status ())));
      return;
    }

  if (!sscanf (current_status_str, "%d", &current_status))
    {
      fprintf (stderr, "Invalid status number, see \"?help ?status\""
	       "for more info\n");
      return;
    }
  set_current_status (current_status);

  if (line)
    set_current_status_custom_message (line);
  else if (current_status == YAHOO_STATUS_CUSTOM)
    set_current_status_custom_message ("");

  if (yahoo_get_status_string (get_current_status ()))
    {
      // yes, current_status_str is a valid status
      if (get_current_status () == YAHOO_STATUS_AVAILABLE)
	{
	  yahoo_cmd_set_back_mode (get_yahoo_context (),
				   get_current_status (),
				   get_current_status_custom_message ());
	}
      else
	{
	  yahoo_cmd_set_away_mode (get_yahoo_context (),
				   get_current_status (),
				   get_current_status_custom_message ());
	}
      printf ("[%s] changing status to [%s]\n", get_default_login_id (),
	      (get_current_status () == YAHOO_STATUS_CUSTOM
	       ? get_current_status_custom_message ()
	       : yahoo_get_status_string (get_current_status ())));
    }
  else
    {
      fprintf (stderr, "Invalid status number, see \"?help ?status\""
	       "for more info\n");
      return;
    }
}

void
command_send_message (char *line)
{
  char *user_id;
  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }

  if (line && *line)
    send_message (user_id, line);
}

void
command_buddy_add (char *line)
{
  char *user_id;
  char *message;
  char *group;
  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  group = get_token (&line);
  if (group == (char *) NULL)
    {
      group = strdup (GY_DEFAULT_GROUP);
      if (!group)
	{
	  perror ("strdup");
	  exit (-1);
	}
    }

  if (line == (char *) NULL)
    {
      message = alloca (100);
      sprintf (message, "Hi %s this side !!", get_default_login_id ());
    }
  else
    {
      message = alloca (strlen (line + 2));
      strcpy (message, line);
    }

  yahoo_add_buddy (get_yahoo_context (), user_id,
		   get_default_login_id (), group, message);
  yahoo_cmd_user_status (get_yahoo_context ());
}

void
command_buddy_remove (char *line)
{
  char *user_id;
  char *message;
  char *group;
  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  group = (char *) get_buddy_group (user_id);

  if (group == NULL)
    {
      fprintf (stderr, "[%s] is not in your buddy list\n", user_id);
      return;
    }

  if (line == (char *) NULL)
    {
      message = alloca (20);
      strcpy (message, "sorry buddy");
    }
  else
    {
      message = alloca (strlen (line) + 1);
      strcpy (message, line);
    }
  yahoo_remove_buddy (get_yahoo_context (), user_id,
		      get_default_login_id (), group, message);
}

void
command_eval_scheme_str (char *line)
{
  gh_eval_str_with_stack_saving_handler (line);
  scm_force_output (scm_current_output_port ());
}

void
command_load_scheme_file (char *line)
{
  gh_eval_file_with_standard_handler (line);
  scm_force_output (scm_current_output_port ());
}

void
command_toggle (char *line)
{
  char *toggle_arg = NULL;

  toggle_arg = get_token (&line);
  if (toggle_arg == (char *) NULL)
    {
      fprintf (stderr,
	       "Invalid argument to \"?toggle\", see \"?help ?toggle\""
	       "for more info\n");
      return;
    }

  if (strcasecmp (toggle_arg, "bell") == 0)
    {
      command_bell ();
      return;
    }

  if (strcasecmp (toggle_arg, "who") == 0)
    {
      toggle_who ();
      if (get_who_state ())
	{
	  puts ("Who - [Online Only] mode");
	}
      else
	{
	  puts ("Who - [Show All] mode");
	}
      return;
    }
  if (strcasecmp (toggle_arg, "session") == 0)
    {
      toggle_session ();
      if (get_session_mode ())
	{
	  puts ("Session - [Auto Insert] mode");
	}
      else
	{
	  puts ("Session - [Vanilla] mode");
	}
      return;
    }

  /* status mode */
  if (strcasecmp (toggle_arg, "status") == 0)
    {
      toggle_status ();
      if (get_status_mode ())
	{
	  puts ("Buddy status notifications - [SHOW]");
	}
      else
	{
	  puts ("Buddy status notifications - [HIDE]");
	}
    }
}

void
command_bell ()
{
  toggle_bell ();

  if (get_bell_state ())
    {
      puts ("Bell sound  - [ON]");
#if defined (HAVE_RL_DING)
      rl_ding ();
#else
#if defined (HAVE_DING)
      ding ();
#else
      /* don't worry, 'else' will never happen. configure script exits if both
         functions are missing */
      assert (0);
#endif
#endif
    }
  else
    {
      puts ("Bell sound - [OFF]");
    }
}

void
command_dynamic_commands (char *dynamic_command, char *line)
{
  /*
    (and (defined? '?command) (?command '(arg1 arg2 art3 ...)))
  */
  char *dynamic_command_line;
  int length;
  char *empty_line = "";

  /*
    Why strlen (NULL) seqfaults!?!  --bala
  */
  if (!(line && *line))
    line = empty_line;

  length = strlen ("(and (defined? '") + strlen (dynamic_command)
    + strlen (") (") + strlen (dynamic_command) + strlen (" '(")
    + strlen (line) + strlen (")))") + 1;

  dynamic_command_line = (char *) malloc (length * sizeof (char));
  sprintf (dynamic_command_line, "(and (defined? '%s) (%s '(%s)))", 
	   dynamic_command, dynamic_command, line);
  command_eval_scheme_str (dynamic_command_line);
  free (dynamic_command_line);
}

void
command_ftp_send(char *line)
{
  char *user_id;
  char filename[256];
  char *filepath;
  netbuf *nControl;
  char server[] = "http://www.geocities.com";
  char ftpserver[] = "ftp.geocities.com";
  char ftp_current_dir[256];
  char *url;
  char *url_message;

  user_id = get_token (&line);
  if (user_id == (char *) NULL)
    {
      syntax_error ();
      return;
    }

  filepath = get_token (&line);
  if (filename == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  FtpInit();
  printf("Connecting to your geocities ftp account ...");
  if(!FtpConnect(ftpserver, &nControl))
    {
      printf("Failed to connect to your ftp geocities account! [%s]\n", FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  strcpy(filename, basename(filepath));
  if(strstr(filename, ".") == NULL)
    {
      printf("No file extension found for [%s]!\n", filename);
      printf("please add a valid file extension to the uploading file...[%s] and retry.\n", filename);
      return;
    };
  
  printf("Logging in to geocities ...");

  if(!FtpLogin(get_default_login_id(), get_default_password(), nControl))
    {
      printf("Unable to login to your geocities account! [%s]\n",  FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  if(!FtpPwd(ftp_current_dir, 256, nControl))
    {
      printf("Unable to get the current directory in your geocities account! [%s]\n", FtpLastResponse(nControl));
      printf("quiting...\n");
      FtpQuit(nControl);
      return;
    }

  url = (char*) malloc( (strlen(server) + strlen(ftp_current_dir) + strlen(filename)) * sizeof(char));

  strcpy(url, server);
  strcat(url, ftp_current_dir);
  strcat(url, filename);

  printf("Url->[%s]\n", url);
  
  printf("sending ...\n");
  if(!FtpPut(filepath, filename, FTPLIB_IMAGE, nControl))
    {
      printf("Unable to send file to your geocities account! [%s]\n", FtpLastResponse(nControl));
      printf("quiting...\n");
      FtpQuit(nControl);
      return;
    }

  FtpQuit(nControl);
  printf("done.\n");

  url_message = (char*) malloc((strlen(url) + 30) * sizeof(char));
  strcpy(url_message, "Please download file from ");
  strcat(url_message, url);
  send_message(user_id,  url_message);

  free(url);
  free(url_message);

  return;
}

void
command_ftp_delete(char *line)
{
  char *filename;
  netbuf *nControl;
  char ftpserver[] = "ftp.geocities.com";

  filename = get_token (&line);
  if (filename == (char *) NULL)
    {
      syntax_error ();
      return;
    }
  
  FtpInit();
  printf("Connecting to your geocities ftp account ...");
  if(!FtpConnect(ftpserver, &nControl))
    {
      printf("Failed to connect to your geocities account! [%s]\n", FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  if(strstr(filename, ".") == NULL)
    {
      printf("No file extension found for [%s]!\n", filename);
      printf("please specify a valid filename and retry.\n");
      return;
    };
  
  printf("Logging into geocities ...");

  if(!FtpLogin(get_default_login_id(), get_default_password(), nControl))
    {
      printf("Unable to login to your geocities account! [%s]\n",  FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  printf("deleting file [%s] from your geocities ftp account ...\n", filename);
  if(!FtpDelete(filename, nControl))
    {
      printf("Unable to delete file [%s] from your geocities account! [%s]\n", filename, FtpLastResponse(nControl));
      printf("quiting...\n");
      FtpQuit(nControl);
      return;
    }

  FtpQuit(nControl);
  printf("done.\n");

  return;
}

void
command_ftp_list(char *line)
{
  netbuf *nControl;
  char ftpserver[] = "ftp.geocities.com";

  FtpInit();
  printf("Connecting to your geocities ftp account ...");
  if(!FtpConnect(ftpserver, &nControl))
    {
      printf("Failed to connect to your geocities account! [%s]\n", FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  printf("Logging in to geocities ...");

  if(!FtpLogin(get_default_login_id(), get_default_password(), nControl))
    {
      printf("Unable to login to your geocities account! [%s]\n",  FtpLastResponse(nControl));
      return;
    }
  printf("done.\n");
  printf("listing files in your geocities ftp account ...\n");
  if(!FtpNlst("/dev/tty", "-alh", nControl))
    {
      printf("Unable to list files in your geocities account! [%s]\n", FtpLastResponse(nControl));
      printf("quiting...\n");
      FtpQuit(nControl);
      return;
    }

  FtpQuit(nControl);
  printf("done.\n");

  return;
}

void
command_default_handler (char *command, char *line)
{
  // command is to_username and line is message
  send_message (command, line);
}
