/*
   messenger.c : main file and heart for the messenger

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <guile/gh.h>
#include <config.h>

#include "interpreter.h"
#include "gy-utils.h"
#include "gnuyahoo.h"
#include "yahoo-wrapper.h"
#include "extension.h"

extern int errno;
extern int rl_point;
int auto_insert_mode = 0;
int quit_flag = 0;
struct yahoo_context *context;
struct yahoo_options options;

static char gnuyahoo_logo[] =
  "  +-----------------------------------------------------+ \n"
  "  |                                                     |+\n"
  "  |                 ...   ...                           ||\n"
  "  |               ..         ..                         ||\n"
  "  |              ..          ..                         ||\n"
  "  |               ...~~`'~~...                          ||\n"
  "  |                '(O##(O).***                         ||\n"
  "  |                  |##......*******......-_           ||\n"
  "  |                  |##......................          ||\n"
  "  |                  ##./ `....GNU Yahoo........        ||\n"
  "  |                 (--)  `.................   ..       ||\n"
  "  |                  ##   `.................     **     ||\n"
  "  |                        .............. .       **    ||\n"
  "  | Thank You for choosing .....    v .. ..        `*   ||\n"
  "  | Free Software          `. ..     ......             ||\n"
  "  | Please tell your        ....      .. ..             ||\n"
  "  | friends about GNU       ....      .....             ||\n"
  "  | <www.gnu.org>           WW WW      WW WW            ||\n"
  "  |                                                     ||\n"
  "  +-----------------------------------------------------+|\n"
  "   +-----------------------------------------------------+\n";

int
get_auto_insert_mode (void)
{
  return auto_insert_mode;
}

void
set_auto_insert_mode (int auto_insert_mode_value)
{
  auto_insert_mode = auto_insert_mode_value;
}


int
get_quit_flag (void)
{
  return quit_flag;
}

void
set_quit_flag (int quit_flag_value)
{
  quit_flag = quit_flag_value;
}

struct yahoo_context *
get_yahoo_context (void)
{
  return context;
}

/* handle SIGTERM or like signals and do a proper logout */
void
signal_handler (int signum)
{
  /* set the quit flag for proper logout and freeing of resources */
  fprintf (stderr, "signal [%d] caught!! better logout safely\n", signum);
  set_quit_flag (1);
}

void
set_yahoo_context (struct yahoo_context *context_value)
{
  context = context_value;
}

// heart of gnuyahoo
void
main_loop ()
{
  fd_set fds;
  quit_flag = 0;

  /* register quit handler for SIGTERM */
  if (signal (SIGTERM, signal_handler) == SIG_ERR)
    perror ("signal(SIGTERM)");

  while (!quit_flag)
    {
      FD_ZERO (&fds);
      FD_SET (fileno (stdin), &fds);
      FD_SET (context->sockfd, &fds);

      while (1)
	{
	  errno = 0;


	  // toggle_session_mode - AUTO-INSEVRT / VANILLA
	  if (get_session_mode ())
	    {
	      if (auto_insert_mode == 1)
		{
		  if (get_current_target_buddy () != NULL)
		    {
		      rl_insert_text (get_current_target_buddy ());
		      rl_insert_text (" ");
		      rl_redisplay ();
		    }
		}

	      auto_insert_mode = 0;
	    }

	  if (select (FD_SETSIZE, &fds, NULL, NULL, NULL) < 0)
	    {
	      if (quit_flag)
		{
		  /* go down and see, there is another check for
		     breaking the main loop */
		  break;
		}
	      if (errno == EINTR)
		{
		  /* resuming from interrupt, so not an error situatuon,
		     this generally happens when you suspend your
		     messenger with "C-z" and then "fg". This is allowed "
		   */
		  rl_reset_line_state ();
		  rl_forced_update_display ();

		  continue;
		}
	      perror ("select()");
	      exit (EXIT_FAILURE);
	    }
	  else
	    break;
	}

      if (quit_flag)
	{
	  /* this is second check, select might have returned after
	     interrupt due to SIGTERM signal */
	  break;
	}

      if (FD_ISSET (fileno (stdin), &fds))
	rl_callback_read_char ();

      if (FD_ISSET (context->sockfd, &fds))
	{
	  if (handle_yahoo_message (get_default_login_id ()) != 0)
	    quit_flag = 1;
	}
    }
}


// messenger starts from here
void
messenger_main (int argc, char **argv)
{
  char *logon_mode = NULL;
  char *login_id = get_default_login_id ();
  char *password = get_default_password ();

  if (!login_id)
    {
      login_id = (char *) malloc (sizeof (char) * USERNAME_LENGTH);
      if (!login_id)
	{
	  fprintf (stderr, "Insufficient memory\n");
	  exit (EXIT_SUCCESS);
	}

      printf ("Yahoo ID: ");
      scanf ("%s", login_id);
      set_default_login_id (login_id);
    }

  if (!password)
    {
      password = (char *) strdup (getpass ("Password: "));
      set_default_password (password);

      if (!password)
	{
	  fprintf (stderr, "Insufficient memory\n");
	  exit (EXIT_SUCCESS);
	}
    }

  if (strlen (password) > PASSWORD_LENGTH)
    {
      fprintf (stderr, "gnuyahoo: error, password exceeds maximum size!!\n");
      exit (EXIT_FAILURE);
    }

  options.connect_mode = YAHOO_CONNECT_NORMAL;

  fprintf (stderr, "Initializing Yahoo! connection ... ");
  if (!(context = yahoo_init (login_id, password, &options)))
    {
      fprintf (stderr, "failed\n");
      exit (EXIT_FAILURE);
    }
  fprintf (stderr, "done\n");

  fprintf (stderr, "Connecting to Yahoo! ... ");
  if (!yahoo_connect (context))
    {
      fprintf (stderr, "failed\n");
      exit (EXIT_FAILURE);
    }
  fprintf (stderr, "done\n");

  if (get_current_status () == -1)
    set_current_status (GY_DEFAULT_STATUS);

  fprintf (stderr, "[%s] logging in [%s] mode ... ",
	   login_id, yahoo_get_status_string (get_current_status ()));
  if (!yahoo_get_config (context))
    {
      fprintf (stderr, "failed in get_config\n");
      exit (EXIT_FAILURE);
    }
  if (!yahoo_cmd_logon (context, get_current_status ()))
    {
      fprintf (stderr, "failed\n");
      exit (EXIT_FAILURE);
    }
  fprintf (stderr, "done\n");

  create_buddy_list ();

  rl_callback_handler_install (get_default_prompt (), interpreter);
  rl_attempted_completion_function = (CPPFunction *) complete_text;

  // add GNU_ROBOT as buddy by default for every user :)
  if (get_setup_mode () == 1)
    yahoo_add_buddy (get_yahoo_context (), GY_ROBOT,
		     get_default_login_id (), GY_ROBOT_GROUP,
		     "New " PACKAGE " user!!");

  /* call "gy-login-post-hook" when gnuyahoo is ready to go
   */
  set_hook_return (0);
  scm_run_hook (get_login_post_hook (), gh_list (SCM_UNDEFINED));
  if (get_hook_return () == 1)
    return;

  main_loop ();

  fprintf (stderr, "\nLogging out ... ");
  yahoo_cmd_logoff (context);
  fprintf (stderr, "done\n");

  puts (gnuyahoo_logo);

  if (login_id)
    free (login_id);
  if (password)
    free (password);
  if (logon_mode)
    free (logon_mode);

  set_terminal_attributes ();
  exit (EXIT_SUCCESS);
}
