/*
   yahoo_wrapper.c: higher level wrapper to libyahoo functions


   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* READ ME FIRST:
   All the process_*  and handle_yahoo_message functions should use
   PRINTF_MESSAGE only to print any message to screen. This will
   ensure proper insertion of message into readline buffer.
*/

#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <guile/gh.h>
#include <readline/readline.h>

#include "gnuyahoo.h"
#include "interpreter.h"
#include "yahoo-wrapper.h"
#include "messenger.h"
#include "gy-utils.h"
#include "extension.h"
#include <config.h>

char *message;
char *message_to;
char *received_message;
char *received_message_from;
char *current_target_buddy;

void
set_message (char *message_value)
{
  message = message_value;
}

char *
get_message (void)
{
  return message;
}

void
set_message_to (char *message_to_value)
{
  message_to = message_to_value;
}

char *
get_message_to (void)
{
  return message_to;
}

void
set_received_message (char *received_message_value)
{
  received_message = received_message_value;
}

char *
get_received_message (void)
{
  return received_message;
}

void
set_received_message_from (char *received_message_from_value)
{
  received_message_from = received_message_from_value;
}

char *
get_received_message_from (void)
{
  return received_message_from;
}

void
set_current_target_buddy (char *current_target_buddy_value)
{
  current_target_buddy = current_target_buddy_value;
}

char *
get_current_target_buddy (void)
{
  return current_target_buddy;
}


struct buddy *buddy_list = (struct buddy *) NULL;

void
send_message (char *to, char *message)
{
  /* hook evaluation */
  set_message_to (to);
  set_message (message);
  set_current_target_buddy (to);	// AUTO-INSERT-MODE

  set_hook_return (0);
  scm_run_hook (get_message_send_hook (),
		gh_list (gh_str02scm (get_message_to ()),
			 gh_str02scm (get_message ()), SCM_UNDEFINED));

  if (get_hook_return () == 1)
    return;

  if (get_yahoo_context () == 0)
    {
      printf ("ERROR: yahoo context is NULL. You need to login first\n");
      return;
    }
  if (is_buddy_online (to))
    {
      yahoo_cmd_msg (get_yahoo_context (), get_default_login_id (),
		     get_message_to (), get_message ());
    }
  else
    {
      printf ("Offline message sent to [%s]\n", to);
      yahoo_cmd_msg_offline (get_yahoo_context (), get_default_login_id (),
			     get_message_to (), get_message ());
    }
}

/* this function is primarily for scm-procedure. using this function
for other purposes is not justified 
*/
void
send_message_no_hook (char *to, char *message)
{

  // dont set current_target_buddy from inside this call. otherwise
  // it will confuse "session (AUTU-INSERT-MODE). 
  // set_current_target_buddy (to); // AUTO-INSERT-MODE

  if (get_yahoo_context () == 0)
    {
      printf ("ERROR: yahoo context is NULL. You need to login first\n");
      return;
    }

  if (is_buddy_online (to))
    {
      yahoo_cmd_msg (get_yahoo_context (), get_default_login_id (), to,
		     message);
    }
  else
    {
      // printf ("Offline message sent to [%s]\n", to);
      // fflush (stdout);
      yahoo_cmd_msg_offline (get_yahoo_context (), get_default_login_id (),
			     to, message);
    }
}


struct buddy *
get_buddy (char *buddy_id)
{

  struct buddy *temp_buddy;

  temp_buddy = buddy_list;
  while (temp_buddy)
    {
      if (strcasecmp (temp_buddy->buddy_id, buddy_id))
	temp_buddy = temp_buddy->next;
      else
	break;
    }
  return temp_buddy;
}


char *
get_buddy_group (char *buddy)
{
  char *group = NULL;
  int i;

  for (i = 0;
       get_yahoo_context ()->buddies && get_yahoo_context ()->buddies[i]; i++)
    {
      if (!strcasecmp (get_yahoo_context ()->buddies[i]->id, buddy))
	{
	  group = get_yahoo_context ()->buddies[i]->group;
	  break;
	}
    }
  return group;
}

struct buddy *
get_buddy_by_index (int index)
{
  int i = 0;
  struct yahoo_buddy *tmpbuddy;
  struct buddy *curr_buddy;

  curr_buddy = (struct buddy *) NULL;

  while (get_yahoo_context ()->buddies && get_yahoo_context ()->buddies[i])
    {
      tmpbuddy = get_yahoo_context ()->buddies[i];
      curr_buddy = get_buddy (tmpbuddy->id);

      if (curr_buddy && (index == i))
	return curr_buddy;

      i++;
    }
  return NULL;
}

int
is_buddy_online (char *buddy_id)
{
  int i;
  struct yahoo_buddy *tmpbuddy;
  struct buddy *curr_buddy;

  curr_buddy = (struct buddy *) NULL;

  i = 0;
  while (get_yahoo_context () &&
	 get_yahoo_context ()->buddies && get_yahoo_context ()->buddies[i])
    {
      tmpbuddy = get_yahoo_context ()->buddies[i];
      curr_buddy = get_buddy (tmpbuddy->id);
      if (curr_buddy && !strcasecmp (curr_buddy->buddy_id, buddy_id))
	{
	  if (curr_buddy->in_pager || curr_buddy->in_chat)
	    return (1);		// online
	  else
	    return (0);		//offline
	}
      i++;
    }
  /*
     FIXME: This code finds online / offline if only he is a buddy. If he
     is not a buddy, he is assumed online -ab <ab@gnu.org.in>
   */
  return (1);
}


int
add_buddy (char *buddy_id, int status, int in_pager, int in_chat, char *custom_message)
{
  struct buddy *temp_buddy;

  if (!buddy_id)
    return -1;

  temp_buddy = (struct buddy *) malloc (sizeof (struct buddy));
  if (!temp_buddy)
    return -1;

  temp_buddy->buddy_id = buddy_id;
  temp_buddy->status = status;
  temp_buddy->in_pager = in_pager;
  temp_buddy->in_chat = in_chat;
  temp_buddy->custom_message = custom_message;
  temp_buddy->next = buddy_list;
  buddy_list = temp_buddy;

  return 0;
}

int
remove_buddy (char *buddy_id)
{
  struct buddy *curr_buddy;
  struct buddy *prev_buddy;

  prev_buddy = (struct buddy *) NULL;
  curr_buddy = buddy_list;
  while (curr_buddy)
    {
      if (strcasecmp (curr_buddy->buddy_id, buddy_id) == 0)
	{
	  if (prev_buddy == (struct buddy *) NULL)
	    {
	      if (curr_buddy == buddy_list)
		buddy_list = curr_buddy->next;
	    }
	  else
	    prev_buddy->next = curr_buddy->next;
	  free (curr_buddy);
	  return 0;
	}
      prev_buddy = curr_buddy;
      curr_buddy = curr_buddy->next;
    }
  return -1;
}

void
create_buddy_list ()
{
  int i;
  struct yahoo_buddy *tmpbuddy;

  i = 0;
  while (get_yahoo_context ()->buddies && get_yahoo_context ()->buddies[i])
    {
      tmpbuddy = get_yahoo_context ()->buddies[i];
      if (add_buddy (tmpbuddy->id, 0, 0, 0, NULL) == -1)
	printf ("\nInsufficient memory\n");
      i++;
    }
}

void
reset_all_status ()
{
  struct buddy *curr_buddy;
  struct buddy *prev_buddy;

  prev_buddy = (struct buddy *) NULL;
  curr_buddy = buddy_list;
  while (curr_buddy)
    {
      prev_buddy = curr_buddy->next;
      curr_buddy->status = curr_buddy->in_pager = curr_buddy->in_pager = 0;
      curr_buddy->custom_message = (char *) NULL;
      curr_buddy = prev_buddy;
    }
}

/* This function is rewritten below for optimization
void
display_buddy (struct buddy *buddy)
{
  // toggle_who_state - SHOW-ALL / ONLINE-ONLY
  if (!get_who_state ())
    {				// SHOW-ALL mode
      printf ("%c %s", (buddy->in_pager || buddy->in_chat ? '*' : ' '),
	      buddy->buddy_id);
      if (buddy->status != YAHOO_STATUS_AVAILABLE
	  && (buddy->in_pager || buddy->in_chat))
	printf (" (%s)", yahoo_get_status_string (buddy->status));
      printf ("\n");
    }
  else if (buddy->in_pager || buddy->in_chat)
    {
      printf ("%c %s", (buddy->in_pager || buddy->in_chat ? '*' : ' '),
	      buddy->buddy_id);
      if (buddy->status != YAHOO_STATUS_AVAILABLE &&
	  (buddy->in_pager || buddy->in_chat))
	printf (" (%s)", yahoo_get_status_string (buddy->status));
      printf ("\n");
    }
}
*/

void
display_buddy (struct buddy *buddy)
{
  if (buddy->in_pager || buddy->in_chat)
    {
      printf ("* %s", buddy->buddy_id);
      if (buddy->status != YAHOO_STATUS_AVAILABLE)
	printf (" (%s)", (buddy->status == YAHOO_STATUS_CUSTOM
			    ? buddy->custom_message
			    : yahoo_get_status_string (buddy->status)));
      printf ("\n");
    }
  else if (!get_who_state ())              // toggle_who_state - SHOW-ALL / ONLINE-ONLY
    printf ("  %s\n", buddy->buddy_id);    // SHOW-ALL
}

void
display_buddy_list ()
{
  int i;
  struct yahoo_buddy *tmpbuddy;
  struct buddy *curr_buddy;
  char *curr_group = NULL;

  curr_buddy = (struct buddy *) NULL;

  i = 0;
  while (get_yahoo_context ()->buddies && get_yahoo_context ()->buddies[i])
    {
      tmpbuddy = get_yahoo_context ()->buddies[i];
      curr_buddy = get_buddy (tmpbuddy->id);

      if (!curr_group
	  || (strcasecmp (curr_group, get_yahoo_context ()->buddies[i]->group)
	      != 0))
	{
	  curr_group = get_yahoo_context ()->buddies[i]->group;
	  printf ("\n[%s]\n", curr_group);
	}
      if (curr_buddy)
	display_buddy (curr_buddy);
      //      else
      //printf ("'%s' not on buddy list\n", tmpbuddy->id);
      i++;
    }
  printf ("\n");
}

int
process_status_packet (struct yahoo_packet *pkt, struct yahoo_rawpacket *rawpkt, char *username)
{
  char *rawpkt_content = (char *) NULL;
  char *delim = (char *) NULL;
  int delim_size = 0, i = 0;

  if (pkt->service == YAHOO_SERVICE_LOGOFF
      && !strcmp (pkt->active_id, username))
    {
      PRINTF_MESSAGE ("\nConnection lost, disconnected\n");
      return -1;
    }

  for (i = 0; i < pkt->idstatus_count; i++)
    {
      struct buddy *curr_buddy;

      curr_buddy = get_buddy (pkt->idstatus[i]->id);
      if (!curr_buddy)
	{
	  // buddy not on buddy list
	  continue;
	}

      if (pkt->service != YAHOO_SERVICE_CHATLOGOFF &&
	  pkt->service != YAHOO_SERVICE_CHATLOGON)
	curr_buddy->status = pkt->idstatus[i]->status;

      curr_buddy->in_pager = pkt->idstatus[i]->in_pager;
      curr_buddy->in_chat = pkt->idstatus[i]->in_chat;

      if (curr_buddy->status == YAHOO_STATUS_CUSTOM)
	{
	  delim_size = strlen (curr_buddy->buddy_id) + strlen ("(99,");
	  delim = (char *) malloc (sizeof (char) * delim_size + 1);
	  strcpy (delim, curr_buddy->buddy_id);
	  strcat (delim, "(99,");

	  rawpkt_content = (char *) rawpkt->content;
	  rawpkt_content = strstr (rawpkt_content, delim);
	  rawpkt_content += delim_size;

	  if (curr_buddy->custom_message)
	    {
	      free (curr_buddy->custom_message);
	      curr_buddy->custom_message = (char *) NULL;
	    }
	  curr_buddy->custom_message = get_token_with_strdelim (&rawpkt_content, ",");
	}
      else
	curr_buddy->custom_message = (char *) NULL;

      if (get_status_mode ())
	{
	  if (pkt->service == YAHOO_SERVICE_LOGON)
	    {
	      PRINTF_MESSAGE ("\r[%s] has connected\n", curr_buddy->buddy_id);
	    }
	  else if (pkt->service == YAHOO_SERVICE_LOGOFF)
	    {
	      PRINTF_MESSAGE ("\r[%s] has disconnected\n",
			      curr_buddy->buddy_id);
	    }
	  else
	    {
	      if (curr_buddy->status == YAHOO_STATUS_CUSTOM)
		{
		  PRINTF_MESSAGE ("\r[%s] is now [%s]\n", curr_buddy->buddy_id,
				  curr_buddy->custom_message);
		}
	      else
		{
		  PRINTF_MESSAGE ("\r[%s] %s\n", curr_buddy->buddy_id,
				  yahoo_get_status_append (curr_buddy->status));
		}
	    }
	}
    }
  return 1;
}

void
process_message_packet (struct yahoo_packet *pkt)
{
  char *str_strip;

  if (pkt->msgtype == YAHOO_MSGTYPE_STATUS)
    {
      struct buddy *curr_buddy;

      curr_buddy = get_buddy (pkt->msg_id);
      if (!curr_buddy)
	return;
      curr_buddy->status = pkt->msg_status;
    }

  if (pkt->msg)
    {
      if (pkt->msgtype == YAHOO_MSGTYPE_BOUNCE)
	{
	  // pkt->mgs_id is null mostly here, so better use active_id
	  // PRINTF_MESSAGE ("\r%s -> <Message not sent, user not online>\n",
	  // pkt->msg_id);
	  PRINTF_MESSAGE ("\r%s -> <Message not sent, user not online>\n",
			  pkt->active_id);
	}
      else
	{
	  str_strip = filter_message (pkt->msg);
	  set_received_message (str_strip);
	  set_received_message_from (pkt->msg_id);
	  set_hook_return (0);

	  scm_run_hook (get_message_receive_hook (),
			gh_list (gh_str02scm (get_received_message_from ()),
				 gh_str02scm (get_received_message ()),
				 SCM_UNDEFINED));

	  if (get_hook_return () == 1)
	    return;

	  set_current_target_buddy (strdup (pkt->msg_id));	// AUTO-INSERT-MODE       

	  PRINTF_MESSAGE ("\r%s -> %s\n", get_received_message_from (),
			  get_received_message ());
	  if (str_strip)
	    {
	      free (str_strip);
	      str_strip = (char *) NULL;
	    }
	}
    }
}

void
process_ping_packet (struct yahoo_packet *pkt)
{
  PRINTF_MESSAGE ("\rPING\n");
}

void
process_mail_notification (struct yahoo_packet *pkt,
			   struct yahoo_rawpacket *rawpkt)
{
  if (yahoo_parsepacket_newmail (get_yahoo_context (), pkt, rawpkt) == 0)
    {
      if (pkt->mail_status != 0)
	PRINTF_MESSAGE ("\rYou've got [%d] New Mail(s) !!\n",
			pkt->mail_status);
    }
  else
    PRINTF_MESSAGE ("\nError fetching mail status\n");
}

void
process_offline_message (struct yahoo_packet *pkt,
			 struct yahoo_rawpacket *inpkt)
{
  if (yahoo_parsepacket_message_offline (get_yahoo_context (), pkt, inpkt) == 0)
    {
      char *pkt_msg = (char *) NULL;
      char *from = (char *) NULL;
      char *time_secs = (char *) NULL;
      char *time_fstr = (char *) NULL;
      char *message = (char *) NULL;
      char *str_strip = (char *) NULL;
      char *delim = (char *) NULL;
      int delim_size = 0;
      int mem_flag = 0;

      delim_size = strlen ("6,6,") + strlen (get_default_login_id ()) + strlen (",");
      delim = (char *) malloc (sizeof (char) * (delim_size + 1));
      strcpy (delim, "6,6,");
      strcat (delim, get_default_login_id ());
      strcat (delim, ",");

      pkt_msg = pkt->msg;
      from = pkt->msg_id;
      time_secs = pkt->msg_timestamp;

      while (1)
	{
	  message = get_token_with_strdelim_i (&pkt_msg, delim);

	  if (time_secs)
	    time_fstr = get_utc_time (time_secs);
	  else
	    time_fstr = (char *) NULL;

	  if (message)
	    str_strip = filter_message (message);
	  else
	    str_strip = (char *) NULL;

	  set_hook_return (0);
	  scm_run_hook (get_message_receive_offline_hook (),
			gh_list (gh_str02scm (from),
				 gh_str02scm (str_strip),
				 gh_str02scm (time_fstr),
				 SCM_UNDEFINED));
	  if (get_hook_return () == 1)
	    return;

	  PRINTF_MESSAGE ("%s [%s] -> %s\n", from, time_fstr, str_strip);

	  if (from && mem_flag)
	    {
	      free (from);
	      from = (char *) NULL;
	    }
	  if (time_secs && mem_flag)
	    {
	      free (time_secs);
	      time_secs = (char *) NULL;
	    }
	  if (time_fstr)
	    {
	      free (time_fstr);
	      time_fstr = (char *) NULL;
	    }
	  if (message)
	    {
	      free (message);
	      message = (char *) NULL;
	    }
	  if (str_strip)
	    {
	      free (str_strip);
	      str_strip = (char *) NULL;
	    }
	  if (!pkt_msg)
	    break;
	  pkt_msg += delim_size;

	  from = get_token_with_strdelim (&pkt_msg, ",");
	  pkt_msg++;
	  time_secs = get_token_with_strdelim (&pkt_msg, ",");
	  pkt_msg++;
	  mem_flag = !mem_flag;
	}
    }
  else
    {
      PRINTF_MESSAGE ("\nError processing offline messages\n");
    }
}

void
process_file_transfer (struct yahoo_packet *pkt)
{
  int fd;

  PRINTF_MESSAGE ("\rYou've received a file from [%s]\n", pkt->file_from);
  PRINTF_MESSAGE ("URL: %s\n", pkt->file_url);

  if ((fd =
       open (get_download_filename (), O_CREAT | O_APPEND | O_WRONLY,
	     S_IRWXU)) < 0)
    {
      perror (get_download_filename ());
      return;
    }

  if (write (fd, "From: ", strlen ("From: ")) != strlen ("From: "))
    {
      perror (get_download_filename ());
      return;
    }

  write (fd, pkt->file_from, strlen (pkt->file_from));
  write (fd, " URL: ", strlen (" URL: "));
  write (fd, pkt->file_url, strlen (pkt->file_url));
  write (fd, "\n", 1);

  if (close (fd) == EOF)
    {
      perror (get_download_filename ());
      return;
    }
  PRINTF_MESSAGE ("URL saved in [%s]\n", get_download_filename ());

}

int
handle_yahoo_message (char *username)
{
  struct yahoo_rawpacket *rawpkt;
  struct yahoo_packet *pkt;

  if (!yahoo_getdata (get_yahoo_context ()))
    {
      PRINTF_MESSAGE ("\nERROR: could not get yahoo context\n");
      return -1;
    }

  if (get_yahoo_context ()->io_buf_curlen <= 103)
    {
      PRINTF_MESSAGE ("got I/O buffer length < 104");
      return 0;
    }

  while ((rawpkt = yahoo_getpacket (get_yahoo_context ())))
    {
      pkt = yahoo_parsepacket (get_yahoo_context (), rawpkt);

#ifdef DEBUG
      PRINTF_MESSAGE ("\nReceived packet:\n");
      PRINTF_MESSAGE ("\tService = %s\n",
		      yahoo_get_service_string (pkt->service));
      PRINTF_MESSAGE ("\tReal ID = %s\n", pkt->real_id);
      PRINTF_MESSAGE ("\tActive ID = %s\n", pkt->active_id);
      PRINTF_MESSAGE ("\tConnection ID = %X\n", pkt->connection_id);
      PRINTF_MESSAGE ("\tMagic ID = %X\n", pkt->magic_id);
      PRINTF_MESSAGE ("\tUnknown Flag 1 = %X\n", pkt->unknown1);
      PRINTF_MESSAGE ("\tMessage Type = %X\n", pkt->msgtype);
      PRINTF_MESSAGE ("\tRaw Content = %s\n", rawpkt->content);
      fflush (stdout);
#endif

      switch (pkt->service)
	{
	case YAHOO_SERVICE_USERSTAT:
	case YAHOO_SERVICE_CHATLOGON:
	case YAHOO_SERVICE_CHATLOGOFF:
	case YAHOO_SERVICE_LOGON:
	case YAHOO_SERVICE_LOGOFF:
	case YAHOO_SERVICE_ISAWAY:
	case YAHOO_SERVICE_ISBACK:
	  if (process_status_packet (pkt, rawpkt, username) == -1)
	    return -1;
	  break;
	case YAHOO_SERVICE_MESSAGE:
	case YAHOO_SERVICE_CHATMSG:
	case YAHOO_SERVICE_SYSMESSAGE:

	  switch (pkt->msgtype)
	    {
	    case YAHOO_MSGTYPE_OFFLINE:
	      process_offline_message (pkt, rawpkt);
	      break;
	    case YAHOO_MSGTYPE_ERROR:
	      // i noticed this type of message, when i got 
	      // "Session expired. Please relogin. Thank you LOGIN-ID" message
	      // in that case packet contains error message string

	      // FIXME: i think there is a bug here. if(pkt->msg) is a
	      // temporary hack. when changing status to 0 for second
	      // time gnuyahoo crashes. becase pkt->msg is
	      // NULL. thanks to visu for identifying the bug
	      if (pkt->msg)
		PRINTF_MESSAGE (pkt->msg);

	      break;
	    default:
	      process_message_packet (pkt);
	    }
	  break;

	case YAHOO_SERVICE_NEWCONTACT:
	  if (pkt->msg)
	    process_message_packet (pkt);
	  else
	    process_status_packet (pkt, rawpkt, username);
	  yahoo_get_config (get_yahoo_context ());
	  create_buddy_list ();
	  break;

	case YAHOO_SERVICE_PING:
	  process_ping_packet (pkt);
	  break;

	case YAHOO_SERVICE_NEWMAIL:
	  process_mail_notification (pkt, rawpkt);
	  break;

	case YAHOO_SERVICE_FILETRANSFER:
	  if (yahoo_parsepacket_filetransfer
	      (get_yahoo_context (), pkt, rawpkt) == 0)
	    process_file_transfer (pkt);
	  else
	    PRINTF_MESSAGE ("\nError fetching file...\n");
	  break;

	default:
	  PRINTF_MESSAGE ("Unhandled Packet -> Please report to ["
			  GY_BUG_REPORT_EMAIL "]\n");
	  PRINTF_MESSAGE ("\nReceived packet:\n");
	  PRINTF_MESSAGE ("\tService = %s\n",
			  yahoo_get_service_string (pkt->service));
	  PRINTF_MESSAGE ("\tReal ID = %s\n", pkt->real_id);
	  PRINTF_MESSAGE ("\tActive ID = %s\n", pkt->active_id);
	  PRINTF_MESSAGE ("\tConnection ID = %X\n", pkt->connection_id);
	  PRINTF_MESSAGE ("\tMagic ID = %X\n", pkt->magic_id);
	  PRINTF_MESSAGE ("\tUnknown Flag 1 = %X\n", pkt->unknown1);
	  PRINTF_MESSAGE ("\tMessage Type = %X\n", pkt->msgtype);
	  PRINTF_MESSAGE ("\tRaw Content = %s\n", rawpkt->content);
	  fflush (stderr);

	}
      yahoo_free_packet (pkt);
      yahoo_free_rawpacket (rawpkt);
    }
  return 0;
}
