/*
Copyright (C) 2002-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "header.h"

static int newanonvar=0;

int newanon(void)
{
  char s[33];
  sprintf(s,"anon_%d",newanonvar++);
  return newentry(strdup(s));
}

int newfun(const char *name)
{
  char *s = malloc(5+strlen(name));
  sprintf(s,"fun_%s",name);
  if (findfunction(s)<0) return newentry(s);
  free(s);
  return newanon();
}

void mkfunc(int n, int p, int flag, int wr)
{
  int x = tree[n].x;
  gpfunc *gp;
  const char *name=entryname(x);
  int nf=findfunction(name);
  if (nf==-1)
  {
    nf = newuserfunc(name);
    gp = lfunc+nf;
  }
  else
  {
    gp=lfunc+nf;
    if (gp->spec!=GPuser)
      die(n,"Trying to redefine function %s",name);
    else
    {
      userfunc *ufunc=gp->user;
      int parent=ufunc->pdefnode;
      int node=ufunc->defnode;
      if (tree[node].y>=0)
        warning(node,"Function %s redefined line %d",name,tree[n].lineno);
      if (tree[parent].x==node)
        tree[parent].x=GNIL;
      else
        tree[parent].y=GNIL;
    }
  }
  gp->user->flag=flag;
  gp->user->wrapper=wr;
  gp->user->pdefnode=p;
  gp->user->defnode=n;
}

static void topfunc(int n, int p, int fun, int pfun, int nf, int wr)
{
  int code = tree[n].y, arglist = tree[n].x;
  int args = arglist>=0 ? tree[arglist].y : -1;
  int fl = wr<0?(1<<UFclosure):(1<<UFclosure)|(1<<UFstatic);
  if (fun==-1 || pfun==-1)
  {
    if(n!=nf) tree[n]=tree[nf];
    mkfunc(nf,p,0,wr);
    gentopfunc(args,n,n,p);
    gentopfunc(code,n,n,p);
    return;
  }
  if (tree[pfun].x==fun)
  {
    int seq = newnode(Fseq,nf,newleaf(fun));
    tree[fun] = tree[seq];
    mkfunc(nf,seq,fl,wr);
    gentopfunc(args,n,nf,seq);
    gentopfunc(code,n,nf,seq);
  }
  else
  {
    int seq = newnode(Fseq,tree[pfun].x,nf);
    tree[pfun].x = seq;
    mkfunc(nf,seq,fl,wr);
    gentopfunc(args,n,nf,seq);
    gentopfunc(code,n,nf,seq);
  }
}

static void topfunclambda(int n, int p, int fun, int pfun, int wr)
{
  int x = tree[n].x, y = tree[n].y;
  int nn = newanon();
  int nf = newnode(Fdeffunc,newnode(Ffunction,nn,x),y);
  int seq = newnode(Fentry,nn,-1);
  topfunc(n,p,fun,pfun,nf,wr);
  if (fun>=0) tree[n] = tree[seq];
}

static int topfuncproto(int n, int fun, int pfun, int nf)
{
  int arg[STACKSZ+1];
  int nb=genlistargs(n,arg,0,STACKSZ);
  gpfunc *gp = lfunc+nf;
  gpwrap *gw = gp->wrap;
  int nbwr = gw ? gw->nb: 0;
  const char *proto=gp->proto.code;
  int i=0;
  int var=-1, seq, a, t, binf, wr;
  char const *p=proto;
  char c;
  PPproto mod;
  if (!proto) return 1;
  while((mod=parseproto(&p,&c)))
  {
    switch(mod)
    {
    case PPauto:
      break;
    case PPstd:
      if (i>=nb || arg[i]==GNOARG)
        die(n,"missing mandatorry argument");
      a = arg[i++];
      wr = i<=nbwr ? gw->w[i-1]:-2;
      switch(c)
      {
        case 'G':
          if (wr>=0 && tree[a].f==Flambda)
            topfunclambda(a, n, fun, pfun, wr);
          break;
        case 'V':
          var = a;
          killlistarg(n,a);
          break;
        case '=':
          genequal(a,lfunc[nf].gpname,&var,&binf,&t);
          tree[a]=tree[binf];
          break;
        case 'I':
        case 'E':
          if (wr>=-1)
          {
        case 'J':
            seq = newnode(Flambda,var,newleaf(a));
            tree[a] = tree[seq];
            topfunclambda(a, n, fun, pfun, wr);
          }
          break;
      }
      break;
    case PPdefault:
      i++;
      a  = i<=nb ? arg[i-1]: GNOARG;
      wr = i<=nbwr ? gw->w[i-1]:-2;
      switch(c)
      {
        case 'I':
        case 'E':
          if (a!=GNOARG && wr>=-1)
          {
            seq = newnode(Flambda,var,newleaf(a));
            tree[a]=tree[seq];
            topfunclambda(a, n, fun, pfun, wr);
          }
          break;
      }
      break;
    case PPdefaultmulti:
      i++;
      break;
    case PPstar:
      break;
    default:
      die(n,"internal error: PPproto %d in genfuncbycode",mod);
    }
  }
  return 0;
}

/*
  n: node
  p:parent node
*/
void gentopfunc(int n, int p, int fun, int pfun)
{
  int x,y;
  if (n<0)
    return;
  x=tree[n].x;
  y=tree[n].y;
  switch(tree[n].f)
  {
  case Flambda:
    topfunclambda(n,p,fun,pfun,-1);
    break;
  case Fdeffunc:
    if (fun>=0)
    {
      int nn = newfun(entryname(x));
      int seq= newnode(Fassign,newnode(Fentry,tree[x].x,-1),newnode(Fentry,nn,-1));
      int nf = newnode(Fdeffunc,newnode(Ffunction,nn,tree[x].y),tree[n].y);
      topfunc(n,p,fun,pfun,nf,-1);
      tree[n] = tree[seq];
    }
    else
      topfunc(n,p,fun,pfun,n,-1);
    break;
  case Fassign:
    if (tree[x].f==Fentry && tree[y].f==Flambda)
    {
      int nn = newfun(entryname(x));
      int seq= newnode(Fentry,nn,-1);
      int nf = newnode(Fdeffunc,newnode(Ffunction,nn,tree[y].x),tree[y].y);
      topfunc(y,n,fun,pfun,nf,-1);
      if (fun>=0) tree[y] = tree[seq];
    }
    else
    {
      gentopfunc(x,n,fun,pfun);
      gentopfunc(y,n,fun,pfun);
    }
    break;
  case Ffunction:
    {
      int nf = findfunction(entryname(n));
      if (nf>=0 && lfunc[nf].spec<0 && !lfunc[nf].iter)
        topfuncproto(n,fun,pfun,nf);
      gentopfunc(y,n,fun,pfun);
    }
    break;
  case Fentry:
  case Fconst:
  case Fsmall:
  case Fnoarg:
    break;
  case Frefarg:
  case Ftag:
    gentopfunc(x,n,fun,pfun);
    break;
  default:
    if (tree[n].f>=FneedENTRY)
      die(n,"Internal error: unknown func %s in gentopfunc",funcname(tree[n].f));
    else
    {
      gentopfunc(x,n,fun,pfun);
      gentopfunc(y,n,fun,pfun);
    }
  }
}
