/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "assert-macros.h"

#include "gpass/configuration.h"

/***********************************************************
 *
 * initialize/terminate, setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
    gpass_configuration_finalize();
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
START_TEST(test_get__window_geometry)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gint x, y, width, height;
    
    g_object_get(config,
                 "window_x", &x, "window_y", &y,
                 "window_width", &width, "window_height", &height, NULL);
    ASSERT_EQUAL_INT(0, x);
    ASSERT_EQUAL_INT(0, y);
    ASSERT_EQUAL_INT(500, width);
    ASSERT_EQUAL_INT(300, height);
}
END_TEST

START_TEST(test_get__visible)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gboolean toolbar_visible, statusbar_visible;
    
    g_object_get(config,
                 "toolbar_visible", &toolbar_visible,
                 "statusbar_visible", &statusbar_visible, NULL);
    ASSERT_TRUE(toolbar_visible);
    ASSERT_TRUE(statusbar_visible);
}
END_TEST

START_TEST(test_get__pane_width)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gint pane_width;

    g_object_get(config, "pane_width", &pane_width, NULL);
    ASSERT_EQUAL_INT(250, pane_width);
}
END_TEST

START_TEST(test_get__undo_levels)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gint undo_levels;

    g_object_get(config, "undo_levels", &undo_levels, NULL);
    ASSERT_EQUAL_INT(15, undo_levels);
}
END_TEST

START_TEST(test_get__visible_secrets)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gboolean visible_secrets;

    g_object_get(config, "visible_secrets", &visible_secrets, NULL);
    ASSERT_FALSE(visible_secrets);
}
END_TEST

START_TEST(test_get__lock)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gboolean lock;

    g_object_get(config, "lock", &lock, NULL);
    ASSERT_TRUE(lock);
}
END_TEST

START_TEST(test_get__lock_timeout)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gint lock_timeout;
    
    g_object_get(config, "lock_timeout", &lock_timeout, NULL);
    ASSERT_EQUAL_INT(15, lock_timeout);
}
END_TEST

START_TEST(test_set__window_geometry)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config,
                 "window_x", 100, "window_y", 200,
                 "window_width", 300, "window_height", 400, NULL);
    ASSERT_EQUAL_INT(100, config->window_x);
    ASSERT_EQUAL_INT(200, config->window_y);
    ASSERT_EQUAL_INT(300, config->window_width);
    ASSERT_EQUAL_INT(400, config->window_height);
}
END_TEST

START_TEST(test_set__visible)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config,
                 "toolbar_visible", FALSE,
                 "statusbar_visible", FALSE, NULL);
    ASSERT_FALSE(config->toolbar_visible);
    ASSERT_FALSE(config->statusbar_visible);
}
END_TEST

START_TEST(test_set__pane_width)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config, "pane_width", 200, NULL);
    ASSERT_EQUAL_INT(200, config->pane_width);
}
END_TEST

START_TEST(test_set__undo_levels)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config, "undo_levels", 5, NULL);
    ASSERT_EQUAL_INT(5, config->undo_levels);
}
END_TEST

START_TEST(test_set__visible_secrets)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config, "visible_secrets", TRUE, NULL);
    ASSERT_TRUE(config->visible_secrets);
    g_object_set(config, "visible_secrets", FALSE, NULL);
    ASSERT_FALSE(config->visible_secrets);
}
END_TEST

START_TEST(test_set__lock)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config, "lock", TRUE, NULL);
    ASSERT_TRUE(config->lock);
    g_object_set(config, "lock", FALSE, NULL);
    ASSERT_FALSE(config->lock);
}
END_TEST

START_TEST(test_set__lock_timeout)
{
    GPassConfiguration *config = gpass_configuration_instance();
    
    g_object_set(config, "lock_timeout", 10, NULL);
    ASSERT_EQUAL_INT(10, config->lock_timeout);
    g_object_set(config, "lock_timeout", 2, NULL);
    ASSERT_EQUAL_INT(2, config->lock_timeout);
    g_object_set(config, "lock_timeout", 5, NULL);
    ASSERT_EQUAL_INT(5, config->lock_timeout);
}
END_TEST

START_TEST(test_get_launcher)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gchar *launcher;

    gpass_configuration_get_launcher(config, "general", &launcher);
    ASSERT_NULL(launcher);
}
END_TEST

START_TEST(test_set_launcher)
{
    GPassConfiguration *config = gpass_configuration_instance();
    gchar *launcher;

    gpass_configuration_set_launcher(config,
                                     "general", "gnome-open @hostname@");
    gpass_configuration_get_launcher(config, "general", &launcher);
    ASSERT_EQUAL_STRING("gnome-open @hostname@", launcher);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassConfiguration");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_get__window_geometry);
    tcase_add_test(tc, test_get__visible);
    tcase_add_test(tc, test_get__pane_width);
    tcase_add_test(tc, test_get__undo_levels);
    tcase_add_test(tc, test_get__visible_secrets);
    tcase_add_test(tc, test_get__lock);
    tcase_add_test(tc, test_get__lock_timeout);
    tcase_add_test(tc, test_set__window_geometry);
    tcase_add_test(tc, test_set__visible);
    tcase_add_test(tc, test_set__pane_width);
    tcase_add_test(tc, test_set__undo_levels);
    tcase_add_test(tc, test_set__visible_secrets);
    tcase_add_test(tc, test_set__lock);
    tcase_add_test(tc, test_set__lock_timeout);
    tcase_add_test(tc, test_get_launcher);
    tcase_add_test(tc, test_set_launcher);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
