/*
  Gnome-o-Phone - A program for internet telephony
  Copyright (C) 1999  Roland Dreier
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  $Id: talk.c 1.21 Sat, 04 Dec 1999 15:20:37 -0600 dreier $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "talk.h"

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <glib.h>

#if defined (HAVE_GSM_H)
#include <gsm.h>
#elif defined (HAVE_GSM_GSM_H)
#include <gsm/gsm.h>
#else
#error "GSM header file not found."
#endif

#include "thread-queue.h"
#include "connection.h"
#include "request.h"
#include "rtp-audio.h"
#include "thread.h"
#include "sound.h"

#include "gphone.h"
#include "gphone-lib.h"

int Silence_Level = 128;

extern char *Test_File;

static Thread_Queue Talk_Queue;
static gsm Talk_Handle;
static Sound_Handle Talk_SH = NULL;

static void
debug_read_sound(gpointer buf, int num_bytes)
{
  int r;
  static int sound_fd = -1;

  if (sound_fd == -1) {
    sound_fd = open(Test_File, O_RDONLY);
    if (sound_fd < 0) {
      gphone_perror_exit("*** debug_read_sound : open", 1);
    }
  }

  r = read(sound_fd, buf, num_bytes);

  if (r < num_bytes) {
    lseek(sound_fd, 0, SEEK_SET);
    read(sound_fd, (gchar *) buf + r, num_bytes - r);
  }
}

void
talk_init(void)
{
  Talk_Queue = thread_queue_new();

  if ((Talk_Handle = gsm_create()) == 0) {
    gphone_print_exit("*** talk_init : no gsm talk handle.", 2);
  }
}

void
talk_add_request(Request req)
{
  thread_queue_add(Talk_Queue, req);
}

static void
do_switch(void)
{
  g_assert(get_sound_duplex() == HALF_DUPLEX);

  switch(get_status()) {
  case STAT_TALK:
    sound_close(Talk_SH);
    Talk_SH = NULL;
    set_status(STAT_LISTEN);
    send_switch_packet();
    break;
  case STAT_LISTEN:
    /* Not implemented yet */
    break;
  default:
    break;
  }
}

void *
talk_side(void *arg)
{
  int i;
  gchar *sndbuf, *gsmbuf;
  int sndbufsize, gsmbufsize;
  Program_Status status;
  int level;

  sndbufsize = NUM_FRAME * get_raw_frame_length();
  sndbuf = g_malloc(sndbufsize);
  gsmbufsize = NUM_FRAME * get_enc_frame_length();
  gsmbuf = g_malloc(gsmbufsize);

  while (1) {
    if (!thread_queue_empty(Talk_Queue)) {
      Request req = (Request) thread_queue_head(Talk_Queue);

      switch (req -> type) {
      case REQUEST_QUIT:
        return NULL;
        break;
      case REQUEST_CALL:
        connection_call(req -> data);
        break;
      case REQUEST_SWITCH:
        do_switch();
        break;
      default:
        g_assert_not_reached();
        break;
      }

      g_free(req -> data);
      thread_queue_remove_free(Talk_Queue);
    }

    if (connection_connected()) {
      /* check if we should send an RTCP packet */
      maybe_send_rtcp();
    }

    status = get_status();

    if (status == STAT_TALK || status == STAT_TALK_FD) {
      /* read some sound, encode it and send it */

      if (Test_File != NULL) {
        debug_read_sound(sndbuf, sndbufsize);
      } else {
        if (Talk_SH == NULL) {
          Talk_SH = sound_open(O_RDONLY);
          if (Talk_SH == NULL) {
            g_warning("*** Sound card refuses to open the microphone\n");
          } else {
            g_log("GPHONE", G_LOG_LEVEL_INFO,
                  "Microphone ON");
          }
        }

        read_samples(Talk_SH, sndbuf, sndbufsize);
      }
      
      level = find_power_level((void *) sndbuf, sndbufsize);

      if (level > Silence_Level) {
        for (i = 0; i < NUM_FRAME; i++) {
          gsm_encode(Talk_Handle, 
                     (gsm_signal *) (sndbuf + i * get_raw_frame_length()),
                     ((gsm_frame *) gsmbuf)[i]);
        }
        
        rtp_send(gsmbuf, gsmbufsize, PAYLOAD_GSM,
                 NUM_FRAME * get_raw_frame_length()
                 / (sizeof (gsm_signal)));
      }

      set_mic_level(level);
    } else {                    /* not talking now */
      set_mic_level(0);

      short_sleep(.05);
    }
  }

  return(NULL);
}



/*
 * Local variables:
 *  compile-command: "make -k gphone"
 * End:
 */
