/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasimage.c - image item.
 */

/**
 * SECTION:goocanvasimage
 * @Title: GooCanvasImage
 * @Short_Description: an image item.
 *
 * GooCanvasImage represents an image item.
 *
 * It is a subclass of #GooCanvasItemSimple and so inherits all of the style
 * properties such as "operator" and "pointer-events".
 *
 * It also implements the #GooCanvasItem interface, so you can use the
 * #GooCanvasItem functions such as goo_canvas_item_raise() and
 * goo_canvas_item_rotate().
 *
 * To create a #GooCanvasImage use goo_canvas_image_new().
 *
 * To get or set the properties of an existing #GooCanvasImage, use
 * g_object_get() and g_object_set().
 *
 * To respond to events such as mouse clicks on the image you must connect
 * to the signal handlers of the corresponding #GooCanvasImageView objects.
 * (See goo_canvas_view_get_item_view() and #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "goocanvasprivate.h"
#include "goocanvasimage.h"
#include "goocanvasimageview.h"
#include "goocanvasutils.h"


enum {
  PROP_0,

  PROP_PATTERN,
  PROP_X,
  PROP_Y,
  PROP_WIDTH,
  PROP_HEIGHT,

  /* Convenience properties. */
  PROP_PIXBUF
};

static void goo_canvas_image_finalize (GObject *object);
static void item_interface_init (GooCanvasItemIface *iface);
static void goo_canvas_image_get_property (GObject            *object,
					   guint               param_id,
					   GValue             *value,
					   GParamSpec         *pspec);
static void goo_canvas_image_set_property (GObject            *object,
					   guint               param_id,
					   const GValue       *value,
					   GParamSpec         *pspec);

G_DEFINE_TYPE_WITH_CODE (GooCanvasImage, goo_canvas_image,
			 GOO_TYPE_CANVAS_ITEM_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM,
						item_interface_init))


static void
goo_canvas_image_class_init (GooCanvasImageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass*) klass;

  gobject_class->finalize = goo_canvas_image_finalize;

  gobject_class->get_property = goo_canvas_image_get_property;
  gobject_class->set_property = goo_canvas_image_set_property;

  g_object_class_install_property (gobject_class, PROP_PATTERN,
                                   g_param_spec_boxed ("pattern",
						       _("Pattern"),
						       _("The cairo pattern to paint"),
						       GOO_TYPE_CAIRO_PATTERN,
						       G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_X,
				   g_param_spec_double ("x",
							"X",
							_("The x coordinate of the image"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_Y,
				   g_param_spec_double ("y",
							"Y",
							_("The y coordinate of the image"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_WIDTH,
				   g_param_spec_double ("width",
							_("Width"),
							_("The width of the image"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_HEIGHT,
				   g_param_spec_double ("height",
							_("Height"),
							_("The height of the image"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_PIXBUF,
				   g_param_spec_object ("pixbuf",
							_("Pixbuf"),
							_("The GdkPixbuf to display"),
							GDK_TYPE_PIXBUF,
							G_PARAM_WRITABLE));
}


static void
goo_canvas_image_init (GooCanvasImage *canvas_image)
{

}


/**
 * goo_canvas_image_new:
 * @parent: the parent item, or %NULL. If a parent is specified, it will assume
 *  ownership of the item, and the item will automatically be freed when it is
 *  removed from the parent. Otherwise call g_object_unref() to free it.
 * @pixbuf: the #GdkPixbuf containing the image data, or %NULL.
 * @x: the x coordinate of the image.
 * @y: the y coordinate of the image.
 * @first_property: the name of the first property to set, or %NULL.
 * @...: the remaining property names and values to set, terminated with a
 *  %NULL.
 * 
 * Creates a new image item.
 * 
 * <!--PARAMETERS-->
 *
 * Here's an example showing how to create an image at (100.0, 100.0), using
 * the given pixbuf at its natural width and height:
 *
 * <informalexample><programlisting>
 *  GooCanvasItem *image = goo_canvas_image_new (mygroup, pixbuf, 100.0, 100.0,
 *                                               NULL);
 * </programlisting></informalexample>
 *
 * Returns: a new image item.
 **/
GooCanvasItem*
goo_canvas_image_new (GooCanvasItem *parent,
		      GdkPixbuf     *pixbuf,
		      gdouble        x,
		      gdouble        y,
		      const gchar   *first_property,
		      ...)
{
  GooCanvasItem *item;
  GooCanvasImage *image;
  va_list args;

  item = g_object_new (GOO_TYPE_CANVAS_IMAGE, NULL);
  image = GOO_CANVAS_IMAGE (item);

  image->x = x;
  image->y = y;

  va_start (args, first_property);
  g_object_set_valist (G_OBJECT (item), first_property, args);
  va_end (args);

  if (pixbuf)
    {
      image->pattern = goo_canvas_cairo_pattern_from_pixbuf (pixbuf);
      image->width = gdk_pixbuf_get_width (pixbuf);
      image->height = gdk_pixbuf_get_height (pixbuf);
    }

  if (parent)
    {
      goo_canvas_item_add_child (parent, item, -1);
      g_object_unref (item);
    }

  return item;
}


static void
goo_canvas_image_finalize (GObject *object)
{
  GooCanvasImage *image = (GooCanvasImage*) object;

  cairo_pattern_destroy (image->pattern);

  G_OBJECT_CLASS (goo_canvas_image_parent_class)->finalize (object);
}


static GooCanvasItemView*
goo_canvas_image_create_view (GooCanvasItem     *item,
			      GooCanvasView     *canvas_view,
			      GooCanvasItemView *parent_view)
{
  return goo_canvas_image_view_new (canvas_view, parent_view,
				    (GooCanvasImage*) item);
}


static void
item_interface_init (GooCanvasItemIface *iface)
{
  iface->create_view = goo_canvas_image_create_view;
}


static void
goo_canvas_image_get_property (GObject              *object,
			       guint                 prop_id,
			       GValue               *value,
			       GParamSpec           *pspec)
{
  GooCanvasImage *image = (GooCanvasImage*) object;

  switch (prop_id)
    {
    case PROP_PATTERN:
      g_value_set_boxed (value, image->pattern);
      break;
    case PROP_X:
      g_value_set_double (value, image->x);
      break;
    case PROP_Y:
      g_value_set_double (value, image->y);
      break;
    case PROP_WIDTH:
      g_value_set_double (value, image->width);
      break;
    case PROP_HEIGHT:
      g_value_set_double (value, image->height);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}



static void
goo_canvas_image_set_property (GObject              *object,
			       guint                 prop_id,
			       const GValue         *value,
			       GParamSpec           *pspec)
{
  GooCanvasImage *image = (GooCanvasImage*) object;
  GdkPixbuf *pixbuf;

  switch (prop_id)
    {
    case PROP_PATTERN:
      cairo_pattern_destroy (image->pattern);
      image->pattern = g_value_get_boxed (value);
      cairo_pattern_reference (image->pattern);
      break;
    case PROP_X:
      image->x = g_value_get_double (value);
      break;
    case PROP_Y:
      image->y = g_value_get_double (value);
      break;
    case PROP_WIDTH:
      image->width = g_value_get_double (value);
      break;
    case PROP_HEIGHT:
      image->height = g_value_get_double (value);
      break;
    case PROP_PIXBUF:
      cairo_pattern_destroy (image->pattern);
      pixbuf = g_value_get_object (value);
      image->pattern = goo_canvas_cairo_pattern_from_pixbuf (pixbuf);
      image->width = gdk_pixbuf_get_width (pixbuf);
      image->height = gdk_pixbuf_get_height (pixbuf);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_signal_emit_by_name (image, "changed", TRUE);
}
