/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: common.h,v 1.41 2005/05/05 04:48:17 dave_malcolm Exp $ */

#ifndef GQ_COMMON_H_INCLUDED
#define GQ_COMMON_H_INCLUDED

#include <lber.h>
#include <ldap.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <config.h>  /* NDEBUG - possibly */
#include <assert.h>

#define MAX_SERVERNAME_LEN    64
#define MAX_HOSTNAME_LEN      1024
#define MAX_SEARCH_HISTORY_LENGTH  10

#define MAX_NUM_ATTRIBUTES   256
#define MAX_DN_LEN          1024
#define MAX_ATTR_LEN          64
#define MAX_BINDPW_LEN        256

#define DEFAULT_SHOWDN         1
#define DEFAULT_SHOWOC         1
#define DEFAULT_SHOW_RDN_ONLY  1

#define SEARCHARG_BEGINS_WITH    0
#define SEARCHARG_ENDS_WITH      1
#define SEARCHARG_CONTAINS       2
#define SEARCHARG_EQUALS         3

#define LDIF_UMICH               0
#define LDIF_V1                  1

struct server_schema {
     GList *oc;
     GList *at;
     GList *mr;
     GList *s;
};


#define SERVER_HAS_NO_SCHEMA     1

struct ldapserver {
     char *name;
     char *ldaphost;
     int   ldapport;
     char *basedn;
     char *binddn;
     char *bindpw;
     char *pwencoding;
     /* split the "configuration" password from the one entered by
	hand. This simplifies the handling of the configured password
	considerably */
     char *enteredpw;
     int   bindtype;
     char *saslmechanism;
     char *searchattr;
     int   maxentries;
     int   cacheconn;
     int   enabletls;
     long  local_cache_timeout;
     int   ask_pw;
     int   show_ref;	/* obsolete - kept for configfile compatibility */
     int   hide_internal;

     /* the canonical name of the host. Essentially this is the
	corresponding LDAP URI for the ldaphost/port combination -
	maintained through canonicalize_ldapserver() */
     char *canon_name;
     
     /* a flag indicating if ldaphost seems to be a URI or not */
     int   is_uri;

     /* if quiet is non-zero open_connection will not pop-up any error
	or questions */
     int   quiet;

     /* reference count. maintained through ldapserver_ref and
	ldapserver_unref. Similar to the reference counting within
	gtk. The last ldapserver_unref will delete the object (when
	refcount becomes <= 0).
	
	Whenever a ldapserver object get stored for a longer period of
	time (that is, whenever a structure holding a pointer to the
	object is not freed within the same function it is create in)
	the object should be ref'd.

	Objects are created with a refcount of 0. This means that such
	an object is actually "floating".
     */
     int   refcount;

     /* internal data */

     LDAP *connection;
     int   incarnation;    /* number of bind operations done so far,
			      never decremented - this is a purely
			      statistical number */
     int   missing_closes; /* incremented on every open_connection,
			      decremented on each close,
			      close_connection really closes only if
			      this drops to zero */
     struct server_schema *ss;
     int   flags;

     int   version;
     /* server_down is a flag set by the SIGPIPE signal handler and in
	case of an LDAP_SERVER_DOWN error. It indicates that we should
	reconnect the next time open_connection gets called. There is
	no (simple) way to find out which connection has been broken
	in case of a SIGPIPE, thus we have to record this for every
	connection. We MIGHT instead check for ld_errno in the LDAP
	structure, but that is neither really documented (some man
	pages mention it though) nor is it actually available through
	ldap.h */
     int   server_down;
};

void ldapserver_ref(struct ldapserver *);
void ldapserver_unref(struct ldapserver *);


#ifdef USE_G_SNPRINTF
#  include <stdio.h>
#  ifdef HAVE_G_SNPRINTF
#    define snprintf g_snprintf
#  endif
#endif

#define g_free_and_dup(target, source) \
	{ if (target) g_free(target); \
	  (target) = (source) ? g_strdup(source) : NULL; }

#define g_free_if(x) \
	{ if (x) g_free(x); (x) = NULL; }

#ifndef HAVE_REORDERABLE_G_STRING_SPRINTF
#    undef g_string_sprintf
#    define g_string_sprintf gq_g_string_sprintf
#    define g_string_printf gq_g_string_sprintf
void gq_g_string_sprintf(GString *string, const gchar *format, ...);
#endif

#ifndef HAVE_LDAP_MEMFREE
#    define ldap_memfree(x)	free(x)
#endif

struct dn_on_server {
     struct ldapserver *server;
     char *dn;
     int flags;				/* used to specify more
					 * information if needed */
};

struct dn_on_server *new_dn_on_server(const char *d, struct ldapserver *s);
void free_dn_on_server(struct dn_on_server *s);

/* Working towards refactoring the tree views from GtkCTree to GtkTreeView: */
typedef GtkCTree     GQTreeWidget;
typedef GtkCTreeNode GQTreeWidgetNode;
typedef GtkCTreeRow  GQTreeWidgetRow;
typedef GtkCTreeFunc GQTreeWidgetFunc;
#define GQ_TREE_WIDGET(x) GTK_CTREE(x)

/* Utilties for working with tree views: */
GQTreeWidget*
gq_tree_widget_new (void);

void
gq_tree_insert_dummy_node (GQTreeWidget *ctree,
			   GQTreeWidgetNode *parent_node);

GQTreeWidgetNode*
gq_tree_insert_node (GQTreeWidget *ctree,
		     GQTreeWidgetNode *parent_node,
		     GQTreeWidgetNode *sibling_node,
		     const gchar* label,
		     gpointer data,
		     void (*destroy_cb)(gpointer data));

void
gq_tree_remove_node (GQTreeWidget *tree_widget,
		     GQTreeWidgetNode *node);

void
gq_tree_remove_children (GQTreeWidget *tree_widget,
			 GQTreeWidgetNode *parent_node);

char*
gq_tree_get_node_text (GQTreeWidget *tree_widget,
		       GQTreeWidgetNode *node);

void
gq_tree_set_node_text (GQTreeWidget *tree_widget,
		       GQTreeWidgetNode *node,
		       const char *text);

gpointer
gq_tree_get_node_data (GQTreeWidget *tree_widget,
		       GQTreeWidgetNode *node);

GQTreeWidgetNode*
gq_tree_get_root_node (GQTreeWidget *tree_widget);

void
gq_tree_fire_expand_callback (GQTreeWidget *tree_widget,
			      GQTreeWidgetNode *node);

GQTreeWidgetNode*
gq_tree_get_parent_node (GQTreeWidget *tree_widget,
			 GQTreeWidgetNode *node);

gboolean
gq_tree_is_node_expanded (GQTreeWidget *tree_widget,
			  GQTreeWidgetNode *node);

void
gq_tree_expand_node (GQTreeWidget *tree_widget,
		     GQTreeWidgetNode *node);

void
gq_tree_toggle_expansion (GQTreeWidget *tree_widget,
			  GQTreeWidgetNode *node);

void
gq_tree_select_node (GQTreeWidget *tree_widget,
		     GQTreeWidgetNode *node);

GQTreeWidgetNode*
gq_tree_get_node_at (GQTreeWidget *tree_widget,
		     gint x,
		     gint y);
#endif

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
