/*
 * Grouch.app				Copyright (C) 2006 Andy Sveikauskas
 * ------------------------------------------------------------------------
 * This program is free software under the GNU General Public License
 * --
 * See note at OscarIncomingSnac.h
 */

#import <Oscar/OscarIncomingSnac.h>
#import <Oscar/OscarTlvList.h>
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>

@implementation OscarIncomingSnac

+ snacAtBuffer:(const void *)buffer withLength:(size_t)length
{
	OscarIncomingSnac *r = [OscarIncomingSnac new];
	if (![r initWithBuffer:buffer withLength:length])
	{
		[r release];
		return nil;
	}
	else
		return r;
}

+ snacAtHeaderlessBuffer:(const void*)buffer withLength:(size_t)length
{
	OscarIncomingSnac *r = [OscarIncomingSnac new];
	if (![r initWithHeaderlessBuffer:buffer withLength:length])
	{
		[r release];
		return nil;
	}
	else
		return r;
}

- init
{
	[super init];
	header.type = header.family = header.flags = header.tid = 0;
	buf = NULL;
	len = 0;
	return self;
}

- initWithBuffer:(const void *)p withLength:(size_t)l
{
	buf = p;
	len = l;

	if( len < SNAC_SIZE )
		return nil;

	header.family = oscar_int16_get(buf);
	header.type = oscar_int16_get(buf+2);
	header.flags = oscar_int16_get(buf+4);
	header.tid = oscar_int32_get(buf+6);	

	buf += SNAC_SIZE;
	len -= SNAC_SIZE;

	return self;
}

- initWithHeaderlessBuffer:(const void *)p withLength:(size_t)l
{
	buf = p;
	len = l;
	return self;
}

- (int)readInt16
{
	int r = 0;
	if( len < 2 )
		return r;
	else
	{
		r = oscar_int16_get(buf);
		buf += 2;
		len -= 2;
		return r;
	}
}

- (long)readInt32
{
	int r = 0;
	if( len < 4 )
		return r;
	else
	{
		r = oscar_int32_get(buf);
		buf += 4;
		len -= 4;
		return r;
	}
}

- (unsigned char)readByte
{
	unsigned char r = 0;
	if( !len )
		return r;
	else
	{
		r = *buf++;
		--len;
		return r;
	}
}

- (NSString *)readPascalString
{
	int l = [self readByte];
	NSString *r = @"";
	if( l <= len )
	{
		r = [NSString stringWithCString:(char*)buf length:l];
		buf += l;
		len -= l;
		return r;
	}
	return r;
}

- (NSString *)readPascalString16
{
	int l = [self readInt16];
	NSString *r = @"";
	if( l <= len )
	{
		r = [NSString stringWithCString:(char*)buf length:l];
		buf += l;
		len -= l;
	}
	return r;
}

- (NSString *)readRawString
{
	return [self readRawStringWithEncoding:
		     [NSString defaultCStringEncoding]];
}

- (NSString *)readRawStringWithEncoding:(NSStringEncoding)e
{
	NSString *r;
	r = [[NSString alloc] initWithBytes:buf length:len encoding:e];
	[r autorelease];
	len = 0;
	return r;
}

- (OscarTlvListIn*)readTLVs
{
	OscarTlvListIn *r;
	r = [OscarTlvListIn listFromBuffer:buf andLength:len andTLVs:NULL];
	[r autorelease];
	len = 0;
	return r;
}

- (OscarTlvListIn*)readTlvList
{
	size_t n = [self readInt16];
	OscarTlvListIn *r;

	r = [OscarTlvListIn listFromBuffer:buf andLength:len andTLVs:&n];
	[r autorelease];
	buf += n;
	len -= n;
	return r;
}

- (OscarTlvListIn*)readTlvList16
{
	size_t l = [self readInt16];
	OscarTlvListIn *r = nil;

	if( len >= l )
	{
		r=[OscarTlvListIn listFromBuffer:buf andLength:l andTLVs:NULL];
		[r autorelease];
		buf += l;
		len -= l;
	}
	return r;
}

- (const void*)buffer
{
	return buf;
}

- (size_t)bytesRemaining
{
	return len;
}

- (void)skipBytes:(size_t)n
{
	if( n < len )
	{
		buf += n;
		len -= n;
	}
	else
		len = 0;
}

- (int)type
{
	return header.type;
}
- (int)family
{
	return header.family;
}
- (int)flags
{
	return header.flags;
}
- (long)tagID
{
	return header.tid;
}

@end
