#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""
Wraps the gtk treeview in something a little nicer.
"""

from GroundControl import GLADE_DIR, PIXMAP_DIR, __appname__
from GroundControl.base import Thread

import os
import gtk
import gobject
import glib
import logging

MISSING = None

class IconManager(object):
    """Manage a set of cached icons"""
    def __init__(self, location):
        self.location = os.path.join(PIXMAP_DIR, location)
        self.cache    = {}
        self.get_icon('default')

    def get_icon(self, name):
        """Simple method for getting a set of pix icons and caching them."""
        if not name:
            name = 'default'
        if not self.cache.has_key(name):
            icon_path = self.icon_path(name)
            if os.path.exists(icon_path):
                try:
                    self.cache[name] = gtk.gdk.pixbuf_new_from_file(icon_path)
                except glib.GError, msg:
                    logging.warn(_("No icon '%s',%s") % (icon_path, msg))
            else:
                self.cache[name] = None
                logging.warning(_("Can't find icon for %s in %s") % (
                    name, self.location))
        if not self.cache.has_key(name) or not self.cache[name]:
            name = 'default'
        return self.cache.get(name, MISSING)

    def icon_path(self, name):
        """Returns the icon path based on stored location"""
        svg_path = os.path.join(self.location, '%s.svg' % name)
        png_path = os.path.join(self.location, '%s.png' % name)
        if os.path.exists(name) and os.path.isfile(name):
            return name
        if os.path.exists(svg_path) and os.path.isfile(svg_path):
            return svg_path
        elif os.path.exists(png_path) and os.path.isfile(png_path):
            return png_path
        return os.path.join(self.location, name)


class GtkApp(object):
    """
    This wraps gtk builder and allows for some extra functionality with
    windows, especially the management of gtk main loops.

    start_loop - If set to true will start a new gtk main loop.
    *args **kwargs - Passed to primary window when loaded.
    """
    gtkfile = None
    windows = None

    def __init__(self, *args, **kwargs):
        self.main_loop = glib.main_depth()
        start_loop = kwargs.pop('start_loop', False)
        self.w_tree = gtk.Builder()
        self.w_tree.set_translation_domain(__appname__)
        self.w_tree.add_from_file(self.gapp_xml)
        self.signals = {}
        self.widget  = self.w_tree.get_object
        # Now start dishing out initalisation
        self.init_gui(*args, **kwargs)
        self.w_tree.connect_signals(self.signals)
        self.signals = None
        # Start up a gtk main loop when requested
        if start_loop:
            logging.debug("Starting new GTK Main Loop.")
            gtk.main()

    @property
    def gapp_xml(self):
        """Load any given gtk builder file from a standard location"""
        path = os.path.join(GLADE_DIR, self.gtkfile)
        if not os.path.exists(path):
            raise Exception("Gtk Builder file is missing: %s" % path)
        return path

    def init_gui(self, *args, **kwards):
        """Initalise all of our windows and load their signals"""
        self._loaded = {}
        self._primary = None
        if self.windows:
            for cls in self.windows:
                window = cls(self)
                self._loaded[window.name] = window
                if window.primary:
                    if not self._primary:
                        
                        window.load(*args, **kwards)
                        self._primary = window
                    else:
                        logging.error(_("More than one window is set Primary!"))

    def add_signal(self, name, function):
        """Add a signal to this gtk wtree object"""
        if self.signals != None:
            self.signals[name] = function
        else:
            raise Exception("Unable to add signal '%s' - too late!" % name)

    def load_window(self, name, *args, **kwargs):
        """Load a specific window from our group of windows"""
        if self._loaded.has_key(name):
            self._loaded[name].load(*args, **kwargs)
            return self._loaded[name]

    def exit(self):
        """Exit our gtk application and kill gtk main if we have to"""
        if self.main_loop < glib.main_depth():
            # Quit Gtk loop if we started one.
            logging.debug("Quit '%s' Main Loop." % self._primary.name)
            gtk.main_quit()
            # You have to return in order for the loop to exit
            return 0


class Window(object):
    """
    This wraps gtk windows and allows for having parent windows as well
    as callback events when done and optional quiting of the gtk stack.

    name = 'name-of-the-window'

    Should the window be the first loaded and control gtk loop:

    primary = True
    """
    name    = None
    primary = True

    def __init__(self, gapp):
        self.gapp = gapp
        # Set object defaults
        self.dead     = False
        self.done     = False
        self.parent   = None
        self.callback = None
        self._args    = {}
        # Setup the gtk app connection
        self.widget = self.gapp.widget
        # Setup the gtk builder window
        self.window = self.widget(self.name)
        if not self.window:
            raise Exception("Missing window '%s'" % (self.name))
        # These are some generic convience signals
        self.window.connect('destroy', self._exiting)
        self.add_generic_signal('destroy', self.destroy_window)
        self.add_generic_signal('close', self.destroy_window)
        self.add_generic_signal('cancel', self.cancel_changes)
        self.add_generic_signal('apply', self.apply_changes)
        self.add_generic_signal('dblclk', self.double_click)
        # Now load any custom signals
        for name, value in self.signals().iteritems():
            self.gapp.add_signal(name, value)

    def load(self, parent=None, callback=None):
        """Show the window to the user and set callbacks"""
        # If we have a parent window, then we expect not to quit
        self.parent = parent
        self.callback = callback
        if self.parent:
            self.parent.set_sensitive(False)
        self.window.show()

    def signals(self):
        """Replace this with a method returning dict of signals"""
        return {}

    def add_generic_signal(self, name, function):
        """Adds a generic signal to the gapp"""
        self.gapp.add_signal("%s_%s" % (self.name, name), function)

    def load_window(self, name, *args, **kwargs):
        """Load child window, automatically sets parent"""
        kwargs['parent'] = self.window
        return self.gapp.load_window(name, *args, **kwargs)

    def get_args(self):
        """Nothing to return"""
        return {}

    def double_click(self, widget, event):
        """This is the cope with gtk's rotten support for mouse events"""
        if event.type == gtk.gdk._2BUTTON_PRESS:
            return self.apply_changes(widget)

    def apply_changes(self, widget=None):
        """Apply any changes as required by callbacks"""
        valid = self.is_valid()
        if valid == True:
            logging.debug("Applying changes")
            self.done = True
            self._args = self.get_args()
            self.post_process()
            self.destroy_window()
        else:
            self.is_valid_failed(valid)

    def post_process(self):
        """Do anything internally that needs doing"""
        pass

    def is_valid(self):
        """Return true is all args are valid."""
        return True

    def is_valid_failed(self, reason=None):
        """This is what happens when we're not valid"""
        logging.error(_("Child arguments aren't valid: %s") % str(reason))

    def cancel_changes(self, widget=None):
        """We didn't like what we did, so don't apply"""
        self.done = False
        self.destroy_window()

    def destroy_window(self, widget=None):
        """We want to make sure that the window DOES quit"""
        if self.primary:
            self.window.destroy()
        else:
            self.window.hide()
            self._exiting(widget)

    def _exiting(self, widget=None):
        """Internal method for what to do when the window has died"""
        if self.callback and self.done:
            logging.debug("Calling callback function for window")
            self.callback(**self._args)
            self.done = False
        else:
            logging.debug("Cancelled or no callback for window")

        if self.parent:
            # We assume the parent didn't load another gtk loop
            self.parent.set_sensitive(True)
        # Exit our entire app if this is the primary window
        if self.primary:
            self.gapp.exit()
            if self.done:
                self.sucess(**self._args)
        self.dead = True

    def sucess(self, **kwargs):
        """A callback for when the gtk is finally gone and we're finished"""
        pass


class ChildWindow(Window):
    """
    Base class for child window objects, these child windows are typically
    window objects in the same gtk builder file as their parents. If you just want
    to make a window that interacts with a parent window, use the normal
    Window class and call with the optional parent attribute.
    """
    primary = False


anicons = IconManager('animation')

class ThreadedWindow(Window):
    """
    This class enables an extra status stream to cross call gtk
    From a threaded process, allowing unfreezing of gtk apps.
    """
    def __init__(self, *args, **kwargs):
        # Common variables for threading
        self._thread = None
        self._closed = False
        self._calls  = [] # Calls Stack
        self._unique = {} # Unique calls stack
        # Back to setting up a window
        super(ThreadedWindow, self).__init__(*args, **kwargs)

    def load(self, *args, **kwargs):
        """Load threads and kick off status"""
        self._anistat = self.widget("astat")
        self._anicount = 1
        # Kick off our initial thread
        self.start_thread(self.inital_thread)
        # Back to loading a window
        super(ThreadedWindow, self).load(*args, **kwargs)

    def start_thread(self, method, *args, **kwargs):
        """Kick off a thread and a status monitor timer"""
        if not self._thread or not self._thread.isAlive() and not self._closed:
            self._thread = Thread(target=method, args=args, kwargs=kwargs)
            self._thread.start()
            logging.debug("-- Poll Start %s --" % self.name)
            # Show an animation to reflect the polling
            if self._anistat:
                self._anistat.show()
            # Kick off a polling service, after a delay
            gobject.timeout_add( 300, self.thread_through )
        else:
            raise Exception("Thread is already running!")

    def thread_through(self):
        """Keep things up to date fromt he thread."""
        self.process_calls()
        if self._thread.isAlive():
            #logging.debug("-- Poll Through %s --" % self.name)
            # This will allow our poll to be visible to the user and
            # Has the advantage of showing stuff is going on.
            if self._anistat:
                self._anicount += 1
                if self._anicount == 9:
                    self._anicount = 1
                image = anicons.get_icon(str(self._anicount))
                self._anistat.set_from_pixbuf(image)
            gobject.timeout_add( 100, self.thread_through )
        else:
            logging.debug("-- Poll Quit %s --" % self.name)
            # Hide the animation by default and exit the thread
            if self._anistat:
                self._anistat.hide()
            self.thread_exited()

    def inital_thread(self):
        """Replace this method with your own app's setup thread"""
        pass

    def thread_exited(self):
        """What is called when the thread exits"""
        pass

    def call(self, name, *args, **kwargs):
        """Call a method outside of the thread."""
        unique = kwargs.pop('unique_call', False)
        if type(name) != str:
            raise Exception("Call name must be a string not %s" % type(name))
        call = (name, args, kwargs)
        # Unique calls replace the previous calls to that method.
        if unique:
            self._unique[name] = call
        else:
            self._calls.append( call )

    def process_calls(self):
        """Go through the calls stack and call them, return if required."""
        while self._calls:
            (name, args, kwargs) = self._calls.pop(0)
            logging.debug("Calling %s" % name)
            ret = getattr(self, name)(*args, **kwargs)
        for name in self._unique.keys():
            (name, args, kwargs) = self._unique.pop(name)
            ret = getattr(self, name)(*args, **kwargs)

    def _exiting(self, widget=None):
        if self._thread and self._thread.isAlive():
            logging.warn(_("Your thread is still active."))
        self._closed = True
        super(ThreadedWindow, self)._exiting(widget)


class BaseView(object):
    """Controls for tree and icon views, a base class"""

    def __init__(self, dbobj, selected=None, unselected=None, name=None):
        self.selected_signal = selected
        self.unselected_signal = unselected
        self._iids = []
        self._list = dbobj
        self._name = name
        self.selected = None
        self._model   = None
        self._data    = None
        self.no_dupes = True
        self.connect_signals()
        self.setup()
        super(BaseView, self).__init__()

    def connect_signals(self):
        """Try and connect signals to and from the view control"""
        raise NotImplementedError, "Signal connection should be elsewhere."

    def setup(self):
        """Setup any required aspects of the view"""
        return self._list

    def clear(self):
        """Clear all items from this treeview"""
        for iter_index in range(0, len(self._model)):
            try:
                del(self._model[0])
            except IndexError:
                logging.error(_("Could not delete item %d") % iter_index)
                return

    def add(self, target, parent=None):
        """Add all items from the target to the treeview"""
        for item in target:
            self.add_item(item, parent=parent)

    def add_item(self, item, parent=None):
        """Add a single item image to the control"""
        if item:
            iid = self.get_item_id(item)
            if iid:
                if iid in self._iids and self.no_dupes:
                    logging.debug("Ignoring item %s in list, duplicate." % iid)
                    return None
                self._iids.append(iid)
            result = self._model.append(parent, [item])
            # item.connect('update', self.updateItem)
            return result
        else:
            raise Exception("Item can not be None.")

    def get_item_id(self, item):
        """Return if possible an id for this item,
           if set forces list to ignore duplicates,
           if returns None, any items added."""
        return None

    def replace(self, new_item, item_iter=None):
        """Replace all items, or a single item with object"""
        if item_iter:
            self.remove_item(target_iter=item_iter)
            self.add_item(new_item)
        else:
            self.clear()
            self._data = new_item
            self.add(new_item)

    def item_selected(self, item=None):
        """Base method result, called as an item is selected"""
        if self.selected != item:
            self.selected = item
            if self.selected_signal and item:
                self.selected_signal(item)
            elif self.unselected_signal and not item:
                self.unselected_signal(item)

    def remove_item(self, item=None, target_iter=None):
        """Remove an item from this view"""
        if target_iter and not item:
            return self._model.remove(target_iter)
        target_iter = self._model.get_iter_first()
        for itemc in self._model:
            if itemc[0] == item:
                return self._model.remove(target_iter)
            target_iter = self._model.iter_next(target_iter)

    def item_double_clicked(self, items):
        """What happens when you double click an item"""
        return items # Nothing

    def get_item(self, iter):
        """Return the object of attention from an iter"""
        return self._model[iter][0]


class TreeView(BaseView):
    """Controls and operates a tree view."""
    def connect_signals(self):
        """Attach the change cursor signal to the item selected"""
        self._list.connect('cursor_changed', self.item_selected_signal)

    def selected_items(self, treeview):
        """Return a list of selected item objects"""
        # This may need more thought, only returns one item
        item_iter = treeview.get_selection().get_selected()[1]
        try:
            return [ self.get_item(item_iter) ]
        except TypeError, msg:
            logging.debug("Error %s" % msg)

    def item_selected_signal(self, treeview):
        """Signal for selecting an item"""
        items = self.selected_items(treeview)
        if items:
            return self.item_selected( items[0] )

    def item_button_clicked(self, treeview, event):
        """Signal for mouse button click"""
        if event.type == gtk.gdk.BUTTON_PRESS:
            self.item_double_clicked( self.selected_items(treeview)[0] )

    def expand_item(self, item):
        """Expand one of our nodes"""
        self._list.expand_row(self._model.get_path(item), True)

    def setup(self):
        """Set up an icon view for showing gallery images"""
        self._model = gtk.TreeStore(gobject.TYPE_PYOBJECT)
        self._list.set_model(self._model)
        return self._list


class IconView(BaseView):
    """Allows a simpler IconView for DBus List Objects"""
    def connect_signals(self):
        """Connect the selection changed signal up"""
        self._list.connect('selection-changed', self.item_selected_signal)

    def setup(self):
        """Setup the icon view control view"""
        self._model = gtk.ListStore(str, gtk.gdk.Pixbuf, gobject.TYPE_PYOBJECT)
        self._list.set_model(self._model)
        return self._list

    def item_selected_signal(self, icon_view):
        """Item has been selected"""
        self.selected = icon_view.get_selected_items()

