/************************************************************************
 *                                                                        *
 *   Copyright (C) 2001 Grub, Inc.                                        *
 *                                                                        *
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 1, or (at your option)  *
 *   any later version.                                                   *
 *                                                                        *
 *   This program is distributed in the hope that it will be useful,      *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *   GNU General Public License for more details.                         *
 *                                                                        *
 *   You should have received a copy of the GNU General Public License    *
 *   along with this program; if not, write to the Free Software          *
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *                                                                        *
 *                                                                        *
 **************************************************************************/

#include "Coordinator.h"

using namespace std;

static Coordinator *cord_sigterm = NULL;

// function for threading off the gui portion
static void *startGui(void *stuff)
{
	((gui*)stuff)->start();
	return NULL;
}

Coordinator::Coordinator()
{
	Client_Protocol = NULL;
	Client_Crawler = NULL;
	
	// create a new client database in which to store the crawled data
	Client_Data = new ClientDB();

	// set up the "verboseness", or amount of data printed to the server
	// NEEDS TO BE UPDATED OR REWORKED - reported 01/14/02 by kord
	if (Crawler_Status_Info.superverbose && Crawler_Status_Info.verbose) 
	{
		setVerboseMode(1, 1); 
	}
	else if (Crawler_Status_Info.verbose) 
	{
		setVerboseMode(1, 0);
	}
	else
	{
		setVerboseMode(0, 0); 
	}
}

Coordinator::~Coordinator()
{
	if (Client_Protocol != NULL) delete Client_Protocol;
	if (Client_Crawler != NULL) delete Client_Crawler ;
	if (Client_Data != NULL) delete Client_Data;
}

void *Coordinator::runCoordinator()
{
	int num_server_errors = 0;
	int reload_settings_count = 0;
	int continue_run = true;

	// create our GUI instance
	gui *grub;

	if (!Crawler_Status_Info.console) 
	{
		// set up the pthread paramaters for the gui
		pthread_attr_t gui_attr;
		pthread_attr_init(&gui_attr);
		pthread_attr_setdetachstate(&gui_attr, PTHREAD_CREATE_DETACHED);
		int p_gui_error;
		pthread_t gui_pid;

		// thread off the gui
		grub = new gui();
		p_gui_error = pthread_create(&gui_pid, &gui_attr, startGui, (void *)grub);
	}

	/*                        */
	/* Start main client loop */
	/*                        */
	while (continue_run)
	{
		try 
		{
			// clear out the limit indicator - this boolean simply
			// show whether or not bandwidth limiting is occuring
			Crawler_Status_Info.bandwidth_limit = 0;
			
			if (reload_settings_count > HUP_INTERVAL)
			{
				hupSettings();	
				reload_settings_count = 0;
			}
			else
			{
				reload_settings_count++;
			}
	
			switch (Crawler_Status_Info.current_action)
			{
				case 0 :// code falling through here is a GET from the server
					// which indicates URLs flowing into the archive to 
					// crawl.  we sleep a little to give things time to
					// settle down in the GUI(s)
					sleep(3);

				case 2 :// code starting here (instead of above) is a PUT to
					// the server which indicates URLs and data uploading 
					// to the server from the crawl
					Crawler_Status_Info.current_url = 0; 

					Client_Protocol = new ClientTalk(Client_Data,
									 Crawler_Status_Info.current_action,
									 Config_File_Info.AltClientID,
									 Config_File_Info.PortNumber,
									 1.0);

					// tell the protocol if we are a GET(0) or a PUT(2)
					Client_Protocol->setAct(Crawler_Status_Info.current_action);

					// start the new protocol transfer
					Client_Protocol->start();

					// if we were a get, we set an indicator to the number of URLs that we just got
					if ( Crawler_Status_Info.current_action == 0 ) 
					{
						Crawler_Status_Info.max_url = Client_Protocol->getURLCount();
					}

					// delete the protocol instance now
					delete Client_Protocol ;
					Client_Protocol = NULL;
					
					// indicate that we are error free for the moment
					num_server_errors = 0;

					// break out
					break;

				case 1 :// code starting here begins a crawl of the URLs pulled
					// down from the server.  the crawler class threads off
					// seperate crawlers once we enter the class itself.
				        int crawl_result;

					Client_Crawler = new Crawler(Client_Data);

					crawl_result = Client_Crawler->start();

					// older code that examined return from crawler errors
					// but newer crawler code always return zero (0) this 
					// test has been left in here for historic reasons -
					// older code is available on CVS
					if (crawl_result != 0) 
					{  
					}
					
					// delete the crawling instance now
					delete Client_Crawler ;
					Client_Crawler = NULL;
					break;
			}

						
		
			// this line cycles current action through 0, 1 and 2
			Crawler_Status_Info.current_action = ( Crawler_Status_Info.current_action + 1 ) % 3;

		}

 		catch (ProtocolExp& e) {
			int code;

 			clog(GCLOG_ERR,"(cord) Caught Protocol Exception %s",e.what());
			clog(GCLOG_INFO,"(cord) Resetting protocol...");

			/* some type of server error was detected
			 or client received no url's from server,
			 client will attempt to wait */
 			if( ((code = e.getcode()) == COMM_ERROR) || code == NO_URLS )
			{
				if ( code == COMM_ERROR )
				{
					Verboseprintf("Unable to connect to the server...");
				}
				else
				{
					Verboseprintf("No URLs received from the server...");
				}
	
				num_server_errors++;       

				// indicate that we are sleeping, call the delay function, which 
				// causes a variable sleep time in relation to the number of errors
				// that we have seen, then indicate we are back from sleeping
				Crawler_Status_Info.sleep_indicator = true;
				Retry_Delay(num_server_errors); 
				Crawler_Status_Info.sleep_indicator = false;
 			}
	
			// reset our current action to a GET(0)
			Crawler_Status_Info.current_action = 0;
	
			// empty our archive - remove this later when we begin
			// using the metakit database.  we should be able to
			// recover the crawl data at that time and return it.
 			// Client_Data->emptyArchive();

			// delete the protocol stuff
 			delete Client_Protocol ;
 			Client_Protocol = NULL;
		}

		// check to see if we are cleared to continue - halt only if we didn't just get through crawling
		// this allows us to finish a crawl, and then send the results back without losing any crawl data
		if ( Crawler_Status_Info.coordinator_quit && (Crawler_Status_Info.current_action == 0 || 
							      Crawler_Status_Info.current_action == 2) )
		{
			continue_run = false;
			clog(GCLOG_INFO,"(cord) Halt signal sent to coordinator thread - shutting down...");
		}

		// check to see if both the crawler and the coordinator have been told to quit, this doesn't
		// make any allowances for sending stuff back to the server - good if we have a bad crawl
		if ( Crawler_Status_Info.coordinator_quit && Crawler_Status_Info.crawler_quit )
		{
			continue_run = false;
			clog(GCLOG_INFO,"(cord) Halt signal sent to coordinator and crawler - shutting down...");
		}
	
		// check to see if a per run limit is enabled and has been reached
		if ( Crawler_Status_Info.url_run > 0 )
		{
			if ( (Crawler_Status_Info.url_run < Crawler_Status_Info.total_urls + 1) && 
			     (Crawler_Status_Info.current_action == 0 || Crawler_Status_Info.current_action == 2) )
			{
				continue_run = false;
				clog(GCLOG_INFO,"(cord) Per run URL limit reached: %d - halting execution...", 
						      Crawler_Status_Info.url_run);
			}
		}

		// check to see if a per run limit is enabled and has been reached
		if ( Crawler_Status_Info.byte_run > 0 )
		{
			if ( (Crawler_Status_Info.byte_run < Crawler_Status_Info.total_bytes) && 
			     (Crawler_Status_Info.current_action == 0 || Crawler_Status_Info.current_action == 2) )
			{
				continue_run = false;
				clog(GCLOG_INFO,"(cord) Per run BYTE limit reached: %d - halting execution...", 
						      Crawler_Status_Info.total_bytes);
			}
		}
	}
	
	return (NULL);	
}

void Coordinator::start()
{
	// much of the settings stuff needs to be moved to the main
	// program group, although rereading the config file should
	// be an option that is available to the coordinator.  we
	// need to reorganize this to make it clearer. 
	
	// mask out any signals so main process can handle them for us
	sigset_t newsig;
	sigemptyset(&newsig);
	sigaddset(&newsig, SIGINT);
	pthread_sigmask(SIG_BLOCK, &newsig, NULL);

	// log the start of our main functions
	clog(GCLOG_INFO, "Grub client starting up...");
	Verboseprintf("Grub client starting up...\n");

	if (Crawler_Status_Info.dumb_mode == true)
	{
		while (Crawler_Status_Info.coordinator_pause == true)
		{
			// wait around for settings to appear from elsewhere
			struct timeval tv;
			tv.tv_sec = 0;
			tv.tv_usec = 1000;
			select (0, NULL, NULL, NULL, &tv);

			// test to see if we are being asked to quit
			if (Crawler_Status_Info.coordinator_quit == true)
			{
				pthread_exit(0);
			}
		}

		clog(GCLOG_INFO,"Settings have been updated from the grubface.");
	}
	else
	{
		// log the proxy info because it is used to contact the configuration server
		clog(GCLOG_INFO,"Proxy is: %s ", Config_File_Info.Proxy);
		clog(GCLOG_INFO,"ProxyPort is: %d ", Config_File_Info.ProxyPort);

		if (Config_File_Info.ClientID > 999998) {
			clog(GCLOG_ERR,"Incorrect client ID specified.");
			fprintf( stderr, "Incorrect client ID specified - did you check your grub.conf file?\n");
			pthread_exit(0);
		}
		
		int settings_loop_count = 0;

		// grab the server suggested settings and version info
		ret = readServerSettings();

		// check to see if we have good results
		if (ret == 1)
		{
			// readServerSettings didn't like us, so bail
			clog(GCLOG_INFO, "Error reading from the remote configuration server!");
			clog(GCLOG_INFO, "Using defaults settings from config file!");

			// can't proceed if we don't have an alternate client id 
			if(Config_File_Info.AltClientID == 0) 
			{
				clog(GCLOG_ERR,"No Alternate Connect ID specified in the config file.");
				clog(GCLOG_ERR,"Settings server was not available to retrieve Alt Connect ID.");
				fprintf( stderr, "\n");
				fprintf( stderr, "Something has gone wrong with contacting the remote setting server.\n");
				fprintf( stderr, "It is likely that your password is either incorrect, or the client \n");
				fprintf( stderr, "was not authenticated with our server for some other reason.  It is\n");
				fprintf( stderr, "still possible to run the client without contacting the remote server\n");
				fprintf( stderr, "first, but you need to have your current alternate connect ID placed\n");
				fprintf( stderr, "in your configuration file - which you don't.  You can investigate \n");
				fprintf( stderr, "further by checking the grubclient.log file in the working directory.\n");
				fprintf( stderr, "\n");
				pthread_exit(0);
			}
		}
		else
		{
			// NewVersionFlag is located in Status_Info because it does not need to
			// be set via the config file routines, but is set via the remote server
			if (Crawler_Status_Info.NewVersionFlag == 1)
			{
				fprintf( stderr, "\n");
				fprintf( stderr, "A newer version of the client is now available for download.  It is\n");
				fprintf( stderr, "suggested, but not required, that you download the new version from\n");
				fprintf( stderr, "us at http://www.grub.org/client.html.\n");
				fprintf( stderr, "\n");
			}

			// if the result is 2, it means that we don't want to let the client run
			// so we warn the user and do an exit of the program
			if (Crawler_Status_Info.NewVersionFlag == 2)
			{
				fprintf( stderr, "\n");
				fprintf( stderr, "A newer version of the client is now available for download.  It's a\n");
				fprintf( stderr, "required upgrade - you can download the updated software from us at\n");
				fprintf( stderr, "http://www.grub.org/client.html.\n");
				fprintf( stderr, "\n");
				pthread_exit(0);
			}
		}

		clog(GCLOG_INFO,"Settings have been updated from remote server.");
	}

	// log all the settings
	clog(GCLOG_INFO,"ClientID is: %d", Config_File_Info.ClientID);
	clog(GCLOG_INFO,"AltClientID is: %u", Config_File_Info.AltClientID);
	clog(GCLOG_INFO,"NumOfCrawlersToRun is: %d", Config_File_Info.NumOfCrawlersToRun);
	clog(GCLOG_INFO,"MaxAmountOfBandwidth is: %d", Config_File_Info.MaxAmountOfBandwidth);
	clog(GCLOG_INFO,"ThreadsPerHost is: %d", Config_File_Info.ThreadsPerHost);
	clog(GCLOG_INFO,"PortNumber is: %d ", Config_File_Info.PortNumber);
	clog(GCLOG_INFO,"NewVersionFlag is : %d", Crawler_Status_Info.NewVersionFlag);

	// begin the coordinator main process
	runCoordinator();

	// on return exit
	pthread_exit(0);
}

// function to reload the server settings and statistics
void Coordinator::hupSettings()
{
	if (Crawler_Status_Info.dumb_mode = false)
	{
		clog(GCLOG_INFO, "Restart signaled, reloading config file and server settings...");
		int ret = readServerSettings();
	}
}

