/* Project: GRUB http://www.grub.org
 * moduel: CSTAT
 * purpose: dealing with logging, error, statistic
 * Ben Guofu Wu bante@mailandnews.com
 * Last revsion: 12 August 2000
 * Files: gc_log.cpp 
 * See also: gc_log.h, gc_log_test.cpp, gc_thread_test.cpp, Thread.t.cpp
 *
 * To do:
 *   1. support for statistic not implemented yet
 *   2. The format of the log record should be configurable
 * 
 * This module is only tested on linuxPPC1999 at the moment. 
 *
 */
/* Maintainng records:
 * 1. 12 August 2000: 
 *    $$ The mutex_log becomes a member of the class rather than a global 
 *       variable. In the GRUB client, there should be only one copy of 
 *       object this class in order to write to the public log file.  
 *    $$ The class maintains a list of the log files. The oldest one will
 *       be deleted when a new one is created and the number of the log files
 *       exceeds the max_num_of_logfiles.
 * 2. 
 *    
*/
#include "gc_log.h"

#include <string.h> //strcpy ..
#include <unistd.h>  // fstat, rename
#include <sys/stat.h>  //stat 
#include <sys/types.h> // stat
#include <time.h> // time_t
#include <fcntl.h> //o_creat
#include <dirent.h> //scan dir. Don't know VC++ has it
//#define _DEBUG

#include <iostream.h>
#include <string>

using std::string;

typedef struct
{
	char *t_name;
	int t_val;
} Log_type;

static const Log_type priorities[]=
{
	{"trace", GCLOG_TRACE},
	{"debug", GCLOG_DEBUG},
	{"info",  GCLOG_INFO},
	{"error", GCLOG_ERR},
	{"alert", GCLOG_ALERT},
	{"crit",  GCLOG_CRIT},
	{"emerg", GCLOG_EMERG},
	{NULL, -1},
};



/* this constructor initializes some configuration variables such as max_size
 * log_level.  */
GC_Log::GC_Log(FILE *stream, int max_size, bool log_to_stdout_also,
		unsigned mask)
{
	remove_onmax = false;  /* this feature is unavailable here */
	open_failed = true;
	open_by_me = false;
	log_stream = stream;
	MAX_SIZE = max_size;
	LOG_TO_STDOUT_ALSO = log_to_stdout_also;
	log_mask = mask;
#ifdef USE_THREADS
	mutex_log = new Mutex();
#endif
	//set the default path. The default path can be changed using a member 
	//function
	default_path = DEFAULT_LOG_PATH; //"." 
	max_num_of_logfiles = DEFAULT_NUM_LOG_FILES; //15

	if(log_stream == NULL)
	{
		fprintf(stderr, "no stream\n");
		return;
	}
	int fd;
	if((fd = fileno(log_stream)) < 0) 
	{
#ifdef _DEBUG
		fprintf(stderr, "can not get fd\n");
#endif
		return;
	}
	//check if the stream is a regular file or not
	struct stat buf;
	if(fstat(fd, &buf) < 0)
	{
		fprintf(stderr, "Can not get the stat of the log stream\n");
		return;
	}
	//if the stream is not a reguar file, do nothing further
	if(!S_ISREG(buf.st_mode))
	{
		open_failed = false;
		return;
	}	

#ifdef USE_THREADS
	//get the log file list when the stream is a regular file
	mutex_log->Lock();
#endif
	if(!get_logfile_list())
	{
#ifdef USE_THREADS
		mutex_log->Unlock();
#endif
		fprintf(stderr, "can not get the log file lists\n");
		return;
	}
#ifdef _DEBUG
	/////////////////////////
	for(int i=0;i<logfile_queue.size();i++)
		cout<<logfile_queue[i].name<<endl;
	/////////////
#endif
	if(!check_log_file()) 
	{
#ifdef USE_THREADS
		mutex_log->Unlock();
#endif
#ifdef _DEBUG
		fprintf(stdout, "mutex_log unlocked after checkfile failed\n");
#endif	
		return;
	}	
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif
	open_failed = false;
#ifdef _DEBUG
#ifdef USE_THREADS
	fprintf(stdout, "mutex_log unlocked \n");
#endif
	fprintf(stdout, "constructor sucessful\n");
#endif
}

/* This constructor will try to open a file with a specified name, if it exists,
 * check its size. If it doesn't exist, create a new one for writing */
GC_Log::GC_Log(char *file, int max_size, bool log_to_stdout_also, 
		unsigned mask, bool remove_on_max = false)
{
	remove_onmax = remove_on_max;
	open_failed = true;
	open_by_me = true;
	MAX_SIZE = max_size;
	LOG_TO_STDOUT_ALSO = log_to_stdout_also;
	log_mask = mask;
	//set the default path. The default path can be changed using a member 
	//function
	default_path = DEFAULT_LOG_PATH; //"."
	max_num_of_logfiles = DEFAULT_NUM_LOG_FILES; //15
#ifdef USE_THREADS
	mutex_log = new Mutex();
	mutex_log->Lock();
#endif
	if(!get_logfile_list())
	{
#ifdef USE_THREADS
		mutex_log->Unlock();
#endif
		fprintf(stderr, "can not get the log file lists\n");
		return;
	}
	//if *file is not in the logfile list, means that it
	//is totally new. push it to the back
	bool in = false;
	for(int i=0; i<logfile_queue.size();i++)
	{
		if(strcmp(file, logfile_queue[i].name.c_str())==0)
		{
			in=true;
			break;
		}
	}
	if(!in) 
	{
		logfile_queue.push_back(filename(file));
	}	
#ifdef _DEBUG
	/////////////////////////
	for(int i=0;i<logfile_queue.size();i++)
		cout<<logfile_queue[i].name<<endl;
	/////////////
#endif
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif

	//check the file name
	if (file == NULL || *file == '\0')
	{
		fprintf(stderr, "Illegal file name.\n");
		return;
	}
	strncpy( orig_filename, file, MAX_STRING_LEN );
	orig_filename[MAX_STRING_LEN-1] = '\0';

	//open the file
#if 0
	int fd = open(file, O_APPEND | O_CREAT, PERMS );
	if(fd<0) 
	{
		fprintf(stderr, "can not open file %s\n", file);
		return;
	}
	log_stream = fdopen(fd, "a");
#endif
	log_stream = fopen(file, "a");
	if(log_stream == NULL)
	{
		fprintf(stderr, "can not open stream %s\n", file);
		//close(fd);
		return;
	}
	//check the size of the file
#ifdef USE_THREADS
	mutex_log->Lock();
#endif
	if(!check_log_file()) 
	{
#ifdef USE_THREADS
		mutex_log->Unlock();
#endif
#ifdef _DEBUG	
		fprintf(stderr, "check file failed in constructor\n");
#endif
		return;
	}
#ifdef USE_THREADS
	mutex_log->Unlock(); 
#endif
	open_failed = false;
#ifdef _DEBUG 
	fprintf(stderr, "Constructor successful\n");
#endif
}

GC_Log::~GC_Log()
{
	if(open_by_me && (!open_failed)) fclose(log_stream);
#ifdef USE_THREADS
	delete mutex_log;
#endif
	//if(open_by_me && (log_stream != NULL)) fclose(log_stream);
}

/* check the size of the log file, if its size exceeds max_size, 
 * open a new log file. if everything ok, return true */
bool GC_Log::check_log_file()
{
	
	//get the file descriptor of the stream 
 	int fd = fileno(log_stream);
	if(fd < 0) 
	{
		fprintf(stderr, 
			"Can not find the file descriptor of the stream\n");
		return false;
	}
	struct stat buf;
	if(fstat(fd, &buf) < 0)
	{
		fprintf(stderr, "Can not get the stat of the log stream\n");
		return false;
	}
	//if the stream is not a reguar file, do nothing
	if(!S_ISREG(buf.st_mode)) return true;
	//if the stream is a regular file
	/* if the size of the log file exceecs max_size,
	 * close the stream. and create a new one with the name like:
	 * default_log_file_name_rawtime.log*/
	if(buf.st_size >= MAX_SIZE)
	{
#ifdef _DEBUG 
		fprintf(stdout, "log file size exceeds mas size\n");
#endif
		//close the stream
		if(fclose(log_stream)<0)
		{
			fprintf(stderr, 
				"can not close the old stream\n");
			return false;
		}
		//create a string as a file name (default_name+time.log)
#ifdef _DEBUG
		fprintf(stdout, "close the original log file\n");
#endif
		time_t raw_time;
		char s_time[256];
		char abs_file_name[MAX_PATH_LEN];
		char new_file_name[MAX_PATH_LEN/2];

		if ( remove_onmax )
		{
			/* ozra's addition:
			 * if a mode that deletes the whole log file once it reaches
			 * the max size is used, just delete the log file and reopen
			 * it
			 */

			if ( ! open_by_me ) {

				fprintf(stderr, 
					"cannot use remove-on-max feature on files "
					"not opened originally by this module\n");
				return false;
			}
			strncpy( new_file_name, orig_filename, MAX_PATH_LEN/2 );
			new_file_name[MAX_PATH_LEN/2-1] = '\0';
		}
		else {

			(void)time(&raw_time);
			sprintf(s_time, "%d", raw_time);
			sprintf(new_file_name,"%s", DEFAULT_LOG_FILE_NAME);
			sprintf(new_file_name+strlen(new_file_name), "%s",s_time);
			sprintf(new_file_name+strlen(new_file_name), "%s",".log");
		}
		//the absolute path = default_path+"/"+"new_file_name"
		sprintf(abs_file_name,"%s%s%s", default_path, PATH_DELIMITOR,
				new_file_name);
		//open the file for writing in the default path
		char current_path[MAX_PATH_LEN];
		getcwd(current_path, MAX_PATH_LEN);
	 	if(chdir(default_path) != 0)
		{
			fprintf(stderr, "can not go to path %s\n",
					default_path);
			open_failed = true;
			return false;
		}
		if ( remove_onmax )
			log_stream = fopen(new_file_name, "w");
		else
			log_stream = fopen(new_file_name, "a");
		if(log_stream ==NULL) 
		{
			fprintf(stderr,
				"can not create a new log file\n");
			open_failed = true;
			chdir(current_path);
			return false;
		}
		//recover the path
		chdir(current_path);
		open_by_me = true;
		open_failed = false;

		if ( remove_onmax ) {

			/* don't need to deal with the queue; just exit */
			return true;
		}

#ifdef _DEBUG
		fprintf(stdout, "New log file %s created\n", abs_file_name);
#endif
		//maintain the log file list. 
		while(logfile_queue.size() >= max_num_of_logfiles)
		{
			if(unlink(logfile_queue.front().name.c_str()) == -1)
			{
		             char msg[1024];
			     sprintf(msg, "fail to delete %s",
			        logfile_queue.front().name.c_str());
			     perror(msg);
			}    
			   
#ifdef _DEBUG 
			else
			cout<<logfile_queue.front().name<<" is deleted"<<endl;
#endif
			logfile_queue.pop_front();
		}
		logfile_queue.push_back(filename(abs_file_name));
	}
  /* if the stream is stdout/socket/pipe/device stream,
   * its "size" is not checked */
	return true;
}//end of check_log_file


/* get the log file list at the default path */
bool GC_Log::get_logfile_list()
{
	//get the log files at the default directory
	//currently it only read files whose names are in the format of
	//gc_log+rawtime.log. It makes things much easier.
	DIR *dp;
	struct dirent *entry;
	struct stat stabuf;
	if((dp = opendir(default_path)) == NULL)
	{
		fprintf(stderr, "can not open directory %s\n", default_path);
		return false;
	}
	//get the current path and store it
	char current_path[MAX_PATH_LEN];
	getcwd(current_path, MAX_PATH_LEN);
	//go to the log file path
	if(chdir(default_path)!=0)
	{
		fprintf(stderr,"can not go to the directory %s\n",default_path);
		return false;
	}
	while((entry = readdir(dp)) != NULL)
	{
		//get the iterator and size of the logfile list
		std::deque<filename>::iterator itr=logfile_queue.end();
		int subscript = logfile_queue.size()-1;
		lstat(entry->d_name, &stabuf);
		//if it is a regular file, check its name
		if(S_ISREG(stabuf.st_mode))
		{
			//if the filename begins with DEFAULT_LOG_FILE_NAME
			if(strncmp(entry->d_name, DEFAULT_LOG_FILE_NAME,
				sizeof(DEFAULT_LOG_FILE_NAME)-1) == 0) 
			{
			   filename tmp = filename(entry->d_name);
			   //insert the filename into the file list
			   while(itr != logfile_queue.begin() && 
					   tmp < logfile_queue[subscript])
			   {
				   subscript--;
				   itr--;
			   }
			   // add the absolute path
			   tmp.name=string(default_path)+
				   string(PATH_DELIMITOR)+tmp.name;
			   logfile_queue.insert(itr, tmp);
			}
		}
		//if the file is not a regular file or the filename does not
		//begin with DEFAULT_LOG_FILE_NAME, do nothing

		//next file name
	}
	closedir(dp);
	//go back to the path stored
	if(chdir(current_path) !=0) 
	{
		fprintf(stderr, "can not recover the path\n");
		return false;
	}

	//if the log file list is too long, delete the old ones
	while(logfile_queue.size() > max_num_of_logfiles)
	{ //delete the front file
		if(unlink(logfile_queue.front().name.c_str())!=0)
		{
			char msg[1024];
			sprintf(msg, "can not delete file %s",
				logfile_queue.front().name.c_str());
			perror(msg);
		}
		logfile_queue.pop_front();
	}

	return true;
}

	

void GC_Log::enableLog(unsigned mask)
{
	log_mask |= mask;
}

void GC_Log::changeMaxSize(int size)
{
	MAX_SIZE = size;
}

/* return -1 if can not write to the stream. return 1 if OK */
int GC_Log::logprintf(int level, char *information)
{
	if(open_failed) return -1;
#ifdef _DEBUG
	if(log_stream == NULL) 
	{
		fprintf(stderr, "No stream for logging\n");
		return -1;
	}
	int fd = fileno(log_stream);
	if(fd<0){
		fprintf(stderr, "can not find fd of the stream in logprintf\n");
		return -1;
	}
#endif
	char logBuf[MAX_STRING_LEN];
	//check whether the log level is masked
	if((log_mask & level) ==0 ) return 1;
	//check the log file. 
#ifdef USE_THREADS
	mutex_log->Lock();
#endif
	if(!check_log_file())
	{
#ifdef USE_THREADS
		mutex_log->Unlock();
#endif
#ifdef _DEBUG
		fprintf(stderr, "check file failed in logprintf\n");
#endif
		return -1;
	}	
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif
	//the log record is in the format of
	//[time] [log level] information
	//first, get the time
	struct tm *tm_ptr;
	time_t t_time;
	char buf[256];
	time(&t_time);
	tm_ptr=localtime(&t_time);
	strftime(buf, 256, "%a %b %d %H:%M:%S %Y", tm_ptr);
	sprintf(logBuf, "[");
	sprintf(logBuf+1, buf);
	sprintf(logBuf+strlen(logBuf), "] [");
	//get the level
	int l;
	for(l=0;priorities[l].t_name != NULL;l++)
		if (priorities[l].t_val == level) break;
	if (priorities[l].t_name != NULL) {
		sprintf(logBuf+strlen(logBuf), priorities[l].t_name);
		sprintf(logBuf+strlen(logBuf),"] ");
	}
	//get the information
	sprintf(logBuf+strlen(logBuf),"%s", information);
	sprintf(logBuf+strlen(logBuf), "\n");//a new line
	//write to streams
#ifdef USE_THREADS
	mutex_log->Lock();
#endif
	if(LOG_TO_STDOUT_ALSO)
		fprintf(stdout,"%s",logBuf);
	fprintf(log_stream,"%s",logBuf);
	//flush it
	fflush(log_stream);
	fflush(stdout);
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif
	return 1;
}

void GC_Log::setMaxNumOfLogfiles(int num)
{ 
	max_num_of_logfiles = num;
#ifdef USE_THREADS
	mutex_log->Lock();
#endif
	while(logfile_queue.size()>max_num_of_logfiles)
	{
		//delete the oldest. Ignore the errors if the file can 
		//not be deleted.(maybe it does not exist any more)
		if(unlink(logfile_queue.front().name.c_str()) == -1)
		{
			char msg[1024];
			sprintf(msg, "fail to delete %s",
					logfile_queue.front().name.c_str());
			perror(msg);
		}
#ifdef _DEBUG
		cout<<logfile_queue.front().name<<" is deleted"<<endl;
#endif
		logfile_queue.pop_front();
	}
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif
}

void GC_Log::setDefaultPath(char* path)
{
	if(strcmp(default_path, path) == 0) return;

	default_path = path;
	//get the log file list at the newly set path
#ifdef USE_THREADS
	mutex_log->Lock(); //maybe a new file is created while
			   //retriving logfile list
#endif
	//if the path does not exit yet, creat it
	DIR *dp;
	if((dp = opendir(default_path)) == NULL) 
	{	
		//PERMS = 0660 but it doesn't work. use 760 instead for now
		mkdir(default_path, 0760);
#ifdef _DEBUG 
		fprintf(stdout, "directory %s is created\n", path);
#endif
	}
	closedir(dp);
	//only the log file under the new path are maintained
	while(!logfile_queue.empty())
		logfile_queue.pop_front();
	get_logfile_list(); //get the log file list now
	////////////////
#ifdef _DEBUG
	for(int i=0;i<logfile_queue.size();i++)
	{
		cout<<logfile_queue[i].name<<endl;
	}
#endif
	//////////////////
#ifdef USE_THREADS
	mutex_log->Unlock();
#endif
}

