//*****************************************************************************
//                                 NbkGnuCap.cpp                              *
//                                ---------------                             *
//  Started     : 05/09/2003                                                  *
//  Last Update : 11/12/2007                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/NbkGnuCap.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( NbkGnuCap, wxNotebook )

  EVT_BUTTON( PnlAnaBase::ID_BTN_OPTIONS, NbkGnuCap::OnBtnOptions )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Arguments:
//   poParent - The parent window
//   oWinID   - The display object ID
//   roPosn   - The notebook position
//   roSize   - The notebook size

NbkGnuCap::NbkGnuCap( wxWindow * poParent, wxWindowID oWinID )
                    : NbkSimrBase( poParent, oWinID ), m_oDlgOptions( poParent )
{
  // Set the simulator engine type specifier
  bSetSimrType( eSIMR_GNUCAP );

  // Create the various display objects
  m_poPnlGnuCapOP = new PnlGnuCapOP( this );
  m_poPnlGnuCapDC = new PnlGnuCapDC( this );
  m_poPnlGnuCapAC = new PnlGnuCapAC( this );
  m_poPnlGnuCapTR = new PnlGnuCapTR( this );
  m_poPnlGnuCapFO = new PnlGnuCapFO( this );

  // Add the display objects to the note book
  AddPage( m_poPnlGnuCapOP, wxT( " Quiescent  " ) );
  AddPage( m_poPnlGnuCapDC, wxT( "     DC      " ) );
  AddPage( m_poPnlGnuCapAC, wxT( "     AC      " ) );
  AddPage( m_poPnlGnuCapTR, wxT( " Transient  " ) );
  AddPage( m_poPnlGnuCapFO, wxT( "   Fourier   " ) );

  // Specify the default page to be displayed
  SetSelection( 0 );
}

//*****************************************************************************
// Default constructor.
// (Used for two stage creation ie. must call Create( ) method.

NbkGnuCap::NbkGnuCap( void ) : NbkSimrBase( ), m_oDlgOptions( this )
{
  // Set the simulator engine type specifier
  bSetSimrType( eSIMR_GNUCAP );
}

//*****************************************************************************
// Destructor.

NbkGnuCap::~NbkGnuCap( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bClear( void )
{
  bool  bRtnValue=TRUE;

  if( ! m_poPnlGnuCapOP->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlGnuCapDC->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlGnuCapAC->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlGnuCapTR->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlGnuCapFO->bClear( ) ) bRtnValue = FALSE;

  if( ! m_oDlgOptions   .bClear( ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  if( ! m_poPnlGnuCapOP->bLoad( roSim ) )               bRtnValue = FALSE;
  if( ! m_poPnlGnuCapDC->bLoad( roSim ) )               bRtnValue = FALSE;
  if( ! m_poPnlGnuCapAC->bLoad( roSim ) )               bRtnValue = FALSE;
  if( ! m_poPnlGnuCapTR->bLoad( roSim ) )               bRtnValue = FALSE;
  if( ! m_poPnlGnuCapFO->bLoad( roSim ) )               bRtnValue = FALSE;

  if( m_oDlgOptions.bSetCmd( roSim.rosGetOPTIONS( ) ) ) bRtnValue = FALSE;

  if( ! bSetPage( roSim.eGetAnaType( ) ) )              bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSave( Simulation & roSim )
{
  switch( GetSelection( ) )
  {
    case 0: // OP analysis
      if( ! m_poPnlGnuCapOP->bSave( roSim ) ) return( FALSE );
      break;
    case 1: // DC analysis
      if( ! m_poPnlGnuCapDC->bSave( roSim ) ) return( FALSE );
      break;
    case 2: // AC analysis
      if( ! m_poPnlGnuCapAC->bSave( roSim ) ) return( FALSE );
      break;
    case 3: // TR analysis
      if( ! m_poPnlGnuCapTR->bSave( roSim ) ) return( FALSE );
      break;
    case 4: // FO analysis
      if( ! m_poPnlGnuCapFO->bSave( roSim ) ) return( FALSE );
      break;
    default:                                  return( FALSE );
  }

  m_oDlgOptions.m_fTEMPAMB = roSim.fGetTempC( );
  m_oDlgOptions.bFormat( );
  roSim.bSetOPTIONS( (wxString) m_oDlgOptions );

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   eType - The enumerated analysis type specifier
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSetPage( eAnaType eType )
{
  int  iPage;

  switch( eType )
  {
    case eANA_OP : iPage = 0; break;
    case eANA_DC : iPage = 1; break;
    case eANA_AC : iPage = 2; break;
    case eANA_TR : iPage = 3; break;
    case eANA_FO : iPage = 4; break;
    default:                  return( FALSE );
  }

  SetSelection( iPage ); // Specify the page to be displayed

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   psType - The two letter analysis type specifier (case ignored)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSetPage( const wxChar * psType )
{
  eAnaType  eType=eANA_NONE;
  wxString  osType;

  osType = wxString( psType ).Upper( );

  if( osType.Length( ) == 2 )
  {
         if( osType == wxT("OP") ) eType = eANA_OP;
    else if( osType == wxT("DC") ) eType = eANA_DC;
    else if( osType == wxT("AC") ) eType = eANA_AC;
    else if( osType == wxT("TR") ) eType = eANA_TR;
    else if( osType == wxT("FO") ) eType = eANA_FO;
    else if( osType == wxT("DI") ) eType = eANA_DI;
    else if( osType == wxT("NO") ) eType = eANA_NO;
    else if( osType == wxT("PZ") ) eType = eANA_PZ;
    else if( osType == wxT("SE") ) eType = eANA_SE;
    else if( osType == wxT("TF") ) eType = eANA_TF;
  }

  return( bSetPage( eType ) );
}

//*****************************************************************************
// Get the two letter page specifier.
//
// Return Values:
//   Success - The two letter analysis type specifier (lower case)
//   Failure - An empty string

const wxString & NbkGnuCap::rosGetPage( void )
{
  static  wxString  osPage;

  switch( eGetPage( ) )
  {
    case eANA_OP : osPage = wxT("op"); break;
    case eANA_DC : osPage = wxT("dc"); break;
    case eANA_AC : osPage = wxT("ac"); break;
    case eANA_TR : osPage = wxT("tr"); break;
    case eANA_FO : osPage = wxT("fo"); break;
    default :      osPage.Empty( );
  }

  return( osPage );
}

//*****************************************************************************
// Get the enumerated page specifier.
//
// Return Values:
//   Success - The enumerated analysis specifier
//   Failure - Simulation::eANA_NONE

eAnaType  NbkGnuCap::eGetPage( void )
{
  eAnaType  ePage;

  switch( GetSelection( ) )
  {
    case 0 :  ePage = eANA_OP; break;
    case 1 :  ePage = eANA_DC; break;
    case 2 :  ePage = eANA_AC; break;
    case 3 :  ePage = eANA_TR; break;
    case 4 :  ePage = eANA_FO; break;
    default : ePage = eANA_NONE;
  }

  return( ePage );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// .OPTIONS command setup button control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  NbkGnuCap::OnBtnOptions( wxCommandEvent & roEvtCmd )
{
  PnlAnaBase * poPnlAna;
  double       df1;
  int          i1;

  // Set the temperature in the OPTIONS dialog
  poPnlAna = (PnlAnaBase *) GetPage( GetSelection( ) );
  if( poPnlAna->bIsCreatedTemp( ) )
  {
    df1 = poPnlAna->m_oPnlTemp.dfGetValue( );
    m_oDlgOptions.bSetValue( DlgGnuCapOPT::ID_PNL_TEMPAMB, (float) df1 );
  }

  // Display the OPTIONS dialog
  m_oDlgOptions.CenterOnParent( );
  i1 = m_oDlgOptions.ShowModal( );

  // Set the temperature in the analysis panel
  if( i1==wxID_OK && poPnlAna->bIsCreatedTemp( ) )
  {
    df1 = m_oDlgOptions.dfGetValue( DlgGnuCapOPT::ID_PNL_TEMPAMB );
    poPnlAna->m_oPnlTemp.bSetValue( df1 );
  }
}

//*****************************************************************************
