/* GStreamer
 *
 * gstv4lradiobin.c: implementation of V4L radio container
 *
 * Copyright (C) 2001-2002 Ronald Bultje <rbultje@ronald.bitfreak.net>
 * Copyright (C) 2005 Nickolay V. Shmyrev <nshmyrev@yandex.ru>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include <string.h>
#include "v4lradio_calls.h"
#include "gstv4lradiotuner.h"

#include <gst/propertyprobe/propertyprobe.h>

/* elementfactory information */
static GstElementDetails gst_v4lradio_bin_details =
GST_ELEMENT_DETAILS ("Container to expose video4linux radio support",
    "Generic/Video",
    "Container to add v4l radio management to audio sources like alsasrc",
    "Nickolay V. Shmyrev <nshmyrev@yandex.ru>");

/* V4lRadioBin signals and args */
enum
{
  SIGNAL_OPEN,
  SIGNAL_CLOSE,
  LAST_SIGNAL
};

enum
{
  ARG_0,
  ARG_DEVICE,
  ARG_DEVICE_NAME,
  ARG_FREQUENCY
};


static void gst_v4lradio_bin_base_init (gpointer g_class);
static void gst_v4lradio_bin_class_init (GstV4lRadioBinClass * klass);
static void gst_v4lradio_bin_init (GstV4lRadioBin * v4lelement);
static void gst_v4lradio_bin_dispose (GObject * object);
static void
gst_v4lradio_bin_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void
gst_v4lradio_bin_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);
static GstElementStateReturn
gst_v4lradio_bin_change_state (GstElement * element);


static GstElementClass *parent_class = NULL;
static guint gst_v4lradio_bin_signals[LAST_SIGNAL] = { 0 };

static gboolean
gst_v4lradio_iface_supported (GstImplementsInterface * iface, GType iface_type)
{
  g_assert (iface_type == GST_TYPE_TUNER);

  return TRUE;
}

static void
gst_v4lradio_interface_init (GstImplementsInterfaceClass * klass)
{
  klass->supported = gst_v4lradio_iface_supported;
}

static const GList *
gst_v4lradio_probe_get_properties (GstPropertyProbe * probe)
{
  GObjectClass *klass = G_OBJECT_GET_CLASS (probe);
  static GList *list = NULL;

  if (!list) {
    list = g_list_append (NULL, g_object_class_find_property (klass, "device"));
  }

  return list;
}

static gboolean
gst_v4lradio_class_probe_devices (GstV4lRadioBinClass * klass, gboolean check)
{
  static gboolean init = FALSE;
  static GList *devices = NULL;

  if (!init && !check) {
    gchar *dev_base[] = { "/dev/radio", "/dev/v4l/radio", NULL };
    gint base, n, fd;

    while (devices) {
      GList *item = devices;
      gchar *device = item->data;

      devices = g_list_remove (devices, item);
      g_free (device);
    }

    /* detect /dev entries */
    for (n = 0; n < 64; n++) {
      for (base = 0; dev_base[base] != NULL; base++) {
        struct stat s;
        gchar *device = g_strdup_printf ("%s%d", dev_base[base], n);

        /* does the /dev/ entry exist at all? */
        if (stat (device, &s) == 0) {
          /* yes: is a device attached? */
          if ((fd = open (device, O_RDONLY)) > 0 || errno == EBUSY) {
            if (fd > 0)
              close (fd);

            devices = g_list_append (devices, device);
            break;
          }
        }
        g_free (device);
      }
    }

    init = TRUE;
  }

  klass->devices = devices;

  return init;
}

static void
gst_v4lradio_probe_probe_property (GstPropertyProbe * probe,
    guint prop_id, const GParamSpec * pspec)
{
  GstV4lRadioBinClass *klass = GST_V4LRADIO_BIN_GET_CLASS (probe);

  switch (prop_id) {
    case ARG_DEVICE:
      gst_v4lradio_class_probe_devices (klass, FALSE);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (probe, prop_id, pspec);
      break;
  }
}

static gboolean
gst_v4lradio_probe_needs_probe (GstPropertyProbe * probe,
    guint prop_id, const GParamSpec * pspec)
{
  GstV4lRadioBinClass *klass = GST_V4LRADIO_BIN_GET_CLASS (probe);
  gboolean ret = FALSE;

  switch (prop_id) {
    case ARG_DEVICE:
      ret = !gst_v4lradio_class_probe_devices (klass, TRUE);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (probe, prop_id, pspec);
      break;
  }

  return ret;
}

static GValueArray *
gst_v4lradio_class_list_devices (GstV4lRadioBinClass * klass)
{
  GValueArray *array;
  GValue value = { 0 };
  GList *item;

  if (!klass->devices)
    return NULL;

  array = g_value_array_new (g_list_length (klass->devices));
  item = klass->devices;
  g_value_init (&value, G_TYPE_STRING);
  while (item) {
    gchar *device = item->data;

    g_value_set_string (&value, device);
    g_value_array_append (array, &value);

    item = item->next;
  }
  g_value_unset (&value);

  return array;
}

static GValueArray *
gst_v4lradio_probe_get_values (GstPropertyProbe * probe,
    guint prop_id, const GParamSpec * pspec)
{
  GstV4lRadioBinClass *klass = GST_V4LRADIO_BIN_GET_CLASS (probe);
  GValueArray *array = NULL;

  switch (prop_id) {
    case ARG_DEVICE:
      array = gst_v4lradio_class_list_devices (klass);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (probe, prop_id, pspec);
      break;
  }

  return array;
}

static void
gst_v4lradio_property_probe_interface_init (GstPropertyProbeInterface * iface)
{
  iface->get_properties = gst_v4lradio_probe_get_properties;
  iface->probe_property = gst_v4lradio_probe_probe_property;
  iface->needs_probe = gst_v4lradio_probe_needs_probe;
  iface->get_values = gst_v4lradio_probe_get_values;
}


GType
gst_v4lradio_bin_get_type (void)
{
  static GType v4lradio_bin_type = 0;

  if (!v4lradio_bin_type) {
    static const GTypeInfo v4lradio_bin_info = {
      sizeof (GstV4lRadioBinClass),
      gst_v4lradio_bin_base_init,
      NULL,
      (GClassInitFunc) gst_v4lradio_bin_class_init,
      NULL,
      NULL,
      sizeof (GstV4lRadioBin),
      0,
      (GInstanceInitFunc) gst_v4lradio_bin_init,
      NULL
    };
    static const GInterfaceInfo v4lradioiface_info = {
      (GInterfaceInitFunc) gst_v4lradio_interface_init,
      NULL,
      NULL,
    };
    static const GInterfaceInfo v4lradio_tuner_info = {
      (GInterfaceInitFunc) gst_v4lradio_tuner_interface_init,
      NULL,
      NULL,
    };
    static const GInterfaceInfo v4lradio_propertyprobe_info = {
      (GInterfaceInitFunc) gst_v4lradio_property_probe_interface_init,
      NULL,
      NULL,
    };

    v4lradio_bin_type = g_type_register_static (GST_TYPE_BIN,
        "GstV4lRadioBin", &v4lradio_bin_info, 0);

    g_type_add_interface_static (v4lradio_bin_type,
        GST_TYPE_IMPLEMENTS_INTERFACE, &v4lradioiface_info);
    g_type_add_interface_static (v4lradio_bin_type,
        GST_TYPE_TUNER, &v4lradio_tuner_info);
    g_type_add_interface_static (v4lradio_bin_type,
        GST_TYPE_PROPERTY_PROBE, &v4lradio_propertyprobe_info);
  }

  return v4lradio_bin_type;
}


static void
gst_v4lradio_bin_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  GstV4lRadioBinClass *klass = GST_V4LRADIO_BIN_CLASS (g_class);

  klass->devices = NULL;

  gst_element_class_set_details (gstelement_class, &gst_v4lradio_bin_details);
}

static void
gst_v4lradio_bin_class_init (GstV4lRadioBinClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_DEVICE,
      g_param_spec_string ("device", "Device", "Device location",
          NULL, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_DEVICE_NAME,
      g_param_spec_string ("device_name", "Device name", "Name of the device",
          NULL, G_PARAM_READABLE));
  g_object_class_install_property (gobject_class, ARG_FREQUENCY,
      g_param_spec_ulong ("frequency", "Tuned frequency",
          "Frequency to tune to (in Hz)", 0, G_MAXULONG, 0, G_PARAM_READWRITE));

  /* signals */
  gst_v4lradio_bin_signals[SIGNAL_OPEN] =
      g_signal_new ("open", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST,
      G_STRUCT_OFFSET (GstV4lRadioBinClass, open),
      NULL, NULL, g_cclosure_marshal_VOID__STRING,
      G_TYPE_NONE, 1, G_TYPE_STRING);
  gst_v4lradio_bin_signals[SIGNAL_CLOSE] =
      g_signal_new ("close", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST,
      G_STRUCT_OFFSET (GstV4lRadioBinClass, close),
      NULL, NULL, g_cclosure_marshal_VOID__STRING,
      G_TYPE_NONE, 1, G_TYPE_STRING);

  gobject_class->set_property = gst_v4lradio_bin_set_property;
  gobject_class->get_property = gst_v4lradio_bin_get_property;

  gstelement_class->change_state = gst_v4lradio_bin_change_state;

  gobject_class->dispose = gst_v4lradio_bin_dispose;
}


static void
gst_v4lradio_bin_init (GstV4lRadioBin * v4lradio_bin)
{
  /* some default values */
  v4lradio_bin->radio_fd = -1;
  v4lradio_bin->radiodev = g_strdup ("/dev/radio0");
  v4lradio_bin->device_name = g_strdup ("Unknown");
  v4lradio_bin->frequency = 100000000;
  v4lradio_bin->channel = NULL;
}


static void
gst_v4lradio_bin_dispose (GObject * object)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (object);

  if (v4lradio_bin->radiodev) {
    g_free (v4lradio_bin->radiodev);
    v4lradio_bin->radiodev = NULL;
  }

  if (v4lradio_bin->device_name) {
    g_free (v4lradio_bin->device_name);
    v4lradio_bin->device_name = NULL;
  }

  if (v4lradio_bin->channel) {
    g_object_unref (v4lradio_bin->channel);
    v4lradio_bin->channel = NULL;
  }

  if (((GObjectClass *) parent_class)->dispose)
    ((GObjectClass *) parent_class)->dispose (object);
}


static void
gst_v4lradio_bin_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  GstV4lRadioBin *v4lradio_bin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_V4LRADIO_BIN (object));
  v4lradio_bin = GST_V4LRADIO_BIN (object);

  switch (prop_id) {
    case ARG_DEVICE:
      if (v4lradio_bin->radiodev)
        g_free (v4lradio_bin->radiodev);
      v4lradio_bin->radiodev = g_strdup (g_value_get_string (value));
      break;
    case ARG_FREQUENCY:
      if (GST_V4LRADIO_IS_OPEN (v4lradio_bin)) {
        GstTuner *tuner = GST_TUNER (v4lradio_bin);

        gst_tuner_set_frequency (tuner, NULL, g_value_get_ulong (value));
      } else {
        v4lradio_bin->frequency = g_value_get_ulong (value);
        g_object_notify (object, "frequency");
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
gst_v4lradio_bin_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  GstV4lRadioBin *v4lradio_bin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_V4LRADIO_BIN (object));
  v4lradio_bin = GST_V4LRADIO_BIN (object);

  switch (prop_id) {
    case ARG_DEVICE:
      g_value_set_string (value, v4lradio_bin->radiodev);
      break;
    case ARG_DEVICE_NAME:{
      gchar *new = NULL;

      if (GST_V4LRADIO_IS_OPEN (v4lradio_bin))
        new = v4lradio_bin->device_name;
      g_value_set_string (value, new);
      break;
    }
    case ARG_FREQUENCY:
      g_value_set_ulong (value, v4lradio_bin->frequency);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static GstElementStateReturn
gst_v4lradio_bin_change_state (GstElement * element)
{
  GstV4lRadioBin *v4lradio_bin;
  GstTuner *tuner;

  g_return_val_if_fail (GST_IS_V4LRADIO_BIN (element), GST_STATE_FAILURE);

  v4lradio_bin = GST_V4LRADIO_BIN (element);
  tuner = GST_TUNER (v4lradio_bin);

  /* if going down into NULL state, close the device if it's open
   * if going to READY, open the device (and set some options)
   */
  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      if (!gst_v4lradio_open (v4lradio_bin))
        return GST_STATE_FAILURE;

      gst_tuner_set_frequency (tuner, gst_tuner_get_channel (tuner),
          v4lradio_bin->frequency);

      g_signal_emit (G_OBJECT (v4lradio_bin),
          gst_v4lradio_bin_signals[SIGNAL_OPEN], 0, v4lradio_bin->radiodev);
      break;

    case GST_STATE_READY_TO_NULL:

      if (!gst_v4lradio_close (v4lradio_bin))
        return GST_STATE_FAILURE;

      g_signal_emit (G_OBJECT (v4lradio_bin),
          gst_v4lradio_bin_signals[SIGNAL_CLOSE], 0, v4lradio_bin->radiodev);
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}
