/* GStreamer
 *
 * gstv4lrtuner.c: tuner interface implementation for V4LR
 *
 * Copyright (C) 2003 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/tuner/tuner.h>

#include "gstv4lradiotuner.h"
#include "gstv4lradiobin.h"
#include "v4lradio_calls.h"

static const GList *gst_v4lradio_tuner_list_channels (GstTuner * mixer);
static void gst_v4lradio_tuner_set_channel (GstTuner * mixer,
    GstTunerChannel * channel);
static GstTunerChannel *gst_v4lradio_tuner_get_channel (GstTuner * mixer);

static void gst_v4lradio_tuner_set_frequency (GstTuner * tuner,
    GstTunerChannel * channel, gulong frequency);
static gulong gst_v4lradio_tuner_get_frequency (GstTuner * tuner,
    GstTunerChannel * channel);
static gint gst_v4lradio_tuner_signal_strength (GstTuner * tuner,
    GstTunerChannel * channel);

void
gst_v4lradio_tuner_interface_init (GstTunerClass * klass)
{
  /* default virtual functions */
  klass->list_channels = gst_v4lradio_tuner_list_channels;
  klass->set_channel = gst_v4lradio_tuner_set_channel;
  klass->get_channel = gst_v4lradio_tuner_get_channel;

  klass->set_frequency = gst_v4lradio_tuner_set_frequency;
  klass->get_frequency = gst_v4lradio_tuner_get_frequency;
  klass->signal_strength = gst_v4lradio_tuner_signal_strength;
}


static const GList *
gst_v4lradio_tuner_list_channels (GstTuner * mixer)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (mixer);

  return g_list_append (NULL, v4lradio_bin->channel);
}

static void
gst_v4lradio_tuner_set_channel (GstTuner * mixer, GstTunerChannel * channel)
{
  return;
}

static GstTunerChannel *
gst_v4lradio_tuner_get_channel (GstTuner * mixer)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (mixer);

  return v4lradio_bin->channel;
}


static void
gst_v4lradio_tuner_set_frequency (GstTuner * tuner,
    GstTunerChannel * channel, gulong frequency)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (tuner);

  /* assert that we're opened and that we're using a known item */
  g_return_if_fail (GST_V4LRADIO_IS_OPEN (v4lradio_bin));

  gst_v4lradio_set_frequency (v4lradio_bin, frequency);
}

static gulong
gst_v4lradio_tuner_get_frequency (GstTuner * tuner, GstTunerChannel * channel)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (tuner);
  gulong frequency = 0;

  /* assert that we're opened and that we're using a known item */
  g_return_val_if_fail (GST_V4LRADIO_IS_OPEN (v4lradio_bin), 0);
  gst_v4lradio_get_frequency (v4lradio_bin, &frequency);

  return frequency;
}

static gint
gst_v4lradio_tuner_signal_strength (GstTuner * tuner, GstTunerChannel * channel)
{
  GstV4lRadioBin *v4lradio_bin = GST_V4LRADIO_BIN (tuner);
  guint signal = 0;

  /* assert that we're opened and that we're using a known item */
  g_return_val_if_fail (GST_V4LRADIO_IS_OPEN (v4lradio_bin), 0);

  gst_v4lradio_get_signal (v4lradio_bin, &signal);

  return (gint) signal;
}
