/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkrgb.h>
#include <libgnomeui/gnome-icon-text.h>
#include <libgnomevfs/gnome-vfs.h>
#include "catalog-png-exporter.h"
#include "comments.h"
#include "file-utils.h"
#include "gthumb-init.h"
#include "my-marshallers.h"
#include "pixbuf-utils.h"
#include "thumb-loader.h"
#include "image-loader.h"


static void begin_export    (CatalogPngExporter *ce);

static void end_export      (CatalogPngExporter *ce);

static void begin_page      (CatalogPngExporter *ce, 
			     int page_n);

static void end_page        (CatalogPngExporter *ce,
			     int page_n);

static void paint_frame     (CatalogPngExporter *ce,
			     GdkRectangle *frame_rect,
			     GdkRectangle *image_rect,
			     gchar *filename);

static void paint_image     (CatalogPngExporter *ce,
			     GdkRectangle *image_rect,
			     GdkPixbuf *image);

static void paint_text      (CatalogPngExporter *ce,
			     int x,
			     int y,
			     char *text,
			     int *height);

static int  get_text_height (CatalogPngExporter *ce,
			     char *text);


#define COL_SPACING  15
#define ROW_SPACING  15
#define FRAME_BORDER 4
#define TEXT_SPACING 3
#define CAPTION_MAX_ROWS 4

/* Separators for text layout */
#define DEFAULT_SEPARATORS " \t-.[]#"

#define FONTSET "-adobe-helvetica-medium-r-normal--*-100-*-*-*-*-*-*," \
                "-*-*-medium-r-normal--10-*-*-*-*-*-*-*," \
                "*"


typedef struct {
	char *           comment;
	char *           filename;
	GnomeVFSFileSize file_size;
	time_t           file_time;
	GdkPixbuf *      thumb;
	int              image_width;
	int              image_height;
	char *           caption_row[CAPTION_MAX_ROWS];
	gboolean         caption_set;
} ImageData;


#define IMAGE_DATA(x) ((ImageData*)(x))


static ImageData *
image_data_new (gchar *filename)
{
	ImageData * idata;
	CommentData *cdata;
	int i;

	idata = g_new (ImageData, 1);

	cdata = comments_load_comment (filename);
	idata->comment = comments_get_comment_as_string (cdata);
	if (cdata != NULL)
		comment_data_free (cdata);

	idata->filename = g_strdup (filename);
	idata->file_size = 0;
	idata->thumb = NULL;
	idata->image_width = 0;
	idata->image_height = 0;

	for (i = 0; i < CAPTION_MAX_ROWS; i++)
		idata->caption_row[i] = NULL;
	idata->caption_set = FALSE;

	return idata;
}


static void
image_data_free (ImageData *idata)
{
	int i;

	if (idata->filename)
		g_free (idata->filename);
	if (idata->thumb)
		gdk_pixbuf_unref (idata->thumb);

	for (i = 0; i < CAPTION_MAX_ROWS; i++)
		if (idata->caption_row[i] != NULL)
			g_free (idata->caption_row[i]);

	g_free (idata);
}


enum {
	DONE,
	PROGRESS,
	LAST_SIGNAL
};


static GtkObjectClass *parent_class = NULL;
static guint catalog_png_exporter_signals[LAST_SIGNAL] = { 0 };


static void 
catalog_png_exporter_destroy (GtkObject *object)
{
	CatalogPngExporter *ce;

        g_return_if_fail (IS_CATALOG_PNG_EXPORTER (object));
  
        ce = CATALOG_PNG_EXPORTER (object);

	if (ce->directory != NULL) {
		g_free (ce->directory);
		ce->directory = NULL;
	}

	if (ce->filename1 != NULL) {
		g_free (ce->filename1);
		ce->filename1 = NULL;
	}

	if (ce->filename2 != NULL) {
		g_free (ce->filename2);
		ce->filename2 = NULL;
	}
		
	if (ce->file_list != NULL) {
		g_list_foreach (ce->file_list, (GFunc) image_data_free, NULL);
		g_list_free (ce->file_list);
		ce->file_list = NULL;
	}

	if (ce->pages_height != NULL) {
		g_free (ce->pages_height);
		ce->pages_height = NULL;
	}

	if (ce->text_font != NULL) {
		g_free (ce->text_font);
		ce->text_font = NULL;
	}

	if (ce->font != NULL)
		gdk_font_unref (ce->font);

	if (ce->tloader != NULL) {
		gtk_object_destroy (GTK_OBJECT (ce->tloader));
		ce->tloader = NULL;
	}

	/* Chain up */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
catalog_png_exporter_class_init (CatalogPngExporterClass *class)
{
	GtkObjectClass *object_class;

	parent_class = gtk_type_class (gtk_object_get_type ());
	object_class = (GtkObjectClass*) class;

	catalog_png_exporter_signals[DONE] =
                gtk_signal_new ("done",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (CatalogPngExporterClass, done),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	catalog_png_exporter_signals[PROGRESS] =
                gtk_signal_new ("progress",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (CatalogPngExporterClass, done),
                                gtk_marshal_NONE__FLOAT,
                                GTK_TYPE_NONE, 1,
				GTK_TYPE_FLOAT);

	gtk_object_class_add_signals (object_class, catalog_png_exporter_signals, 
                                      LAST_SIGNAL);

	object_class->destroy = catalog_png_exporter_destroy;
}


static void
catalog_png_exporter_init (CatalogPngExporter *ce)
{
	ce->file_list = NULL;

	ce->page_width = 0;
	ce->page_height = 0;

	ce->thumb_width = 0;
	ce->thumb_height = 0;

	ce->text_font = NULL;

	ce->tloader = NULL;
	ce->file_to_load = NULL;

	ce->caption_fields = CAPTION_FILE_NAME;

	/* Colors. */
	ce->page_color.red   = 65535 * 1.0;
	ce->page_color.green = 65535 * 1.0;
	ce->page_color.blue  = 65535 * 1.0;

	ce->text_color.red   = 65535 * 0.0;
	ce->text_color.green = 65535 * 0.0;
	ce->text_color.blue  = 65535 * 0.0;

	ce->frame_color.red   = 65535 * 0.8;
	ce->frame_color.green = 65535 * 0.8;
	ce->frame_color.blue  = 65535 * 0.8;

	ce->frame_style = FRAME_STYLE_SIMPLE;

	ce->directory = NULL;
	ce->filename1 = NULL;
	ce->filename2 = NULL;

	ce->font = gdk_fontset_load (FONTSET);

	ce->pages_height = NULL;

	ce->write_image_map = FALSE;
	ce->imap_handle = NULL;
}


GtkType
catalog_png_exporter_get_type ()
{
	static guint catalog_png_exporter_type = 0;

	if (! catalog_png_exporter_type) {
		GtkTypeInfo catalog_png_exporter_info = {
			"CatalogPngExporter",
			sizeof (CatalogPngExporter),
			sizeof (CatalogPngExporterClass),
			(GtkClassInitFunc) catalog_png_exporter_class_init,
			(GtkObjectInitFunc) catalog_png_exporter_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		catalog_png_exporter_type = gtk_type_unique (
			gtk_object_get_type (), 
			&catalog_png_exporter_info);
	}
	
	return catalog_png_exporter_type;
}


CatalogPngExporter *
catalog_png_exporter_new (GList *file_list)
{
	CatalogPngExporter *ce;
	GList *scan;

	ce = CATALOG_PNG_EXPORTER (gtk_type_new (catalog_png_exporter_get_type ()));

	for (scan = file_list; scan; scan = scan->next) {
		ImageData *idata;

		idata = image_data_new ((gchar*) scan->data);
		ce->file_list = g_list_prepend (ce->file_list, idata);
	}
	ce->file_list = g_list_reverse (ce->file_list);

	return CATALOG_PNG_EXPORTER (ce);
}


void
catalog_png_exporter_set_directory (CatalogPngExporter *ce,
				    char *directory)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	if (ce->directory)
		g_free (ce->directory);
	ce->directory = g_strdup (directory);
}


void
catalog_png_exporter_set_name (CatalogPngExporter *ce,
			       char *filename1,
			       char *filename2)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	if (ce->filename1)
		g_free (ce->filename1);
	ce->filename1 = g_strdup (filename1);

	if (ce->filename2)
		g_free (ce->filename2);
	ce->filename2 = g_strdup (filename2);
}


void
catalog_png_exporter_set_page_size (CatalogPngExporter *ce,
				    int width,
				    int height)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->page_width = width;
	ce->page_height = height;
}


void
catalog_png_exporter_set_page_size_row_col (CatalogPngExporter *ce,
					    int rows,
					    int cols)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->page_size_use_row_col = TRUE;
	ce->page_rows = rows;
	ce->page_cols = cols;
}


void
catalog_png_exporter_set_thumb_size (CatalogPngExporter *ce,
				     int width,
				     int height)

{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->thumb_width = width;
	ce->thumb_height = height;

	ce->frame_width = width + FRAME_BORDER * 2;
	ce->frame_height = height + FRAME_BORDER * 2;
}


void
catalog_png_exporter_set_text_font (CatalogPngExporter *ce,
				    char *text_font)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	if (ce->text_font != NULL)
		g_free (ce->text_font);
	ce->text_font = g_strdup (text_font);

	if (ce->font != NULL)
		gdk_font_unref (ce->font);
	ce->font = gdk_font_load (ce->text_font);
}


void
catalog_png_exporter_set_page_color (CatalogPngExporter *ce,
				     gushort r,
				     gushort g,
				     gushort b)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->page_color.red   = r;
	ce->page_color.green = g;
	ce->page_color.blue  = b;
}


void
catalog_png_exporter_set_text_color (CatalogPngExporter *ce,
				     gushort r,
				     gushort g,
				     gushort b)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->text_color.red   = r;
	ce->text_color.green = g;
	ce->text_color.blue  = b;
}


void
catalog_png_exporter_set_frame_color (CatalogPngExporter *ce,
				      gushort r,
				      gushort g,
				      gushort b)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	ce->frame_color.red   = r;
	ce->frame_color.green = g;
	ce->frame_color.blue  = b;
}


void
catalog_png_exporter_set_caption (CatalogPngExporter *ce,
				  CaptionFields caption)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));
	ce->caption_fields = caption;
}


void
catalog_png_exporter_set_frame_style (CatalogPngExporter *ce,
				      FrameStyle style)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));
	ce->frame_style = style;
}


void
catalog_png_exporter_write_image_map (CatalogPngExporter *ce,
				      gboolean do_write)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));
	ce->write_image_map = do_write;
}


void
catalog_png_exporter_set_sort_method (CatalogPngExporter *ce,
				      SortMethod method)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));
	ce->sort_method = method;
}


void
catalog_png_exporter_set_sort_type (CatalogPngExporter *ce,
				    GtkSortType sort_type)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));
	ce->sort_type = sort_type;
}


static int
get_max_text_height (CatalogPngExporter *ce,
		     GList *first_item,
		     GList *last_item)
{
	int max_height = 0, tmp;
	int not_empty;
	GList *scan;
	char **row;

	for (scan = first_item; scan != last_item; scan = scan->next) {
		row = IMAGE_DATA (scan->data)->caption_row;
		not_empty = 0;

		tmp = 0;
		tmp += row[0] ? not_empty++, get_text_height (ce, row[0]) : 0;
		tmp += row[1] ? not_empty++, get_text_height (ce, row[1]) : 0;
		tmp += row[2] ? not_empty++, get_text_height (ce, row[2]) : 0;
		tmp += row[3] ? not_empty++, get_text_height (ce, row[3]) : 0;
		tmp += not_empty * TEXT_SPACING;

		max_height = MAX (max_height, tmp);
	}

	return max_height;
}


static void
set_item_caption (CatalogPngExporter *ce,
		  ImageData *idata)
{
	int row = 0;

	if (idata->caption_set)
		return;

	if ((ce->caption_fields & CAPTION_COMMENT) &&
	    (idata->comment != NULL))
		idata->caption_row[row++] = g_strdup (idata->comment);
	if ((ce->caption_fields & CAPTION_FILE_PATH) 
	    && (ce->caption_fields & CAPTION_FILE_NAME)) 
		idata->caption_row[row++] = g_strdup (idata->filename);
	else {
		if (ce->caption_fields & CAPTION_FILE_PATH) 
			idata->caption_row[row++] = remove_level_from_path (idata->filename);
		else if (ce->caption_fields & CAPTION_FILE_NAME)
			idata->caption_row[row++] = g_strdup (file_name_from_path (idata->filename));
	}
	
	if (ce->caption_fields & CAPTION_FILE_SIZE) 
		idata->caption_row[row++] = gnome_vfs_format_file_size_for_display (idata->file_size);

	if (ce->caption_fields & CAPTION_IMAGE_DIM) 
		idata->caption_row[row++] = g_strdup_printf (
			"%dx%d", 
			idata->image_width,
			idata->image_height);
	idata->caption_set = TRUE;
}


static gint 
get_page_height (CatalogPngExporter *ce,
		 int page_n)
{
	int page_height = 0;

	if (ce->page_size_use_row_col 
	    && ! png_exporter_pref.all_pages_same_size) 
		page_height = ce->pages_height[page_n - 1];
	else
		page_height = ce->page_height;

	return page_height;

}


static void
export (CatalogPngExporter *ce)
{
	int cols;
	int i;
	int x, y;
	GList *scan_file, *row_first_item, *row_last_item, *scan_row;
	int twidth, theight;
	int page_n = 0;
	gboolean first_row;
	ImageData *idata;
	int page_height;


	cols = (ce->page_width - COL_SPACING) / (ce->frame_width + COL_SPACING);

	begin_export (ce);

	first_row = TRUE;
	begin_page (ce, ++page_n);
	page_height = get_page_height (ce, page_n);
	y = ROW_SPACING;

	scan_file = ce->file_list;
	idata = (ImageData*) scan_file->data;
	do {
		int row_height;

		/* get items to paint. */
		row_first_item = scan_file;
		row_last_item = NULL;
		for (i = 0; i < cols; i++) {
			if (scan_file == NULL) {
				cols = i;
				break;
			}

			set_item_caption (ce, idata);

			row_last_item = scan_file = scan_file->next;
			if (scan_file != NULL)
				idata = (ImageData*) scan_file->data;
		}

		if (cols == 0) {
			end_page (ce, page_n);
			goto label_end;
		}

		/* check whether the row fit the current page. */
		row_height = (ce->frame_height 
			      + get_max_text_height (ce, 
						     row_first_item, 
						     row_last_item)
			      + ROW_SPACING);

		while (y + row_height > ce->page_height) {
			if (first_row == TRUE) {
				/* this row has an height greater than the
				 * page height, close and exit. */
				goto label_end;
			}
			
			/* the row does not fit this page, create a new
			 * page. */
			if (page_n > 0)
				end_page (ce, page_n);

			first_row = TRUE;
			begin_page (ce, ++page_n);
			page_height = get_page_height (ce, page_n);
			y = ROW_SPACING;
		}

		/* paint the row. */
		x = COL_SPACING;
		for (scan_row = row_first_item; scan_row != row_last_item; scan_row = scan_row->next) {
			GdkPixbuf *pixbuf = IMAGE_DATA (scan_row->data)->thumb;
			char **row = IMAGE_DATA (scan_row->data)->caption_row;
			char *filename = IMAGE_DATA (scan_row->data)->filename;
			gint y1, h, i;
			GdkRectangle frame_rect, image_rect;

			twidth = gdk_pixbuf_get_width (pixbuf);
			theight = gdk_pixbuf_get_height (pixbuf);

			frame_rect.x = x;
			frame_rect.y = y;
			frame_rect.width = ce->frame_width;
			frame_rect.height = ce->frame_height;

			image_rect.x = x + (ce->frame_width - twidth) / 2;
			image_rect.y = y + (ce->frame_height - theight) / 2;
			image_rect.width = twidth;
			image_rect.height = theight;

			image_rect.x--;
			image_rect.y--;
			image_rect.width++;
			image_rect.height++;
			paint_frame (ce, &frame_rect, &image_rect, filename);

			image_rect.x++;
			image_rect.y++;
			image_rect.width--;
			image_rect.height--;
			paint_image (ce, &image_rect, pixbuf);

			y1 = y + ce->frame_height + TEXT_SPACING;

			for (i = 0; i < 3; i++)
				if (row[i] != NULL) {
					paint_text (ce, x, y1, row[i], &h);
					y1 += h + TEXT_SPACING;
				}

			x += ce->frame_width + COL_SPACING;
		}
		y += row_height;
		first_row = FALSE;
	} while (TRUE);

 label_end:
	end_export (ce);
	gtk_signal_emit (GTK_OBJECT (ce), catalog_png_exporter_signals[DONE]);
}


static void
compute_pages_size (CatalogPngExporter *ce)
{
	GList *scan_file, *row_first_item, *row_last_item;
	int cols, rows;
	int c, r;
	int pages, page_n;
	gboolean not_done;
	ImageData *idata;

	ce->page_width = COL_SPACING + ce->page_cols * (ce->frame_width + COL_SPACING);
	ce->page_height = 0;

	cols = ce->page_cols;
	rows = ce->page_rows;
	pages = ce->n_images / (cols * rows) + 2;
	ce->pages_height = g_new (int, pages);

	not_done = TRUE;
	page_n = 0;
	scan_file = ce->file_list;
	idata = (ImageData*) scan_file->data;
	do {
		int row_height;
		int page_height;

		page_height = ROW_SPACING;

		for (r = 0; r < rows; r++) {
			/* get row items. */
			row_first_item = scan_file;
			row_last_item = NULL;
			for (c = 0; c < cols; c++) {
				if (scan_file == NULL) {
					cols = c;
					break;
				}
				
				set_item_caption (ce, idata);
				
				row_last_item = scan_file = scan_file->next;
				if (scan_file != NULL)
					idata = (ImageData*) scan_file->data;
			}
			
			if (cols == 0) {
				not_done = FALSE;
				break;
			}

			/* check whether the row fit the current page. */
			row_height = (ce->frame_height 
				      + get_max_text_height (ce, 
							     row_first_item, 
							     row_last_item)
				      + ROW_SPACING);
			
			page_height += row_height;
		}

		ce->pages_height[page_n] = page_height;
		ce->page_height = MAX (ce->page_height, page_height);
		page_n++;
	} while (not_done);
}


static gint
comp_func_name (gconstpointer a, gconstpointer b)
{
	ImageData *data_a, *data_b;

	data_a = IMAGE_DATA (a);
	data_b = IMAGE_DATA (b);

	return strcasecmp (file_name_from_path (data_a->filename), 
			   file_name_from_path (data_b->filename));
}


static gint
comp_func_path (gconstpointer a, gconstpointer b)
{
	ImageData *data_a, *data_b;

	data_a = IMAGE_DATA (a);
	data_b = IMAGE_DATA (b);

	return strcasecmp (data_a->filename, data_b->filename);
}


static gint
comp_func_time (gconstpointer a, gconstpointer b)
{
	ImageData *data_a, *data_b;

	data_a = IMAGE_DATA (a);
	data_b = IMAGE_DATA (b);

	if (data_a->file_time == data_b->file_time)
		return 0;
	else if (data_a->file_time > data_b->file_time)
		return 1;
	else
		return -1;
}


static gint
comp_func_size (gconstpointer a, gconstpointer b)
{
	ImageData *data_a, *data_b;

	data_a = IMAGE_DATA (a);
	data_b = IMAGE_DATA (b);

	if (data_a->file_size == data_b->file_size)
		return 0;
	else if (data_a->file_size > data_b->file_size)
		return 1;
	else
		return -1;
}


static gint
comp_func_none (gconstpointer a, gconstpointer b)
{
	return 0;
}


static GCompareFunc
get_sortfunc (CatalogPngExporter *ce)
{
	GCompareFunc func;

	switch (ce->sort_method) {
	case SORT_BY_NAME:
		func = comp_func_name;
		break;
	case SORT_BY_TIME:
		func = comp_func_time;
		break;
	case SORT_BY_SIZE:
		func = comp_func_size;
		break;
	case SORT_BY_PATH:
		func = comp_func_path;
		break;
	case SORT_NONE:
		func = comp_func_none;
		break;
	default:
		func = comp_func_none;
	}

	return func;
}


static void
load_next_file (CatalogPngExporter *ce)
{
	gchar *filename;

	gtk_signal_emit (GTK_OBJECT (ce), 
			 catalog_png_exporter_signals[PROGRESS],
			 ((float) ++ce->n_images_done) / ce->n_images);

	ce->file_to_load = ce->file_to_load->next;
	if (ce->file_to_load == NULL) {
		/* sort list */
		ce->file_list = g_list_sort (ce->file_list, get_sortfunc (ce));
		if (ce->sort_type == GTK_SORT_DESCENDING)
			ce->file_list = g_list_reverse (ce->file_list);

		/* compute the page size if needed. */
		if (ce->page_size_use_row_col) 
			compute_pages_size (ce);

		export (ce);

		return;
	}

	filename = IMAGE_DATA (ce->file_to_load->data)->filename;
	thumb_loader_set_path (ce->tloader, filename);
	thumb_loader_start (ce->tloader);
}


static void
thumb_loader_done (ThumbLoader *tloader, 
		   gpointer data)
{
	CatalogPngExporter *ce = data;
	ImageLoader *il;
	GdkPixbuf *pixbuf;
	ImageData *idata;

	idata = (ImageData*) ce->file_to_load->data;

	/* thumbnail. */
	pixbuf = thumb_loader_get_pixbuf (tloader);
	gdk_pixbuf_ref (pixbuf);
	idata->thumb = pixbuf;
	
	/* image width and height. */
	il = thumb_loader_get_image_loader (tloader);
	pixbuf = image_loader_get_pixbuf (il);
	idata->image_width = gdk_pixbuf_get_width (pixbuf);
	idata->image_height = gdk_pixbuf_get_height (pixbuf);

	/* file size. */
	idata->file_size = get_file_size (idata->filename);

	/* file time. */
	idata->file_time = get_file_mtime (idata->filename);

	load_next_file (ce);
}


static void
thumb_loader_error (ThumbLoader *tloader, 
		    gpointer data)
{
	CatalogPngExporter *ce = data;

	load_next_file (ce);
}


void
catalog_png_exporter_export (CatalogPngExporter *ce)
{
	g_return_if_fail (IS_CATALOG_PNG_EXPORTER (ce));

	g_return_if_fail (ce->page_size_use_row_col || ce->page_width != 0);
	g_return_if_fail (ce->page_size_use_row_col || ce->page_height != 0);
	g_return_if_fail (ce->thumb_width != 0);
	g_return_if_fail (ce->thumb_height != 0);

	if (ce->tloader != NULL)
		gtk_object_destroy (GTK_OBJECT (ce->tloader));

	ce->tloader = THUMB_LOADER (thumb_loader_new (NULL, 
						      ce->thumb_width,
						      ce->thumb_height));
	thumb_loader_use_cache (ce->tloader, FALSE);
	gtk_signal_connect (GTK_OBJECT (ce->tloader), "done",
			    thumb_loader_done,
			    ce);
	gtk_signal_connect (GTK_OBJECT (ce->tloader), "error",
			    thumb_loader_error,
			    ce);

	if (ce->file_list == NULL) 
		return;

	ce->n_images = g_list_length (ce->file_list);
	ce->n_images_done = 0;
		
	ce->file_to_load = ce->file_list;
	thumb_loader_set_path (ce->tloader, 
			       IMAGE_DATA (ce->file_to_load->data)->filename);
	thumb_loader_start (ce->tloader);
}



/* ----- */


static char *
leading_zero (gint n)
{
	static char s[4];
	int i;

	sprintf (s, "%3d", n);
	for (i = 0; i < 4; i++)
		if (s[i] == ' ')
			s[i] = '0';

	return s;
}


static void 
begin_export (CatalogPngExporter *ce)
{
	ce->pixmap = gdk_pixmap_new ((GdkWindow*) &gdk_root_parent,
				     ce->page_width, ce->page_height, 
				     -1);
	ce->gc = gdk_gc_new (ce->pixmap);
	gdk_gc_set_font (ce->gc, ce->font);

	ce->white.red   = 65535;
	ce->white.green = 65535;
	ce->white.blue  = 65535;
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->white, FALSE, TRUE);

	ce->black.red   = 0;
	ce->black.green = 0;
	ce->black.blue  = 0;
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->black, FALSE, TRUE);

	ce->gray.red   = 65535 * 0.8;
	ce->gray.green = 65535 * 0.8;
	ce->gray.blue  = 65535 * 0.8;
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->gray, FALSE, TRUE);

	ce->dark_gray.red   = 65535 * 0.5;
	ce->dark_gray.green = 65535 * 0.5;
	ce->dark_gray.blue  = 65535 * 0.5;
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->dark_gray, FALSE, TRUE);

	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->page_color, FALSE, TRUE);
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->text_color, FALSE, TRUE);
	gdk_colormap_alloc_color (gdk_colormap_get_system (),
				  &ce->frame_color, FALSE, TRUE);
}


static void
end_export (CatalogPngExporter *ce)
{
	gdk_gc_unref (ce->gc);
	gdk_pixmap_unref (ce->pixmap);
}


static void
begin_page (CatalogPngExporter *ce, 
	    int page_n)
{
	GnomeVFSURI *uri;
	GnomeVFSResult result;
	GnomeVFSFileSize temp;
	int width, height;
	gchar path[256];
	gchar *line;

	width = ce->page_width;
	height = get_page_height (ce, page_n);

	gdk_gc_set_foreground (ce->gc, &ce->page_color); 
	gdk_draw_rectangle (ce->pixmap,
			    ce->gc,
			    TRUE,
			    0, 0,
			    width,
			    height);

	/* image map file. */

	if (! ce->write_image_map)
		return;

	sprintf (path, "%s/%s%s%s.html", ce->directory, ce->filename1,
		 leading_zero (page_n), ce->filename2);

	uri = gnome_vfs_uri_new (path);
	if (uri == NULL) {
		g_warning ("URI not valid : %s", path);
		return;
	}

	ce->imap_handle = NULL;
	result = gnome_vfs_create_uri (&ce->imap_handle, uri, 
				       GNOME_VFS_OPEN_WRITE, FALSE, 0664);
	if (result != GNOME_VFS_OK) {
		g_warning ("Cannot create file %s", path);
		return;
	}

	sprintf (path, "%s%s%s.png", ce->filename1,
		 leading_zero (page_n), ce->filename2);

	line = g_strdup_printf ("<IMG SRC=\"%s\" WIDTH=%d HEIGHT=%d BORDER=0 USEMAP=\"#map\">\n\n", path, width, height);
	gnome_vfs_write (ce->imap_handle, line, strlen (line), &temp);
	g_free (line);

	line = g_strdup_printf ("<MAP NAME=\"map\">\n");
	gnome_vfs_write (ce->imap_handle, line, strlen (line), &temp);
	g_free (line);
}


static void
end_page (CatalogPngExporter *ce,
	  int page_n)
{
	GdkPixbuf *pixbuf;
	gchar path[256];
	int width, height;
	GnomeVFSFileSize temp;
	gchar *line;

	width = ce->page_width;
	height = get_page_height (ce, page_n);

	pixbuf = gdk_pixbuf_get_from_drawable (NULL,
					       ce->pixmap,
					       gdk_colormap_get_system (),
					       0, 0,
					       0, 0,
					       width,
					       height);

	sprintf (path, "%s/%s%s%s.png", ce->directory, ce->filename1,
		 leading_zero (page_n), ce->filename2);

	pixbuf_to_file_as_png (pixbuf, path);

	gdk_pixbuf_unref (pixbuf);

	/* image map file. */

	if (! ce->write_image_map || (ce->imap_handle == NULL))
		return;

	line = g_strdup_printf ("</MAP>\n");
	gnome_vfs_write (ce->imap_handle, line, strlen (line), &temp);
	g_free (line);

	gnome_vfs_close (ce->imap_handle);
}


static void
paint_frame (CatalogPngExporter *ce,
	     GdkRectangle *frame_rect,
	     GdkRectangle *image_rect,
	     gchar *filename)
{
	gint x, y, width, height;
	GnomeVFSFileSize temp;
	gchar *line;

	switch (ce->frame_style) {
	case FRAME_STYLE_NONE:
		break;

	case FRAME_STYLE_SLIDE:
		/* Outer frame. */
		x = frame_rect->x;
		y = frame_rect->y;
		width = frame_rect->width;
		height = frame_rect->height;
		
		gdk_gc_set_foreground (ce->gc, &ce->frame_color); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    TRUE,
				    x, 
				    y,
				    width, 
				    height);
		
		gdk_gc_set_foreground (ce->gc, &ce->white); 
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y,
			       x, 
			       y + height);
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y,
			       x + width, 
			       y);

		gdk_gc_set_foreground (ce->gc, &ce->dark_gray); 
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x + width, 
			       y,
			       x + width, 
			       y + height);
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y + height,
			       x + width, 
			       y + height);
		
		/* Image background. */
		x = image_rect->x;
		y = image_rect->y;
		width = image_rect->width;
		height = image_rect->height;
		
		gdk_gc_set_foreground (ce->gc, &ce->white); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    TRUE,
				    x, y,
				    width,
				    height);
		
		/* Inner frame. */
		gdk_gc_set_foreground (ce->gc, &ce->dark_gray); 
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y,
			       x, 
			       y + height);
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y,
			       x + width, 
			       y);
		
		gdk_gc_set_foreground (ce->gc, &ce->white); 
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x + width, 
			       y,
			       x + width, 
			       y + height);
		gdk_draw_line (ce->pixmap,
			       ce->gc,
			       x, 
			       y + height,
			       x + width, 
			       y + height);
		break;

	case FRAME_STYLE_SIMPLE:
	case FRAME_STYLE_SHADOW:
		x = image_rect->x;
		y = image_rect->y;
		width = image_rect->width;
		height = image_rect->height;

		gdk_gc_set_foreground (ce->gc, &ce->white); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    TRUE,
				    x, y,
				    width,
				    height);
		
		gdk_gc_set_foreground (ce->gc, &ce->frame_color); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    FALSE,
				    x, y,
				    width,
				    height);

		if (ce->frame_style == FRAME_STYLE_SIMPLE)
			break;

		gdk_gc_set_foreground (ce->gc, &ce->frame_color); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    TRUE,
				    x + 5, y + 5,
				    width,
				    height);

		gdk_gc_set_foreground (ce->gc, &ce->white); 
		gdk_draw_rectangle (ce->pixmap,
				    ce->gc,
				    TRUE,
				    x + 1, y + 1,
				    width - 1,
				    height - 1);

		break;
	}

	/* image map file. */

	if (! ce->write_image_map || (ce->imap_handle == NULL))
		return;

	line = g_strdup_printf ("<AREA SHAPE=\"RECT\" COORDS=\"%d,%d,%d,%d\" HREF=\"%s\">\n", 
				frame_rect->x, 
				frame_rect->y, 
				frame_rect->x + frame_rect->width, 
				frame_rect->y + frame_rect->height,
				file_name_from_path (filename));
	gnome_vfs_write (ce->imap_handle, line, strlen (line), &temp);
	g_free (line);	
}


static void
paint_image (CatalogPngExporter *ce,
	      GdkRectangle *image_rect,
	      GdkPixbuf *image)
{
	gint x, y, width, height;

	x = image_rect->x;
	y = image_rect->y;
	width = image_rect->width;
	height = image_rect->height;

	if (gdk_pixbuf_get_has_alpha (image))
		gdk_pixbuf_render_to_drawable_alpha (image,
						     ce->pixmap, 
						     0, 0,
						     x, y,
						     width, height,
						     GDK_PIXBUF_ALPHA_BILEVEL,
						     112,
						     GDK_RGB_DITHER_MAX, 0, 0);
	else
		gdk_pixbuf_render_to_drawable (image,
					       ce->pixmap, 
					       ce->gc,
					       0, 0,
					       x, y,
					       width, height,
					       GDK_RGB_DITHER_MAX, 0, 0);
}


static void
paint_text (CatalogPngExporter *ce,
	    int x,
	    int y,
	    char *text,
	    int *height)
{
	GnomeIconTextInfo *giti;

	giti = gnome_icon_layout_text (ce->font,
				       text,
				       DEFAULT_SEPARATORS,
				       ce->thumb_width, TRUE);

	gdk_gc_set_foreground (ce->gc, &ce->text_color); 
	gdk_gc_set_background (ce->gc, &ce->page_color); 
	gnome_icon_paint_text (giti,
			       ce->pixmap, 
			       ce->gc,
			       x + (ce->frame_width - giti->width) / 2, 
			       y,
			       GTK_JUSTIFY_CENTER);

	*height = giti->height;

	gnome_icon_text_info_free (giti);
}


static int 
get_text_height (CatalogPngExporter *ce,
		 char *text)
{
	GnomeIconTextInfo *giti;
	gint text_height;

	giti = gnome_icon_layout_text (ce->font,
				       text,
				       DEFAULT_SEPARATORS,
				       ce->thumb_width, TRUE);
	text_height = giti->height;
	gnome_icon_text_info_free (giti);

	return text_height;
}
