/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <errno.h>
#include <math.h>
#include <string.h>
#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>

#include "auto-completion.h"
#include "thumb-cache.h"
#include "catalog.h"
#include "clist-utils.h"
#include "file-utils.h"
#include "file-list.h"
#include "gthumb-app.h"
#include "image-list-utils.h"
#include "image-viewer.h"
#include "main.h"
#include "menu-callbacks.h"
#include "menu.h"
#include "nav-window.h"
#include "pixbuf-utils.h"
#include "toolbar.h"
#include "window.h"

#include "gedo-paned.h"
#include "gedo-vpaned.h"
#include "gedo-hpaned.h"

#include "icons/dir.xpm"
#include "icons/catalog.xpm"
#include "icons/catalog_search.xpm"
#include "icons/nav_button.xpm"

#define ACTIVITY_DELAY 100

enum {
	TARGET_STRING,
	TARGET_URL
};

static GtkTargetEntry target_table[] = {
	{ "STRING",        0, TARGET_STRING },
	{ "text/plain",    0, TARGET_STRING },
	{ "text/uri-list", 0, TARGET_URL }
};

static guint n_targets = sizeof (target_table) / sizeof (target_table[0]);


static void
close_window_cb (GtkWidget *caller, 
		 GdkEvent *event, 
		 GThumbWindow *window)
{
	close_cb (caller, window);
}


static void update_statusbar_list_info (GThumbWindow *window);


static int
file_selected_cb (GtkWidget *widget, 
		  gint pos, 
		  GdkEvent *event,
		  gpointer data)
{
	GThumbWindow *window = data;

	window_update_menu_sensitivity (window);
	update_statusbar_list_info (window);

	return TRUE;
}


static void
view_image_at_pos (GThumbWindow *window, 
		   gint pos)
{
	gchar *path;
	ImageList *ilist = IMAGE_LIST (window->file_list->ilist);
	
	/* If the image is from a catalog remember the catalog name. */
	if (window->image_catalog != NULL) {
		g_free (window->image_catalog);
		window->image_catalog = NULL;
	}

	if (window->sidebar_content == CATALOG_LIST)
		window->image_catalog = g_strdup (window->catalog_path);
	
	/* Load the image. */
	path = file_list_path_from_row (window->file_list, pos);
	window_load_image (window, path);
	g_free (path);
	
	/* Center row if not fully visible. */
	if (image_list_image_is_visible (ilist, pos) != GTK_VISIBILITY_FULL)
		image_list_moveto (ilist, pos, 0.5); 
}


static int
file_button_press_cb (GtkWidget *widget, 
		      GdkEventButton *event,
		      gpointer data)
{
#define HIDE(x) gtk_widget_hide (x);
#define SHOW(x) gtk_widget_show (x);
	ImageList *ilist = IMAGE_LIST (widget);
	GThumbWindow *window = data;
	gint pos;

	if ((event->button != 1) && (event->button != 3))
		return FALSE;

	if ((event->state & GDK_SHIFT_MASK)
	    || (event->state & GDK_CONTROL_MASK))
		return FALSE;

	pos = image_list_get_image_at (ilist, event->x, event->y);
	if (pos == -1)
		return FALSE;

	if (event->button == 1) {
		view_image_at_pos (window, pos);
		return TRUE;
	}

	/* Button 3 pressed. */

	gtk_widget_set_sensitive (window->popupmenu_file[2],
				  ilist_utils_only_one_is_selected (ilist));
	gtk_widget_set_sensitive (window->popupmenu_file[12],
				  ilist_utils_only_one_is_selected (ilist));

	if (window->sidebar_content == CATALOG_LIST) {
		HIDE (window->popupmenu_file[10]);
		SHOW (window->popupmenu_file[11]);
		SHOW (window->popupmenu_file[12]);
	} else {
		SHOW (window->popupmenu_file[10]);
		HIDE (window->popupmenu_file[11]);
		HIDE (window->popupmenu_file[12]);
	}

	gtk_menu_popup (GTK_MENU (window->file_popup_menu),
			NULL, NULL, NULL, 
			window, 
			event->button,
			event->time);

	return TRUE;
#undef HIDE
#undef SHOW
}


static int
file_unselected_cb (GtkWidget *widget, 
		    gint pos, 
		    GdkEvent *event,
		    gpointer data)
{
	GThumbWindow *window = data;	

	window_update_menu_sensitivity (window);
	update_statusbar_list_info (window);

	return TRUE;
}


static void 
dir_selected_cb (GtkWidget *widget, 
		 gint row, 
		 gint col,
		 GdkEvent *event,
		 gpointer data)
{
	GThumbWindow *window = data;
	gchar *path;

	path = dir_list_path_from_row (window->dir_list, row);
	window_go_to_directory (window, path);
	g_free (path);
}


static void window_set_file_list (GThumbWindow *window, 
				  GList *list,
				  DoneFunc done_func,
				  gpointer done_func_data);


static void
catalog_selected_cb_continue (gpointer data)
{
	GThumbWindow *window = data;

	/* Add to history list if not present as last entry. */
	if ((preferences.history->list == NULL) 
	    || ((window->catalog_path != NULL)
		&& (strcmp (window->catalog_path, pref_util_remove_prefix (preferences.history->list->data)) != 0))) {
		gint row;
		gboolean is_search;

		row = catalog_list_row_from_path (window->catalog_list,
						  window->catalog_path);
		is_search = catalog_list_is_search (window->catalog_list, row);

		bookmarks_add_with_prefix (preferences.history, 
					   window->catalog_path,
					   is_search ? SEARCH_PREFIX : CATALOG_PREFIX);
		bookmarks_write_to_disk (preferences.history);
		all_windows_update_history_list ();
	}
}


static void 
catalog_selected_cb (GtkWidget *widget, 
		     gint row, 
		     gint col,
		     GdkEvent *event,
		     gpointer data)
{
	GThumbWindow *window = data;
	gchar *path;
	Catalog *catalog;

	path = catalog_list_path_from_row (window->catalog_list, row);

	if (path_is_dir (path)) {
		window_set_file_list (window, NULL, NULL, NULL);
		if (window->catalog_path)
			g_free (window->catalog_path);
		window->catalog_path = NULL;
	
		catalog_list_change_to (window->catalog_list, path);
		g_free (path);
	} else {
		catalog = catalog_new ();
		catalog_load_from_disk (catalog, path);
		window_set_file_list (window, 
				      catalog->list,
				      catalog_selected_cb_continue,
				      window);
		catalog_free (catalog);

		if (window->catalog_path)
			g_free (window->catalog_path);
		window->catalog_path = path;

		window_update_menu_sensitivity (window);
	}
}


static int
catalog_button_press_cb (GtkWidget *widget, 
			 GdkEventButton *event,
			 gpointer data)
{
	GtkCList *clist = GTK_CLIST (widget);
	GThumbWindow *window = data;
	gint row, col;
	gchar *name;
	gboolean is_search;

	if (event->button != 3)
		return FALSE;

	if (!gtk_clist_get_selection_info (clist, event->x, event->y, &row, &col))
		return FALSE;

	if ((event->state & GDK_SHIFT_MASK) 
	    || (event->state & GDK_CONTROL_MASK))
		return FALSE;

	/* Ignore the ".." item. */
	gtk_clist_get_pixtext (clist, row, col, &name, NULL, NULL, NULL);
	if (strcmp (name, "..") == 0)
		return FALSE;

	/* If the item is a directory hide the "move to" command. */
	if (path_is_dir (gtk_clist_get_row_data (clist, row)))
		gtk_widget_hide (window->popupmenu_catalog[2]);
	else
		gtk_widget_show (window->popupmenu_catalog[2]);

	/**/
	is_search = catalog_list_is_search (window->catalog_list, row);
	gtk_widget_set_sensitive (window->popupmenu_catalog[4], is_search);
	gtk_widget_set_sensitive (window->popupmenu_catalog[5], is_search);

	if (!clist_utils_row_is_selected (clist, row)) {
		gtk_signal_handler_block_by_data (GTK_OBJECT (clist), window);
		gtk_clist_unselect_all (clist);
		gtk_clist_select_row (clist, row, col);
		gtk_signal_handler_unblock_by_data (GTK_OBJECT (clist), window);
	}

	gtk_menu_popup (GTK_MENU (window->catalog_popup_menu),
			NULL, NULL, NULL, 
			window, 
			event->button,
			event->time);
	window_update_menu_sensitivity (window);

	return TRUE;
}


static void 
bookmark_selected_cb (GtkWidget *widget, 
		      gpointer data)
{
	GThumbWindow *window = data;
	gchar *path;
	const gchar *no_prefix_path;

	path = gtk_object_get_data (GTK_OBJECT (widget), "full_path");
	no_prefix_path = pref_util_remove_prefix (path);

	if (pref_util_location_is_catalog (path) 
	    || pref_util_location_is_search (path)) 
		window_go_to_catalog (window, no_prefix_path);
	else 
		window_go_to_directory (window, no_prefix_path);
}


static gint
location_entry_key_press_cb (GtkWidget *widget, 
			     GdkEventKey *event,
			     GThumbWindow *window)
{
	gchar *path;
	gint   n;

	g_return_val_if_fail (window != NULL, FALSE);
	
	switch (event->keyval) {
	case GDK_Return:
	case GDK_KP_Enter:
		path = gtk_entry_get_text (GTK_ENTRY (widget));
		window_go_to_directory (window, path);
                return TRUE;

	case GDK_Tab:
		if (event->state & GDK_CONTROL_MASK) {
			gtk_signal_emit_by_name (GTK_OBJECT (window->app),
						 "focus_changed",
						 TRUE);
			return TRUE;
		}

		path = gtk_entry_get_text (GTK_ENTRY (widget));
		n = auto_compl_get_n_alternatives (path);

		if (n > 0) { 
			gchar *text;
			text = auto_compl_get_common_prefix ();

			if (n == 1) {
				auto_compl_hide_alternatives ();
				if ((window->dir_list->path != NULL)
				    && strcmp (window->dir_list->path, text))
					window_go_to_directory (window, text);
				else {
					/* Add a separator at the end. */
					gchar *new_path;
					gint len = strlen (path);

					if (strcmp (path, text) != 0) {
						/* Reset the right name. */
						gtk_entry_set_text (GTK_ENTRY (widget), text);
						return TRUE;
					}
					
					/* Ending separator, do nothing. */
					if ((len <= 1) 
					    || (path[len - 1] == '/'))
						return TRUE;

					new_path = g_strconcat (path,
								"/",
								NULL);
					gtk_entry_set_text (GTK_ENTRY (widget), new_path);
					g_free (new_path);

					/* Re-Tab */
					gtk_widget_event (widget, (GdkEvent*)event);
				}
			} else {
				gtk_entry_set_text (GTK_ENTRY (widget), text);
				auto_compl_show_alternatives (widget);
			}

			if (text)
				g_free (text);
		}
		return TRUE;

	case GDK_Right:
	case GDK_Left:
	case GDK_Up:
	case GDK_Down:
		return TRUE;
	}

	return TRUE;
}


static gint
image_clicked_cb (GtkWidget *widget, 
		  GThumbWindow *window)
{
	window_show_next_image (window);
	return TRUE;
}


void
window_update_statusbar (GThumbWindow *window)
{
	gchar *info_txt;
	gchar  time_txt[50];
	gchar *path;
	gint width, height;
	time_t timer;
	struct tm *tm;
	gdouble sec;

	width = image_viewer_get_image_width (IMAGE_VIEWER (window->viewer));
	height = image_viewer_get_image_height (IMAGE_VIEWER (window->viewer));

	path = window->image_path;
	
	timer = get_file_mtime (path);
	tm = localtime (&timer);
	strftime (time_txt, 50, "%d %b %Y, %X", tm);
	sec = g_timer_elapsed (image_loader_get_timer (IMAGE_VIEWER (window->viewer)->loader),  NULL);

	info_txt = g_strdup_printf ("(%d x %d) [%s] {%2.2fs}",
				    width,
				    height,
				    time_txt,
				    sec);
	gtk_label_set (GTK_LABEL (window->image_info), info_txt);

	g_free (info_txt);
}


static gint
image_loaded_cb (GtkWidget *widget, 
		 GThumbWindow *window)
{
	window_update_statusbar (window);
	window_update_title (window);
	window_update_menu_sensitivity (window);
	return TRUE;
}


static gint
zoom_changed_cb (GtkWidget *widget, 
		 GThumbWindow *window)
{
	window_update_title (window);

	gtk_signal_handler_block_by_data (GTK_OBJECT (window->tool_item_zoom_to_fit), window);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (window->tool_item_zoom_to_fit), IMAGE_VIEWER (window->viewer)->zoom_fit);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (window->tool_item_zoom_to_fit), window);

	return TRUE;	
}


static gint
key_press_cb (GtkWidget *widget, 
	      GdkEventKey *event,
	      gpointer data)
{
	GThumbWindow *window = data;
	ImageViewer *viewer = IMAGE_VIEWER (window->viewer);

	if (GTK_WIDGET_HAS_FOCUS (window->location_entry))
		return FALSE;

	if (image_list_editing (IMAGE_LIST (window->file_list->ilist)))
		return FALSE;

	if (event->state & GDK_SHIFT_MASK) {
		switch (event->keyval) {
		case GDK_Down:
		case GDK_KP_Down:
			image_viewer_scroll_step_y (viewer, TRUE);
			return TRUE;

		case GDK_Up:
		case GDK_KP_Up:
			image_viewer_scroll_step_y (viewer, FALSE);
			return TRUE;

		case GDK_Right:
		case GDK_KP_Right:
			image_viewer_scroll_step_x (viewer, TRUE);
			return TRUE;

		case GDK_Left:
		case GDK_KP_Left:
			image_viewer_scroll_step_x (viewer, FALSE);
			return TRUE;
		}
	}

	if (event->state & GDK_CONTROL_MASK) {
		switch (event->keyval) {
		case GDK_Down:
		case GDK_KP_Down:
			image_viewer_scroll_page_y (viewer, TRUE);
			return TRUE;

		case GDK_Up:
		case GDK_KP_Up:
			image_viewer_scroll_page_y (viewer, FALSE);
			return TRUE;

		case GDK_Right:
		case GDK_KP_Right:
			image_viewer_scroll_page_x (viewer, TRUE);
			return TRUE;

		case GDK_Left:
		case GDK_KP_Left:
			image_viewer_scroll_page_x (viewer, FALSE);
			return TRUE;
		}
	}

	if ((event->state & GDK_CONTROL_MASK)
	    || (event->state & GDK_MOD1_MASK))
		return FALSE;

	switch (event->keyval) {
		/* Close window. */
	case GDK_q:
		close_cb (widget, window);
		return TRUE;

		/* Zoom in. */
	case GDK_plus:
	case GDK_equal:
	case GDK_KP_Add:
		image_viewer_zoom_in (viewer);
		break;

		/* Zoom out. */
	case GDK_minus:
	case GDK_KP_Subtract:
		image_viewer_zoom_out (viewer);
		break;
		
		/* Actual size. */
	case GDK_KP_Divide:
	case GDK_1:
	case GDK_z:
		image_viewer_set_zoom (viewer, 1.0);
		break;

		/* Zomm to fit. */
	case GDK_x:
		image_viewer_zoom_to_fit (viewer);
		break;

		/* Full screen view. */
	case GDK_v:
	case GDK_F11:
		fullscreen_start (fullscreen, window);
		break;

		/* Interrupt thumbnails creation. */
	case GDK_Escape:
		if (window->file_list->doing_thumbs)
			file_list_interrupt_thumbs (window->file_list);
		break;

		/* View/hide thumbnails. */
	case GDK_t:
		file_list_enable_thumbs (window->file_list, 
					 ! (window->file_list->enable_thumbs));
		break;

		/* Start/Stop Slideshow. */
	case GDK_s:
		slideshow_cb (widget, window);
		break;

		/* Toggle animation. */
	case GDK_g:
		toggle_animation_cb (widget, window);
		break;

		/* Step animation. */
	case GDK_j:
		step_animation_cb (widget, window);
		break;

		/* Flip image. */
	case GDK_f:
	case GDK_F:
		flip_cb (widget, window);
		break;

		/* Rotate. */
	case GDK_bracketright:
	case GDK_r:
		image_viewer_alter (viewer, ALTER_ROTATE_90);
		break;

	case GDK_bracketleft:
		image_viewer_alter (viewer, ALTER_ROTATE_90_CC);
		break;

		/* Mirror. */
	case GDK_m:
	case GDK_M:
		mirror_cb (widget, window);
		break;

		/* View/Hide Browser. */
	case GDK_Return:
		browser_cb (NULL, window);
		break;

	default:
		break;
	}

	if (GTK_WIDGET_HAS_FOCUS (window->dir_list->clist))
		return FALSE;

	if (GTK_WIDGET_HAS_FOCUS (window->catalog_list->clist))
		return FALSE;

	switch (event->keyval) {
		/* Show previous image. */
	case GDK_b:
	case GDK_BackSpace:
	case GDK_Up:
	case GDK_Page_Up:
	case GDK_Left: 
	case GDK_KP_Up:
	case GDK_KP_Page_Up:
	case GDK_KP_Left:
		window_show_prev_image (window);
		break;

		/* Show next image. */
	case GDK_space:
	case GDK_Right:
	case GDK_Down:
	case GDK_Page_Down: 
	case GDK_KP_Right:
	case GDK_KP_Down:
	case GDK_KP_Page_Down:
		window_show_next_image (window);
		break;

		/* Show first image. */
	case GDK_Home: 
	case GDK_KP_Home:
		window_show_first_image (window);
		break;
		
		/* Show last image. */
	case GDK_End: 
	case GDK_KP_End:
		window_show_last_image (window);
		break;

		/* Delete selection. */
	case GDK_Delete: 
	case GDK_KP_Delete:
		if (window->sidebar_content == DIR_LIST)
			delete_cb (NULL, window);
		else if (window->sidebar_content == CATALOG_LIST)
			remove_from_catalog_cb (NULL, window);
		break;

	default:
		break;
	}

	return TRUE;
}


static void
move_focus_forward (GThumbWindow *window)
{
	GtkWidget *dir_or_catalog_list;

	if (window->sidebar_content == DIR_LIST)
		dir_or_catalog_list = window->dir_list->clist;
	else
		dir_or_catalog_list = window->catalog_list->clist;

	if (window->focused == window->file_list->ilist)
		window->focused = dir_or_catalog_list;
	else if (window->focused == dir_or_catalog_list)
		window->focused = window->location_entry;
	else if (window->focused == window->location_entry)
		window->focused = window->file_list->ilist;

	if (! window->sidebar_visible && 
	    (window->focused == dir_or_catalog_list))
		move_focus_forward (window);
}


static void
move_focus_backward (GThumbWindow *window)
{
	GtkWidget *dir_or_catalog_list;

	if (window->sidebar_content == DIR_LIST)
		dir_or_catalog_list = window->dir_list->clist;
	else
		dir_or_catalog_list = window->catalog_list->clist;

	if (window->focused == window->file_list->ilist)
		window->focused = window->location_entry;
	else if (window->focused == dir_or_catalog_list)
		window->focused = window->file_list->ilist;
	else if (window->focused == window->location_entry)
		window->focused = dir_or_catalog_list;

	if (! window->sidebar_visible && 
	    (window->focused == dir_or_catalog_list))
		move_focus_backward (window);
}


static gint
focus_changed_cb (GtkWidget *widget, 
		  gboolean forward,
		  gpointer data)
{
	GThumbWindow *window = data;
	GdkEventFocus fevent;

	window->focused = GTK_WINDOW (window->app)->focus_widget;

	if (window->focused != NULL) {
		fevent.type = GDK_FOCUS_CHANGE;
		fevent.window =  window->focused->window;
		fevent.in = FALSE;
		gtk_widget_event (window->focused, (GdkEvent*) &fevent);
		gtk_widget_queue_draw (window->focused);
		
		if (forward) 
			move_focus_forward (window);
		else 
			move_focus_backward (window);
	} else
		window->focused = window->file_list->ilist;

	gtk_window_set_focus (GTK_WINDOW (window->app), window->focused);
	gtk_widget_queue_draw (window->focused);

	return TRUE;
}


void
give_focus_to_image_list (GThumbWindow *window)
{
	if (window->focused != window->file_list->ilist) {
		if ((window->focused == window->catalog_list->clist) 
		    || (window->focused == window->dir_list->clist)) 
			focus_changed_cb (window->app, FALSE, window);
		else
			focus_changed_cb (window->app, TRUE, window);
	}
}


static int
image_button_press_cb (GtkWidget *widget, 
		       GdkEventButton *event,
		       gpointer data)
{
	GThumbWindow *window = data;
	ImageViewer *viewer = IMAGE_VIEWER (window->viewer);
	ImageList *ilist;

	switch (event->button) {
	case 1:
	case 2:
		break;
		
	case 3:
		ilist = IMAGE_LIST (window->file_list->ilist);

		gtk_widget_set_sensitive (image_popup_menu_data[7].widget, ilist_utils_selection_not_null (ilist));
		gtk_widget_set_sensitive (image_popup_menu_data[9].widget, ilist_utils_only_one_is_selected (ilist));
		gtk_widget_set_sensitive (image_popup_menu_data[10].widget, ilist_utils_selection_not_null (ilist));
		gtk_widget_set_sensitive (image_popup_menu_data[11].widget, ilist_utils_selection_not_null (ilist));
		gtk_widget_set_sensitive (image_popup_menu_data[12].widget, ilist_utils_selection_not_null (ilist));

		if (window->fullscreen) {
			gtk_widget_hide (image_popup_menu_data[6].widget);
			gtk_widget_hide (image_popup_menu_data[7].widget);
			gtk_widget_hide (image_popup_menu_data[8].widget);
			gtk_widget_hide (image_popup_menu_data[9].widget);
			gtk_widget_hide (image_popup_menu_data[10].widget);
			gtk_widget_hide (image_popup_menu_data[11].widget);
			gtk_widget_hide (image_popup_menu_data[12].widget);
			
		} else {
			gtk_widget_show (image_popup_menu_data[6].widget);
			gtk_widget_show (image_popup_menu_data[7].widget);
			gtk_widget_show (image_popup_menu_data[8].widget);
			gtk_widget_show (image_popup_menu_data[9].widget);
			gtk_widget_show (image_popup_menu_data[10].widget);
			gtk_widget_show (image_popup_menu_data[11].widget);
			gtk_widget_show (image_popup_menu_data[12].widget);
		}

		gtk_menu_popup (GTK_MENU (window->image_popup_menu),
				NULL, NULL, NULL, 
				window, 
				event->button,
				event->time);
		
		return TRUE;

		/* -- Mouse Wheel -- */
	case 4:
		if (event->state & GDK_CONTROL_MASK) 
			image_viewer_zoom_in (viewer);
		else if (event->state & GDK_SHIFT_MASK) 
			image_viewer_scroll_step_y (viewer, FALSE);
		else 
			window_show_prev_image (window);
		return TRUE;

	case 5:
		if (event->state & GDK_CONTROL_MASK) 
			image_viewer_zoom_out (viewer);
		else if (event->state & GDK_SHIFT_MASK) 
			image_viewer_scroll_step_y (viewer, TRUE);
		else 
			window_show_next_image (window);
		return TRUE;
	}

	return FALSE;
}


static void
window_progress (gfloat percent, 
		 gpointer data)
{
	GThumbWindow *window = data;
	gtk_progress_bar_update (GTK_PROGRESS_BAR (window->progress), percent);
}


static void  
viewer_drag_data_get  (GtkWidget          *widget,
		       GdkDragContext     *context,
		       GtkSelectionData   *selection_data,
		       guint               info,
		       guint               time,
		       gpointer            data)
{
	GThumbWindow *window = data;
	gchar *path;

	if (IMAGE_VIEWER (window->viewer)->is_void) 
		return;

	path = image_viewer_get_image_filename (IMAGE_VIEWER (window->viewer));
	if (info == TARGET_URL) {
		gchar *url;

		url = g_strconcat ("file://",
				   path,
				   NULL);
		gtk_selection_data_set (selection_data,
					selection_data->target,
					8, 
					url, strlen (url));
		g_free (url);
	} else 
		gtk_selection_data_set (selection_data,
					selection_data->target,
					8, 
					path, strlen (path));
	g_free (path);
}


static void  
viewer_drag_data_delete  (GtkWidget          *widget,
			  GdkDragContext     *context,
			  gpointer            data)
{
	g_warning ("DragDataDelete called !!!");
}


static gchar *
make_url_list (GList *list, 
	       gboolean plain_text)
{
	gchar *url_list;
	gint   url_list_length;
	gchar *prefix;
	gint   prefix_length;
	gchar *url_sep = "\r\n";
	gint   url_sep_length;
	GList *scan;

	if (list == NULL)
		return NULL;
	
	prefix = (plain_text) ? g_strdup ("") : g_strdup ("file://");
	prefix_length = strlen (prefix);
	url_sep_length = strlen (url_sep);

	url_list_length = 0;
	for (scan = list; scan; scan = scan->next)
		url_list_length += (prefix_length 
				    + strlen (scan->data)
				    + url_sep_length);

	url_list = g_malloc (url_list_length + 1);
	*url_list = 0;

	for (scan = list; scan; scan = scan->next) {
		url_list = strncat (url_list, prefix, prefix_length);
		url_list = strncat (url_list, scan->data, strlen (scan->data));
		url_list = strncat (url_list, url_sep, url_sep_length);
	}

	g_free (prefix);

	return url_list;
}


static void  
file_list_drag_data_get  (GtkWidget          *widget,
			  GdkDragContext     *context,
			  GtkSelectionData   *selection_data,
			  guint               info,
			  guint               time,
			  gpointer            data)
{
	GThumbWindow *window = data;
	ImageList *ilist;
	GList *list;
	gchar *url_list;

	ilist = IMAGE_LIST (window->file_list->ilist);

	list = ilist_utils_get_file_list_selection (ilist);
	url_list = make_url_list (list, (info == TARGET_STRING));
	path_list_free (list);

	if (url_list == NULL) 
		return;

	gtk_selection_data_set (selection_data, 
				selection_data->target,
				8, 
				url_list, strlen (url_list));
	
	g_free (url_list);
}


static GList *
get_file_list_from_url_list (gchar *url_list)
{
	GList *list = NULL;
	gint i;
	gchar *url_start, *url_end;

	i = 0;
	url_start = url_list;
	while (url_list[i] != '\0')	{
		while ((url_list[i] != '\0')
		       && (url_list[i] != '\r')
		       && (url_list[i] != '\n')) i++;

		url_end = url_list + i;
		if (strncmp (url_start, "file:", 5) == 0) {
			url_start += 5;
			if ((url_start[0] == '/') 
			    && (url_start[1] == '/')) url_start += 2;
		}
		list = g_list_prepend (list, g_strndup (url_start, url_end - url_start));

		while ((url_list[i] != '\0')
		       && ((url_list[i] == '\r')
			   || (url_list[i] == '\n'))) i++;
		url_start = url_list + i;
	}
	
	return g_list_reverse (list);
}


static GtkWidget *
get_button_from_sidebar_content (GThumbWindow *window, 
				 gint sidebar_content)
{
	GtkWidget *button;

	switch (sidebar_content) {
	case DIR_LIST:
		button = window->tool_item_show_dir;
		break;

	case CATALOG_LIST:
		button = window->tool_item_show_catalog;
		break;

	default:
		button = NULL;
	}

	return button;
}


static void
set_toggle_button_state (GThumbWindow *window,
			 GtkWidget *button, 
			 gboolean active)
{
	gtk_signal_handler_block_by_data (GTK_OBJECT (button), window);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), active);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (button), window);
}


static void
_window_set_sidebar (GThumbWindow *window,
		     gint sidebar_content)
{
	GtkWidget *old_button, *new_button;

	old_button = get_button_from_sidebar_content (window, 
						      window->sidebar_content);
	if (old_button != NULL)
		set_toggle_button_state (window, old_button, FALSE);

	new_button = get_button_from_sidebar_content (window, sidebar_content);
	if ((new_button != NULL) && window->sidebar_visible) 
		set_toggle_button_state (window, new_button, TRUE);

	window->sidebar_content = sidebar_content;
	gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), 
			       sidebar_content - 1);

	give_focus_to_image_list (window);
	window_update_menu_sensitivity (window);
}


void  
window_drag_data_received  (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time,
			    gpointer            extra_data)
{
	GThumbWindow *window = extra_data;
	Catalog *catalog;
	gchar *catalog_path;
	gchar *catalog_name;
	GList *list;
	GList *scan;

	if (! ((data->length >= 0) && (data->format == 8))) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	gtk_drag_finish (context, TRUE, FALSE, time);

	list = get_file_list_from_url_list ((gchar *)data->data);

	/* Create a catalog with the Drag&Drop list. */

	catalog = catalog_new ();
	catalog_name = g_strconcat (_("Drag & Drop"),
				    CATALOG_EXT,
				    NULL);
	catalog_path = get_catalog_full_path (catalog_name);
	g_free (catalog_name);

	catalog_set_path (catalog, catalog_path);

	for (scan = list; scan; scan = scan->next)
		catalog_add_item (catalog, (gchar*) scan->data);

	catalog_write_to_disk (catalog);
	catalog_free (catalog);
	path_list_free (list);

	/* View the Drag&Drop catalog. */
	window_go_to_catalog (window, catalog_path);
	g_free (catalog_path);
}


static gboolean
file_list_text_changed (ImageList *ilist,
			gint pos,
			gchar *new_filename,
			gpointer data)
{
	GThumbWindow *window;
	gchar *old_filename;
	gchar *current_dir;
	gchar *old_filepath;
	gchar *new_filepath;
	GList *selection;
	gboolean result;

	window = data;

	old_filename = image_list_get_old_text (ilist);
	if (strcmp (new_filename, old_filename) == 0)
		return FALSE;

	selection = file_list_get_selection (window->file_list);
	if (selection == NULL)
		return FALSE;
	
	current_dir = remove_level_from_path (selection->data);
	path_list_free (selection);
	g_return_val_if_fail (current_dir != NULL, FALSE);

	result = TRUE;

	old_filepath = g_strconcat (current_dir,
				    "/",
				    old_filename,
				    NULL);
	new_filepath = g_strconcat (current_dir,
				    "/",
				    new_filename,
				    NULL);
	g_free (current_dir);

	if (path_is_file (new_filepath)) {
		GtkWidget *gdialog;
		gchar *msg;

		msg = g_strdup_printf (_("The name \"%s\" is already used in this directory. " "Please use a different name."), new_filename);
		gdialog = gnome_error_dialog_parented (msg, GTK_WINDOW (window->app));
		gtk_widget_show (gdialog);
		g_free (msg);

		result = FALSE;
	}

	if (! rename (old_filepath, new_filepath)) {
		cache_move (old_filepath, new_filepath);
		all_windows_notify_file_rename (old_filepath, new_filepath);
	} else {
		GtkWidget *dialog;
		gchar *msg;
		
		msg = g_strdup_printf ("%s\n%s\n\n%s.", 
				       _("Could not rename the file:"),
				       old_filepath, 
				       strerror (errno));
		dialog = gnome_error_dialog_parented (msg, GTK_WINDOW (window->app));
		gtk_widget_show (dialog);

		result = FALSE;
	}

	g_free (old_filepath);
	g_free (new_filepath);

	return result;
}


static void
set_check_menu_item_state (GThumbWindow *window,
			   GtkWidget *mitem, 
			   gboolean active)
{
	gtk_signal_handler_block_by_data (GTK_OBJECT (mitem), window);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), active);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (mitem), window);
}



GThumbWindow *
window_new (gboolean view_first_image)
{
	GThumbWindow *window;
	GtkWidget    *paned1;      /* Main paned widget. */
	GtkWidget    *paned2;      /* Secondary paned widget. */
	GtkWidget    *table;
	GtkWidget    *vscrollbar;
	GtkWidget    *hscrollbar;
	GtkWidget    *event_box;
	GtkWidget    *pixmap;
	GtkWidget    *frame;
	GtkWidget    *toolbar;
	gint          e_width;
	gint          i;
	GtkToolbarStyle toolbar_style;

	window = g_new (GThumbWindow, 1);

	window->app = gthumb_app_new ("main", _("gthumb"));
	gnome_window_icon_set_from_default (GTK_WINDOW (window->app));
	gtk_widget_realize (window->app);
	gtk_signal_connect (GTK_OBJECT (window->app), "delete_event",
			    GTK_SIGNAL_FUNC (close_window_cb),
			    window);
	gtk_window_set_default_size (GTK_WINDOW (window->app), 
				     preferences.main_win_width, 
				     preferences.main_win_height);

	gtk_drag_dest_set (window->app,
			   GTK_DEST_DEFAULT_ALL,
			   target_table, n_targets,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE);

	/* Create the widgets. */
	if ((preferences.layout_type == 0) || (preferences.layout_type == 2))
		window->paned = paned1 = gedo_hpaned_new (); 
	else
		window->paned = paned1 = gedo_vpaned_new (); 
	gedo_paned_child1_use_minsize (GEDO_PANED (paned1), TRUE, 80);
	gnome_app_set_contents (GNOME_APP (window->app), paned1);

	if ((preferences.layout_type == 0) || (preferences.layout_type == 2))
		window->content_paned = paned2 = gedo_vpaned_new (); 
	else
		window->content_paned = paned2 = gedo_hpaned_new (); 
	gedo_paned_child1_use_minsize (GEDO_PANED (paned2), TRUE, 80);
	gedo_paned_child2_use_minsize (GEDO_PANED (paned2), TRUE, 80);
	gedo_paned_add1 (GEDO_PANED (paned1), paned2);

	window->file_list = file_list_new ();
	file_list_set_progress_func (window->file_list, window_progress, window);
        gtk_drag_source_set (window->file_list->ilist, 
			     GDK_BUTTON2_MASK,
			     target_table, n_targets, 
			     GDK_ACTION_COPY | GDK_ACTION_MOVE);

	gtk_signal_connect (GTK_OBJECT (window->app), 
			    "drag_data_received",
			    GTK_SIGNAL_FUNC (window_drag_data_received), 
			    window);

	window->dir_list = dir_list_new ();
	window->catalog_list = catalog_list_new ();

	window->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (window->notebook), FALSE);
	gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook), 
				  window->dir_list->root_widget,
				  NULL);
	gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook), 
				  window->catalog_list->root_widget,
				  NULL);

	gedo_paned_add1 (GEDO_PANED (paned2), window->notebook);
	if (preferences.layout_type < 2)
		gedo_paned_add2 (GEDO_PANED (paned2), window->file_list->root_widget);
	else
		gedo_paned_add2 (GEDO_PANED (paned1), window->file_list->root_widget);

	gtk_signal_connect (GTK_OBJECT (window->file_list->ilist), 
			    "select_image",
			    (GtkSignalFunc) file_selected_cb, 
			    window);
	gtk_signal_connect_after (GTK_OBJECT (window->file_list->ilist), 
			    "button_press_event",
			    (GtkSignalFunc) file_button_press_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->file_list->ilist), 
			    "unselect_image",
			    (GtkSignalFunc) file_unselected_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->file_list->ilist), 
			    "drag_data_get",
			    GTK_SIGNAL_FUNC (file_list_drag_data_get), 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->file_list->ilist),
			    "text_changed",
			    GTK_SIGNAL_FUNC (file_list_text_changed),
			    window);

	gtk_signal_connect (GTK_OBJECT (window->dir_list->clist), 
			    "select_row",
			    (GtkSignalFunc) dir_selected_cb, 
			    window);

	gtk_signal_connect (GTK_OBJECT (window->catalog_list->clist), 
			    "select_row",
			    (GtkSignalFunc) catalog_selected_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->catalog_list->clist), 
			    "button_press_event",
			    (GtkSignalFunc) catalog_button_press_cb, 
			    window);

	/* Image viewer. */
	
	window->viewer = image_viewer_new ();
	image_viewer_set_zoom_quality (IMAGE_VIEWER (window->viewer),
				       preferences.zoom_quality);
	image_viewer_set_zoom_change (IMAGE_VIEWER (window->viewer),
				      preferences.zoom_change);
	image_viewer_set_check_type (IMAGE_VIEWER (window->viewer),
				     preferences.check_type);
	image_viewer_set_check_size (IMAGE_VIEWER (window->viewer),
				     preferences.check_size);
	image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer),
				      preferences.transp_type);
	gtk_drag_source_set (IMAGE_VIEWER (window->viewer)->drawing_area, 
			     GDK_BUTTON2_MASK,
			     target_table, n_targets, 
			     GDK_ACTION_COPY | GDK_ACTION_MOVE);

	window->viewer_container = frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (frame), window->viewer);

	table = gtk_table_new (2, 2, FALSE);
	vscrollbar = gtk_vscrollbar_new (IMAGE_VIEWER (window->viewer)->vadj);
	hscrollbar = gtk_hscrollbar_new (IMAGE_VIEWER (window->viewer)->hadj);
	event_box = gtk_event_box_new ();
	pixmap = gnome_pixmap_new_from_xpm_d (nav_button_xpm);
	gtk_container_add (GTK_CONTAINER (event_box), pixmap);

	gtk_table_attach (GTK_TABLE (table), frame, 0, 1, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_table_attach (GTK_TABLE (table), vscrollbar, 1, 2, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_table_attach (GTK_TABLE (table), hscrollbar, 0, 1, 1, 2,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_table_attach (GTK_TABLE (table), event_box, 1, 2, 1, 2,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	if (preferences.layout_type < 2)
		gedo_paned_add2 (GEDO_PANED (paned1), table);
	else
		gedo_paned_add2 (GEDO_PANED (paned2), table);

	gtk_signal_connect (GTK_OBJECT (window->viewer), 
			    "clicked",
			    (GtkSignalFunc) image_clicked_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->viewer), 
			    "image_loaded",
			    (GtkSignalFunc) image_loaded_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->viewer), 
			    "zoom_changed",
			    (GtkSignalFunc) zoom_changed_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->viewer), 
			    "button_press_event",
			    (GtkSignalFunc) image_button_press_cb, 
			    window);

	gtk_signal_connect (GTK_OBJECT (IMAGE_VIEWER (window->viewer)->drawing_area), 
			    "drag_data_get",
			    GTK_SIGNAL_FUNC (viewer_drag_data_get), 
			    window);
	gtk_signal_connect (GTK_OBJECT (IMAGE_VIEWER (window->viewer)->drawing_area), 
			    "drag_data_delete",
			    GTK_SIGNAL_FUNC (viewer_drag_data_delete), 
			    window);

	gtk_signal_connect (GTK_OBJECT (window->app), 
			    "key_press_event",
			    (GtkSignalFunc) key_press_cb, 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->app), 
			    "focus_changed",
			    (GtkSignalFunc) focus_changed_cb, 
			    window);

	gtk_signal_connect (GTK_OBJECT (event_box), 
			    "button_press_event",
			    (GtkSignalFunc) nav_button_clicked_cb, 
			    window->viewer);

	/* Create menus. */
	gnome_app_create_menus_with_data (GNOME_APP (window->app), main_menu, window);

	/* Create popup menus. */
	window->file_popup_menu = gtk_menu_new ();
	gnome_app_fill_menu_with_data (GTK_MENU_SHELL (window->file_popup_menu),
				       file_popup_menu_data,
				       (GtkAccelGroup*) NULL,
				       FALSE,
				       0,
				       window);
	window->catalog_popup_menu = gtk_menu_new ();
	gnome_app_fill_menu_with_data (GTK_MENU_SHELL (window->catalog_popup_menu),
				       catalog_popup_menu_data,
				       (GtkAccelGroup*) NULL,
				       FALSE,
				       0,
				       window);
	window->image_popup_menu = gtk_menu_new ();
	gnome_app_fill_menu_with_data (GTK_MENU_SHELL (window->image_popup_menu),
				       image_popup_menu_data,
				       (GtkAccelGroup*) NULL,
				       FALSE,
				       0,
				       window);

	/* Create the toolbar. */
	switch (preferences.toolbar_style) {
	case TOOLBAR_STYLE_DEFAULT:
		toolbar_style = GTK_TOOLBAR_BOTH;
		break;

	case TOOLBAR_STYLE_ICONS:
	default:
		toolbar_style = GTK_TOOLBAR_ICONS;
		break;
	}
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   toolbar_style);
	gnome_app_fill_toolbar_with_data (GTK_TOOLBAR (toolbar), 
					  toolbar_data, 
					  (GtkAccelGroup*) NULL,
					  window);

	/* Add a location entry to the toolbar. */
	window->location_entry = gtk_entry_new ();
	gtk_signal_connect_after (GTK_OBJECT (window->location_entry),
				  "key_press_event",
				  (GtkSignalFunc) location_entry_key_press_cb,
				  window);
	gtk_widget_show (window->location_entry);

	if (preferences.location_on_main_bar) {
		gtk_toolbar_insert_widget (GTK_TOOLBAR (toolbar),
					   window->location_entry,
					   NULL,
					   NULL, 0);

		/* Set up the entry width. */
		e_width = gdk_string_width (window->location_entry->style->font, 
					    "MMMMMMMMMMMMMMMMMMM");
		gtk_widget_set_usize (window->location_entry, e_width, -1);
	}

	gnome_app_set_toolbar (GNOME_APP (window->app), GTK_TOOLBAR (toolbar));

	if (! preferences.location_on_main_bar) {
		GtkWidget *hbox;
		GtkWidget *label;
		GnomeDockItemBehavior toolbar_props;
		gboolean toolbar_relief;

		toolbar_props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
				 GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL);
		if (! gnome_preferences_get_toolbar_detachable ())
			toolbar_props |= GNOME_DOCK_ITEM_BEH_LOCKED;

		label = gtk_label_new (_("Location: "));
		hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), window->location_entry, 
				    TRUE, TRUE, 5);
		gtk_widget_show_all (hbox);

		gnome_app_add_docked (GNOME_APP (window->app),
				      hbox,
				      "location_bar",
				      toolbar_props,
				      GNOME_DOCK_TOP,
				      2, 1, 0);

		toolbar_relief = gnome_preferences_get_toolbar_relief ();

		/* set border appropriate to whether or not there's relief */
		gtk_container_set_border_width (GTK_CONTAINER (hbox->parent), 
						toolbar_relief ? 5 : 0);

		/* set shadowing following gnome pref */
		gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (hbox->parent), (toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE));
	}

	/* Create a statusbar. */
	window->statusbar = gnome_appbar_new (FALSE, TRUE, 
					      GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar (GNOME_APP (window->app), window->statusbar);
	gnome_app_install_appbar_menu_hints (GNOME_APPBAR (window->statusbar),
					     main_menu);

	/* * Image info. */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	
	window->image_info = gtk_label_new (" ");
	gtk_misc_set_alignment (GTK_MISC (window->image_info), 0.5, 0.5);
	gtk_widget_set_usize (window->image_info, 
                              gdk_string_width (gtk_widget_get_style (window->image_info)->font, "MMMMMMMMMMMMMMMMMMMMMMMM"), -1);
	gtk_container_add (GTK_CONTAINER (frame), window->image_info);
        gtk_box_pack_start (GTK_BOX (window->statusbar), frame, 
			    FALSE, TRUE, 0);

	/* * Progress bar. */
	window->progress = gtk_progress_bar_new ();
        gtk_box_pack_start (GTK_BOX (window->statusbar), window->progress, 
			    FALSE, TRUE, 0);

	/* Sync widgets and visualization options. */

	window->sidebar_visible = TRUE;
	window->sidebar_width = preferences.sidebar_width;
	gedo_paned_set_position (GEDO_PANED (paned1), preferences.sidebar_width);
	gedo_paned_set_position (GEDO_PANED (paned2), preferences.sidebar_content_height);

	window->mitem_bookmarks = GTK_MENU_ITEM (main_menu[4].widget)->submenu;
	window->mitem_open_with = file_menu[1].widget;
#ifdef HAVE_PRINT
	window->mitem_print = file_menu[3].widget;
#endif

	window->mitem_rename              = edit_menu[0].widget;
	window->mitem_delete              = edit_menu[1].widget;
	window->mitem_copy                = edit_menu[2].widget;
	window->mitem_move                = edit_menu[3].widget;
	window->mitem_modify_image        = edit_menu[8].widget;
	window->mitem_add_comment         = edit_menu[9].widget;
	window->mitem_delete_comment      = edit_menu[10].widget;
	window->mitem_add_to_catalog      = edit_menu[12].widget;
	window->mitem_remove_from_catalog = edit_menu[13].widget;

	window->mitem_edit_catalog = edit_menu[4].widget;
	window->mitem_catalog_search[0] = edit_catalog_menu[4].widget;
	window->mitem_catalog_search[1] = edit_catalog_menu[5].widget;

	window->mitem_add_bookmark = bookmarks_menu[0].widget;

	window->mitem_thumbnails = view_menu[2].widget;
	window->mitem_browser    = view_menu[10].widget;

	window->mitem_go_to_container = go_menu[2].widget;

	window->mitem_go = GTK_MENU_ITEM (main_menu[3].widget)->submenu;

	for (i = 0; i < 4; i++) 
		window->mitem_transparenty[i] = trans_type_radio_list[i].widget;
	for (i = 0; i < 2; i++) 
		window->mitem_zoom_quality[i] = zoom_quality_radio_list[i].widget;

	window->mitem_slideshow = tools_menu[0].widget;
	window->mitem_wallpaper = tools_menu[1].widget;
	window->mitem_export = tools_menu[4].widget;

	window->popmitem_fullscreen = image_popup_menu_data[5].widget;
	for (i = 0; i < 13; i++) 
		window->popupmenu_file[i] = file_popup_menu_data[i].widget;
	for (i = 0; i < 6; i++) 
		window->popupmenu_catalog[i] = catalog_popup_menu_data[i].widget;

	window->tool_item_up_dir = toolbar_data[0].widget;
	window->tool_item_show_dir = toolbar_data[4].widget;
	window->tool_item_show_catalog = toolbar_data[5].widget;
	window->tool_item_zoom_to_fit = toolbar_data[10].widget;

	gtk_widget_show_all (GTK_WIDGET (window->notebook));

	/* Update data. */

	window->sidebar_content = NO_LIST;
	window->catalog_path = NULL;
	window->image_path = NULL;
	window->image_catalog = NULL;

	window->fullscreen = FALSE;
	window->slideshow = FALSE;
	window->timer = 0;

	window->bookmarks_tooltips = NULL;
	window->history_tooltips = NULL;
	window_update_bookmark_list (window);
	window_update_history_list (window);

	window->dir_load_timeout_handle = 0;
	window->activity_ref = 0;

	window->setting_file_list = FALSE;
	window->changing_directory = FALSE;

	window->focused = NULL;
	give_focus_to_image_list (window);

	window->view_first_image = view_first_image;

	/* Update menu items status. */

	set_check_menu_item_state (window, window->mitem_thumbnails, 
			      preferences.enable_thumbnails);

	switch (preferences.transp_type) {
	case TRANSP_TYPE_WHITE:   i = 0; break;
	case TRANSP_TYPE_GRAY:    i = 1; break;
	case TRANSP_TYPE_BLACK:   i = 2; break;
	case TRANSP_TYPE_CHECKED: i = 3; break;
	}
	set_check_menu_item_state (window, window->mitem_transparenty[i], TRUE);

	switch (preferences.zoom_quality) {
	case ZOOM_QUALITY_HIGH: i = 0; break;
	case ZOOM_QUALITY_LOW:  i = 1; break;
	}
	set_check_menu_item_state (window, window->mitem_zoom_quality[i], TRUE);

	/* Sort type item. */

	set_check_menu_item_state (window, 
				   sort_by_radio_list[window->file_list->sort_method - 1].widget, 
				   TRUE);

	/* 'reversed' item. */

	set_check_menu_item_state (window, 
				   sort_menu[2].widget,
				   (window->file_list->sort_type != GTK_SORT_ASCENDING));

	/* 'browser' item.  */

	set_check_menu_item_state (window, window->mitem_browser, TRUE);

	/* Add the window to the window's list */

	window_list = g_list_prepend (window_list, window);

	/* Initial location. */

	if (pref_util_location_is_catalog (preferences.starting_location)) {
		window->catalog_path = g_strdup (pref_util_get_catalog_location (preferences.starting_location));
		window_go_to_catalog (window, window->catalog_path);
	} else {
		const gchar *path;

		if (pref_util_location_is_file (preferences.starting_location))
			path = pref_util_get_file_location (preferences.starting_location);
		else {  /* we suppose it is a directory name without prefix. */
			path = preferences.starting_location;
			if (! path_is_dir (path))
				path = g_get_home_dir ();
		}

		window_go_to_directory (window, path);
	}

	return window;
}


static void
window_free (GThumbWindow *window)
{
	g_return_if_fail (window != NULL);

	file_list_free (window->file_list);
	dir_list_free (window->dir_list);
	catalog_list_free (window->catalog_list);

	if (window->catalog_path)
		g_free (window->catalog_path);
	if (window->image_path)
		g_free (window->image_path);
	if (window->image_catalog)
		g_free (window->image_catalog);	

	g_free (window);
}


void
window_close (GThumbWindow *window)
{
	ImageViewer *viewer;

	g_return_if_fail (window != NULL);

	/* interrupt any activity. */

	if (window->changing_directory) {
		dir_list_interrupt_change_to (window->dir_list);
		window_stop_activity_mode (window);
	}
	if (window->setting_file_list) 
		file_list_interrupt_set_list (window->file_list, NULL, NULL);
	if (window->file_list->doing_thumbs)
		file_list_interrupt_thumbs (window->file_list);
	if (window->slideshow)
		window_stop_slideshow (window);

	/* Save visualization options. */

	if (window->sidebar_visible) {
		preferences.sidebar_width = gedo_paned_get_position (GEDO_PANED (window->paned));
		preferences.sidebar_content_height = gedo_paned_get_position (GEDO_PANED (window->content_paned));
	}

	gdk_window_get_size (window->app->window, 
			     &preferences.main_win_width,
			     &preferences.main_win_height);
	preferences.enable_thumbnails = window->file_list->enable_thumbs;
	preferences.file_list_sort = window->file_list->sort_method;
	preferences.file_list_sort_type = window->file_list->sort_type;

	if (preferences.go_to_last_location) {
		if (preferences.startup_location != NULL) {
			g_free (preferences.startup_location);
			preferences.startup_location = NULL;
		}

		if (window->sidebar_content == DIR_LIST) 
			preferences.startup_location = 
				g_strconcat ("file://",
					     window->dir_list->path,
					     NULL);
		else if (window->sidebar_content == CATALOG_LIST) 
			preferences.startup_location = 
				g_strconcat ("catalog://",
					     window->catalog_path,
					     NULL);
	}

	viewer = IMAGE_VIEWER (window->viewer);
	preferences.transp_type = image_viewer_get_transp_type (viewer);
	preferences.check_type = image_viewer_get_check_type (viewer);
	preferences.check_size = image_viewer_get_check_size (viewer);

	/* Destroy the main window. */
	gtk_widget_destroy (window->app);

	window_list = g_list_remove (window_list, window);
	window_free (window);
}


/* Try to make the current image visible in the file list. */
static void
window_make_current_image_visible (GThumbWindow *window)
{
	gchar *path;
	gint row;
	ImageList *ilist;

	path = image_viewer_get_image_filename (IMAGE_VIEWER (window->viewer));
	if (path == NULL)
		return;

	row = file_list_row_from_path (window->file_list, path);
	g_free (path);

	if (row == -1)
		return;

	ilist = IMAGE_LIST (window->file_list->ilist);
	
	gtk_signal_handler_block_by_data (GTK_OBJECT (ilist), window);
	image_list_select_image (ilist, row);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (ilist), window);
	image_list_moveto (ilist, row, 0.5);
}


void
window_update_menu_sensitivity (GThumbWindow *window)
{
	gint sidebar_content = window->sidebar_content;
	gboolean sel_not_null;
	gboolean image_is_void;
	gboolean viewing_dir;
	gboolean viewing_catalog;
	gboolean is_search;
	gint row;

	sel_not_null = ilist_utils_selection_not_null (IMAGE_LIST (window->file_list->ilist));
	image_is_void = image_viewer_is_void (IMAGE_VIEWER (window->viewer));
	viewing_dir = sidebar_content == DIR_LIST;
	viewing_catalog = sidebar_content == CATALOG_LIST; 

	/* File menu. */

#ifdef HAVE_PRINT
	gtk_widget_set_sensitive (window->mitem_print, ! image_is_void);
#endif

	/* Edit menu. */

	gtk_widget_set_sensitive (window->mitem_rename, 
				   ilist_utils_only_one_is_selected (IMAGE_LIST (window->file_list->ilist)));
	
	gtk_widget_set_sensitive (window->mitem_delete, sel_not_null);
	gtk_widget_set_sensitive (window->mitem_copy, sel_not_null);
	gtk_widget_set_sensitive (window->mitem_move, sel_not_null);

	gtk_widget_set_sensitive (window->mitem_modify_image, ! image_is_void);
	gtk_widget_set_sensitive (window->mitem_add_comment, sel_not_null);
	gtk_widget_set_sensitive (window->mitem_delete_comment, sel_not_null);
	gtk_widget_set_sensitive (window->mitem_add_to_catalog, 
				  viewing_dir && sel_not_null);
	gtk_widget_set_sensitive (window->mitem_remove_from_catalog, 
				  viewing_catalog && sel_not_null);
	gtk_widget_set_sensitive (window->mitem_go_to_container, 
				  viewing_catalog && ilist_utils_only_one_is_selected (IMAGE_LIST (window->file_list->ilist)));

	/* Edit Catalog menu. */

	if (! viewing_catalog 
	    || (GTK_CLIST (window->catalog_list->clist)->selection == NULL))
		row = -1;
	else 
		row = GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data);

	gtk_widget_set_sensitive (window->mitem_edit_catalog, row != -1);
	is_search = ((row != -1) 
		     && (catalog_list_is_search (window->catalog_list, row)));
	gtk_widget_set_sensitive (window->mitem_catalog_search[0], is_search);
	gtk_widget_set_sensitive (window->mitem_catalog_search[1], is_search);

	/* Tools menu. */

	gtk_widget_set_sensitive (window->mitem_wallpaper, ! image_is_void);
	gtk_widget_set_sensitive (window->mitem_open_with, 
				  sel_not_null || ! image_is_void);
	gtk_widget_set_sensitive (window->mitem_export, sel_not_null);
}


void
window_set_sidebar_content (GThumbWindow *window,
			    gint sidebar_content)
{
	switch (sidebar_content) {
	case DIR_LIST: 
		if (window->sidebar_content != sidebar_content) {
			if (window->dir_list->path == NULL) 
				window_go_to_directory (window, g_get_home_dir ());
			else 
				window_go_to_directory (window, window->dir_list->path);
		}
		break;

	case CATALOG_LIST:
		if (window->sidebar_content != sidebar_content) {
			gchar *path;
	
			if (window->catalog_path == NULL)
				path = get_catalog_full_path (NULL);
			else 
				path = remove_level_from_path (window->catalog_path);
			catalog_list_change_to (window->catalog_list, path);

			g_free (path);

			if (window->catalog_path != NULL) {
				gint row;

				if (window->slideshow)
					window_stop_slideshow (window);
				row = catalog_list_row_from_path (window->catalog_list, window->catalog_path);
				gtk_clist_select_row (GTK_CLIST (window->catalog_list->clist), row, 0);
			} else
				window_set_file_list (window, NULL, 
						      NULL, NULL);
			window_update_title (window);
		}
		break;
	default:
	}

	_window_set_sidebar (window, sidebar_content);
	window_show_sidebar (window);
}


void
window_hide_sidebar (GThumbWindow *window)
{
	GtkWidget *button;

	if (! window->sidebar_visible)
		return;

	window->sidebar_visible = FALSE;
	window->sidebar_width = gedo_paned_get_position (GEDO_PANED (window->paned));
	gedo_paned_hide_child1 (GEDO_PANED (window->paned));

	/* Sync menu and toolbar. */
	set_check_menu_item_state (window, window->mitem_browser, FALSE);
	button = get_button_from_sidebar_content (window, 
						  window->sidebar_content);
	if (button != NULL)
		set_toggle_button_state (window, button, FALSE);
}


void
window_show_sidebar (GThumbWindow *window)
{
	GtkWidget *button;

	if (window->sidebar_visible)
		return;

	window->sidebar_visible = TRUE;
	gedo_paned_set_position (GEDO_PANED (window->paned), window->sidebar_width); 

	/* Sync menu and toolbar. */
	set_check_menu_item_state (window, window->mitem_browser, TRUE);
	button = get_button_from_sidebar_content (window, 
						  window->sidebar_content);
	if (button != NULL)
		set_toggle_button_state (window, button, TRUE);
}


void
window_refresh (GThumbWindow *window)
{
	window->refreshing = TRUE;
	window_update_file_list (window);
}


void
window_update_file_list (GThumbWindow *window)
{
	if (window->sidebar_content == DIR_LIST)
		window_go_to_directory (window, window->dir_list->path);

	else if (window->sidebar_content == CATALOG_LIST) {
		gchar *catalog_path;

		if (GTK_CLIST (window->catalog_list->clist)->selection == NULL)
			return;

		catalog_path = catalog_list_path_from_row (window->catalog_list, GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data));

		window_go_to_catalog (window, catalog_path);
		g_free (catalog_path);
	}
}


void
window_update_catalog_list (GThumbWindow *window)
{
	gchar *catalog_dir;
	gchar *base_dir;

	if (window->sidebar_content != CATALOG_LIST) 
		return;

	/* If the catalog still exists, show the directory it belongs to. */
	if ((window->catalog_path != NULL) 
	    && path_is_file (window->catalog_path)) {
		GtkCList *clist;
		gint row;

		catalog_dir = remove_level_from_path (window->catalog_path);
		catalog_list_change_to (window->catalog_list, catalog_dir);
		g_free (catalog_dir);

		row = catalog_list_row_from_path (window->catalog_list, 
						  window->catalog_path);

		g_return_if_fail (row != -1);

		/* Select (without updating the file list) and view 
		 * the catalog. */
		clist = GTK_CLIST (window->catalog_list->clist);
		gtk_signal_handler_block_by_data (GTK_OBJECT (clist),
						  window);
		gtk_clist_select_row (clist, row, 0);
		gtk_signal_handler_unblock_by_data (GTK_OBJECT (clist),
						    window);
		gtk_clist_moveto (clist, row, 0, 0.5, 0);
		return;
	} 

	/* No catalog selected. */

	if (window->catalog_path != NULL) {
		g_free (window->catalog_path);
		window->catalog_path = NULL;

		/* Update file list. */
		window_set_file_list (window, NULL, NULL, NULL);
	}

	g_return_if_fail (window->catalog_list->path != NULL);

	/* If directory exists then update. */
	if (path_is_dir (window->catalog_list->path)) {
		catalog_list_refresh (window->catalog_list);
		return;
	}

	/* Else go up one level until a directory exists. */
	base_dir = g_strconcat (g_get_home_dir(),
				"/",
				RC_CATALOG_DIR,
				NULL);
	catalog_dir = g_strdup (window->catalog_list->path);
	
	while ((strcmp (base_dir, catalog_dir) != 0)
	       && ! path_is_dir (catalog_dir)) {
		gchar *new_dir;
		
		new_dir = remove_level_from_path (catalog_dir);
		g_free (catalog_dir);
		catalog_dir = new_dir;
	}

	catalog_list_change_to (window->catalog_list, catalog_dir);
	
	g_free (catalog_dir);
	g_free (base_dir);
}


static GtkWidget *
bookmarks_create_menu_item (Bookmarks *bookmarks, const gchar *path)
{
        GtkWidget *pixmap;
        GtkWidget *item = gtk_pixmap_menu_item_new ();
        GtkWidget *l;
        GtkWidget *hb = gtk_hbox_new (FALSE, 0);
	GdkPixmap *book_pixmap;
	GdkBitmap *mask;
	const gchar *name_only;

	g_return_val_if_fail (path != NULL, NULL);

	name_only = bookmarks_get_menu_name (bookmarks, path);
	if (name_only == NULL)
		name_only = "???";

	if (strlen (name_only) > BOOKMARKS_MENU_MAX_LENGTH) {
		gchar *s, *t;
		gint offset;
		
		offset = strlen (name_only) - BOOKMARKS_MENU_MAX_LENGTH;
		s = g_strndup (name_only + offset, BOOKMARKS_MENU_MAX_LENGTH);
		t = g_strconcat ("...", s, NULL);
		l = gtk_label_new (t);
		g_free (t);
		g_free (s);
	} else
		l = gtk_label_new (name_only);
	
	gtk_box_pack_start (GTK_BOX (hb), l, FALSE, FALSE, 2);

	if (pref_util_location_is_catalog (path))
		pixmap_from_xpm ((const char **) catalog_xpm, &book_pixmap, &mask);
	else if (pref_util_location_is_search (path))
		pixmap_from_xpm ((const char **) catalog_search_xpm, &book_pixmap, &mask);
	else
		pixmap_from_xpm ((const char **) dir_xpm, &book_pixmap, &mask);
	pixmap = gtk_pixmap_new (book_pixmap, mask);

        gtk_container_add (GTK_CONTAINER (item), hb);
        gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (item), pixmap);
        gtk_widget_show_all (hb);

        if (gnome_preferences_get_menus_have_icons ())
                gtk_widget_show (pixmap);

	gdk_pixmap_unref (book_pixmap);
	gdk_bitmap_unref (mask);

        return item;
}


static gint
bookmark_sort (gconstpointer p1,
	       gconstpointer p2)
{
	const gchar *s1, *s2;
	const gchar *n1, *n2;

	s1 = pref_util_remove_prefix ((const gchar *) p1);
	s2 = pref_util_remove_prefix ((const gchar *) p2);

	if (strcmp (s1, "/") == 0)
		n1 = s1;
	else
		n1 = file_name_from_path (s1);

	if (strcmp (s2, "/") == 0)
		n2 = s2;
	else
		n2 = file_name_from_path (s2);
	
	return strcasecmp (n1, n2);
}


void
window_update_bookmark_list (GThumbWindow *window)
{
	GList *scan, *l, *names;

	bookmarks_load_from_disk (preferences.bookmarks);

	if (window->bookmarks_tooltips) 
                gtk_object_destroy (GTK_OBJECT (window->bookmarks_tooltips));
	window->bookmarks_tooltips = gtk_tooltips_new ();

	/* Delete bookmarks menu. */
	l = gtk_container_children (GTK_CONTAINER (window->mitem_bookmarks));

	g_return_if_fail (l != NULL);
	g_return_if_fail (l->next != NULL);
	g_return_if_fail (l->next->next != NULL);

	if (preferences.menu_have_tearoff) {
		g_return_if_fail (l->next->next->next != NULL);
		l = l->next->next->next->next;
	} else
		l = l->next->next->next;

	for (scan = l; scan; scan = scan->next)
		if (GTK_IS_WIDGET (scan->data))
			gtk_widget_destroy (GTK_WIDGET (scan->data));

	if (l) g_list_free (l);

	if (preferences.bookmarks->list == NULL)
		return;

	/* Sort list. */
	names = g_list_copy (preferences.bookmarks->list);
	names = g_list_sort (names, bookmark_sort);

	/* Update bookmarks menu. */
	for (scan = names; scan; scan = scan->next) {
		GtkWidget *mitem;

		mitem = bookmarks_create_menu_item (preferences.bookmarks,
						    scan->data);
		gtk_widget_show (mitem);

		gtk_tooltips_set_tip (window->bookmarks_tooltips, 
				      GTK_WIDGET (mitem),
				      bookmarks_get_menu_tip (preferences.bookmarks, scan->data),
				      NULL);
		gtk_object_set_data (GTK_OBJECT (mitem), "tooltips", 
				     window->bookmarks_tooltips);
		gtk_object_set_data_full (GTK_OBJECT (mitem), "full_path", 
					  g_strdup (scan->data),
					  g_free);
		gtk_menu_append (GTK_MENU (window->mitem_bookmarks), mitem);

		gtk_signal_connect (GTK_OBJECT (mitem), 
				    "activate", 
				    GTK_SIGNAL_FUNC (bookmark_selected_cb), 
				    window);
	}
	g_list_free (names);
}


void
window_update_history_list (GThumbWindow *window)
{
	GList *scan, *l;
	gint i;

	bookmarks_load_from_disk (preferences.history);

	if (window->history_tooltips != NULL) 
                gtk_object_destroy (GTK_OBJECT (window->history_tooltips));
	window->history_tooltips = gtk_tooltips_new ();

	/* Delete bookmarks menu. */
	l = gtk_container_children (GTK_CONTAINER (window->mitem_go));

	g_return_if_fail (l != NULL);

	/* make l point to the first history item. */

	l = l->next->next->next->next->next->next;
	if (preferences.menu_have_tearoff) 
		l = l->next;

	for (scan = l; scan; scan = scan->next)
		if (GTK_IS_WIDGET (scan->data))
			gtk_widget_destroy (GTK_WIDGET (scan->data));

	if (l) g_list_free (l);

	if (preferences.history->list == NULL)
		return;

	/* Update history menu. */

	i = 0;
	scan = preferences.history->list;
	while (scan && (i++ < preferences.history_max_length)) {
		GtkWidget *mitem;

		mitem = bookmarks_create_menu_item (preferences.history,
						    scan->data);
		gtk_widget_show (mitem);

		gtk_tooltips_set_tip (window->history_tooltips, 
				      GTK_WIDGET (mitem),
				      bookmarks_get_menu_tip (preferences.history, scan->data),
				      NULL);
		gtk_object_set_data (GTK_OBJECT (mitem), "tooltips", 
				     window->history_tooltips);

		gtk_object_set_data_full (GTK_OBJECT (mitem), "full_path", 
					  g_strdup (scan->data),
					  g_free);
		gtk_menu_append (GTK_MENU (window->mitem_go), mitem);

		gtk_signal_connect (GTK_OBJECT (mitem), 
				    "activate", 
				    GTK_SIGNAL_FUNC (bookmark_selected_cb), 
				    window);

		scan = scan->next;
	}
}


static void
set_dir_list_continue (gpointer data)
{
	GThumbWindow *window = data;

	window_make_current_image_visible (window);
	window_update_title (window);
	window_update_menu_sensitivity (window);
}


static void
go_to_directory_continue (DirList *dir_list,
			  gpointer data)
{
	GThumbWindow *window = data;
	gchar *path;
	GList *file_list;

	window_stop_activity_mode (window);

	if (dir_list->result != GNOME_VFS_ERROR_EOF) {
		char *msg;
		GtkWidget *dialog;

		msg = g_strdup_printf (_("Cannot load directory %s : %s\n"), dir_list->try_path, gnome_vfs_result_to_string (dir_list->result));

		dialog = gnome_error_dialog_parented (msg, GTK_WINDOW (window->app));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

		g_free (msg);
		window->changing_directory = FALSE;
		return;
	}

	path = dir_list->path;

	/* Make the "Go Up" button insensitive if the current dir is "/". */
	gtk_widget_set_sensitive (window->tool_item_up_dir,
				  strcmp (path, "/") != 0);
	
	/* Update location entry. */
	gtk_entry_set_text (GTK_ENTRY (window->location_entry), path);
	
	/* Add to history list if not present as last entry. */
	if ((preferences.history->list == NULL) 
	    || (strcmp (path, pref_util_remove_prefix (preferences.history->list->data)) != 0)) {
		bookmarks_add_with_prefix (preferences.history, path,
					   FILE_PREFIX);
		bookmarks_write_to_disk (preferences.history);
		all_windows_update_history_list ();
	}

	/* Select the directory view. */

	_window_set_sidebar (window, DIR_LIST);
	if (! window->refreshing)
		window_show_sidebar (window);
	else
		window->refreshing = FALSE;

	file_list = dir_list_get_file_list (window->dir_list);
	window_set_file_list (window, file_list, 
			      set_dir_list_continue, window);
	g_list_free (file_list);

	window->changing_directory = FALSE;
}


/* used in : goto_dir_set_list_interrupted, window_go_to_directory. */
static void
real_go_to_directory (GThumbWindow *window,
		      const gchar *dir_path)
{
	gchar *path;

	if (window->changing_directory) {
		dir_list_interrupt_change_to (window->dir_list);
		window_stop_activity_mode (window);
	}

	window->changing_directory = TRUE;

	if (window->file_list->doing_thumbs)
		file_list_interrupt_thumbs (window->file_list);

	if (window->slideshow)
		window_stop_slideshow (window);

	path = remove_ending_separator (dir_path);

	window_start_activity_mode (window);
	dir_list_change_to (window->dir_list, 
			    path, 
			    go_to_directory_continue,
			    window);
	g_free (path);
}


typedef struct {
	GThumbWindow *window;
	gchar *dir_path;
} GoToDir_SetListInterruptedData;


static void
go_to_dir_set_list_interrupted (gpointer callback_data)
{
	GoToDir_SetListInterruptedData *data = callback_data;

	data->window->setting_file_list = FALSE;
	window_stop_activity_mode (data->window);

	real_go_to_directory (data->window, data->dir_path);
	g_free (data->dir_path);
	g_free (data);
}



void
window_go_to_directory (GThumbWindow *window,
			const gchar *dir_path)
{
	g_return_if_fail (window != NULL);

	if (window->slideshow)
		window_stop_slideshow (window);

	if (window->setting_file_list) {
		GoToDir_SetListInterruptedData *sli_data;

		sli_data = g_new (GoToDir_SetListInterruptedData, 1);
		sli_data->window = window;
		sli_data->dir_path = g_strdup (dir_path);
		file_list_interrupt_set_list (window->file_list,
					      go_to_dir_set_list_interrupted,
					      sli_data);
		return;
	}

	real_go_to_directory (window, dir_path);
}


void
window_go_to_catalog (GThumbWindow *window,
		      const gchar *catalog_path)
{
	gchar * catalog_dir;
	gint row;

	g_return_if_fail (window != NULL);
	
	if (! path_is_file (catalog_path)) {
		GtkWidget *dialog;
		
		dialog = gnome_error_dialog_parented (_("The specified catalog does not exist."), GTK_WINDOW (window->app));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		return;
	}

	if (window->file_list->doing_thumbs)
		file_list_interrupt_thumbs (window->file_list);

	if (window->slideshow)
		window_stop_slideshow (window);

	if (window->catalog_path)
		g_free (window->catalog_path);
	window->catalog_path = g_strdup (catalog_path);

	catalog_dir = remove_level_from_path (catalog_path);
	catalog_list_change_to (window->catalog_list, catalog_dir);
	g_free (catalog_dir);

	row = catalog_list_row_from_path (window->catalog_list, catalog_path);
	gtk_clist_select_row (GTK_CLIST (window->catalog_list->clist), row, 0);

	_window_set_sidebar (window, CATALOG_LIST); 

	if (! window->refreshing)
		window_show_sidebar (window);
	else
		window->refreshing = FALSE;
}


gboolean
window_show_next_image (GThumbWindow *window)
{
	gint row;

	g_return_val_if_fail (window != NULL, FALSE);

	if ((window->setting_file_list) || (window->changing_directory)) 
		return FALSE;

	if (window->image_path == NULL) 
		row = -1;
	else 
		row = file_list_row_from_path (window->file_list, 
					       window->image_path);

	row = file_list_next_image (window->file_list, row, TRUE);
	if (row == -1) 
		return FALSE;

	file_list_select_image_by_row (window->file_list, row);
	view_image_at_pos (window, row);

	return TRUE;
}


gboolean
window_show_prev_image (GThumbWindow *window)
{
	gint row;

	g_return_val_if_fail (window != NULL, FALSE);

	if ((window->setting_file_list) || (window->changing_directory))
		return FALSE;

	if (window->image_path == NULL)
		row = IMAGE_LIST (window->file_list->ilist)->images;
	else
		row = file_list_row_from_path (window->file_list, window->image_path);

	row = file_list_prev_image (window->file_list, row, TRUE);
	if (row == -1)
		return FALSE;

	file_list_select_image_by_row (window->file_list, row);
	view_image_at_pos (window, row);

	return TRUE;
}


gboolean
window_show_first_image (GThumbWindow *window)
{
	if (IMAGE_LIST (window->file_list->ilist)->images == 0)
		return FALSE;

	if (window->image_path) {
		g_free (window->image_path);
		window->image_path = NULL;
	}

	return window_show_next_image (window);
}


gboolean
window_show_last_image (GThumbWindow *window)
{
	if (IMAGE_LIST (window->file_list->ilist)->images == 0)
		return FALSE;

	if (window->image_path) {
		g_free (window->image_path);
		window->image_path = NULL;
	}

	return window_show_prev_image (window);
}


gint
slideshow_timeout_cb (gpointer data)
{
	GThumbWindow *window = data;
	gboolean go_on;

	if (preferences.ss_direction == DIRECTION_FORWARD)
		go_on = window_show_next_image (window);
	else
		go_on = window_show_prev_image (window);

	if (! go_on) {
		if (! preferences.ss_wrap_around) {
			window_stop_slideshow (window);
			return FALSE;
		}

		if (preferences.ss_direction == DIRECTION_FORWARD)
			go_on = window_show_first_image (window);
		else
			go_on = window_show_last_image (window);

		if (! go_on) {
			/* No image to show, stop. */
			window_stop_slideshow (window);
			return FALSE;
		}

		return TRUE;
	}

	return go_on;
}


void
window_start_slideshow (GThumbWindow *window)
{
	gboolean go_on = TRUE;
	gint row;

	g_return_if_fail (window != NULL);

	window->slideshow = TRUE;

	/* Check menu item. */
	set_check_menu_item_state (window, window->mitem_slideshow, TRUE);

	row = file_list_row_from_path (window->file_list, window->image_path);
	if (row == -1) {
		if (preferences.ss_direction == DIRECTION_FORWARD)
			go_on = window_show_next_image (window);
		else
			go_on = window_show_prev_image (window);
	}

	if (! go_on) {
		window_stop_slideshow (window);
		return;
	}

	window->timer = gtk_timeout_add (preferences.ss_delay * 1000, 
					 slideshow_timeout_cb, window);
}


void
window_stop_slideshow (GThumbWindow *window)
{
	window->slideshow = FALSE;
	if (window->timer != 0) {
		gtk_timeout_remove (window->timer);
		window->timer = 0;
	}

	/* Uncheck menu item. */
	set_check_menu_item_state (window, window->mitem_slideshow, FALSE);
}


void
window_load_image (GThumbWindow *window, 
		   const gchar *filename)
{
	g_return_if_fail (window != NULL);
	
	if (window->image_path)
		g_free (window->image_path);
	window->image_path = NULL;

	if (filename == NULL) {
		image_viewer_set_void (IMAGE_VIEWER (window->viewer));
		window_update_title (window);
		window_update_menu_sensitivity (window);
		return;
	}
		
	window->image_path = g_strdup (filename);
	image_viewer_load_image (IMAGE_VIEWER (window->viewer), filename);
}


static void
update_statusbar_list_info (GThumbWindow *window)
{
	gchar *info, *size_txt, *sel_size_txt;
	gint tot_n, sel_n;
	GnomeVFSFileSize tot_size, sel_size;
	GList *scan;
	GList *selection;

	tot_n = 0;
	tot_size = 0;

	for (scan = window->file_list->list; scan; scan = scan->next) {
		FileData *fd = scan->data;

		tot_n++;
		tot_size += fd->size;
	}

	sel_n = 0;
	sel_size = 0;
	selection = file_list_get_selection_as_fd (window->file_list);

	for (scan = selection; scan; scan = scan->next) {
		FileData *fd = scan->data;

		sel_n++;
		sel_size += fd->size;
	}

	g_list_free (selection);

	size_txt = gnome_vfs_format_file_size_for_display (tot_size);
	sel_size_txt = gnome_vfs_format_file_size_for_display (sel_size);
	info = g_strdup_printf ("%d %s (%s) %s: %d %s (%s)",
				tot_n,
				(tot_n == 1) ? _("file") : _("files"),
				size_txt,
				_("Selected"),
				sel_n,
				(sel_n == 1) ? _("file") : _("files"),
				sel_size_txt);

	gnome_appbar_set_default (GNOME_APPBAR (window->statusbar), info);

	g_free (size_txt);
	g_free (sel_size_txt);
	g_free (info);
}


typedef struct {
	GThumbWindow *window;
	DoneFunc done_func;
	gpointer done_func_data;
} WindowSetListData;


static void
window_set_file_list_continue (gpointer callback_data)
{
	WindowSetListData *data = callback_data;
	GThumbWindow *window = data->window;

	window_stop_activity_mode (window);
	update_statusbar_list_info (window);
	window->setting_file_list = FALSE;

	if (window->view_first_image) {
		window->view_first_image = FALSE;
		window_show_first_image (window);
		window_update_menu_sensitivity (window);
	}

	if (data->done_func != NULL)
		(*data->done_func) (data->done_func_data);
	g_free (data);
}


typedef struct {
	WindowSetListData * wsl_data;
	GList *list;
	GThumbWindow *window;
} SetListInterruptedData; 


static void
set_list_interrupted_cb (gpointer callback_data)
{
	SetListInterruptedData * sli_data = callback_data;

	sli_data->window->setting_file_list = TRUE;
	file_list_set_list (sli_data->window->file_list, 
			    sli_data->list, 
			    window_set_file_list_continue, 
			    sli_data->wsl_data);

	g_list_foreach (sli_data->list, (GFunc) g_free, NULL);
	g_list_free (sli_data->list);
	g_free (sli_data);
}


static void
window_set_file_list (GThumbWindow *window, 
		      GList *list,
		      DoneFunc done_func,
		      gpointer done_func_data)
{
	WindowSetListData *data;

	if (window->slideshow)
		window_stop_slideshow (window);

	data = g_new (WindowSetListData, 1);
	data->window = window;
	data->done_func = done_func;
	data->done_func_data = done_func_data;

	if (window->setting_file_list) {
		SetListInterruptedData * sli_data;
		GList *scan;

		sli_data = g_new (SetListInterruptedData, 1);

		sli_data->wsl_data = data;
		sli_data->window = window;

		sli_data->list = NULL;
		for (scan = list; scan; scan = scan->next) {
			gchar *path = g_strdup ((char*)(scan->data));
			sli_data->list = g_list_prepend (sli_data->list, path);
		}

		file_list_interrupt_set_list (window->file_list,
					      set_list_interrupted_cb,
					      sli_data);
		return;
	}

	window->setting_file_list = TRUE;
	window_start_activity_mode (window);
	file_list_set_list (window->file_list, list, 
			    window_set_file_list_continue, data);
}


void 
window_update_title (GThumbWindow *window)
{
	gchar *info_txt = NULL;
	gchar *path;
	gint zoom;

	g_return_if_fail (window != NULL);

	path = window->image_path;

	if (path == NULL) {
		if ((window->sidebar_content == DIR_LIST)
		    && (window->dir_list->path != NULL)) {

			info_txt = g_strdup_printf ("%s - %s",
						    _("gThumb"),
						    window->dir_list->path);
		} else
			info_txt = g_strdup_printf ("%s", _("gThumb"));
	} else {
		zoom = (gint) (IMAGE_VIEWER (window->viewer)->zoom_level
			       * 100.0);
		if (window->image_catalog != NULL) {
			gchar *cat_name = g_strdup (file_name_from_path (window->image_catalog));

			/* Cut out the file extension. */
			cat_name[strlen (cat_name) - 4] = 0;
			
			info_txt = g_strdup_printf ("%s - %s : %s (%d%%)",
						    _("gThumb"),
						    cat_name,
						    path,
						    zoom);
			g_free (cat_name);
		} else 
			info_txt = g_strdup_printf ("%s - %s (%d%%)",
						    _("gThumb"),
						    path ? path : "",
						    zoom);
	}

	g_assert (info_txt != NULL);
	gtk_window_set_title (GTK_WINDOW (window->app), info_txt);
	g_free (info_txt);
}


static void
notify_files_deleted (GThumbWindow *window,
		      GList *list)
{
	GList *scan;
	gchar *filename;
	gint row = -1;
	gint smallest_row, image_row;
	gboolean current_image_deleted = FALSE;
	gboolean no_image_viewed;
	ImageList *ilist;

	ilist = IMAGE_LIST (window->file_list->ilist);
	smallest_row = -1;

	image_row = -1;
	if (window->image_path)
		image_row = file_list_row_from_path (window->file_list, 
						     window->image_path);
	no_image_viewed = (image_row == -1);

	for (scan = list; scan; scan = scan->next) {
		filename = scan->data;

		row = file_list_row_from_path (window->file_list, filename);
		if (row == -1) 
			continue;

		if (image_row == row) {
			/* the current image will be deleted. */
			image_row = -1;
			current_image_deleted = TRUE;
		} else if (image_row > row)
			/* a previous image will be deleted, so image_row 
			 * decrements its value. */
			image_row--;

		if (scan == list)
			smallest_row = row;
		else
			smallest_row = MIN (smallest_row, row);

		file_list_delete_row (window->file_list, row);
	}

	/* Try to visualize the old row. */
	if (ilist->images != 0) {
		if (image_row != -1)
			row = image_row;
		else
			row = smallest_row;

		if (row > ilist->images - 1)
			row = ilist->images -1;
		if (row < 0)
			row = 0;

		image_list_moveto (ilist, row, 0.5);
	}

	if (! no_image_viewed) {
		if (current_image_deleted) {
			/* delete the image from the viewer. */
			window_load_image (window, NULL);

			if ((ilist->images > 0) && (row != -1)) {
				view_image_at_pos (window, row);
				file_list_select_image_by_row (window->file_list, row);
			}
		} else
			file_list_select_image_by_row (window->file_list, 
						       image_row);
	}

	update_statusbar_list_info (window);
}


void
window_notify_files_deleted (GThumbWindow *window,
			     GList *list)
{
	g_return_if_fail (window != NULL);

	if ((window->sidebar_content == CATALOG_LIST)
	    && (window->catalog_path != NULL)) { /* update the catalog. */
		Catalog *catalog;
		GList *scan;

		catalog = catalog_new ();
		catalog_load_from_disk (catalog, window->catalog_path);

		for (scan = list; scan; scan = scan->next)
			catalog_remove_item (catalog, (gchar*) scan->data);
		catalog_write_to_disk (catalog);
		catalog_free (catalog);
	}

	notify_files_deleted (window, list);
}


void
window_notify_cat_files_deleted (GThumbWindow *window,
				 const gchar *catalog_name,
				 GList *list)
{
	g_return_if_fail (window != NULL);

	if (window->sidebar_content != CATALOG_LIST)
		return;
	if (window->catalog_path == NULL)
		return;
	if (strcmp (window->catalog_path, catalog_name) != 0)
		return;

	notify_files_deleted (window, list);
}

	
void
window_notify_file_rename (GThumbWindow *window,
			   const gchar *old_name,
			   const gchar *new_name)
{
	gint pos;

	g_return_if_fail (window != NULL);

	if ((window->sidebar_content == CATALOG_LIST)
	    && (window->catalog_path != NULL)) { /* update the catalog. */
		Catalog *catalog;
		GList *scan;
		gboolean changed = FALSE;

		catalog = catalog_new ();
		catalog_load_from_disk (catalog, window->catalog_path);

		for (scan = catalog->list; scan; scan = scan->next)
			if (strcmp ((gchar*) scan->data, old_name) == 0) {
				catalog_remove_item (catalog, old_name);
				catalog_add_item (catalog, new_name);
				changed = TRUE;
			}
		if (changed)
			catalog_write_to_disk (catalog);
		catalog_free (catalog);
	}

	pos = file_list_row_from_path (window->file_list, old_name);
	if (pos != -1)
		file_list_rename_row (window->file_list, pos, new_name);

	if ((window->image_path != NULL) 
	    && strcmp (old_name, window->image_path) == 0) {
		/* current image renamed, update infos. */
		g_free (window->image_path);
		window->image_path = g_strdup (new_name);
		window_update_statusbar (window);
		window_update_title (window);
	}
}


void
window_notify_update_comment (GThumbWindow *window,
			      const gchar *filename)
{
	gint pos;

	g_return_if_fail (window != NULL);

	pos = file_list_row_from_path (window->file_list, filename);
	if (pos != -1)
		file_list_update_comment (window->file_list, pos);
}


void
window_notify_update_directory (GThumbWindow *window,
				const gchar *dir_path)
{
	g_return_if_fail (window != NULL);

	if ((window->dir_list->path == NULL) 
	    || (strcmp (window->dir_list->path, dir_path) != 0)) 
		return;

	window_update_file_list (window);
}


static gint
load_progress (gpointer data)
{
	GThumbWindow *window = data;

	window->dir_load_progress += 0.1;
	if (window->dir_load_progress > 1.0)
		window->dir_load_progress = 0.0;

	window_progress (window->dir_load_progress, data);

	return TRUE;
}


void
window_start_activity_mode (GThumbWindow *window)
{
	g_return_if_fail (window != NULL);

	if (window->activity_ref++ > 0)
		return;

	window->dir_load_progress = 0.0;
	gtk_progress_set_activity_mode (GTK_PROGRESS (window->progress), TRUE);
	window->dir_load_timeout_handle = gtk_timeout_add (ACTIVITY_DELAY, 
							   load_progress, 
							   window);
}


void
window_stop_activity_mode (GThumbWindow *window)
{
	g_return_if_fail (window != NULL);

	if (--window->activity_ref > 0)
		return;

	if (window->dir_load_timeout_handle == 0)
		return;

	gtk_timeout_remove (window->dir_load_timeout_handle);
	window->dir_load_timeout_handle = 0;
	gtk_progress_set_activity_mode (GTK_PROGRESS (window->progress), 
					FALSE);
	window_progress (0.0, window);
}
