(define gtk-sources-version "0.6")

(define gtk-sources-copyright
  (concat "gtk-sources.lisp v" gtk-sources-version "\n"
	  
	  "Copyright 2001, Matthew Danish <mdanish@andrew.cmu.edu>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

The GNU General Public License is available from http://www.gnu.org
or in the directory /usr/share/common-licenses/GPL on Debian."))

(define gtk-sources-info
  "The purpose of this program is to provide a graphical interface for
manipulating the sources.list file that defines where the APT system
retrieves packages from, in Debian.

This program has been designed to run with the rep interpreter/compiler
(http://librep.sourceforge.net) and the rep-gtk extensions
(http://rep-gtk.sourceforge.net).  Both are available as Debian packages.")

(define gtk-sources-usage
  "Usage: 'rep gtk-sources.lisp [-f <sources list>]'
If the environment variable SOURCES_LIST is set, it will use that as
the default, otherwise it will use /etc/apt/sources.list")

(define gtk-sources-todo
  '("Move line"
    "Update mirror list"))

(define gtk-sources-changes
  '(("0.6"
     ("Can choose a site from mirror list now"))
    ("0.5"
     ("Will perform byte-compilation at runtime, if needed, now."
      "Added ssh method"
      "Places windows at mouse pointer now"
      "sources.list can be specified by environment variable SOURCES_LIST."))
    ("0.4"
     ("Fixed up some window dimensions"
      "Byte-compiles on install"))
    ("0.3"
     ("Centralized information"
      "Added Cut&Paste ability"
      "Added Update button"
      "Added Save button"
      "Timeouts now work"
      "Fixed some minor bugs"
      "Added About dialog"
      "Lispified program information"))
    ("0.2"
     ("Fixed a bug in Edit"))
    ("0.1"
     ("First release"))))

(define gtk-sources-note-to-hackers
  "Note to hackers:
I am trying to keep constants and other centralized information in one
place, the 'global-info' closure.  Please keep that in mind, thanks.")


;; The program itself
(require 'gtk)
(require 'rep.io.timers)
(require 'rep.regexp)

(define default-sources.list 
  (let ((env-var (getenv "SOURCES_LIST")))
    (if (null env-var)
	"/etc/apt/sources.list"
      env-var)))
    

(define stdout standard-output)

;; For message-passing object based programming
(define (send obj . args)
  (if (not (consp (cdr args)))
      ((obj (car args)))
    (apply (obj (car args)) (cdr args))))


;; global constant info

(define make-global-info
  (lambda ()
    (define master-mirrors-list "/usr/share/gtk-sources/Mirrors.masterlist")

    ;; here's where you'd add a new URI type
    (define uri-type-list '("http" "ftp" "file" "copy" "cdrom" "ssh"))

    ;; Put it into one of the following, if necessary

    ;; if it's gonna have brackets (as in "cdrom:[...]")
    (define uri-type-needs-brackets '("cdrom"))

    ;; if it only has a colon, not :// (as in "file:...")
    (define uri-type-no-// '("file" "copy" "cdrom"))

    (define default-uri-type-separator "http://")

    (define list-column-names '("Active" "Type" "URI" "Distribution" "Components"))

    (define uri-separator
      (lambda (uri-type)
	(if (member uri-type uri-type-no-//)
	    ":"
	  "://")))

    (define default-distributions '("stable"
				    "frozen"
				    "testing"
				    "unstable"
				    "potato"
				    "woody"
				    "sid"
				    "stable/non-US"
				    "testing/non-US"
				    "unstable/non-US"
				    "potato/non-US"
				    "woody/non-US"
				    "sid/non-US"
				    "progeny"))

    (define default-components '("main"
				 "main contrib"
				 "main contrib non-free"))
				
    

    (define about-msg (concat gtk-sources-copyright "\n"
			      "--\n"
			      gtk-sources-info "\n"
			      "--\n"
			      gtk-sources-usage "\n"
			      "--\nTo Do:\n"
			      (apply concat (mapcar (lambda (x)
						      (concat x "\n"))
						    gtk-sources-todo))
			      "--\nChanges:\n"
			      (apply concat (mapcar (lambda (x)
						      (apply concat (append (list " " (car x) "\n")
									    (mapcar (lambda (x2)
										      (concat "  * " x2 "\n"))
										    (cadr x)))))
						    gtk-sources-changes))))
								     

    (lambda (msg)
      (cond

       ((equal msg 'about-text)
	(lambda () about-msg))
       ((equal msg 'master-mirrors-list)
	(lambda () master-mirrors-list)) 
       ((equal msg 'version)
	(lambda () gtk-sources-version))
       ((equal msg 'default-components)
	(lambda () default-components))
       ((equal msg 'default-distributions)
	(lambda () default-distributions))
       ((equal msg 'vector-column-names)
	(lambda () (apply vector list-column-names)))
       ((equal msg 'list-column-names)
	(lambda () list-column-names))
       ((equal msg 'uri-type-list)
	(lambda () uri-type-list))
       ((equal msg 'uri-separator)
	uri-separator)
       ((equal msg 'need-brackets?)
	(lambda (type)
	  (if (member type uri-type-needs-brackets) #t #f)))
       ((equal msg 'no-//?)
	(lambda (type)
	  (if (member type uri-type-no-//) #t #f)))
       ((equal msg 'uri-type-separator-list)
	(lambda () 
	  (mapcar (lambda (x)
		    (concat x (uri-separator x)))
		  uri-type-list)))
       ((equal msg 'default-uri-type-separator)
	(lambda () default-uri-type-separator))
       ((equal msg 'toplevel-x)
	(lambda () 600))
       ((equal msg 'toplevel-y)
	(lambda () 400))
       ((equal msg 'about-x)
	(lambda () 500))
       ((equal msg 'about-y)
	(lambda () 400))
       ((equal msg 'update-x)
	(lambda () 500))
       ((equal msg 'update-y)
	(lambda () 400))
       ((equal msg 'mirror-dialog-x)
	(lambda () 700))
       ((equal msg 'mirror-dialog-y)
	(lambda () 600))
       ((equal msg 'mirror-column-widths)
	(lambda () #(150 100 150 50 150)))
       ((equal msg 'mirror-column-names)
	(lambda () #("Site" "Country" "Location" "Type" "Comment")))
       (t
	(format standard-output "Reaching a place we shouldn't be... (global-info '%s)\n" msg)
	#f)))))

(define global-info (make-global-info))



;; the Common Lisp function 'position' (simplified)
(define position
  (lambda (item seq #!optional (n 0))
    (if (consp seq)
	(if (equal item (car seq))
	    n
	  (position item (cdr seq) (+ n 1)))
      nil)))

;; convert vector to list
(define vector->list
  (lambda (v)
    (let loop ((i 0)
	       (len (length v))
	       (l '()))
	 (if (< i len)
	     (loop (+ i 1) len (append l (list (aref v i))))
	   l))))
	       

;; exit by throwing 'ready-to-exit-program
(define exit-program
  (lambda (n)
    (throw 'ready-to-exit-program)))



;; Parse a mirrors list

;; vector of regular expressions to use
(define regular-expressions 
  #("^([^: ]+):[ \t]+([^\n\r]+)"
    "^[\t ]+([^\n\r]+)"
    "^[\t ]*[\r\n]+$"))

;; form an associative list of keys and values
;; example: (("Site" "ftp.debian.org") ("Type" "Push-Primary"))
(define read-in-mirror
  (lambda (line lineno #!optional (m '()))
    (cond
     ((string-match (aref regular-expressions 0) line)
      (cons (append m (list (list (expand-last-match "\\1")
				  (expand-last-match "\\2"))))
       'partial-record))
     ((string-match (aref regular-expressions 1) line)
      (cons (let* ((len (length m))
		   (last-elem (nth (- len 1) m)))
	      (rplaca (nthcdr (- len 1) m)
		      (append (list (car last-elem))
			      (cdr last-elem)
			      (list (expand-last-match "\\1"))))
	      m)
	    'partial-record))
     ((string-match (aref regular-expressions 2) line)
      (cons m
	    'complete-record))
     (t
      (format standard-output "Parse error: line %d: %s\n" lineno line)
      (throw 'parse-error)))))

;; form a list of the associative lists of mirrors
(define read-in-mirror-list
  (lambda (fp #!optional (m '()) (ml '()) (lineno 1))
    (let ((line (read-line fp)))
      (if (null line)
	  (append ml (list m))
	(let ((ret-val (read-in-mirror line lineno m))) 
	  ;; why doesnt rep have multiple-values-bind??!
	  (if (equal (cdr ret-val) 'complete-record)
	      (read-in-mirror-list fp '() (append ml (list (car ret-val))) (+ lineno 1))
	    (read-in-mirror-list fp (car ret-val) ml (+ lineno 1))))))))

;; What order should the mirrors be sorted in?
(define mirrors-list-p
  (lambda (a b)
    (let ((a-country (apply concat (cdr (assoc "Country" a))))
	  (b-country (apply concat (cdr (assoc "Country" b)))))
      (cond 
       ((string< a-country b-country)
	#t)
       ((string= a-country b-country)
	(let ((a-type (apply concat (cdr (assoc "Type" a))))
	      (b-type (apply concat (cdr (assoc "Type" b)))))
	  (cond
	   ((string< a-type b-type)
	    #t)
	   ((string= a-type b-type)
	    (let ((a-loc (apply concat (cdr (assoc "Location" a))))
		  (b-loc (apply concat (cdr (assoc "Country" a)))))
	      (cond
	       ((string< a-loc b-loc)
		#t)
	       ((string= a-loc b-loc)
		#t)
	       (t
		#f))))
	   (t
	    #f))))
	      
       (t 
	#f)))))
	  	
	  
;; mirror info dialog
(define create-mirror-info-dialog
  (lambda (mirror)
      (let* ((dialog (gtk-dialog-new))
	     (vbox (gtk-dialog-vbox dialog))
	     (action-area (gtk-dialog-action-area dialog)))
	(let ((closeb (gtk-button-new-with-label "Close")))
	  (gtk-window-set-position dialog 'mouse)
	  (gtk-window-set-title dialog "Mirror Info")
	  (gtk-signal-connect closeb
			      "clicked"
			      (lambda ()
				(gtk-widget-destroy dialog)))
	  (let ((labels-list 
		 (mapcar (lambda (x)
			   (gtk-label-new (concat (car x) ": "
						  (apply concat
							 (mapcar (lambda (y)
								   (concat y
									   " "))
								 (cdr x))))))
			 mirror)))
	    (mapc (lambda (x)
		    (gtk-container-add vbox x))
		  labels-list))
	  (gtk-container-add action-area closeb)
	  (gtk-widget-show-all dialog)))))



;; mirrors list object
(define make-mirrors-list
  (lambda (filename)
    (let ((mirrors (sort (read-in-mirror-list (open-file filename 'read))
			 mirrors-list-p)))
      (define choose-mirror-dialog
	(lambda (okay)
	  (let* ((dialog (gtk-dialog-new))
		 (vbox (gtk-dialog-vbox dialog))
		 (action-area (gtk-dialog-action-area dialog))
		 (list-of-sites-by-type
		  (lambda (type)
		    (filter (lambda (x)
			      (if (assoc type x) #t #f))
			    mirrors)))
		 (clist (gtk-clist-new-with-titles (send global-info 'mirror-column-names)))
		 (get-uri-type
		  (lambda (type)
		    (string-match "([A-Za-z]+)-([A-Za-z]+)" type)
		    (expand-last-match "\\2")))
		 (current-row #f)
		 (current-col #f)
		 (selectionp #f)
		 (current-type "Archive-http"))
	    (let ((type-menu (gtk-menu-new))
		  (type-option (gtk-option-menu-new))
		  (type-http-item (gtk-menu-item-new-with-label "HTTP"))
		  (type-ftp-item (gtk-menu-item-new-with-label "FTP"))
		  ;; for NON-US, while it lasts...
		  (type-nonus-http-item (gtk-menu-item-new-with-label "NON-US HTTP"))
		  (type-nonus-ftp-item (gtk-menu-item-new-with-label "NON-US FTP"))
		  (infob (gtk-button-new-with-label "More Info"))
		  (okayb (gtk-button-new-with-label "Okay"))
		  (cancelb (gtk-button-new-with-label "Cancel"))
		  (dist-combo (gtk-combo-new))
		  (comp-combo (gtk-combo-new))
		  (scrolledwin (gtk-scrolled-window-new (gtk-adjustment-new 0 1000 0 5 20 50)
							(gtk-adjustment-new 0 1000 0 5 20 50)))


		  (create-deb-line
		   (lambda (n type dist components)
		     (let ((mirror (nth n (list-of-sites-by-type type))))
		       ;; while non-US persists, lets check it
		       (if (and (string-head-eq type "NonUS")
				(not (string-match "[A-Za-z]+\/non-US" dist)))
			   (gtk-yes-no-cancel-dialog "Note: You might have meant to choose a non-US distribution.\nPlease edit the entry and select the appropriate distribution, if so." (lambda () #t) #f #f))
		       (concat "deb " (get-uri-type type) "://" (cadr (assoc "Site" mirror))
			       (cadr (assoc type mirror))
			       " " dist " " components))))
		  (make-menu-item-handler
		   (lambda (type)
		     (lambda (e)
		       (gtk-clist-freeze clist)
		       (setq selectionp #f)
		       (setq current-row #f)
		       (setq current-col #f)
		       (gtk-clist-clear clist)
		       (let ((mirror-column-names (vector->list (send global-info 
								      'mirror-column-names))))
			 (mapc 
			  (lambda (x)
			    (gtk-clist-append clist
					      (apply vector 
						     (mapcar (lambda (y)
							       (apply concat 
								      (cdr (assoc y x))))
							     mirror-column-names))))
			  
			  
			  (list-of-sites-by-type type)))
		       (gtk-clist-thaw clist)
		       (setq current-type type)))))
		  
	      (let ((mirror-column-widths (send global-info 
						'mirror-column-widths)))
		(let loop ((i 0)
			   (len (length mirror-column-widths)))
		     (if (< i len)
			 (progn
			   (gtk-clist-set-column-width clist 
						       i 
						       (aref 
							mirror-column-widths 
							i))
			   
			   (loop (+ i 1) len)))))
		       
	      
	      (gtk-window-set-default-size dialog (send global-info 'mirror-dialog-x) (send global-info 'mirror-dialog-y))
	      

	      (gtk-window-set-position dialog 'mouse)
	      (gtk-window-set-title dialog "Add from Mirrors List")

	      (gtk-combo-set-popdown-strings dist-combo 
					     (send global-info 'default-distributions))
	      (gtk-combo-set-popdown-strings comp-combo
					     (send global-info 'default-components))

	      (gtk-clist-set-selection-mode clist 'single)

	      (gtk-signal-connect clist
				  "select_row"
				  (lambda (o r c e)
				    (setq selectionp #t)
				    (setq current-row r)
				    (setq current-col c)))
	      
	      (gtk-signal-connect clist
				  "unselect_row"
				  (lambda (o r c e)
				    (setq selectionp #f)
				    (setq current-row r)
				    (setq current-col c)))
	      
	      (gtk-menu-append type-menu type-http-item)
	      (gtk-menu-append type-menu type-ftp-item)
	      (gtk-menu-append type-menu type-nonus-http-item)
	      (gtk-menu-append type-menu type-nonus-ftp-item)

	      (gtk-option-menu-set-menu type-option type-menu)


	      
	      (gtk-signal-connect type-http-item
				  "activate"
				  (make-menu-item-handler "Archive-http"))
				    
	      (gtk-signal-connect type-ftp-item
				  "activate"
				  (make-menu-item-handler "Archive-ftp"))

	      (gtk-signal-connect type-nonus-http-item
				  "activate"
				  (make-menu-item-handler "NonUS-http"))

	      (gtk-signal-connect type-nonus-ftp-item
				  "activate"
				  (make-menu-item-handler "NonUS-ftp"))

	      (gtk-menu-item-activate type-http-item)


	      (let ((make-temp-hbox
		     (lambda x
		       (let ((hbox (gtk-hbox-new #f 10)))
			 (apply gtk-container-add* (append (list hbox) x))
			 hbox))))
		
		(gtk-box-pack-start vbox (make-temp-hbox type-option) #f #f 5)
		
		(gtk-container-add scrolledwin clist)
		(gtk-container-add vbox (make-temp-hbox scrolledwin))

		(gtk-box-pack-start vbox (make-temp-hbox infob) #f #f 5))

	      (let ((hbox (gtk-hbox-new #t 5)))
		(gtk-box-pack-start hbox (gtk-label-new "Distribution:") #f #f 0)
		(gtk-box-pack-start hbox dist-combo #f #f 0)
		(gtk-box-pack-start vbox hbox #f #f 0))
	      
	      (let ((hbox (gtk-hbox-new #t 5)))
		(gtk-box-pack-start hbox (gtk-label-new "Components:") #f #f 0)
		(gtk-box-pack-start hbox comp-combo #f #f 0)
		(gtk-box-pack-start vbox hbox #f #f 0))


	      (gtk-container-add action-area okayb)
	      (gtk-container-add action-area cancelb)
	      


	      (gtk-signal-connect infob
				  "clicked"
				  (lambda ()
				    (if (not current-row)
					(gtk-yes-no-cancel-dialog "You must select a site to get info on" #f #f #f)
				      (create-mirror-info-dialog (nth current-row (list-of-sites-by-type current-type))))))
				      

	      (gtk-signal-connect okayb
				  "clicked"
				  (lambda ()
				    (if (not current-row)
					(gtk-yes-no-cancel-dialog "You must select a site, or cancel" #f #f #f)
				      (progn 
					(okay (create-deb-line current-row
							       current-type
							       (gtk-entry-get-text (gtk-combo-entry dist-combo))
							       (gtk-entry-get-text (gtk-combo-entry comp-combo)))))
				      (gtk-widget-destroy dialog))))

	      (gtk-signal-connect cancelb
				  "clicked"
				  (lambda ()
				    (gtk-widget-destroy dialog)))

	      (gtk-widget-show-all dialog)))))
		  
	      

      (lambda (msg)
	(cond
	 ((equal msg 'choose-mirror-dialog)
	  choose-mirror-dialog)
	 ((equal msg 'create-deb-line)
	  create-deb-line)
	 ((equal msg 'list-of-sites-by-type)
	  list-of-sites-by-type)

	 (t
	  #f))))))


;; The sources.list object generator

(define make-sources-list
  (lambda (filename)
    (define full-filename (canonical-file-name filename))
    (define okay-to-get-list 
      '(type uri-type uri-address uri-type-separator uri-separator uri type-string distribution components))
    
    (define read-sources-line
      (lambda (fp)
	(read-line fp)))

    (define make-source-line
      (lambda (commented? line-type proto-type name dist trees)

	(define uri-separator
	  (lambda ()
	    (send global-info 'uri-separator proto-type)))
		    

	(cond 
	 ((stringp commented?)
	  (make-source-line (if (string= commented? "#") #t #f)
			    line-type
			    proto-type
			    name
			    dist
			    trees))
	 ((stringp line-type)
	  (make-source-line commented?
			    (if (string= line-type "deb") 'binary 'source)
			    proto-type
			    name
			    dist
			    trees))
	 (t
	  (lambda (msg)
	    (cond
	     ((equal msg 'activatedp)
	      (lambda () (not commented?)))
	     ((equal msg 'activated-string)
	      (lambda () (if commented? "No" "Yes")))
	     ((equal msg 'activate)
	      (lambda () (setq commented? #f)))
	     ((equal msg 'deactivate)
	      (lambda () (setq commented? #t)))
	     ((equal msg 'type)
	      (lambda () line-type))
	     ((equal msg 'set-type)
	      (lambda (n) 
		(if (or (equal n 'binary)
			(equal n 'source))
		    (setq line-type n)
		  #f)))
	     ((equal msg 'type-string)
	      (lambda () (if (equal line-type 'binary) "Binary" "Source")))
	     ((equal msg 'uri-type)
	      (lambda () proto-type))
	     ((equal msg 'set-uri-type)
	      (lambda (n)
		(if (member n (send global-info 'uri-type-list))
		    (setq proto-type n)
		  #f)))
	     ((equal msg 'uri-address)
	      (lambda () name))
	     ((equal msg 'set-uri-address)
	      (lambda (n)
		(setq name n)))
	     ((equal msg 'uri)
	      (lambda () (concat proto-type (uri-separator) name)))
	     ((equal msg 'uri-separator)
	      uri-separator)
	     ((equal msg 'uri-type-separator)
	      (lambda () (concat proto-type (uri-separator))))
	     ((equal msg 'distribution)
	      (lambda () dist))
	     ((equal msg 'set-distribution)
	      (lambda (n)
		(setq dist n)))
	     ((equal msg 'components)
	      (lambda () trees))
	     ((equal msg 'set-components)
	      (lambda (n)
		(setq trees n)))
	     (t
	      (lambda () #f))))))))
	
    (define parse-one-line
      (lambda (l)
	(if (string-match "(#?)[ ]*(deb|deb-src)[ ]+([^\r\n]+)" l)
	    (progn
	      (let ((commented? (expand-last-match "\\1"))
		    (line-type (expand-last-match "\\2"))
		    (url-tree (expand-last-match "\\3")))
		(string-match "([^:]+):(.+)" url-tree)
		(let ((proto-type (expand-last-match "\\1"))
		      (remainder (expand-last-match "\\2")))
		  (cond
		   ((send global-info 'need-brackets? proto-type)
		    (if (string-match "(\\[.+\\][^ ]*)[ ]+(.*)" remainder)
			(let ((cd-name (expand-last-match "\\1"))
			      (dist-tree (expand-last-match "\\2")))
			  (string-match "([^ ]+)([ ]+)?(.*)?" dist-tree)
			  (let ((dist (expand-last-match "\\1"))
				(trees (expand-last-match "\\3")))
			    (make-source-line commented? line-type proto-type cd-name dist trees)))
		      
		      #f))
		   ((send global-info 'no-//? proto-type)
		    (string-match "([^ ]+)[ ]+([^ ]+)([ ]+)?(.*)?" remainder)
		    (let ((url (expand-last-match "\\1"))
			  (dist (expand-last-match "\\2"))
			  (trees (expand-last-match "\\4")))
		      (make-source-line commented? line-type proto-type url dist trees)))
		   (t
		    (string-match "//([^ ]+)[ ]+([^ ]+)([ ]+)?(.*)?" remainder)
		    (let ((url (expand-last-match "\\1"))
			  (dist (expand-last-match "\\2"))
			  (trees (expand-last-match "\\4")))
		      (make-source-line commented? line-type proto-type url dist trees)))))))
	  #f)))

    (define read-all-lines
      (lambda (l lines fp)
	(if (not l)
	    lines
	  (let ((lo (parse-one-line l)))
	    (if (not lo)
		(read-all-lines (read-sources-line fp) lines fp)
	      (read-all-lines (read-sources-line fp) (append lines (list lo)) fp))))))

    (define write-to-file
      (lambda (fp)
	(mapcar 
	 (lambda (x)
	   (format fp 
		   "%s%s %s %s %s\n"
		   (if (send x 'activatedp) "" "#")
		   (if (equal (send x 'type) 'binary) "deb" "deb-src")
		   (send x 'uri)
		   (send x 'distribution)
		   (send x 'components)))
	 lines-list)))
			   
    (define fill-clist
      (lambda (clist-info)
	(let ((clist (send clist-info 'clist))
	      (r (send clist-info 'current-row))
	      (c (send clist-info 'current-col)))
	  (gtk-clist-freeze clist) ; enforce atomicity in major graphical updates
	  (gtk-clist-clear clist)

	  (mapcar
	   (lambda (x)
	     (gtk-clist-append clist 
			       (apply vector (mapcar 
					      (lambda (y) 
						(send x y))
					      '(activated-string type-string uri distribution components)))))
	   lines-list)

	  (if (and (numberp r) (numberp c))
	      (progn
		(gtk-clist-select-row clist r c)
		(gtk-clist-moveto clist r 0 0.5 0)))
	  
	  (gtk-clist-thaw clist))))

    (define in-obj (open-file full-filename 'read))    

    (define lines-list (read-all-lines (read-sources-line in-obj) '() in-obj))
    (close-file in-obj)
    
    (lambda (msg)
      (cond
       ((equal msg 'fill-clist)
	fill-clist)

       ((equal msg 'write-file)
	(lambda () 
	  (let ((out-obj (open-file full-filename 'write)))
	    (write-to-file out-obj)
	    (close-file out-obj))))

       ((equal msg 'activate)
	(lambda (clist-info n) 
	  (send (nth n lines-list) 'activate)
	  (fill-clist clist-info)))

       ((equal msg 'deactivate)
	(lambda (clist-info n) 
	  (send (nth n lines-list) 'deactivate)
	  (fill-clist clist-info)))

       ((equal msg 'add-line)
	(lambda (clist-info l) 
	  (let ((new-line (parse-one-line l)))
	    (if (not new-line)
		#f
	      (progn 
		(setq lines-list (append lines-list (list new-line)))
		(fill-clist clist-info)
		#t)))))

       ((equal msg 'get)
	(lambda (n what)
	  (if (member what okay-to-get-list)
	      (send (nth n lines-list) what)
	    #f)))

       ((equal msg 'edit-line)
	(lambda (clist-info n deb-line)
	  (let ((lo (nth n lines-list))
		(lo2 (parse-one-line deb-line)))
	    
	    (send lo 'set-type (send lo2 'type))
	    (send lo 'set-uri-type (send lo2 'uri-type))
	    (send lo 'set-uri-address (send lo2 'uri-address))
	    (send lo 'set-distribution (send lo2 'distribution))
	    (send lo 'set-components (send lo2 'components))
	    (fill-clist clist-info))))
       
       ((equal msg 'remove-line)
	(lambda (clist-info n)
	  (setq lines-list (remove (nth n lines-list) lines-list))
	  (gtk-clist-unselect-row (send clist-info 'clist) n 0)
	  (fill-clist clist-info)))

       (t
	#f)))))
	       



;; Some useful GTK functionality extensions
(define gtk-container-add*
  (lambda (c . l)
    (cond
     ((consp l)
      (gtk-container-add c (car l))
      (apply gtk-container-add* (append (list c) (cdr l))))
     (t
      c))))

(define gtk-box-pack-start*
  (lambda (c e f p . l)
    (cond
     ((consp l)
      (gtk-box-pack-start c (car l) e f p)
      (apply gtk-box-pack-start* (append (list c e f p) (cdr l))))
     (t
      c))))

;; question = well, duh
;; yes, no, cancel = functions to be called when that button is pressed. if all 3 are #f, then it will
;;   use the first 'opts' argument as the number of seconds to wait before destroying the dialog (default: 5)
;; opts = if yes, no, or cancel is not #f, then the first three 'opts' are alternatives names for those buttons.
(define gtk-yes-no-cancel-dialog
  (lambda (question yes no cancel . opts)
    (let ((dialog (gtk-dialog-new))
	  (label (gtk-label-new question))
	  (hbox (gtk-hbox-new #f 10))
	  (e #t)
	  (f #t)
	  (p 10)
	  (label1 (if (consp opts)
		      (car opts)
		    "Yes"))
	  (label2 (if (and (consp opts)
			   (consp (cdr opts)))
		      (cadr opts)
		    "No"))
	  (label3 (if (and (consp opts)
			   (consp (cdr opts))
			   (consp (cddr opts)))
		      (caddr opts)
		    "Cancel")))
			
      (gtk-window-set-position dialog 'mouse)
   
      (gtk-box-pack-start (gtk-dialog-action-area dialog) hbox #t #t 0)
      (gtk-box-pack-start (gtk-dialog-vbox dialog) label #t #t 0)
      (gtk-window-set-modal dialog #t)
      (gtk-window-set-title dialog question)

      (if (functionp yes)
	  (let ((yesb (gtk-button-new-with-label label1)))
	    (gtk-signal-connect yesb
				"clicked"
				(lambda ()
				  (if (yes)
				      (gtk-widget-destroy dialog))))
	    (gtk-box-pack-start hbox yesb e f p)))
      (if (functionp no)
	  (let ((nob (gtk-button-new-with-label label2)))
	    (gtk-signal-connect nob
				"clicked"
				(lambda ()
				  (if (no)
				      (gtk-widget-destroy dialog))))
	    (gtk-box-pack-start hbox nob e f p)))
      (if (functionp cancel)
	  (let ((cancelb (gtk-button-new-with-label label3)))
	    (gtk-signal-connect cancelb
				"clicked"
				(lambda ()
				  (if (cancel)
				      (gtk-widget-destroy dialog))))
	    (gtk-box-pack-start hbox cancelb e f p)))
      (if (and (not (functionp yes))
	       (not (functionp no))
	       (not (functionp cancel)))
	  (letrec ((delay (if (numberp label1) label1 5))
		   (update-timeout-string
		    (lambda (d)
		      (concat "This message will disappear in "
			      (number->string d)
			      " seconds")))
		   (l1 (gtk-label-new (update-timeout-string delay)))
		   (timeout (make-timer (lambda ()
					  (setq delay (1- delay))
					  (if (<= delay 0) 
					      (gtk-widget-destroy dialog)
					    (progn
					      (gtk-label-set l1 (update-timeout-string delay))
					      (set-timer timeout))))
					1)))
		  (gtk-box-pack-start hbox 
				      l1
				      #t #f 0)))
	  
      (gtk-widget-show-all dialog))))
				    

;; create an editor dialog. okay and cancel are functions to call when those buttons are pressed
;; the rest of the options are possible values to fill in for the dialog	  
(define edit-source-line-dialog
  (lambda (okay cancel #!optional (default-type "Binary") (default-uri-type "http://")
		(default-uri "") (default-dist "stable") (default-comp "main"))

    (let* ((dialog (gtk-dialog-new))
	   (vbox (gtk-dialog-vbox dialog))
	   (action-area (gtk-dialog-action-area dialog))
	   (uri-types (send global-info 'uri-type-separator-list)))
      (let ((okayb (gtk-button-new-with-label "Okay"))
	    (cancelb (gtk-button-new-with-label "Cancel"))
	    (uri-entry (gtk-entry-new))
	    (dist-combo (gtk-combo-new))
	    (comp-combo (gtk-combo-new))
	    (type-option (gtk-option-menu-new))
	    (type-menu (gtk-menu-new))
	    (binary-menuitem (gtk-menu-item-new-with-label "Binary"))
	    (source-menuitem (gtk-menu-item-new-with-label "Source"))
	    (hbox1 (gtk-hbox-new #f 5))
	    (hbox2 (gtk-hbox-new #f 5))
	    (menu-choice "deb")
	    (proto-option (gtk-option-menu-new))
	    (proto-menu (gtk-menu-new))
	    (uri-type (send global-info 'default-uri-type-separator))
	    (proto-menuitems (mapcar gtk-menu-item-new-with-label uri-types)))
	
	
	(gtk-window-set-position dialog 'mouse)
	
	(gtk-box-pack-start* vbox #t #f 0 hbox1 hbox2)

	(gtk-menu-append type-menu binary-menuitem)
	(gtk-menu-append type-menu source-menuitem)

	(mapcar (lambda (x)
		  (gtk-menu-append proto-menu x))
		proto-menuitems)
	(gtk-option-menu-set-menu proto-option proto-menu)
	
	(gtk-option-menu-set-menu type-option type-menu)

	(gtk-combo-set-popdown-strings dist-combo (send global-info 'default-distributions))


	(gtk-combo-set-popdown-strings comp-combo (send global-info 'default-components))

	(gtk-signal-connect binary-menuitem
			    "activate"
			    (lambda (e)
			      (setq menu-choice "deb")))

	(gtk-signal-connect source-menuitem
			    "activate"
			    (lambda (e)
			      (setq menu-choice "deb-src")))

	(mapcar (lambda (x)
		  (gtk-signal-connect x
				      "activate"
				      (lambda (e)
					(setq uri-type (nth (position x proto-menuitems) uri-types)))))
		proto-menuitems)



	(if (or (string= default-type "Binary")
		(string= default-type "deb"))
	    (progn
	      (gtk-menu-set-active type-menu 0)
	      (gtk-option-menu-set-history type-option 0)
	      (gtk-menu-item-activate binary-menuitem))
	  (progn
	    (gtk-menu-set-active type-menu 1)
	    (gtk-option-menu-set-history type-option 1)
	    (gtk-menu-item-activate source-menuitem)))

	
	(let ((pos (position default-uri-type uri-types)))
	  (if (not pos)
	      (progn
		(gtk-menu-set-active proto-menu 0)
		(gtk-option-menu-set-history proto-option 0)
		(gtk-menu-item-activate (nth 0 proto-menuitems)))
	    (progn
	      (gtk-menu-set-active proto-menu pos)
	      (gtk-option-menu-set-history proto-option pos)
	      (gtk-menu-item-activate (nth pos proto-menuitems)))))


	(gtk-entry-set-text uri-entry default-uri)
	(gtk-entry-set-text (gtk-combo-entry dist-combo) default-dist)
	(gtk-entry-set-text (gtk-combo-entry comp-combo) default-comp)

	(gtk-box-pack-start hbox1 
			    (gtk-label-new "Type:")
			    #t #f 0)
	(gtk-box-pack-start hbox1
			    type-option
			    #t #t 10)
	(gtk-box-pack-start hbox1
			    (gtk-label-new "URI:")
			    #t #f 0)
	(gtk-box-pack-start hbox1
			    proto-option
			    #t #t 0)
	(gtk-box-pack-start hbox1
			    uri-entry
			    #t #t 10)

	(gtk-box-pack-start hbox2
			    (gtk-label-new "Distribution:")
			    #t #f 0)
	(gtk-box-pack-start hbox2
			    dist-combo
			    #t #t 10)
	(gtk-box-pack-start hbox2
			    (gtk-label-new "Components:")
			    #t #f 0)
	(gtk-box-pack-start hbox2
			    comp-combo
			    #t #t 10)


	(gtk-box-pack-start action-area okayb #t #t 10)
	(gtk-box-pack-start action-area cancelb #t #t 10)

	(gtk-window-set-modal dialog #t)
	(gtk-window-set-title dialog "Source line info")

	(let ((valid-entry
	       (lambda ()
		 (if (and (> (length (gtk-entry-get-text uri-entry)) 0)
			  (> (length (gtk-entry-get-text (gtk-combo-entry dist-combo))) 0))
		     #t #f)))
	      (create-deb-line
	       (lambda ()
		 (concat menu-choice " " uri-type (gtk-entry-get-text uri-entry) 
			 " " (gtk-entry-get-text (gtk-combo-entry dist-combo))
			 " " (gtk-entry-get-text (gtk-combo-entry comp-combo))))))
	  
	  (gtk-signal-connect okayb
			      "clicked"
			      (lambda ()
				(if (valid-entry)
				    (progn
				      (okay (create-deb-line))
				      (gtk-widget-destroy dialog))
				  (gtk-yes-no-cancel-dialog "You need to enter in a URI and a distribution." #f #f #f))))
	  (gtk-signal-connect cancelb
			      "clicked"
			      (lambda ()
				(cancel (create-deb-line))
				(gtk-widget-destroy dialog))))

	
	(gtk-widget-show-all dialog)))))
  
      

;; Returns a list of the buttons at the bottom
(define create-button-list
  (lambda (clist-info sources)
    (let ((b1 (gtk-button-new-with-label "Activate Selection"))
	  (b2 (gtk-button-new-with-label "Deactivate Selection"))
	  (b3 (gtk-button-new-with-label "Add to List"))
	  (b4 (gtk-button-new-with-label "Edit Selection"))
	  (b5 (gtk-button-new-with-label "Remove Selection")))
      
      (gtk-signal-connect b1
			  "clicked"
			  (lambda ()
			    (if (send clist-info 'selectionp)
				(send sources 'activate clist-info (send clist-info 'current-row))
			      
			      (gtk-yes-no-cancel-dialog "No selection was made." #f #f #f 3))))

      (gtk-signal-connect b2
			  "clicked"
			  (lambda ()
			    (if (send clist-info 'selectionp)
				(send sources 'deactivate clist-info (send clist-info 'current-row))

			      (gtk-yes-no-cancel-dialog "No selection was made." #f #f #f 3))))

      (gtk-signal-connect b3
			  "clicked"
			  (lambda ()
			    (edit-source-line-dialog (lambda (text)
						       (send sources 'add-line clist-info text))
						     (lambda (text)
						       #t))))


      (gtk-signal-connect b4
			  "clicked"
			  (lambda ()
			    (if (send clist-info 'selectionp)
				(let ((n (send clist-info 'current-row)))
				  (edit-source-line-dialog (lambda (text)
							     (send sources 'edit-line
								   clist-info
								   (send clist-info 'current-row)
								   text))
							   (lambda (text)
							     #t)
							   (send sources 'get n 'type-string)
							   (send sources 'get n 'uri-type-separator)
							   (send sources 'get n 'uri-address)
							   (send sources 'get n 'distribution)
							   (send sources 'get n 'components)))
			      (gtk-yes-no-cancel-dialog "No selection was made." #f #f #f 3))))

      (gtk-signal-connect b5
			  "clicked"
			  (lambda ()
			    (if (send clist-info 'selectionp)
				(send sources 'remove-line clist-info (send clist-info 'current-row))
			      (gtk-yes-no-cancel-dialog "No selection was made." #f #f #f 3))))
				  

      (list b1 b2 b3 b4 b5))))

;; Update dialog
(define create-update-dialog
  (lambda (out-str process)
    (let* ((dialog (gtk-dialog-new))
	   (vbox (gtk-dialog-vbox dialog))
	   (action-area (gtk-dialog-action-area dialog))
	   (textarea (gtk-text-new nil nil)))
      (let ((label (gtk-label-new "Cancel"))
	    (hbox (gtk-hbox-new #f 0))
	    (vscrollbar (gtk-vscrollbar-new (gtk-text-vadj textarea)))
	    (closeb (gtk-button-new)))

	(gtk-window-set-position dialog 'mouse)
	(gtk-window-set-title dialog "Updating...")
	(gtk-window-set-default-size dialog
				     (send global-info 'update-x) 
				     (send global-info 'update-y))	      

	(gtk-container-add closeb label)
	
	(gtk-container-add action-area closeb)

	(gtk-box-pack-start hbox textarea #t #t 0)
	(gtk-box-pack-start hbox vscrollbar #f #f 0)

	(gtk-container-add vbox hbox)

	(gtk-text-set-line-wrap textarea #t)
	(gtk-text-set-editable textarea #f)

	(set-process-function process (lambda ()
					(if (gtk-widget-p dialog)
					    (let ((text "\n***FINISHED***\n")
						  (pos (vector (gtk-editable-get-position textarea))))
					      (gtk-label-set label "Close")
					      (gtk-editable-insert-text textarea
									text
									(length text)
									pos)))))

	(letrec ((timeout (make-timer (lambda ()
					(if (and (process-running-p process)
						 (gtk-widget-p dialog))
					    (let ((text (get-output-stream-string out-str))
						  (pos (vector (gtk-editable-get-position textarea))))
					      
					      (gtk-editable-insert-text textarea
									text
									(length text)
									pos)
					      
					      (set-timer timeout))))
				      0 100))))
					

	(gtk-signal-connect closeb
			    "clicked"
			    (lambda ()
			      (gtk-widget-destroy dialog)
			      (if (process-running-p process)
				  (kill-process process))))


	(gtk-widget-show-all dialog)))))

(define create-about-dialog
  (lambda ()
    (let* ((dialog (gtk-dialog-new))
	   (vbox (gtk-dialog-vbox dialog))
	   (textarea (gtk-text-new nil nil))
	   (action-area (gtk-dialog-action-area dialog)))
      (let ((closeb (gtk-button-new-with-label "Close"))
	    (pos (vector 0))
	    (hbox (gtk-hbox-new #f 0))
	    (vscrollbar (gtk-vscrollbar-new (gtk-text-vadj textarea)))
	    (text (send global-info 'about-text)))

	(gtk-window-set-position dialog 'mouse)

	(gtk-window-set-default-size dialog
				     (send global-info 'about-x) 
				     (send global-info 'about-y))	      
	(gtk-text-set-line-wrap textarea #t)
	(gtk-text-set-editable textarea #f)
	(gtk-window-set-title dialog "About")

	(gtk-box-pack-start hbox textarea #t #t 0)
	(gtk-box-pack-start hbox vscrollbar #f #f 0)

	(gtk-container-add action-area closeb)
	(gtk-container-add vbox hbox)

	(gtk-editable-insert-text textarea
				  text
				  (length text)
				  pos)


	(gtk-signal-connect closeb
			    "clicked"
			    (lambda ()
			      (gtk-widget-destroy dialog)))

	(gtk-widget-show-all dialog)))))


;; Make another button row
(define create-button-list-2
  (lambda (clist-info sources)
    (let ((b-list
	   (list
	    (list
	     (gtk-button-new-with-label "Save to File")
	     (lambda ()
	       (send sources 'write-file)
	       (gtk-yes-no-cancel-dialog "Saved." #f #f #f 2)))
	    (list
	     (gtk-button-new-with-label "Update APT")
	     (lambda ()
	       (let* ((out-str (make-string-output-stream))
		      (process (make-process standard-output)))
		 
		 (set-process-output-stream process out-str)
		 (start-process process "apt-get" "update")
		 (create-update-dialog out-str process))
	       #t))

	    (list
	     (gtk-button-new-with-label "About this Program")
	     (lambda ()
	       (create-about-dialog)))
	    (list
	     (gtk-button-new-with-label "Quit")
	     (lambda ()
	       (gtk-yes-no-cancel-dialog "Save changes before quitting?" 
					 (lambda ()
					   (send sources 'write-file)
					   (exit-program 0))
					 (lambda ()
					   (exit-program 0))
					 (lambda () #t)))))))
      (mapcar (lambda (x)
		(gtk-signal-connect (car x)
				    "clicked"
				    (cadr x)))
	      b-list)

      (mapcar (lambda (x) (car x)) b-list))))


;; Add line hbox
(define create-add-line-hbox-list
  (lambda (clist-info sources)
    (let ((l1 (gtk-label-new "Add a deb line:"))
	  (e1 (gtk-entry-new))
	  (b1 (gtk-button-new-with-label "Add this Line")))
      
      (gtk-signal-connect b1
			  "clicked"
			  (lambda ()
			    (if (send sources 'add-line clist-info (gtk-entry-get-text e1))
				(gtk-entry-set-text e1 "")
			      (gtk-yes-no-cancel-dialog "That was not a valid deb line." #f #f #f 4))))
      (list l1 e1 b1))))
      
(define create-buttons-hbox-list
  (lambda (clist-info sources buttons)
    (let ((button-list (mapcar (lambda (x) 
				 (gtk-button-new-with-label (car x)))
			       buttons)))
      (let loop ((i 0)
		 (len (length button-list)))
	   
	   (if (< i len)
	       (progn
		 (gtk-signal-connect (nth i button-list)
				     "clicked"
				     (cadr (nth i buttons)))
		 (loop (+ i 1) len))
	     #t))
      
      (let ((hbox (gtk-hbox-new #t 5)))
	(mapcar (lambda (x)
		  (gtk-container-add hbox x))
		button-list)
	hbox))))

;; Possible other list object (currently not in use)
;(define make-widget-list
;  (lambda ()
;    (let* ((wlist (gtk-list-new))
;	  (items (mapcar gtk-list-item-new-with-label (send global-info 'list-column-names))))
;
;      (mapcar (lambda (x)
;		(gtk-list-append-item wlist x)) 
;	      items)
;
;      (lambda (msg)
;	(cond
;	 ((equal msg 'list)
;	  (lambda () wlist)))))))


;; GTK CList handler object generator
(define make-clist
  (lambda (sources)
    (define current-row #f)
    (define current-col #f)
    (define selectionp #f)

    (let ((clist (gtk-clist-new-with-titles (send global-info 'vector-column-names))))
      (gtk-clist-set-selection-mode clist 'single)
      (gtk-clist-set-column-width clist 2 320)
      (gtk-clist-set-column-width clist 1 48)

      (gtk-signal-connect clist
			  "select_row"
			  (lambda (o r c e)
			    (setq selectionp #t)
			    (setq current-row r)
			    (setq current-col c)))

      (gtk-signal-connect clist
			  "unselect_row"
			  (lambda (o r c e)
			    (setq selectionp #f)
			    (setq current-row r)
			    (setq current-col c)))

      
      ;(gtk-clist-append clist (vector "Yes" "Binary" "ftp://emu.res.cmu.edu/debian/" "unstable" "main contrib non-free"))

      (lambda (msg)
	(cond
	 ((equal msg 'clist)
	  (lambda () clist))
	 ((equal msg 'selectionp)
	  (lambda () selectionp))
	 ((equal msg 'current-row)
	  (lambda () current-row))
	 ((equal msg 'current-col)
	  (lambda () current-col)))))))




;; The main program code

(define sources.list-path (get-command-line-option "-f" t))


(define sources (make-sources-list (if (stringp sources.list-path) sources.list-path default-sources.list)))

(define mirrors-list (make-mirrors-list (send global-info 'master-mirrors-list)))

;;(mapcar
;; (lambda (x)
;;   (format stdout "!%s %s\n" (send x 'uri) (send x 'activated-string)))
;; (send sources 'lines-list))


(let* ((window (gtk-window-new 'toplevel))
      (scrolledwin (gtk-scrolled-window-new (gtk-adjustment-new 0 1000 0 5 20 50)
					    (gtk-adjustment-new 0 1000 0 5 20 50)))
;      (list-info (make-widget-list))
      (clist-info (make-clist sources))
      (vbox (gtk-vbox-new #f 5))
      (label (gtk-label-new "Select an entry to use the buttons"))
      (hbox (gtk-hbox-new #f 5))
      (add-line-hbox (gtk-hbox-new #f 5))
      (buttons-2-hbox (gtk-hbox-new #f 5))
      (button-list (create-button-list clist-info sources))
      (button-list-2 (create-button-list-2 clist-info sources))
      (add-line-hbox-list (create-add-line-hbox-list clist-info sources)))
  
  (send sources 'fill-clist clist-info)

  (gtk-window-set-title window "gtk-sources")

  (gtk-window-set-default-size window 
  			       (send global-info 'toplevel-x) 
			       (send global-info 'toplevel-y))	      
  
  (gtk-window-set-position window 'mouse)

  (gtk-container-border-width window 10)

  (gtk-signal-connect window
		      "delete_event"
		      (lambda (w)
			(if (gtk-standalone-p)
			    (exit-program 0)
			  (gtk-widget-destroy w))))


;  (gtk-scrolled-window-add-with-viewport scrolledwin (send list-info 'list))

  (gtk-container-add scrolledwin (send clist-info 'clist))

  (apply gtk-box-pack-start* (append (list hbox #t #t 10) button-list))
  (gtk-box-pack-start add-line-hbox (car add-line-hbox-list) #f #f 10)
  (apply gtk-box-pack-start* (append (list add-line-hbox #t #t 10) (cdr add-line-hbox-list)))
  (apply gtk-box-pack-start* (append (list buttons-2-hbox #t #t 10) button-list-2))
 
  (gtk-box-pack-start vbox scrolledwin #t #t 0)
  (gtk-box-pack-start vbox label #f #f 0)
  (gtk-box-pack-start vbox hbox #f #f 0)
  (gtk-box-pack-start vbox (gtk-hseparator-new))
  (gtk-box-pack-start vbox add-line-hbox #f #f 0)
  (gtk-box-pack-start vbox (gtk-hseparator-new))

  (define add-from-mirror-list-callback
    (lambda ()
      (send mirrors-list 'choose-mirror-dialog (lambda (l)
						 (send sources 'add-line clist-info l)))))
  
  (gtk-box-pack-start vbox (create-buttons-hbox-list clist-info
						     sources
						     (list 
						      (list 
						       "Add from Mirrors List"
						       add-from-mirror-list-callback))))
  
  (gtk-box-pack-start vbox (gtk-hseparator-new))
  (gtk-box-pack-start vbox buttons-2-hbox #f #f 0)

  (gtk-container-add window vbox)

  (gtk-widget-show-all window)

  (when (gtk-standalone-p)
    (catch 'ready-to-exit-program
      (recursive-edit))))

;; Local variables:
;; mode: lisp
;; End:
