/*
 *   Copyright (C) 2007-2009 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <gtk/gtk.h>
#include <mhash.h>

#include "hash.h"
#include "main.h"
#include "gui.h"
#include "list.h"

#define BUFFER_SIZE 4096 // File reading buffer
#define PROGRESS_UPDATE 1024 // Reads between progress bar updates

struct hash_s hash[HASH_N] = {
	{ .type = MHASH_MD2,		.name = "MD2" },
	{ .type = MHASH_MD4,		.name = "MD4" },
	{ .type = MHASH_MD5,		.name = "MD5" },
	{ .type = MHASH_SHA1,		.name = "SHA1" },
	{ .type = MHASH_SHA224,		.name = "SHA224" },
	{ .type = MHASH_SHA256,		.name = "SHA256" },
	{ .type = MHASH_SHA384,		.name = "SHA384" },
	{ .type = MHASH_SHA512,		.name = "SHA512" },
	{ .type = MHASH_RIPEMD128,	.name = "RIPEMD128" },
	{ .type = MHASH_RIPEMD160,	.name = "RIPEMD160" },
	{ .type = MHASH_RIPEMD256,	.name = "RIPEMD256" },
	{ .type = MHASH_RIPEMD320,	.name = "RIPEMD320" },
	{ .type = MHASH_HAVAL128,	.name = "HAVAL128" },
	{ .type = MHASH_HAVAL160,	.name = "HAVAL160" },
	{ .type = MHASH_HAVAL192,	.name = "HAVAL192" },
	{ .type = MHASH_HAVAL224,	.name = "HAVAL224" },
	{ .type = MHASH_HAVAL256,	.name = "HAVAL256" },
	{ .type = MHASH_TIGER128,	.name = "TIGER128" },
	{ .type = MHASH_TIGER160,	.name = "TIGER160" },
	{ .type = MHASH_TIGER192,	.name = "TIGER192" },
	{ .type = MHASH_GOST,		.name = "GOST" },
	{ .type = MHASH_WHIRLPOOL,	.name = "WHIRLPOOL" },
	{ .type = MHASH_SNEFRU128,	.name = "SNEFRU128" },
	{ .type = MHASH_SNEFRU256,	.name = "SNEFRU256" },
	{ .type = MHASH_CRC32,		.name = "CRC32" },
	{ .type = MHASH_CRC32B,		.name = "CRC32B" },
	{ .type = MHASH_ADLER32,	.name = "ADLER32" }
};

static void hash_init(const int id)
{
	hash[id].thread = mhash_init(hash[id].type);
	g_assert(hash[id].thread != MHASH_FAILED);
}

static char *hash_end(const int id)
{
	unsigned char *bin = mhash_end_m(hash[id].thread, g_malloc);
	GString *digest = g_string_sized_new(128);

	for (unsigned int i = 0; i < mhash_get_block_size(hash[id].type); i++)
		g_string_append_printf(digest, "%.2x", bin[i]);

	g_free(bin);

	return g_string_free(digest, false);
}

char *hash_string(const char *string, const int id)
{
	hash_init(id);

	// Hash all at once
	mhash(hash[id].thread, string, strlen(string));

	return hash_end(id);
}

bool hash_file(const char *filename)
{
	char *digest;
	char buffer[BUFFER_SIZE];
	GIOChannel *file;
	unsigned int filesize, done = 0;
	GtkProgressBar *progress = GTK_PROGRESS_BAR(gui_get_widget("progressbar"));

	for (int i = 0; i < HASH_N; i++)
		if (gtk_toggle_button_get_active(hash[i].button))
			hash_init(i);

	file = g_io_channel_new_file(filename, "r", NULL);
	g_assert(file);
	g_io_channel_set_encoding(file, NULL, NULL);
	filesize = get_filesize(filename);

	for (gsize just_read = 0; !hash_stop; done++) {
		g_io_channel_read_chars(file, buffer, BUFFER_SIZE, &just_read, NULL);

		if (G_UNLIKELY(!just_read))
			break; // All done!

		for (int i = 0; i < HASH_N; i++)
			if (gtk_toggle_button_get_active(hash[i].button)) {
				mhash(hash[i].thread, buffer, just_read);
				gtk_main_iteration_do(false);
			}

		if (!(done % PROGRESS_UPDATE)) {
			// Update progress bar
			if (filesize)
				gtk_progress_bar_set_fraction(progress,
					(double)done * BUFFER_SIZE / filesize);
			else
				gtk_progress_bar_pulse(progress);
		}
	}
	g_io_channel_unref(file);

	if (hash_stop) {
		for (int i = 0; i < HASH_N; i++) {
			if (gtk_toggle_button_get_active(hash[i].button))
				g_free(hash_end(i));
		}
		hash_stop = false;
		return false;
	}

	for (int i = 0; i < HASH_N; i++) {
		if (gtk_toggle_button_get_active(hash[i].button)) {
			digest = hash_end(i);
			if (gui_view() == VIEW_FILE)
				gtk_entry_set_text(hash[i].entry, digest);
			else if (gui_view() == VIEW_FILE_LIST) {
				list_set_digest(filename, i, digest);
			} else
				g_assert_not_reached();
			g_free(digest);
		}
	}

	return true;
}
