/* Copyright (C) 1998-1999 Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
 * Copyright (C) 1999-2000 Matias Mutchinick
 *
 * The GtkSheet widget was written by Adrian Feiguin.
 * The GtkMatrix widget, a modification of GtkSheet was
 * written by Matias Mutchinick
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtksignal.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkadjustment.h>
#include <gtk/gtktable.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkmain.h>
#include <gtk/gtktypeutils.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkcombo.h>
#include "gtkmatrix.h"



#define CELL_SPACING 1
#define DRAG_WIDTH 6
#define TIMEOUT_SCROLL 20
#define TIMEOUT_FLASH 200
#define TIME_INTERVAL 8
#define COLUMN_MIN_WIDTH 10
#define MINROWS 1
#define MINCOLS 1
#define CELLOFFSET 4
#define DEFAULT_COLUMN_WIDTH 110
#define ROW_TITLE_WIDTH 60





/**
 * DEFAULT_ROW_HEIGHT:
 * @widget : The GtkMatrix.
 *
 * Returns : The default row heignt accordin to the matrix's style.
 */
#define DEFAULT_ROW_HEIGHT(widget) (widget->style->font->ascent+\
				    widget->style->font->descent+\
				    2 * CELLOFFSET)



/**
 * MATRIX_ROW_HEIGHT:
 * @MatriX : The GtkMatrix
 * @RoW   : A row number
 *
 * Returns : The height of that row.
 */
#define MATRIX_ROW_HEIGHT(MatriX,RoW) MAX(MatriX->row[RoW].height,\
                                    DEFAULT_ROW_HEIGHT(GTK_WIDGET(MatriX)))




/**
 * MATRIX_COL_TITLE_HEIGHT:
 * @MatriX : The GtkMatrix
 *
 * Returns : The column title height
 */
#define MATRIX_COL_TITLE_HEIGHT(MatriX)       MAX(MatriX->column_title_area.height,\
                                            DEFAULT_ROW_HEIGHT(GTK_WIDGET(MatriX)))



/**
 * MATRIX_FONT:
 * @MatriX : The GtkMatrix
 *
 * Returns : The font of the matrix's style.
 */
#define MATRIX_FONT(MatriX)                   GTK_WIDGET(MatriX)->style->font 




/* scrollbar spacing class macro */
#define SCROLLBAR_SPACING(w) (GTK_MATRIX_CLASS (GTK_OBJECT (w)->klass)->scrollbar_spacing)




/**
 * CALC_TOP_YPIXEL:
 * @matrix : The GtkMatrix
 * @row   : The number of a row
 *
 * Calculates recursivly the position of the top 'y' pixel of the row.
 */
static inline gint
CALC_TOP_YPIXEL(GtkMatrix *matrix, gint row)
{
	if (row == 0)
		return MATRIX_COL_TITLE_HEIGHT(matrix);
	else {
		return CALC_TOP_YPIXEL(matrix,row-1) + 
			MATRIX_ROW_HEIGHT(matrix,row-1);
	}
	
}



/**
 * ROW_TOP_YPIXEL
 * @matrix  : The GtkMatrix
 * @row    : The nomber of a row
 * 
 * Calculates recursivly the position of top 'y' pixel of @row
 * within the voffset context.
 */
static inline gint
ROW_TOP_YPIXEL(GtkMatrix *matrix,
	       gint      row)
{
	return matrix->voffset + matrix->row[row].top_ypixel; 
}





/**
 * ROW_FROM_YPIXEL:
 * @matrix : The GtkMatrix
 * @y     : The position of some row's top 'y' pixel
 *
 * Returns : The row index from a @y pixel location in the 
 *           context of the matrix's voffset 
 */
static inline gint
ROW_FROM_YPIXEL(GtkMatrix *matrix,
		gint      y)
{
	gint i, cxt_y;
	
	cxt_y = matrix->voffset;
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		cxt_y += MATRIX_COL_TITLE_HEIGHT(matrix);
	
	if (y < cxt_y)
		return 0;
	
	for (i = 0; i <= matrix->maxrow; i++) {
		
		if (y >= cxt_y && y <= (cxt_y + MATRIX_ROW_HEIGHT(matrix,i) ) && 
		    matrix->row[i].is_visible)
			return i;
		
		if (matrix->row[i].is_visible)
			cxt_y += MATRIX_ROW_HEIGHT(matrix,i);
		
	}
	/* no match */
	return matrix->maxrow;
}



/* gives the left pixel of the given column in context of
 * the matrix's hoffset */
static inline gint
COLUMN_LEFT_XPIXEL(GtkMatrix * matrix, gint ncol)
{
	return (matrix->hoffset + matrix->column[ncol].left_xpixel);
}




/* returns the column index from a x pixel location in the 
 * context of the matrix's hoffset */
static inline gint
COLUMN_FROM_XPIXEL(GtkMatrix *matrix,
		   gint      x)
{
	gint i, cx;
	
	cx = matrix->hoffset;
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		cx += matrix->row_title_area.width;
	if (x < cx)
		return 0;
	for (i = 0; i <= matrix->maxcol; i++) {
		if (x >= cx && x <= (cx + matrix->column[i].width) && matrix->column[i].is_visible)
			return i;
		if (matrix->column[i].is_visible)
			cx += matrix->column[i].width;

	}

	/* no match */
	return matrix->maxcol;
}



/* returns the total height of the matrix */
static inline gint MATRIX_HEIGHT(GtkMatrix * matrix)
{
	gint i, cx;

	cx = 0;
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		cx += MATRIX_COL_TITLE_HEIGHT(matrix);
	for (i = 0; i <= matrix->maxrow; i++)
		if (matrix->row[i].is_visible)
			cx += MATRIX_ROW_HEIGHT(matrix,i);

	return cx;
}


/* returns the total width of the matrix */
static inline gint MATRIX_WIDTH(GtkMatrix * matrix)
{
	gint i, cx;

	cx = 0;
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		cx += matrix->row_title_area.width;
	for (i = 0; i <= matrix->maxcol; i++)
		if (matrix->column[i].is_visible)
			cx += matrix->column[i].width;

	return cx;
}


#define MIN_VISIBLE_ROW(matrix) matrix->view.row0
#define MAX_VISIBLE_ROW(matrix) matrix->view.rowi
#define MIN_VISIBLE_COLUMN(matrix) matrix->view.col0
#define MAX_VISIBLE_COLUMN(matrix) matrix->view.coli


static inline gint
POSSIBLE_XDRAG(GtkMatrix *matrix, 
	       gint      x,
	       gint     *drag_column)
{
	gint column, xdrag;

	//g_print("POSIBLE_XDRAG\n");
	
	column = COLUMN_FROM_XPIXEL(matrix, x);
	*drag_column = column;

	xdrag = COLUMN_LEFT_XPIXEL(matrix, column) + CELL_SPACING;
	if (x <= xdrag + DRAG_WIDTH / 2 && column != 0) {
		while (!matrix->column[column - 1].is_visible && column > 0)
			column--;
		*drag_column = column - 1;
		return matrix->column[column - 1].is_sensitive;
	}
	xdrag += matrix->column[column].width;
	if (x >= xdrag - DRAG_WIDTH / 2 && x <= xdrag + DRAG_WIDTH / 2)
		return matrix->column[column].is_sensitive;

	return FALSE;
}


static inline gint
POSSIBLE_YDRAG(GtkMatrix *matrix, 
	       gint      y,
	       gint     *drag_row)
{
	gint row, ydrag;

	//g_print("POSIBLE_YDRAG\n");
	
	row = ROW_FROM_YPIXEL(matrix, y);
	*drag_row = row;

	ydrag = ROW_TOP_YPIXEL(matrix, row) + CELL_SPACING;
	if (y <= ydrag + DRAG_WIDTH / 2 && row != 0) {
		while (!matrix->row[row - 1].is_visible && row > 0)
			row--;
		*drag_row = row - 1;
		return matrix->row[row - 1].is_sensitive;
	}
	ydrag += MATRIX_ROW_HEIGHT(matrix,row);

	if (y >= ydrag - DRAG_WIDTH / 2 && y <= ydrag + DRAG_WIDTH / 2)
		return matrix->row[row].is_sensitive;


	return FALSE;
}


typedef gboolean(*GtkMatrixSignal1) (GtkObject * object,
				     gint arg1, gint arg2, gint * arg3, gint * arg4,
				     gpointer user_data);

typedef gboolean(*GtkMatrixSignal2) (GtkObject * object,
				     gint arg1, gint arg2,
				     gpointer user_data);


static void gtk_matrix_class_init(GtkMatrixClass * klass);
static void gtk_matrix_init(GtkMatrix * matrix);
static void gtk_matrix_destroy(GtkObject * object);
static void gtk_matrix_finalize(GtkObject * object);
static void gtk_matrix_style_set(GtkWidget * widget,
				GtkStyle * previous_style);
static void gtk_matrix_realize(GtkWidget * widget);
static void gtk_matrix_unrealize(GtkWidget * widget);
static void gtk_matrix_map(GtkWidget * widget);
static void gtk_matrix_unmap(GtkWidget * widget);
static void gtk_matrix_draw(GtkWidget * widget,
			   GdkRectangle * area);
static gint gtk_matrix_expose(GtkWidget * widget,
			     GdkEventExpose * event);

static void gtk_matrix_set_scroll_adjustments(GtkMatrix * matrix,
					     GtkAdjustment * hadjustment,
					     GtkAdjustment * vadjustment);

static gint gtk_matrix_button_press(GtkWidget * widget,
				   GdkEventButton * event);
static gint gtk_matrix_button_release(GtkWidget * widget,
				     GdkEventButton * event);
static gint gtk_matrix_motion(GtkWidget * widget,
			     GdkEventMotion * event);
static gint gtk_matrix_entry_key_press(GtkWidget * widget,
				      GdkEventKey * key);
static gint gtk_matrix_key_press(GtkWidget * widget,
				GdkEventKey * key);
static void gtk_matrix_size_request(GtkWidget * widget,
				   GtkRequisition * requisition);
static void gtk_matrix_size_allocate(GtkWidget * widget,
				    GtkAllocation * allocation);

/* Matrix queries */

static gint gtk_matrix_range_isvisible(GtkMatrix * matrix,
				      GtkMatrixRange range);
static gint gtk_matrix_cell_isvisible(GtkMatrix * matrix,
				     gint row, gint column);



/* Drawing Routines */

/* draw cell background and frame */
static void gtk_matrix_cell_draw_default(GtkMatrix * matrix,
					gint row, gint column);

/* draw cell border */
static void gtk_matrix_cell_draw_border(GtkMatrix * matrix,
				       gint row, gint column);

/* draw cell contents */
static void gtk_matrix_cell_draw_label(GtkMatrix * matrix,
				      gint row, gint column);

/* draw visible part of range. If range==NULL then draw the whole screen */
static void gtk_matrix_range_draw(GtkMatrix * matrix,
				 GtkMatrixRange * range);





/* Active Cell handling */

static void gtk_matrix_entry_changed(GtkWidget * widget,
				    gpointer data);
static gint gtk_matrix_deactivate_cell(GtkMatrix * matrix);
static void gtk_matrix_hide_active_cell(GtkMatrix * matrix);
static gint gtk_matrix_activate_cell(GtkMatrix * matrix,
				    gint row, gint col);
static void gtk_matrix_draw_active_cell(GtkMatrix * matrix);
static void gtk_matrix_show_active_cell(GtkMatrix * matrix);



static void gtk_matrix_click_cell(GtkMatrix *matrix,
				 gint      row,
				 gint      column,
				 gboolean *veto);


/* Backing Pixmap */
static void gtk_matrix_make_backing_pixmap(GtkMatrix * matrix,
					  gint width, gint height);
static void gtk_matrix_draw_backing_pixmap(GtkMatrix * matrix,
					  GtkMatrixRange range);
/* Scrollbars */

static void adjust_scrollbars(GtkMatrix * matrix);
static void vadjustment_changed(GtkAdjustment * adjustment,
				gpointer data);
static void hadjustment_changed(GtkAdjustment * adjustment,
				gpointer data);
static void vadjustment_value_changed(GtkAdjustment * adjustment,
				      gpointer data);
static void hadjustment_value_changed(GtkAdjustment * adjustment,
				      gpointer data);


static void 
draw_xor_vline(GtkMatrix *matrix);
static void 
draw_xor_hline(GtkMatrix *matrix);

static gint 
new_column_width(GtkMatrix *matrix,
		 gint      column,
		 gint     *x);


static gint 
new_row_height(GtkMatrix *matrix,
	       gint      row,
	       gint     *y);



static void gtk_matrix_set_entry(GtkMatrix *matrix,
				 GtkWidget *new_entry);

static void gtk_matrix_size_allocate_entry(GtkMatrix * matrix);


/* Matrix button gadgets */
static void 
size_allocate_column_title_buttons(GtkMatrix *matrix);

static void 
size_allocate_row_title_buttons(GtkMatrix *matrix);

static void 
size_allocate_pivot_window(GtkMatrix *matrix);


static void gtk_matrix_recalc_left_xpixels(GtkMatrix *matrix);
static void gtk_matrix_recalc_top_ypixels(GtkMatrix *matrix);



static void row_button_set(GtkMatrix * matrix,
			   gint row);
static void column_button_set(GtkMatrix * matrix,
			      gint column);
static void row_button_release(GtkMatrix * matrix,
			       gint row);
static void column_button_release(GtkMatrix * matrix,
				  gint column);
static void gtk_matrix_button_draw(GtkMatrix * matrix,
				  gint row, gint column);


/* Memory allocation routines */
static void gtk_matrix_real_range_clear(GtkMatrix * matrix,
				       GtkMatrixRange * range,
				       gboolean delete);

static void gtk_matrix_real_cell_clear(GtkMatrix * matrix,
				      gint row,
				      gint column,
				      gboolean delete);


static GtkMatrixCell *gtk_matrix_cell_new(void);
static gint AddRow(GtkMatrix * matrix, gint nrows);
static gint AddColumn(GtkMatrix * matrix, gint ncols);
static gint InsertRow(GtkMatrix * matrix, gint row, gint nrows);
static gint InsertColumn(GtkMatrix * matrix, gint col, gint ncols);
static gint DeleteRow(GtkMatrix * matrix, gint row, gint nrows);
static gint DeleteColumn(GtkMatrix * matrix, gint col, gint ncols);
static gint GrowMatrix(GtkMatrix * matrix,
		      gint newrows, gint newcols);
static gint CheckBounds(GtkMatrix * matrix,
			gint row, gint col);

/* Container Functions */
static void gtk_matrix_remove(GtkContainer * container,
			     GtkWidget * widget);
static void gtk_matrix_realize_child(GtkMatrix * matrix,
				    GtkMatrixChild * child);
static void gtk_matrix_position_child(GtkMatrix * matrix,
				     GtkMatrixChild * child);
static void gtk_matrix_position_children(GtkMatrix * matrix);

/* Signals */

enum {
	TRAVERSE,
	DEACTIVATE,
	ACTIVATE,
	SET_CELL,
	CLEAR_CELL,
	NEW_COL_WIDTH,
	NEW_ROW_HEIGHT,
	ENTRY_CHANGED,
	ROW_BUTTON_PRESS,
	LAST_SIGNAL
};


static void
gtk_matrix_marshal_BOOL__INT_INT(GtkObject * object,
				GtkSignalFunc func,
				gpointer func_data,
				GtkArg * args);
static void
gtk_matrix_marshal_BOOL__INT_INT_POINTER_POINTER(GtkObject * object,
						GtkSignalFunc func,
						gpointer func_data,
						GtkArg * args);

static GtkContainerClass *parent_class = NULL;
static guint matrix_signals[LAST_SIGNAL] =
{0};


guint
gtk_matrix_get_type()
{
	static guint matrix_type = 0;
	if (!matrix_type) {
		GtkTypeInfo matrix_info =
		{
			"GtkMatrix",
			sizeof(GtkMatrix),
			sizeof(GtkMatrixClass),
			(GtkClassInitFunc) gtk_matrix_class_init,
			(GtkObjectInitFunc) gtk_matrix_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		matrix_type = gtk_type_unique(GTK_TYPE_CONTAINER, &matrix_info);
	}
	return matrix_type;
}

static void gtk_matrix_class_init(GtkMatrixClass * klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkContainerClass *container_class;
	
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;
	container_class = (GtkContainerClass *) klass;
	
	parent_class = gtk_type_class(GTK_TYPE_CONTAINER);
	
	matrix_signals[TRAVERSE] =
		gtk_signal_new("traverse",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, traverse),
			       gtk_matrix_marshal_BOOL__INT_INT_POINTER_POINTER,
			       GTK_TYPE_BOOL, 4, GTK_TYPE_INT, GTK_TYPE_INT,
			       GTK_TYPE_POINTER, GTK_TYPE_POINTER);
	matrix_signals[DEACTIVATE] =
		gtk_signal_new("deactivate",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, deactivate),
			       gtk_matrix_marshal_BOOL__INT_INT,
			       GTK_TYPE_BOOL, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	matrix_signals[ACTIVATE] =
		gtk_signal_new("activate",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, activate),
			       gtk_matrix_marshal_BOOL__INT_INT,
			       GTK_TYPE_BOOL, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	matrix_signals[SET_CELL] =
		gtk_signal_new("set_cell",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, set_cell),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	matrix_signals[CLEAR_CELL] =
		gtk_signal_new("clear_cell",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, clear_cell),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	matrix_signals[NEW_COL_WIDTH] =
		gtk_signal_new("new_column_width",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass,new_column_width),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	matrix_signals[NEW_ROW_HEIGHT] =
		gtk_signal_new("new_row_height",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass,new_row_height),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	
	/* (Matias) */
	matrix_signals[ENTRY_CHANGED] =
		gtk_signal_new("entry_changed",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, entry_changed),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	
	matrix_signals[ROW_BUTTON_PRESS] =
		gtk_signal_new("row_button_press",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, row_button_press),
			       gtk_marshal_NONE__INT_INT,
			       GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	

	
	gtk_object_class_add_signals(object_class, matrix_signals, LAST_SIGNAL);
	
	container_class->add = NULL;
	container_class->remove = gtk_matrix_remove;
	container_class->forall = NULL;
	
	object_class->destroy = gtk_matrix_destroy;
	object_class->finalize = gtk_matrix_finalize;
	
	widget_class->set_scroll_adjustments_signal =
		gtk_signal_new("set_scroll_adjustments",
			       GTK_RUN_LAST,
			       object_class->type,
			       GTK_SIGNAL_OFFSET(GtkMatrixClass, set_scroll_adjustments),
			       gtk_marshal_NONE__POINTER_POINTER,
			       GTK_TYPE_NONE, 2, GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);
	
	widget_class->realize = gtk_matrix_realize;
	widget_class->unrealize = gtk_matrix_unrealize;
	widget_class->map = gtk_matrix_map;
	widget_class->unmap = gtk_matrix_unmap;
	widget_class->draw = gtk_matrix_draw;
	widget_class->style_set = gtk_matrix_style_set;
	widget_class->button_press_event = gtk_matrix_button_press;
	widget_class->button_release_event = gtk_matrix_button_release;
	widget_class->motion_notify_event = gtk_matrix_motion;
	widget_class->key_press_event = gtk_matrix_key_press;
	widget_class->expose_event = gtk_matrix_expose;
	widget_class->size_request = gtk_matrix_size_request;
	widget_class->size_allocate = gtk_matrix_size_allocate;
	widget_class->focus_in_event = NULL;
	widget_class->focus_out_event = NULL;

	klass->set_scroll_adjustments = gtk_matrix_set_scroll_adjustments;
	
	klass->traverse = NULL;
	klass->deactivate = NULL;
	klass->activate = NULL;
	klass->set_cell = NULL;
	klass->clear_cell = NULL;
	klass->entry_changed = NULL;
	
}

static void 
gtk_matrix_marshal_BOOL__INT_INT_POINTER_POINTER(GtkObject * object,
						 GtkSignalFunc func,
						 gpointer func_data,
						 GtkArg * args)
{
	GtkMatrixSignal1 rfunc;
	gboolean *veto;
	veto = GTK_RETLOC_BOOL(args[4]);
	
	rfunc = (GtkMatrixSignal1) func;
	
	*veto = (*rfunc) (object, GTK_VALUE_INT(args[0]),
			  GTK_VALUE_INT(args[1]),
			  GTK_VALUE_POINTER(args[2]),
			  GTK_VALUE_POINTER(args[3]),
			  func_data);
}


static void 
gtk_matrix_marshal_BOOL__INT_INT(GtkObject * object,
				 GtkSignalFunc func,
				 gpointer func_data,
				 GtkArg * args)
{
	GtkMatrixSignal2 rfunc;
	gboolean *veto;
	veto = GTK_RETLOC_BOOL(args[2]);
	
	rfunc = (GtkMatrixSignal2) func;
	
	*veto = (*rfunc) (object, GTK_VALUE_INT(args[0]),
			  GTK_VALUE_INT(args[1]),
			  func_data);
}




/**
 * gtk_matrix_emit__entry_changed:
 * matrix : The GtkMatrix
 *
 * Emits the signal "entry_changed", on the active cell.
 */
void
gtk_matrix_emit__entry_changed(GtkWidget *matrix)
{
	gint row,col;

	col = GTK_MATRIX(matrix)->active_cell.col;
	row = GTK_MATRIX(matrix)->active_cell.row;
	
	gtk_signal_emit(GTK_OBJECT(matrix),
			matrix_signals[ENTRY_CHANGED],
			row,col);
}



/**
 * gtk_matrix_emit__row_button_press:
 * matrix : The GtkMatrix
 *
 * Emits the signal "row_button_press", on the active cell.
 */
void
gtk_matrix_emit__row_button_press(GtkWidget  *matrix,
				  gint        row)
{
	gtk_signal_emit(GTK_OBJECT(matrix),
			matrix_signals[ROW_BUTTON_PRESS],
			row,-1);
}





/**
 * gtk_matrix_init:
 *
 * Intitializes the GtkMatrix.
 */
static void 
gtk_matrix_init(GtkMatrix *matrix)
{
	matrix->children = NULL;
	
	matrix->flags = 0;
	
	GTK_WIDGET_UNSET_FLAGS(matrix, GTK_NO_WINDOW);
	GTK_WIDGET_SET_FLAGS(matrix, GTK_CAN_FOCUS);
	
	matrix->maxrow = 0;
	matrix->maxcol = 0;

	matrix->view.row0 = 0;
	matrix->view.col0 = 0;
	matrix->view.rowi = 0;
	matrix->view.coli = 0;

	matrix->maxallocrow = 0;
	matrix->maxalloccol = 0;

	matrix->column_title_window = NULL;
	matrix->column_title_area.x = 0;
	matrix->column_title_area.y = 0;
	matrix->column_title_area.width = 0;
	matrix->column_title_area.height = 0;
	
	matrix->row_title_window = NULL;
	matrix->row_title_area.x = 0;
	matrix->row_title_area.y = 0;
	matrix->row_title_area.width = ROW_TITLE_WIDTH;
	matrix->row_title_area.height = 0;

	matrix->active_cell.row = 0;
	matrix->active_cell.col = 0;

	matrix->matrix_entry = NULL;
	matrix->pixmap = NULL;

	matrix->attributes = NULL;
	matrix->maxrange = 0;

	matrix->matrix_window = NULL;
	matrix->matrix_window_width = 0;
	matrix->matrix_window_height = 0;

	matrix->hoffset = 0;
	matrix->voffset = 0;

	matrix->hadjustment = NULL;
	matrix->vadjustment = NULL;
	
	matrix->cursor = gdk_cursor_new(GDK_PLUS);
	matrix->xor_gc = NULL;
	matrix->fg_gc = NULL;
	matrix->bg_gc = NULL;
	matrix->x_drag = 0;
	matrix->y_drag = 0;
}




/**
 * gtk_matrix_change_entry
 * @matrix  : A GtkMatrix widget
 * @widget : The entry to use for the editing, 
 *           can be a GtkCombo also.
 *
 * Change the widget beign used to edit the GtkMatrix.
 */
void 
gtk_matrix_change_entry(GtkMatrix *matrix, GtkWidget *widget)
{
	GtkWidget     *entry;
	gchar         *text;

	//g_print("gtk_matrix_change_entry\n");
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	
	entry = gtk_matrix_get_entry(matrix);
	gtk_signal_disconnect_by_func(GTK_OBJECT(entry),
				      (GtkSignalFunc) gtk_matrix_entry_changed,
				      GTK_OBJECT(GTK_WIDGET(matrix)));
	
	
	/* TMP : set the text in the cell */
	text = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
	gtk_matrix_hide_active_cell(matrix);
	
	gtk_matrix_set_entry(matrix,widget);
	
	gtk_matrix_show_active_cell(matrix);
	
	/* TMP : set the text in the entry */
	entry = gtk_matrix_get_entry(matrix);
	if (text == NULL)
		text = g_strdup("");
	gtk_entry_set_text(GTK_ENTRY(entry), text);
	
	gtk_signal_connect(GTK_OBJECT(entry),"changed",
			   GTK_SIGNAL_FUNC(gtk_matrix_entry_changed),
			   GTK_OBJECT(GTK_WIDGET(matrix)));
}





/*
 * gtk_matrix_optimal_column_width
 * @matrix : The GtkMatrix
 * @col   : A column in the GtkMatrix
 *
 * Return the optimal with of the GtkMatrix column. 
 */
gint
gtk_matrix_optimal_column_width(GtkMatrix *matrix,
			       gint      col)
{
	GtkMatrixCell **cell = NULL;
	gint text_width, row, optimal = 0;
	GtkMatrixCellAttr attributes;
	
	
	g_return_val_if_fail(matrix != NULL, 0);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), 0);
	if (col > matrix->maxcol)
		return 0;
	if (col < 0)
		return 0;
	
	for (row = 0; row <= matrix->maxrow; row++) {
		
		cell = &matrix->data[row][col];
		
		if ((*cell) != NULL) {
			attributes = (*cell)->attributes;
			
			if (attributes.is_visible) {
				
				text_width = gdk_string_width
					(MATRIX_FONT(matrix)
					 , (*cell)->text);
				
				optimal = MAX(optimal, text_width + 2 * CELLOFFSET);
			}
		}
	}
	return optimal;
}




void 
gtk_matrix_freeze(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_freeze\n");
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_IS_FROZEN);
}

void gtk_matrix_thaw(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_thaw\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	adjust_scrollbars(matrix);

	GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_IS_FROZEN);

	matrix->old_vadjustment = -1.;
	matrix->old_hadjustment = -1.;
	
	if (matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");
	if (matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");
	
}

void gtk_matrix_set_row_titles_width(GtkMatrix * matrix, gint width)
{
	//g_print("gtk_matrix_set_row_titles_width\n");

	if (width < COLUMN_MIN_WIDTH)
		return;
	
	matrix->row_title_area.width = width;
	matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, matrix->row_title_area.width + 1);
	matrix->view.coli = COLUMN_FROM_XPIXEL(matrix, matrix->matrix_window_width);
	
	adjust_scrollbars(matrix);
	
	matrix->old_hadjustment = -1.;
	if (matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");
}



void gtk_matrix_set_column_titles_height(GtkMatrix * matrix, gint height)
{
	//g_print("gtk_matrix_set_column_titles_height\n");

	if (height < DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix)))
		return;
	
	matrix->column_title_area.height = height;
	matrix->view.row0 = ROW_FROM_YPIXEL(matrix, MATRIX_COL_TITLE_HEIGHT(matrix) + 1);
	matrix->view.rowi = ROW_FROM_YPIXEL(matrix, matrix->matrix_window_height - 1);
	
	adjust_scrollbars(matrix);
	
	matrix->old_vadjustment = -1.;
	if (matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");
}

void gtk_matrix_show_column_titles(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_show_column_titles\n");

	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		return;
	
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_COL_TITLES_VISIBLE);
	
	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		gdk_window_show(matrix->column_title_window);
		adjust_scrollbars(matrix);
	}
	matrix->old_vadjustment = -1.;
	if (matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");
}

void gtk_matrix_show_row_titles(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_show_row_titles\n");
	
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		return;
	
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_ROW_TITLES_VISIBLE);

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		gdk_window_show(matrix->row_title_window);
		adjust_scrollbars(matrix);
	}
	matrix->old_hadjustment = -1.;
	if (matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");
}

void gtk_matrix_hide_column_titles(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_hide_column_titles\n");
	
	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		return;
	
	GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_COL_TITLES_VISIBLE);

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		if (matrix->column_title_window)
			gdk_window_hide(matrix->column_title_window);
		adjust_scrollbars(matrix);
	}
	matrix->old_vadjustment = -1.;
	if (matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");
}

void gtk_matrix_hide_row_titles(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_show_row_titles\n");

	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		return;
	
	GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_ROW_TITLES_VISIBLE);

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		if (matrix->row_title_window)
			gdk_window_hide(matrix->row_title_window);
		adjust_scrollbars(matrix);
	}
	matrix->old_hadjustment = -1.;
	if (matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");
}

void gtk_matrix_set_column_title(GtkMatrix * matrix,
				gint column,
				gchar * title)
{
	//g_print("gtk_matrix_set_column_title\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	
	if (matrix->column[column].name)
		g_free(matrix->column[column].name);
	
	matrix->column[column].name = g_strdup(title);
}

void gtk_matrix_set_row_title(GtkMatrix * matrix,
			     gint row,
			     gchar * title)
{
	//g_print("gtk_matrix_set_row_title\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (matrix->row[row].name)
		g_free(matrix->row[row].name);

	matrix->row[row].name = g_strdup(title);
}

void 
gtk_matrix_row_button_add_label(GtkMatrix *matrix, 
				gint      row, 
				gchar    *label)
{
	GtkMatrixButton *button;
	gint            label_height;
	gchar          *words;
	
	//g_print("gtk_matrix_row_button_add_label\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (row < 0 || row > matrix->maxrow)
		return;


	if (matrix->row[row].button.label)
		g_free(matrix->row[row].button.label);

	matrix->row[row].button.label = g_strdup(label);

	button = &matrix->row[row].button;
	label_height = 0;
	if (button->label && strlen(button->label) > 0) {
		words = button->label;
		while (words && *words != '\0') {
			if (*words == '\n' || *(words + 1) == '\0') {
				label_height += MATRIX_FONT(matrix)->ascent +
				    MATRIX_FONT(matrix)->descent;
			}
			words++;
		}
	}
	if (label_height + 2 * CELLOFFSET > MATRIX_COL_TITLE_HEIGHT(matrix))
		gtk_matrix_set_row_height(matrix, row, label_height + 2 * CELLOFFSET);
	
	if (!GTK_MATRIX_IS_FROZEN(matrix))
		gtk_matrix_button_draw(matrix, row, -1);
}




void gtk_matrix_column_button_add_label(GtkMatrix    *matrix, 
				       gint         column, 
				       gchar       *label)
{
	GtkMatrixButton *button;
	gint label_height = 0;
	gchar *words;

	//g_print("gtk_matrix_column_button_add_label\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (column < 0 || column > matrix->maxcol)
		return;

	if (matrix->column[column].button.label)
		g_free(matrix->column[column].button.label);


	matrix->column[column].button.label = g_strdup(label);

	button = &matrix->column[column].button;
	if (button->label && strlen(button->label) > 0) {
		words = button->label;
		while (words && *words != '\0') {
			if (*words == '\n' || *(words + 1) == '\0') {
				label_height += MATRIX_FONT(matrix)->ascent +
				    MATRIX_FONT(matrix)->descent;
			}
			words++;
		}
	}
	if (label_height + 2 * CELLOFFSET > MATRIX_COL_TITLE_HEIGHT(matrix))
		gtk_matrix_set_column_titles_height(matrix, label_height + 2 * CELLOFFSET);
	
	if (!GTK_MATRIX_IS_FROZEN(matrix))
		gtk_matrix_button_draw(matrix, -1, column);
}




void gtk_matrix_column_set_sensitivity(GtkMatrix * matrix, gint column, gint sensitive)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (column < 0 || column > matrix->maxcol)
		return;

	matrix->column[column].is_sensitive = sensitive;
	if (!sensitive)
		matrix->column[column].button.state = GTK_STATE_INSENSITIVE;
	else
		matrix->column[column].button.state = GTK_STATE_NORMAL;

	if (GTK_WIDGET_REALIZED(matrix) && !GTK_MATRIX_IS_FROZEN(matrix))
		gtk_matrix_button_draw(matrix, -1, column);
}

void gtk_matrix_columns_set_sensitivity(GtkMatrix * matrix, gint sensitive)
{
	gint i;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	for (i = 0; i <= matrix->maxcol; i++)
		gtk_matrix_column_set_sensitivity(matrix, i, sensitive);
}

void gtk_matrix_row_set_sensitivity(GtkMatrix * matrix, gint row, gint sensitive)
{

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (row < 0 || row > matrix->maxrow)
		return;

	matrix->row[row].is_sensitive = sensitive;
	if (!sensitive)
		matrix->row[row].button.state = GTK_STATE_INSENSITIVE;
	else
		matrix->row[row].button.state = GTK_STATE_NORMAL;

	if (GTK_WIDGET_REALIZED(matrix) && !GTK_MATRIX_IS_FROZEN(matrix))
		gtk_matrix_button_draw(matrix, row, -1);
}



void gtk_matrix_rows_set_sensitivity(GtkMatrix * matrix, gint sensitive)
{
	gint i;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	for (i = 0; i <= matrix->maxrow; i++)
		gtk_matrix_row_set_sensitivity(matrix, i, sensitive);
}



void gtk_matrix_column_set_visibility(GtkMatrix * matrix, gint column, gint visible)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (column < 0 || column > matrix->maxcol)
		return;
	if (matrix->column[column].is_visible == visible)
		return;

	matrix->column[column].is_visible = visible;
	
	if (!GTK_MATRIX_IS_FROZEN(matrix) &&
	gtk_matrix_cell_isvisible(matrix, MIN_VISIBLE_ROW(matrix), column)) {
		gtk_matrix_range_draw(matrix, NULL);
		size_allocate_pivot_window(matrix);
		size_allocate_column_title_buttons(matrix);
	}
}



void gtk_matrix_row_set_visibility(GtkMatrix * matrix, gint row, gint visible)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (row < 0 || row > matrix->maxrow)
		return;
	if (matrix->row[row].is_visible == visible)
		return;

	matrix->row[row].is_visible = visible;
	
	if (!GTK_MATRIX_IS_FROZEN(matrix) &&
	    gtk_matrix_cell_isvisible(matrix, row, MIN_VISIBLE_COLUMN(matrix))) {
		gtk_matrix_range_draw(matrix, NULL);
		size_allocate_pivot_window(matrix);
		size_allocate_row_title_buttons(matrix);
	}
}




static gint
gtk_matrix_range_isvisible(GtkMatrix * matrix,
			   GtkMatrixRange range)
{
	//g_print("gtk_matrix_range_isvisible\n");

	g_return_val_if_fail(matrix != NULL, FALSE);

	if (range.row0 < 0 || range.row0 > matrix->maxrow)
		return FALSE;

	if (range.rowi < 0 || range.rowi > matrix->maxrow)
		return FALSE;

	if (range.col0 < 0 || range.col0 > matrix->maxcol)
		return FALSE;

	if (range.coli < 0 || range.coli > matrix->maxcol)
		return FALSE;

	if (range.rowi < MIN_VISIBLE_ROW(matrix))
		return FALSE;

	if (range.row0 > MAX_VISIBLE_ROW(matrix))
		return FALSE;

	if (range.coli < MIN_VISIBLE_COLUMN(matrix))
		return FALSE;

	if (range.col0 > MAX_VISIBLE_COLUMN(matrix))
		return FALSE;

	return TRUE;
}


static gint
gtk_matrix_cell_isvisible(GtkMatrix * matrix,
			 gint row, gint column)
{
	GtkMatrixRange range;

	//g_print("gtk_matrix_cell_isvisible\n");

	range.row0 = row;
	range.col0 = column;
	range.rowi = row;
	range.coli = column;
	
	return gtk_matrix_range_isvisible(matrix, range);
}

void gtk_matrix_get_visible_range(GtkMatrix * matrix, GtkMatrixRange * range)
{
	//g_print("gtk_matrix_get_visible_range\n");
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	g_return_if_fail(range != NULL);

	range->row0 = MIN_VISIBLE_ROW(matrix);
	range->col0 = MIN_VISIBLE_COLUMN(matrix);
	range->rowi = MAX_VISIBLE_ROW(matrix);
	range->coli = MAX_VISIBLE_COLUMN(matrix);

}

GtkAdjustment *
gtk_matrix_get_vadjustment(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_get_vadjustment\n");

	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);
	
	return matrix->vadjustment;
}

GtkAdjustment *
gtk_matrix_get_hadjustment(GtkMatrix * matrix)
{
	//g_print("gtk_matrix_get_hadjustment\n");

	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);

	return matrix->hadjustment;
}

void gtk_matrix_set_vadjustment(GtkMatrix * matrix,
			       GtkAdjustment * adjustment)
{
	GtkAdjustment *old_adjustment;

	//g_print("gtk_matrix_set_vadjustment\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (adjustment)
		g_return_if_fail(GTK_IS_ADJUSTMENT(adjustment));

	if (matrix->vadjustment == adjustment)
		return;

	old_adjustment = matrix->vadjustment;

	if (matrix->vadjustment) {
		gtk_signal_disconnect_by_data(GTK_OBJECT(matrix->vadjustment), matrix);
		gtk_object_unref(GTK_OBJECT(matrix->vadjustment));
	}
	matrix->vadjustment = adjustment;

	if (matrix->vadjustment) {
		gtk_object_ref(GTK_OBJECT(matrix->vadjustment));
		gtk_object_sink(GTK_OBJECT(matrix->vadjustment));

		gtk_signal_connect(GTK_OBJECT(matrix->vadjustment), "changed",
				   (GtkSignalFunc) vadjustment_changed,
				   (gpointer) matrix);
		gtk_signal_connect(GTK_OBJECT(matrix->vadjustment), "value_changed",
			       (GtkSignalFunc) vadjustment_value_changed,
				   (gpointer) matrix);
	}
	if (!matrix->vadjustment || !old_adjustment) {
		gtk_widget_queue_resize(GTK_WIDGET(matrix));
		return;
	}
	matrix->old_vadjustment = matrix->vadjustment->value;
}

void gtk_matrix_set_hadjustment(GtkMatrix * matrix,
			       GtkAdjustment * adjustment)
{
	GtkAdjustment *old_adjustment;

	//g_print("gtk_matrix_set_hadjustment\n");

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (adjustment)
		g_return_if_fail(GTK_IS_ADJUSTMENT(adjustment));

	if (matrix->hadjustment == adjustment)
		return;

	old_adjustment = matrix->hadjustment;

	if (matrix->hadjustment) {
		gtk_signal_disconnect_by_data(GTK_OBJECT(matrix->hadjustment), matrix);
		gtk_object_unref(GTK_OBJECT(matrix->hadjustment));
	}
	matrix->hadjustment = adjustment;

	if (matrix->hadjustment) {
		gtk_object_ref(GTK_OBJECT(matrix->hadjustment));
		gtk_object_sink(GTK_OBJECT(matrix->hadjustment));

		gtk_signal_connect(GTK_OBJECT(matrix->hadjustment), "changed",
				   (GtkSignalFunc) hadjustment_changed,
				   (gpointer) matrix);
		gtk_signal_connect(GTK_OBJECT(matrix->hadjustment), "value_changed",
			       (GtkSignalFunc) hadjustment_value_changed,
				   (gpointer) matrix);
	}
	if (!matrix->hadjustment || !old_adjustment) {
		gtk_widget_queue_resize(GTK_WIDGET(matrix));
		return;
	}
	matrix->old_hadjustment = matrix->hadjustment->value;
}

static void gtk_matrix_set_scroll_adjustments(GtkMatrix * matrix,
					     GtkAdjustment * hadjustment,
					     GtkAdjustment * vadjustment)
{
	//g_print("gtk_matrix_set_scroll_adjustments\n");

	if (matrix->hadjustment != hadjustment)
		gtk_matrix_set_hadjustment(matrix, hadjustment);
	if (matrix->vadjustment != vadjustment)
		gtk_matrix_set_vadjustment(matrix, vadjustment);
}

static void gtk_matrix_destroy(GtkObject * object)
{
	GtkMatrix *matrix;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTK_IS_MATRIX(object));

	matrix = GTK_MATRIX(object);

	/* get rid of all the cells */
	gtk_matrix_range_clear(matrix, NULL);
/*
	if (matrix->timer) {
		gtk_timeout_remove(matrix->timer);
		matrix->timer = 0;
	}
*/
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
	
}

static void gtk_matrix_finalize(GtkObject * object)
{

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTK_IS_MATRIX(object));

	if (GTK_OBJECT_CLASS(parent_class)->finalize)
		(*GTK_OBJECT_CLASS(parent_class)->finalize) (object);

}

static void gtk_matrix_style_set(GtkWidget *widget,
				GtkStyle  *previous_style)
{
	GtkMatrix *matrix;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));

	if (GTK_WIDGET_CLASS(parent_class)->style_set)
		(*GTK_WIDGET_CLASS(parent_class)->style_set) (widget, previous_style);

	matrix = GTK_MATRIX(widget);

	if (GTK_WIDGET_REALIZED(widget)) {
		widget->style = gtk_style_attach(widget->style, widget->window);
		gtk_style_set_background(widget->style, widget->window, widget->state);
	}
}





static void
gtk_matrix_realize(GtkWidget * widget)
{
	GtkMatrix        *matrix;
	GdkWindowAttr    attributes;
	gint             attributes_mask;
	GdkGCValues      values, auxvalues;
	GdkColormap     *colormap;
	GtkMatrixChild   *child;
	GList           *children;

	//g_print("gtk_matrix_realize\n");
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_MATRIX (widget));
	
	matrix = GTK_MATRIX (widget);
	
	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
	
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);
	
	attributes.event_mask = gtk_widget_get_events (widget);
	attributes.event_mask |= (GDK_EXPOSURE_MASK |
				  GDK_BUTTON_PRESS_MASK |
				  GDK_BUTTON_RELEASE_MASK |
				  GDK_KEY_PRESS_MASK |
				  GDK_POINTER_MOTION_MASK |
				  GDK_POINTER_MOTION_HINT_MASK);
	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP |
		GDK_WA_CURSOR;
	
	attributes.cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);
	
	/* main window */
	widget->window = gdk_window_new(gtk_widget_get_parent_window(widget),
					&attributes, attributes_mask);
	
	gdk_window_set_user_data(widget->window,matrix);
	
	widget->style = gtk_style_attach (widget->style, widget->window);
	
	gtk_style_set_background(widget->style,widget->window,GTK_STATE_NORMAL);
	
	
	/* column-title window */
	attributes.x = 0;
	if(GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		attributes.x = matrix->row_title_area.width;
	attributes.y = 0;
	attributes.width = matrix->column_title_area.width;
	attributes.height = MATRIX_COL_TITLE_HEIGHT(matrix);
	
	matrix->column_title_window = gdk_window_new(widget->window,&attributes,
						    attributes_mask);
	gdk_window_set_user_data(matrix->column_title_window, matrix);
	gtk_style_set_background(widget->style,matrix->column_title_window,GTK_STATE_NORMAL);
	
	/* row-title window */
	attributes.x = 0;
	attributes.y = 0;
	if(GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		attributes.y = MATRIX_COL_TITLE_HEIGHT(matrix);
	attributes.width  = matrix->row_title_area.width;
	attributes.height = matrix->row_title_area.height;
	
	matrix->row_title_window = gdk_window_new(widget->window,&attributes,
						 attributes_mask);
	gdk_window_set_user_data (matrix->row_title_window, matrix);
	gtk_style_set_background (widget->style,matrix->row_title_window,GTK_STATE_NORMAL);
	

	/* pivot window */
	attributes.x = 0;
	attributes.y = 0;
	attributes.height = ( GTK_MATRIX_COL_TITLES_VISIBLE(matrix) ?
			      MATRIX_COL_TITLE_HEIGHT(matrix) : 0 );
	
	attributes.width =  ( GTK_MATRIX_ROW_TITLES_VISIBLE(matrix) ?
			      matrix->row_title_area.width : 0 );
	
	matrix->pivot_window = gdk_window_new(widget->window,&attributes,
					     attributes_mask);
	gdk_window_set_user_data(matrix->pivot_window, matrix);
	gtk_style_set_background(widget->style,matrix->pivot_window,GTK_STATE_NORMAL);
	

	/* matrix-window */
	attributes.cursor = gdk_cursor_new(GDK_PLUS);
	
	attributes.width = matrix->matrix_window_width;
	attributes.height = matrix->matrix_window_height;
	attributes.y = 0;
	attributes.x = 0;
	
	matrix->matrix_window = gdk_window_new(widget->window,&attributes,attributes_mask);
	gdk_window_set_user_data(matrix->matrix_window, matrix);
	gdk_window_show (matrix->matrix_window);
	
	/* backing_pixmap */
	gtk_matrix_make_backing_pixmap(matrix, 0, 0);  
	
	/* GCs */
	if(matrix->fg_gc) 
		gdk_gc_unref(matrix->fg_gc);
	if(matrix->bg_gc) 
		gdk_gc_unref(matrix->bg_gc);
	matrix->fg_gc = gdk_gc_new (widget->window);
	matrix->bg_gc = gdk_gc_new (widget->window);
	
	colormap = gtk_widget_get_colormap(widget);
	
	gdk_color_white(colormap, &widget->style->white);
	gdk_color_black(colormap, &widget->style->black);
	
	gdk_gc_get_values(matrix->fg_gc, &auxvalues);
	
	values.foreground = widget->style->white;
	values.function = GDK_INVERT;
	values.subwindow_mode = GDK_INCLUDE_INFERIORS;
	if(matrix->xor_gc)
		gdk_gc_unref(matrix->xor_gc);
	matrix->xor_gc = gdk_gc_new_with_values(widget->window,
					       &values,
					       GDK_GC_FOREGROUND |
					       GDK_GC_FUNCTION |
					       GDK_GC_SUBWINDOW);
	
	/* create matrix_entry_window */
	
	if(GTK_WIDGET_NO_WINDOW(matrix->matrix_entry)){
		
		attributes.window_type = GDK_WINDOW_CHILD;
		attributes.x = 0;
		attributes.y = 0;
		attributes.width = matrix->matrix_entry->requisition.width;
		attributes.height = matrix->matrix_entry->requisition.height;
		attributes.wclass = GDK_INPUT_OUTPUT;
		attributes.visual = gtk_widget_get_visual (widget);
		attributes.colormap = gtk_widget_get_colormap (widget);
		attributes.event_mask = GDK_EXPOSURE_MASK;
		
		attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
		matrix->matrix_entry_window =  gdk_window_new (matrix->matrix_window,
							     &attributes, attributes_mask);
		gdk_window_set_user_data (matrix->matrix_entry_window, widget);
	}
	
	if(matrix->matrix_entry->parent){
		gtk_widget_ref(matrix->matrix_entry);
		gtk_widget_unparent(matrix->matrix_entry);
	}
	gtk_widget_set_parent(matrix->matrix_entry, GTK_WIDGET(matrix));
	
	gtk_widget_set_parent_window (matrix->matrix_entry,
				      matrix->matrix_entry_window ? 
				      matrix->matrix_entry_window : 
				      matrix->matrix_window);
	
	gtk_matrix_activate_cell(matrix,
				 matrix->active_cell.row,
				 matrix->active_cell.col);
	
	if(!matrix->cursor)
		matrix->cursor = gdk_cursor_new(GDK_PLUS);
	

	/* HERE ??? */
	size_allocate_row_title_buttons(matrix);
	size_allocate_column_title_buttons(matrix);
	if(GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		gdk_window_show(matrix->column_title_window);
	if(GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		gdk_window_show(matrix->row_title_window);
	
	size_allocate_pivot_window(matrix);
	if( GTK_MATRIX_COL_TITLES_VISIBLE(matrix) &&
	    GTK_MATRIX_ROW_TITLES_VISIBLE(matrix) )
		gdk_window_show(matrix->pivot_window);

	
	children = matrix->children;
	while(children)
	{
		child = children->data;
		children = children->next;
		
		gtk_matrix_realize_child(matrix, child);
	}
}


static void gtk_matrix_unrealize(GtkWidget * widget)
{
	GtkMatrix *matrix;
	GtkMatrixChild *child;
	GList *children;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));

	matrix = GTK_MATRIX(widget);
	GTK_WIDGET_UNSET_FLAGS(widget, GTK_REALIZED | GTK_MAPPED);

	gdk_cursor_destroy(matrix->cursor);
	gdk_gc_destroy(matrix->xor_gc);
	gdk_gc_destroy(matrix->fg_gc);
	gdk_gc_destroy(matrix->bg_gc);

	gtk_style_detach(widget->style);

	gdk_window_destroy(matrix->matrix_window);
	gdk_window_destroy(matrix->column_title_window);
	gdk_window_destroy(matrix->row_title_window);
	gdk_window_set_user_data(widget->window, NULL);
	gdk_window_destroy(widget->window);

	if (matrix->pixmap)
		gdk_pixmap_unref(matrix->pixmap);

	widget->window = NULL;
	matrix->column_title_window = NULL;
	matrix->matrix_window = NULL;
	matrix->matrix_entry_window = NULL;
	matrix->cursor = NULL;
	matrix->xor_gc = NULL;
	matrix->fg_gc = NULL;
	matrix->bg_gc = NULL;

	children = matrix->children;
	while (children) {
		child = children->data;
		children = children->next;

		if (child->window) {
			gdk_window_set_user_data(child->window, NULL);
			gdk_window_destroy(child->window);
			child->window = NULL;
		}
	}

}

static void
gtk_matrix_map (GtkWidget * widget)
{
	GtkMatrix *matrix;
	GtkMatrixChild *child;
	GList *children;

	//g_print("gtk_matrix_map\n");
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_MATRIX (widget));
	
	matrix = GTK_MATRIX (widget);
	
	if (!GTK_WIDGET_MAPPED (widget))
	{
		GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
		
		if(!matrix->cursor) 
			matrix->cursor = gdk_cursor_new(GDK_PLUS);
		
		gdk_window_show (widget->window);
		
		gdk_window_set_background(matrix->matrix_window,
					  &widget->style->bg[GTK_STATE_NORMAL]);
		gdk_window_show (matrix->matrix_window);
		
		if(matrix->matrix_entry_window)
			gdk_window_show(matrix->matrix_entry_window);
		
		if(GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
			gdk_window_show (matrix->column_title_window);
		
		if(GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			gdk_window_show (matrix->row_title_window);
		
		if( GTK_MATRIX_COL_TITLES_VISIBLE(matrix) && 
		    GTK_MATRIX_ROW_TITLES_VISIBLE(matrix) )
			gdk_window_show (matrix->pivot_window);
		
		if(!GTK_WIDGET_MAPPED (matrix->matrix_entry)){
			gtk_widget_show (matrix->matrix_entry);
			gtk_widget_map (matrix->matrix_entry);
			gtk_matrix_hide_active_cell(matrix);
		}
		
		gtk_matrix_range_draw(matrix, NULL);
		
		children = matrix->children;
		while (children)
		{
			child = children->data;
			children = children->next;
			
			if (GTK_WIDGET_VISIBLE (child->widget) &&
			    !GTK_WIDGET_MAPPED (child->widget)){
				gtk_widget_map (child->widget);
				gtk_widget_size_allocate(child->widget,  
							 &child->widget->allocation);
				if (GTK_WIDGET_NO_WINDOW(child->widget) && child->window) 
					gdk_window_show(child->window);
			}
		}
	}
}


static void gtk_matrix_unmap(GtkWidget * widget)
{
	GtkMatrix *matrix;
	GtkMatrixChild *child;
	GList *children;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));

	matrix = GTK_MATRIX(widget);

	if (GTK_WIDGET_MAPPED(widget)) {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_MAPPED);
		
		gdk_window_hide(matrix->matrix_window);
		if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
			gdk_window_hide(matrix->column_title_window);
		if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			gdk_window_hide(matrix->row_title_window);
		gdk_window_hide(widget->window);
		
		if (matrix->matrix_entry_window)
			gdk_window_hide(matrix->matrix_entry_window);

		if (GTK_WIDGET_MAPPED(matrix->matrix_entry))
			gtk_widget_unmap(matrix->matrix_entry);

		children = matrix->children;
		while (children) {
			child = children->data;
			children = children->next;

			if (GTK_WIDGET_VISIBLE(child->widget) &&
			    GTK_WIDGET_MAPPED(child->widget)) {
				gtk_widget_unmap(child->widget);
				if (child->window)
					gdk_window_hide(child->window);
			}
		}

	}
}


static void 
gtk_matrix_cell_draw_default(GtkMatrix *matrix,
			    gint      row,
			    gint      col)
{
	GtkWidget     *widget;
	GdkGC         *fg_gc, *bg_gc;
	GdkRectangle   area;
	
	g_return_if_fail(matrix != NULL);

	/* bail now if we arn't drawable yet */
	if (!GTK_WIDGET_DRAWABLE(matrix))
		return;

	if (row < 0 || row > matrix->maxrow)
		return;
	if (col < 0 || col > matrix->maxcol)
		return;
	if (!matrix->column[col].is_visible)
		return;
	if (!matrix->row[row].is_visible)
		return;

	widget = GTK_WIDGET(matrix);

	/* select GC for background rectangle */
	gdk_gc_set_foreground(matrix->fg_gc,&(GTK_WIDGET(matrix)->style->black));
	gdk_gc_set_foreground(matrix->bg_gc,&(GTK_WIDGET(matrix)->style->white));
	
	fg_gc = matrix->fg_gc;
	bg_gc = matrix->bg_gc;

	area.x = COLUMN_LEFT_XPIXEL(matrix, col);
	area.y = ROW_TOP_YPIXEL(matrix, row);
	area.width = matrix->column[col].width;
	area.height = MATRIX_ROW_HEIGHT(matrix,row);

 	gdk_draw_rectangle(matrix->pixmap,
	  		   bg_gc,
		 	   TRUE,
			   area.x,
			   area.y,
			   area.width,
			   area.height);

	gdk_gc_set_line_attributes(matrix->fg_gc, 1, 0, 0, 0);
}






/**
 * gtk_matrix_cell_draw_border:
 * @matrix  : The GtkMatrix
 * @row    : The row of a cell
 * @col    : The col of a cell
 *
 * Draws the borders for the cell placed at (@row,@col).
 */
static void 
gtk_matrix_cell_draw_border(GtkMatrix    *matrix, 
			   gint         row, 
			   gint         col)
{
	GtkWidget *widget;
	GdkGC *fg_gc, *bg_gc;
	GdkRectangle area;
		
	g_return_if_fail(matrix != NULL);

	/* bail now if we arn't drawable yet */
	if (!GTK_WIDGET_DRAWABLE(matrix))
		return;

	if (row < 0 || row > matrix->maxrow)
		return;
	if (col < 0 || col > matrix->maxcol)
		return;
	if (!matrix->column[col].is_visible)
		return;
	if (!matrix->row[row].is_visible)
		return;

	widget = GTK_WIDGET(matrix);

	/* select GC for background rectangle */
	gdk_gc_set_foreground(matrix->fg_gc,&(GTK_WIDGET(matrix)->style->black));
	gdk_gc_set_foreground(matrix->bg_gc,&(GTK_WIDGET(matrix)->style->white));
	
	fg_gc = matrix->fg_gc;
	bg_gc = matrix->bg_gc;

	area.x = COLUMN_LEFT_XPIXEL(matrix, col);
	area.y = ROW_TOP_YPIXEL(matrix, row);
	area.width = matrix->column[col].width;
	area.height = MATRIX_ROW_HEIGHT(matrix,row);

	gdk_gc_set_line_attributes(matrix->fg_gc, 2,
				   GDK_LINE_SOLID,
				   GDK_CAP_NOT_LAST,
				   GDK_JOIN_MITER);
	
	gdk_draw_line(matrix->pixmap, matrix->fg_gc,
		      area.x , area.y - 1,
		      area.x , area.y + area.height + 1);
	
	gdk_draw_line(matrix->pixmap, matrix->fg_gc,
		      area.x + area.width, area.y - 1,
		      area.x + area.width,
		      area.y + area.height + 1);
	
	gdk_draw_line(matrix->pixmap, matrix->fg_gc,
		      area.x - 1 , area.y,
		      area.x + area.width + 1,
		      area.y);
	
	gdk_draw_line(matrix->pixmap, matrix->fg_gc,
		      area.x - 1 , area.y + area.height,
		      area.x + area.width + 1,
		      area.y + area.height);
}



static void 
gtk_matrix_cell_draw_label(GtkMatrix  *matrix, 
			   gint        row, 
			   gint        col)
{
	GdkRectangle area, clip_area;
	gint         text_width, text_height, y;
	gint         xoffset = 1;
	GdkGC       *fg_gc, *bg_gc;
	gchar       *label;


	g_return_if_fail(matrix != NULL);

	/* bail now if we arn't drawable yet */
	if (!GTK_WIDGET_DRAWABLE(matrix))
		return;

	if (row > matrix->maxallocrow)
		return;
	if (col > matrix->maxalloccol)
		return;
	if (!matrix->data[row][col])
		return;
	if (!matrix->data[row][col]->text || strlen(matrix->data[row][col]->text) == 0)
		return;

	if (row < 0 || row > matrix->maxrow)
		return;
	if (col < 0 || col > matrix->maxcol)
		return;
	if (!matrix->column[col].is_visible)
		return;
	if (!matrix->row[row].is_visible)
		return;


	label = matrix->data[row][col]->text;
		
	/* select GC for background rectangle */
	gdk_gc_set_foreground(matrix->fg_gc,&(GTK_WIDGET(matrix)->style->black));
	gdk_gc_set_foreground(matrix->bg_gc,&(GTK_WIDGET(matrix)->style->white));
	gdk_gc_set_font(matrix->fg_gc, MATRIX_FONT(matrix));
	
	fg_gc = matrix->fg_gc;
	bg_gc = matrix->bg_gc;

	area.x = COLUMN_LEFT_XPIXEL(matrix, col);
	area.y = ROW_TOP_YPIXEL(matrix, row);
	area.width = matrix->column[col].width;
	area.height = MATRIX_ROW_HEIGHT(matrix,row);;

	clip_area = area;
	
	text_width = gdk_string_width(MATRIX_FONT(matrix), label);
	text_height = MATRIX_FONT(matrix)->ascent + MATRIX_FONT(matrix)->descent;
	y = area.y + area.height - CELLOFFSET;
	y = y - text_height + MATRIX_FONT(matrix)->ascent;
	
	gdk_gc_set_clip_rectangle(fg_gc, &clip_area);
	gdk_draw_string(matrix->pixmap,
			MATRIX_FONT(matrix),
			fg_gc,
			area.x + xoffset + CELLOFFSET,
			y,
			label);

	gdk_gc_set_clip_rectangle(fg_gc, NULL);
	gdk_draw_pixmap(matrix->matrix_window,
			GTK_WIDGET(matrix)->style->fg_gc[GTK_STATE_NORMAL],
			matrix->pixmap,
			area.x,
			area.y,
			area.x,
			area.y,
			area.width,
			area.height);
}



static void 
gtk_matrix_range_draw(GtkMatrix      *matrix,
		     GtkMatrixRange *range)
{
	gint i, j;
	GtkMatrixRange drawing_range;
	GdkRectangle area;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_MATRIX(matrix));

	if (!GTK_WIDGET_DRAWABLE(GTK_WIDGET(matrix)))
		return;
	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;

	if (range == NULL) {
		
		drawing_range.row0 = MIN_VISIBLE_ROW(matrix);
		drawing_range.col0 = MIN_VISIBLE_COLUMN(matrix);
		drawing_range.rowi = MAX_VISIBLE_ROW(matrix);
		drawing_range.coli = MAX_VISIBLE_COLUMN(matrix);
		
		gdk_draw_rectangle(matrix->pixmap,
				   GTK_WIDGET(matrix)->style->bg_gc[GTK_STATE_NORMAL],
				   TRUE, 0, 0,
				   matrix->matrix_window_width, 
				   matrix->matrix_window_height);
		
	} else {
		drawing_range.row0 = MAX(range->row0, MIN_VISIBLE_ROW(matrix));
		drawing_range.col0 = MAX(range->col0, MIN_VISIBLE_COLUMN(matrix));
		drawing_range.rowi = MIN(range->rowi, MAX_VISIBLE_ROW(matrix));
		drawing_range.coli = MIN(range->coli, MAX_VISIBLE_COLUMN(matrix));
	}

	gtk_matrix_recalc_left_xpixels(matrix);
	gtk_matrix_recalc_top_ypixels(matrix);
	
	if (drawing_range.coli == matrix->maxcol) {
		
		area.x = COLUMN_LEFT_XPIXEL(matrix, matrix->maxcol) +
			matrix->column[matrix->maxcol].width + 1;
		
		area.y = 0; 
		gdk_draw_rectangle(matrix->pixmap,
				   GTK_WIDGET(matrix)->style->bg_gc[GTK_STATE_NORMAL],
				   TRUE,
				   area.x, area.y,
				   matrix->matrix_window_width - area.x,
				   matrix->matrix_window_height);
		
		gdk_draw_pixmap(matrix->matrix_window,
				GTK_WIDGET(matrix)->style->bg_gc[GTK_STATE_NORMAL],
				matrix->pixmap,
				area.x,
				area.y,
				area.x,
				area.y,
				matrix->matrix_window_width - area.x,
				matrix->matrix_window_height);
	}
	
	if (drawing_range.rowi == matrix->maxrow) {
		area.x = 0;
		area.y = ROW_TOP_YPIXEL(matrix, matrix->maxrow) + 
			MATRIX_ROW_HEIGHT(matrix,matrix->maxrow) + 1;
		
		gdk_draw_rectangle(matrix->pixmap,
				   GTK_WIDGET(matrix)->style->bg_gc[GTK_STATE_NORMAL],
				   TRUE,
				   area.x, area.y,
				   matrix->matrix_window_width,
				   matrix->matrix_window_height - area.y);

		gdk_draw_pixmap(matrix->matrix_window,
				GTK_WIDGET(matrix)->style->bg_gc[GTK_STATE_NORMAL],
				matrix->pixmap,
				area.x,
				area.y,
				area.x,
				area.y,
				matrix->matrix_window_width,
				matrix->matrix_window_height - area.y);
	}

	for (i = drawing_range.row0; i <= drawing_range.rowi; i++)
		for (j = drawing_range.col0; j <= drawing_range.coli; j++) {
			gtk_matrix_cell_draw_default(matrix, i, j);
		}

	for (i = drawing_range.row0; i <= drawing_range.rowi; i++)
		for (j = drawing_range.col0; j <= drawing_range.coli; j++) {
			gtk_matrix_cell_draw_border(matrix, i , j);
		}
	
	for (i = drawing_range.row0; i <= drawing_range.rowi; i++)
		for (j = drawing_range.col0; j <= drawing_range.coli; j++)
			if (i <= matrix->maxallocrow && j <= matrix->maxalloccol && matrix->data[i][j])
				gtk_matrix_cell_draw_label(matrix, i, j);
	
	for (i = drawing_range.row0; i <= drawing_range.rowi; i++)
		for (j = matrix->column[drawing_range.col0].left_text_column; j < drawing_range.col0; j++)
			if (i <= matrix->maxallocrow && j <= matrix->maxalloccol && matrix->data[i][j])
				gtk_matrix_cell_draw_label(matrix, i, j);

	for (i = drawing_range.row0; i <= drawing_range.rowi; i++)
		for (j = drawing_range.coli + 1; j <= matrix->column[drawing_range.coli].right_text_column; j++)
			if (i <= matrix->maxallocrow && j <= matrix->maxalloccol && matrix->data[i][j])
				gtk_matrix_cell_draw_label(matrix, i, j);

	gtk_matrix_draw_backing_pixmap(matrix,drawing_range);
	
	if ( matrix->active_cell.row >= drawing_range.row0 &&
	     matrix->active_cell.row <= drawing_range.rowi &&
	     matrix->active_cell.col >= drawing_range.col0 &&
	     matrix->active_cell.col <= drawing_range.coli)
		gtk_matrix_show_active_cell(matrix);
}



static void 
gtk_matrix_draw_backing_pixmap(GtkMatrix * matrix, GtkMatrixRange range)
{
	gint x, y, width, height;

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;
	
	x = COLUMN_LEFT_XPIXEL(matrix, range.col0);
	y = ROW_TOP_YPIXEL(matrix, range.row0);
	width = COLUMN_LEFT_XPIXEL(matrix, range.coli) - x + 
		matrix->column[range.coli].width;
	height = ROW_TOP_YPIXEL(matrix, range.rowi) - y + 
		MATRIX_ROW_HEIGHT(matrix,range.rowi);
	
	x = MAX(x,0);
	y = MAX(y,0);
	width = MIN(width, matrix->matrix_window_width - x);
	height = MIN(height, matrix->matrix_window_height - y);
	
	x--;
	y--;
	width += 2;
	height += 2;
	
	if (range.coli == matrix->maxcol)
		width = matrix->matrix_window_width - x;
	if (range.rowi == matrix->maxrow)
		height = matrix->matrix_window_height - y;

	gdk_draw_pixmap(matrix->matrix_window,
			GTK_WIDGET(matrix)->style->fg_gc[GTK_STATE_NORMAL],
			matrix->pixmap,
			x,
			y,
			x,
			y,
			width + 1,
			height + 1);
}


static GtkMatrixCell *
gtk_matrix_cell_new()
{
	GtkMatrixCell *cell;
	cell = (GtkMatrixCell *) g_malloc(sizeof(GtkMatrixCell));
	cell->text = NULL;
	cell->link = NULL;
	return cell;
}

void gtk_matrix_set_cell_text(GtkMatrix * matrix, int row, int col, gchar * text)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (col > matrix->maxcol || row > matrix->maxrow)
		return;
	if (col < 0 || row < 0)
		return;

	gtk_matrix_set_cell(matrix, row, col, text);
}



void gtk_matrix_set_cell(GtkMatrix    *matrix, 
			gint         row, 
			gint         col,
			gchar        *text)
{
	GtkMatrixCell     **cell;
	GtkMatrixRange      range;
	gint               text_width;
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (col > matrix->maxcol || row > matrix->maxrow)
		return;
	if (col < 0 || row < 0)
		return;

	CheckBounds(matrix, row, col);
	
	cell = &matrix->data[row][col];
	
	if ((*cell) == NULL)
		(*cell) = gtk_matrix_cell_new();
	
	(*cell)->row = row;
	(*cell)->col = col;
	
	if ((*cell)->text)
		g_free((*cell)->text);
	
	(*cell)->text = g_strdup(text);
	
	if ((*cell)->text)
		text_width = gdk_string_width(MATRIX_FONT(matrix),(*cell)->text);
	else
		text_width = 0;

	range.row0 = row;
	range.rowi = row;
	range.col0 = matrix->view.col0;
	range.coli = matrix->view.coli;
	
	if (GTK_MATRIX_AUTORESIZE(matrix) && !GTK_MATRIX_IS_FROZEN(matrix) &&
	    text_width > matrix->column[col].width - 2 * CELLOFFSET - 1) {
		gtk_matrix_set_column_width(matrix, col, 
					   text_width + 2 * CELLOFFSET + 1);
	} else if (!GTK_MATRIX_IS_FROZEN(matrix))
		gtk_matrix_range_draw(matrix, &range);
	
}




void gtk_matrix_cell_clear(GtkMatrix * matrix, gint row, gint column)
{
	GtkMatrixRange range;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (column > matrix->maxcol || row > matrix->maxrow)
		return;
	if (column > matrix->maxalloccol || row > matrix->maxallocrow)
		return;
	if (column < 0 || row < 0)
		return;

	range.row0 = row;
	range.rowi = row;
	range.col0 = matrix->view.col0;
	range.coli = matrix->view.coli;

	gtk_matrix_real_cell_clear(matrix, row, column, FALSE);

	if (!GTK_MATRIX_IS_FROZEN(matrix)) {
		gtk_matrix_range_draw(matrix, &range);
	}
}

void gtk_matrix_cell_delete(GtkMatrix * matrix, gint row, gint column)
{
	GtkMatrixRange range;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (column > matrix->maxcol || row > matrix->maxrow)
		return;
	if (column > matrix->maxalloccol || row > matrix->maxallocrow)
		return;
	if (column < 0 || row < 0)
		return;

	range.row0 = row;
	range.rowi = row;
	range.col0 = matrix->view.col0;
	range.coli = matrix->view.coli;

	gtk_matrix_real_cell_clear(matrix, row, column, TRUE);

	if (!GTK_MATRIX_IS_FROZEN(matrix)) {
		gtk_matrix_range_draw(matrix, &range);
	}
}


static void gtk_matrix_real_cell_clear(GtkMatrix  *matrix, 
				      gint       row, 
				      gint       column, 
				      gboolean   delete)
{
	gchar        *text;
	GtkMatrixCell *cell;
	gpointer      link;

	cell = matrix->data[row][column];
	if (cell == NULL)
		return;

	text = gtk_matrix_cell_get_text(matrix, row, column);
	link = gtk_matrix_get_link(matrix, row, column);


	if (text) {
		g_free(text);
		matrix->data[row][column]->text = NULL;

		gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[CLEAR_CELL], row, column);
	}
	
	/* This will erase cell's attributes. The proper way to set them
	 * is using gtk_matrix_range_set_...  
	 */
	if (link == NULL)
		delete = TRUE;
	
	if (delete) {
		matrix->data[row][column]->link = NULL;
		g_free(matrix->data[row][column]);
		matrix->data[row][column] = NULL;
	}
}

void gtk_matrix_range_clear(GtkMatrix * matrix, GtkMatrixRange * range)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	gtk_matrix_real_range_clear(matrix, range, FALSE);
}

void gtk_matrix_range_delete(GtkMatrix * matrix, GtkMatrixRange * range)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	gtk_matrix_real_range_clear(matrix, range, TRUE);
}

static void gtk_matrix_real_range_clear(GtkMatrix * matrix, GtkMatrixRange * range,
				       gboolean delete)
{
	gint i, j;
	GtkMatrixRange clear;

	if (!range) {
		clear.row0 = 0;
		clear.rowi = matrix->maxallocrow;
		clear.col0 = 0;
		clear.coli = matrix->maxalloccol;
	} else
		clear = *range;

	clear.row0 = MAX(clear.row0, 0);
	clear.col0 = MAX(clear.col0, 0);
	clear.rowi = MIN(clear.rowi, matrix->maxallocrow);
	clear.coli = MIN(clear.coli, matrix->maxalloccol);

	for (i = clear.row0; i <= clear.rowi; i++)
		for (j = clear.col0; j <= clear.coli; j++) {
			gtk_matrix_real_cell_clear(matrix, i, j, delete);
		}

	gtk_matrix_range_draw(matrix, NULL);
}


gchar *
gtk_matrix_cell_get_text(GtkMatrix * matrix, gint row, gint col)
{
	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);

	if (col > matrix->maxcol || row > matrix->maxrow)
		return NULL;
	if (col < 0 || row < 0)
		return NULL;
	if (row > matrix->maxallocrow || col > matrix->maxalloccol)
		return NULL;
	if (!matrix->data[row][col])
		return NULL;
	if (!matrix->data[row][col]->text)
		return NULL;
	if (strlen(matrix->data[row][col]->text) == 0)
		return NULL;

	return (matrix->data[row][col]->text);
}

void gtk_matrix_link_cell(GtkMatrix * matrix, int row, int col, gpointer link)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (col > matrix->maxcol || row > matrix->maxrow)
		return;
	if (col < 0 || row < 0)
		return;

	if (row > matrix->maxallocrow || col > matrix->maxalloccol ||
	    !matrix->data[row][col])
		gtk_matrix_set_cell_text(matrix, row, col, "");

	matrix->data[row][col]->link = link;
}

gpointer
gtk_matrix_get_link(GtkMatrix * matrix, int row, int col)
{
	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);
	if (col > matrix->maxcol || row > matrix->maxrow)
		return NULL;
	if (col < 0 || row < 0)
		return NULL;

	if (row > matrix->maxallocrow || col > matrix->maxalloccol)
		return NULL;
	if (!matrix->data[row][col])
		return NULL;	/* Added by Bob Lissner */

	return (matrix->data[row][col]->link);
}

void gtk_matrix_remove_link(GtkMatrix * matrix, int row, int col)
{
	gpointer link;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	if (col > matrix->maxcol || row > matrix->maxrow)
		return;
	if (col < 0 || row < 0)
		return;

	link = gtk_matrix_get_link(matrix, row, col);
	if (link)
		link = NULL;
}



gint
gtk_matrix_get_pixel_info(GtkMatrix * matrix,
			 gint x,
			 gint y,
			 gint * row,
			 gint * column)
{
	gint trow, tcol;

	g_return_val_if_fail(matrix != NULL, 0);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), 0);

	/* bounds checking, return false if the user clicked 
	 * on a blank area */
	trow = ROW_FROM_YPIXEL(matrix, y);
	if (trow > matrix->maxrow)
		return FALSE;

	*row = trow;

	tcol = COLUMN_FROM_XPIXEL(matrix, x);
	if (tcol > matrix->maxcol)
		return FALSE;

	*column = tcol;

	return TRUE;
}

gint
gtk_matrix_get_cell_area(GtkMatrix * matrix,
			gint row,
			gint column,
			GdkRectangle * area)
{
	g_return_val_if_fail(matrix != NULL, 0);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), 0);

	if (row > matrix->maxrow || column > matrix->maxcol)
		return FALSE;
	if (row < 0 || column < 0)
		return FALSE;

	area->x = COLUMN_LEFT_XPIXEL(matrix, column);
	area->y = ROW_TOP_YPIXEL(matrix, row);
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		area->x -= matrix->row_title_area.width;
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		area->y -= MATRIX_COL_TITLE_HEIGHT(matrix);

	area->width = matrix->column[column].width;
	area->height = MATRIX_ROW_HEIGHT(matrix,row);

	return TRUE;
}



gint
gtk_matrix_set_active_cell(GtkMatrix *matrix, gint row, gint column)
{
	g_return_val_if_fail(matrix != NULL, 0);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), 0);

	if (row < 0 || column < 0)
		return FALSE;
	if (row > matrix->maxrow || column > matrix->maxcol)
		return FALSE;

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		if (!gtk_matrix_deactivate_cell(matrix))
			return FALSE;
	}
	matrix->active_cell.row = row;
	matrix->active_cell.col = column;

	if (!gtk_matrix_activate_cell(matrix, row, column))
		return FALSE;

	return TRUE;
}

void gtk_matrix_get_active_cell(GtkMatrix * matrix, gint * row, gint * column)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	*row = matrix->active_cell.row;
	*column = matrix->active_cell.col;
}

static void 
gtk_matrix_entry_changed(GtkWidget * widget, gpointer data)
{
	GtkMatrix    *matrix;
	gint         row, col;
	gchar       *text;
	
	
	g_return_if_fail(data != NULL);
	g_return_if_fail(GTK_IS_MATRIX(data));

	matrix = GTK_MATRIX(data);

	if (!GTK_WIDGET_VISIBLE(widget))
		return;
	
	row = matrix->active_cell.row;
	col = matrix->active_cell.col;

	if (row < 0 || col < 0)
		return;

	matrix->active_cell.row = -1;
	matrix->active_cell.col = -1;
	
	text = gtk_entry_get_text(GTK_ENTRY(gtk_matrix_get_entry(matrix)));

	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_IS_FROZEN);
	
	if (text && strlen(text) != 0)
		gtk_matrix_set_cell(matrix, row, col, text);
        else
		gtk_matrix_cell_clear(matrix, row, col);
	
		
	GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_IS_FROZEN);

	matrix->active_cell.row = row;
	matrix->active_cell.col = col;
	
	/* (Matias) */
	gtk_matrix_emit__entry_changed(GTK_WIDGET(matrix));
}



static gint
gtk_matrix_deactivate_cell(GtkMatrix * matrix)
{
	gchar  *text;
	gint    veto = TRUE;
	gint    row, col;

	//g_print("gtk_matrix_deactivate_cell\n");

	g_return_val_if_fail(matrix != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), FALSE);

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return FALSE;
	
	col = matrix->active_cell.col;
	row = matrix->active_cell.row;
	
	gtk_signal_disconnect_by_func(GTK_OBJECT(gtk_matrix_get_entry(matrix)),
				      (GtkSignalFunc) gtk_matrix_entry_changed,
				      GTK_OBJECT(GTK_WIDGET(matrix)));
	
	text = gtk_entry_get_text(GTK_ENTRY(gtk_matrix_get_entry(matrix)));
	if (text && strlen(text) != 0) {
		gtk_matrix_set_cell(matrix, row, col, text);
		gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[SET_CELL], row, col);
	} else
		gtk_matrix_cell_clear(matrix, row, col);
	
	
	gtk_matrix_hide_active_cell(matrix);
	
	matrix->active_cell.row = -1;
	matrix->active_cell.col = -1;
	
	gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[DEACTIVATE],
			row, col, &veto);

	if (!veto)
		return FALSE;
	
	return TRUE;
}


static void 
gtk_matrix_hide_active_cell(GtkMatrix *matrix)
{
	gint    row, col;
	
	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;
	
	row = matrix->active_cell.row;
	col = matrix->active_cell.col;

	if (row < 0 || col < 0)
		return;

	GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_IS_FROZEN);

	row = matrix->active_cell.row;
	col = matrix->active_cell.col;

	column_button_release(matrix, col);
	row_button_release(matrix, row);

	if (matrix->matrix_entry_window)
		gdk_window_hide(matrix->matrix_entry_window);
	else
		gdk_window_hide(matrix->matrix_entry->window);
	
	gdk_draw_pixmap(matrix->matrix_window,
			GTK_WIDGET(matrix)->style->fg_gc[GTK_STATE_NORMAL],
			matrix->pixmap,
			COLUMN_LEFT_XPIXEL(matrix, col) - 1,
			ROW_TOP_YPIXEL(matrix, row) - 1,
			COLUMN_LEFT_XPIXEL(matrix, col) - 1,
			ROW_TOP_YPIXEL(matrix, row) - 1,
			matrix->column[col].width + 4,
		        MATRIX_ROW_HEIGHT(matrix,row) + 4);
	
	GTK_WIDGET_UNSET_FLAGS(matrix->matrix_entry, GTK_HAS_FOCUS);
	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(matrix), GTK_HAS_FOCUS);
	gtk_widget_grab_focus(GTK_WIDGET(matrix));

	GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(matrix->matrix_entry), GTK_VISIBLE);

}



static gint
gtk_matrix_activate_cell(GtkMatrix *matrix, 
			 gint       row, 
			 gint       col)
{
	gint veto = TRUE;
	gchar *text = NULL;
	GtkMatrixCell      *cell;
	GtkEntry          *matrix_entry;

	g_return_val_if_fail(matrix != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), FALSE);

	if (row < 0 || col < 0)
		return FALSE;
	if (row > matrix->maxrow || col > matrix->maxcol)
		return FALSE;

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return veto;
	
	matrix->active_cell.row = row;
	matrix->active_cell.col = col;
	row_button_set(matrix, row);
	column_button_set(matrix, col);

	
	matrix_entry = GTK_ENTRY(gtk_matrix_get_entry(matrix));
	if (row <= matrix->maxallocrow && col <= matrix->maxalloccol) {
		if (matrix->data[row][col] != NULL) {
			cell = matrix->data[row][col];
			if (cell->text)
				text = g_strdup(cell->text);
		}
	}
	if (!text)
		text = g_strdup("");
	
	gtk_entry_set_text(GTK_ENTRY(matrix_entry), text);
	
	gtk_signal_connect(GTK_OBJECT(gtk_matrix_get_entry(matrix)),
			   "changed",
			   (GtkSignalFunc) gtk_matrix_entry_changed,
			   GTK_OBJECT(GTK_WIDGET(matrix)));
	
	gtk_matrix_show_active_cell(matrix);
	
	gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[ACTIVATE], row, col, &veto);
	if (!veto)
		return FALSE;
	
	return TRUE;
}



static void 
gtk_matrix_show_active_cell(GtkMatrix *matrix)
{
	GtkEntry          *matrix_entry;
	gint               row, col;
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;
		
	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(matrix->matrix_entry), GTK_VISIBLE);
	
	matrix_entry = GTK_ENTRY(gtk_matrix_get_entry(matrix));
	row = matrix->active_cell.row;
	col = matrix->active_cell.col;
	
	
	if (gtk_matrix_cell_isvisible(matrix,matrix->active_cell.row,
				     matrix->active_cell.col)) {
		gtk_matrix_size_allocate_entry(matrix);
		if (GTK_WIDGET_REALIZED(matrix->matrix_entry)) {
			if (matrix->matrix_entry_window)
				gdk_window_show(matrix->matrix_entry_window);
			else
				gdk_window_show(matrix->matrix_entry->window);
			gtk_widget_queue_draw(matrix->matrix_entry);
		}
		gtk_matrix_draw_active_cell(matrix);
		gtk_widget_grab_focus(GTK_WIDGET(matrix_entry));
		GTK_WIDGET_SET_FLAGS(GTK_WIDGET(matrix_entry), GTK_HAS_FOCUS);
		GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(matrix), GTK_HAS_FOCUS);
	}
}



static void 
gtk_matrix_draw_active_cell(GtkMatrix *matrix)
{
	GtkMatrixRange range;
	gint           row, col;

	//g_print("gtk_matrix_draw_activate_cell\n");
	
	if (!GTK_WIDGET_DRAWABLE(GTK_WIDGET(matrix)))
		return;
	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;
	
	row = matrix->active_cell.row;
	col = matrix->active_cell.col;
	range.row0 = row;
	range.col0 = col;
	range.rowi = row;
	range.coli = col;
	
	if (row < 0 || col < 0)
		return;

	if (!gtk_matrix_cell_isvisible(matrix, row, col))
		return;

	row_button_set(matrix, row);
	column_button_set(matrix, col);

	gtk_matrix_draw_backing_pixmap(matrix, range);
}



static void 
gtk_matrix_make_backing_pixmap(GtkMatrix   *matrix, 
			      gint        width, 
			      gint        height)
{
	gint pixmap_width, pixmap_height;

	if (width == 0 && height == 0) {
		width = matrix->matrix_window_width + 80;
		height = matrix->matrix_window_height + 80;
	}
	if (!matrix->pixmap) {
		/* allocate */
		matrix->pixmap = gdk_pixmap_new(matrix->matrix_window,
					       width, height,
					       -1);
		if (!GTK_MATRIX_IS_FROZEN(matrix))
			gtk_matrix_range_draw(matrix, NULL);
	} else {
		/* reallocate if sizes don't match */
		gdk_window_get_size(matrix->pixmap,
				    &pixmap_width, &pixmap_height);
		if ((pixmap_width != width) || (pixmap_height != height)) {
			gdk_pixmap_unref(matrix->pixmap);
			matrix->pixmap = gdk_pixmap_new(matrix->matrix_window,
						       width, height,
						       -1);
			if (!GTK_MATRIX_IS_FROZEN(matrix))
				gtk_matrix_range_draw(matrix, NULL);
		}
	}
}

static void gtk_matrix_draw(GtkWidget    *widget,
			    GdkRectangle *area)
{
	GtkMatrix *matrix;
	GtkMatrixRange range;
	GtkMatrixChild *child;
	GdkRectangle child_area;
	GList *children;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));
	g_return_if_fail(area != NULL);

	if (GTK_WIDGET_DRAWABLE(widget)) {
		matrix = GTK_MATRIX(widget);

		range.row0 = ROW_FROM_YPIXEL(matrix, area->y);
		range.rowi = ROW_FROM_YPIXEL(matrix, area->y + area->height);
		range.col0 = COLUMN_FROM_XPIXEL(matrix, area->x);
		range.coli = COLUMN_FROM_XPIXEL(matrix, area->x + area->width);
		
		gtk_matrix_range_draw(matrix, &range);

		if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
			gdk_window_show(matrix->column_title_window);

		if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			gdk_window_show(matrix->row_title_window);
		
		if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix) && 
		    GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			gdk_window_show(matrix->pivot_window);
		
		children = matrix->children;
		while (children) {
			child = children->data;
			children = children->next;

			if (gtk_widget_intersect(child->widget, area, &child_area))
				gtk_widget_draw(child->widget, &child_area);
		}

	}
}


static gint
gtk_matrix_expose(GtkWidget      *widget,
		  GdkEventExpose *event)
{
	GtkMatrix *matrix;
	GtkMatrixRange range;
	GtkMatrixChild *child;
	GList *children;
	GdkEventExpose child_event;

	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_MATRIX(widget), FALSE);
	g_return_val_if_fail(event != NULL, FALSE);

	matrix = GTK_MATRIX(widget);

	if (GTK_WIDGET_DRAWABLE(widget)) {
		range.row0 = ROW_FROM_YPIXEL(matrix, event->area.y);
		range.col0 = COLUMN_FROM_XPIXEL(matrix, event->area.x);
		range.rowi = ROW_FROM_YPIXEL(matrix, event->area.y + event->area.height);
		range.coli = COLUMN_FROM_XPIXEL(matrix, event->area.x + event->area.width);

		/* exposure events on the matrix */

		if (event->window == matrix->row_title_window) {
			size_allocate_row_title_buttons(matrix);
			gdk_window_show(matrix->row_title_window);
		}
		if (event->window == matrix->column_title_window) {
			size_allocate_column_title_buttons(matrix);
			gdk_window_show(matrix->column_title_window);
		}
		/* HERE ??? */
		if (event->window == matrix->pivot_window) {
			size_allocate_pivot_window(matrix);
			gdk_window_show(matrix->pivot_window);
		}

		
		if (event->window == matrix->matrix_window) {
			gtk_matrix_draw_backing_pixmap(matrix, range);

			/* matrix children events */
			child_event = *event;
			children = matrix->children;
			while (children) {
				child = children->data;
				children = children->next;

				if (GTK_WIDGET_NO_WINDOW(child->widget)) {
					GdkRectangle child_area;

					child_area.x = child->x;
					child_area.y = child->y;
					child_area.width = child->widget->allocation.width;
					child_area.height = child->widget->allocation.height;
					gdk_rectangle_intersect(&child_area, &event->area, &child_event.area);
					child_event.window = event->window;
					if (child->window)
						child_event.window = child->window;
					gtk_widget_event(child->widget, (GdkEvent *) & child_event);
				}
			}

		}
	}
	
	return FALSE;
}



static gint
gtk_matrix_button_press(GtkWidget      *widget,
			GdkEventButton *event)
{
	GtkMatrix *matrix;
	GdkModifierType mods;
	gint x, y, row, column;

	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_MATRIX(widget), FALSE);
	g_return_val_if_fail(event != NULL, FALSE);

	if (event->type != GDK_BUTTON_PRESS)
		return TRUE;
	gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
	if (!(mods & GDK_BUTTON1_MASK))
		return TRUE;
	
	matrix = GTK_MATRIX(widget);

	/* press on resize windows */
	if (event->window == matrix->column_title_window &&
	    !GTK_MATRIX_COLUMN_FROZEN(matrix)) {
		gtk_widget_get_pointer(widget, &matrix->x_drag, NULL);
		if (POSSIBLE_XDRAG(matrix, matrix->x_drag, &matrix->col_drag)) {
			
			GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_IN_XDRAG);
			gdk_pointer_grab(matrix->column_title_window, FALSE,
					 GDK_POINTER_MOTION_HINT_MASK |
					 GDK_BUTTON1_MOTION_MASK |
					 GDK_BUTTON_RELEASE_MASK,
					 NULL, NULL, event->time);

			draw_xor_vline(matrix);
			return TRUE;
		}
	}
	if (event->window == matrix->row_title_window && !GTK_MATRIX_ROW_FROZEN(matrix)) {
		gtk_widget_get_pointer(widget, NULL, &matrix->y_drag);

		if (POSSIBLE_YDRAG(matrix, matrix->y_drag, &matrix->row_drag)) {
			GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_IN_YDRAG);
			gdk_pointer_grab(matrix->row_title_window, FALSE,
					 GDK_POINTER_MOTION_HINT_MASK |
					 GDK_BUTTON1_MOTION_MASK |
					 GDK_BUTTON_RELEASE_MASK,
					 NULL, NULL, event->time);

			draw_xor_hline(matrix);
			return TRUE;
		}
	}
	
	/* selections on the matrix */
	if (event->window == matrix->matrix_window) {
		
		gtk_widget_get_pointer(widget, &x, &y);
		gtk_matrix_get_pixel_info(matrix, x, y, &row, &column);
		gdk_pointer_grab(matrix->matrix_window, FALSE,
				 GDK_POINTER_MOTION_HINT_MASK |
				 GDK_BUTTON1_MOTION_MASK |
				 GDK_BUTTON_RELEASE_MASK,
				 NULL, NULL, event->time);
		
		gtk_grab_add(GTK_WIDGET(matrix));
		
		gtk_matrix_deactivate_cell(matrix);
		gtk_matrix_activate_cell(matrix,row,column);
	}
	

	if (event->window == matrix->row_title_window) {
		gtk_widget_get_pointer(widget, &x, &y);
		row = ROW_FROM_YPIXEL(matrix, y);
		gtk_matrix_emit__row_button_press(GTK_WIDGET(matrix),row);
	}
	
	return TRUE;
}




static void 
gtk_matrix_click_cell(GtkMatrix     *matrix, 
		      gint           row, 
		      gint           column, 
		      gboolean      *veto)
{
	*veto = TRUE;
	
	if (row > matrix->maxrow || column > matrix->maxcol) {
		veto = FALSE;
		return;
	}
	
	if (column >= 0 && row >= 0)
		if (!matrix->column[column].is_visible || !matrix->row[row].is_visible) {
			veto = FALSE;
			return;
		}
	
	gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[TRAVERSE],
			matrix->active_cell.row,
			matrix->active_cell.col,
			&row,
			&column,
			veto);
	
	if (!*veto)
		return;
	
	if (row != -1 && column != -1) {
		gtk_matrix_deactivate_cell(matrix);
		gtk_matrix_activate_cell(matrix,row,column);
	}
}



static gint
gtk_matrix_button_release(GtkWidget * widget,
			  GdkEventButton * event)
{
	GtkMatrix *matrix;
	gint       x, y;

	
	matrix = GTK_MATRIX(widget);
	
	/* release on resize windows */
	if (GTK_MATRIX_IN_XDRAG(matrix)) {
		GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_IN_XDRAG);
		gtk_widget_get_pointer(widget, &x, NULL);
		gdk_pointer_ungrab(event->time);
		draw_xor_vline(matrix);

		gtk_matrix_set_column_width(matrix,matrix->col_drag,
					    new_column_width(matrix,matrix->col_drag,&x));
		matrix->old_hadjustment = -1.;
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment), "value_changed");
		return TRUE;
	}
	
	if (GTK_MATRIX_IN_YDRAG(matrix)) {
		GTK_MATRIX_UNSET_FLAGS(matrix, GTK_MATRIX_IN_YDRAG);
		gtk_widget_get_pointer(widget, NULL, &y);
		gdk_pointer_ungrab(event->time);
		draw_xor_hline(matrix);
		
		gtk_matrix_set_row_height(matrix,matrix->row_drag,
					  new_row_height(matrix,matrix->row_drag,&y));
		matrix->old_vadjustment = -1.;
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment), "value_changed");
		return TRUE;
	}
	
	if (event->window == matrix->matrix_window){
		gdk_pointer_ungrab(event->time);
		gtk_grab_remove(GTK_WIDGET(matrix));
	}
	
	return TRUE;
}




static gint
gtk_matrix_motion(GtkWidget      *widget,
		 GdkEventMotion *event)
{
	GtkMatrix         *matrix;
	GdkModifierType   mods;
	gint              x, y, row, column;
	
	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_MATRIX(widget), FALSE);
	g_return_val_if_fail(event != NULL, FALSE);


	matrix = GTK_MATRIX(widget);

	/* selections on the matrix */
	x = event->x;
	y = event->y;
	
	/* COLUMN RESIZE */
	if (event->window == matrix->column_title_window && 
	    !GTK_MATRIX_COLUMN_FROZEN(matrix)) {
		gtk_widget_get_pointer(widget, &x, &y);
		if (POSSIBLE_XDRAG(matrix, x, &column)) {
			if (matrix->cursor->type != GDK_SB_H_DOUBLE_ARROW) {
				gdk_cursor_destroy(matrix->cursor);
				matrix->cursor = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
				gdk_window_set_cursor(matrix->column_title_window, 
						      matrix->cursor);
			}
		} else {
			if (!GTK_MATRIX_IN_XDRAG(matrix) && 
			    GDK_TOP_LEFT_ARROW != matrix->cursor->type) {
				gdk_cursor_destroy(matrix->cursor);
				matrix->cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);
				gdk_window_set_cursor(matrix->column_title_window, 
						      matrix->cursor);
			}
		}
	}
	

	/* ROW RESIZE */
	if (event->window == matrix->row_title_window && !GTK_MATRIX_ROW_FROZEN(matrix)) {
		gtk_widget_get_pointer(widget, &x, &y);
		if ( POSSIBLE_YDRAG(matrix, y, &row)) {
			if (matrix->cursor->type != GDK_SB_V_DOUBLE_ARROW) {
				gdk_cursor_destroy(matrix->cursor);
				matrix->cursor = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
				gdk_window_set_cursor(matrix->row_title_window, 
						      matrix->cursor);
			}
		} else {
			if (!GTK_MATRIX_IN_YDRAG(matrix) &&
			    GDK_TOP_LEFT_ARROW != matrix->cursor->type) {
				gdk_cursor_destroy(matrix->cursor);
				matrix->cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);
				gdk_window_set_cursor(matrix->row_title_window,
						      matrix->cursor);
			}
		}
	}
	
	gdk_window_get_pointer(widget->window, &x, &y, &mods);
	if (!(mods & GDK_BUTTON1_MASK))
		return FALSE;
	
	/* RESIZING X */
	if (GTK_MATRIX_IN_XDRAG(matrix)) {
		if (event->is_hint || event->window != widget->window)
			gtk_widget_get_pointer(widget, &x, NULL);
		else
			x = event->x;

		new_column_width(matrix, matrix->col_drag, &x);
		if (x != matrix->x_drag) {
			draw_xor_vline(matrix);
			matrix->x_drag = x;
			draw_xor_vline(matrix);
		}
		return TRUE;
	}

	/* RESIZING Y */
	if (GTK_MATRIX_IN_YDRAG(matrix)) {
		if (event->is_hint || event->window != widget->window)
			gtk_widget_get_pointer(widget, NULL, &y);
		else
			y = event->y;

		new_row_height(matrix, matrix->row_drag, &y);
		if (y != matrix->y_drag) {
			draw_xor_hline(matrix);
			matrix->y_drag = y;
			draw_xor_hline(matrix);
		}
		return TRUE;
	}
	
	return TRUE;
}





static gint
gtk_matrix_entry_key_press(GtkWidget   *widget,
			   GdkEventKey *key)
{
	GtkMatrix *matrix;
	GtkEntry  *entry;

	matrix = GTK_MATRIX(widget);
	entry  = GTK_ENTRY(gtk_matrix_get_entry(matrix));
	
	gtk_matrix_key_press(widget,key);
	
	switch (key->keyval){
	case GDK_Return:
	case GDK_KP_Enter:
	case GDK_ISO_Left_Tab:
	case GDK_Tab:
	case GDK_Page_Up:
	case GDK_Up:
	case GDK_Page_Down:
	case GDK_Down:
	case GDK_Home:
	case GDK_End:	
		gtk_signal_emit_stop_by_name
			(GTK_OBJECT(entry),"key_press_event");
		break;
	}
	return TRUE;
}




static gint
gtk_matrix_key_press(GtkWidget   *widget,
		     GdkEventKey *key)
{
	GtkMatrix   *matrix;
	gint         row, col;
	gboolean     veto = TRUE;
	gint         scroll = 1;

	matrix = GTK_MATRIX(widget);

	if (key->state & GDK_CONTROL_MASK || key->keyval == GDK_Control_L ||
	    key->keyval == GDK_Control_R)
		return TRUE;

	switch (key->keyval) {
	case GDK_Return:
	case GDK_KP_Enter:
		
		row = matrix->active_cell.row;
		col = matrix->active_cell.col;
		if (row < matrix->maxrow) {
			row = row + scroll;
			while (!matrix->row[row].is_visible && row < matrix->maxrow)
				row++;
		}
		gtk_matrix_click_cell(matrix, row, col, &veto);
		break;
	
	case GDK_ISO_Left_Tab:
		row = matrix->active_cell.row;
		col = matrix->active_cell.col;

		if (col > 0) {
			col = col - scroll;
			while (!matrix->column[col].is_visible && col > 0)
				col--;
			col = MAX(0, col);
		}
		gtk_matrix_click_cell(matrix, row, col, &veto);
		break;
	
	case GDK_Tab:
		row = matrix->active_cell.row;
		col = matrix->active_cell.col;
		if (col < matrix->maxcol) {
			col = col + scroll;
			while (!matrix->column[col].is_visible && col < matrix->maxcol)
				col++;
		} 
		else if (row < matrix->maxrow) {
			col = 0;
			while (!matrix->column[col].is_visible && col < matrix->maxcol)
				col++;
			
			row = row + scroll;
			while (!matrix->row[row].is_visible && row < matrix->maxrow)
				row++;
		}
		
		gtk_matrix_click_cell(matrix, row, col, &veto);
		break;

	case GDK_Page_Up:
		scroll = MAX_VISIBLE_ROW(matrix) - MIN_VISIBLE_ROW(matrix) + 1;
	
	case GDK_Up:
		
		col = matrix->active_cell.col;
		row = matrix->active_cell.row;
		row = row - scroll;
		while (!matrix->row[row].is_visible && row > 0)
			row--;
		row = MAX(0, row);
		gtk_matrix_click_cell(matrix, row, col, &veto);
		break;

	case GDK_Page_Down:
		scroll = MAX_VISIBLE_ROW(matrix) - MIN_VISIBLE_ROW(matrix) + 1;

	case GDK_Down:
		
		col = matrix->active_cell.col;
		row = matrix->active_cell.row;
		if (matrix->active_cell.row < matrix->maxrow) {
			row = row + scroll;
			while (!matrix->row[row].is_visible && row < matrix->maxrow)
				row++;
			row = MIN(matrix->maxrow, row);
		}
		gtk_matrix_click_cell(matrix, row, col, &veto);
		break;
		
	case GDK_Home:
		row = 0;
		while (!matrix->row[row].is_visible && row < matrix->maxrow)
			row++;
		gtk_matrix_click_cell(matrix, row, matrix->active_cell.col, &veto);
		break;

	case GDK_End:
		row = matrix->maxrow;
		while (!matrix->row[row].is_visible && row > 0)
			row--;
		gtk_matrix_click_cell(matrix, row, matrix->active_cell.col, &veto);
		break;
	}
	return TRUE;
}






static void gtk_matrix_size_request(GtkWidget * widget,
				    GtkRequisition * requisition)
{
	GtkMatrix *matrix;
	GList *children;
	GtkMatrixChild *child;
	GtkRequisition child_requisition;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));
	g_return_if_fail(requisition != NULL);

	matrix = GTK_MATRIX(widget);
	
	requisition->width = 3 * DEFAULT_COLUMN_WIDTH;
	requisition->height = 3 * MATRIX_ROW_HEIGHT(matrix,0);
	
	/* compute the size of the column title area */
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		requisition->height += MATRIX_COL_TITLE_HEIGHT(matrix);

	/* compute the size of the row title area */
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		requisition->width += matrix->row_title_area.width;

	matrix->view.row0 = ROW_FROM_YPIXEL(matrix, MATRIX_COL_TITLE_HEIGHT(matrix) + 1);
	matrix->view.rowi = ROW_FROM_YPIXEL(matrix, matrix->matrix_window_height - 1);
	matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, matrix->row_title_area.width + 1);
	matrix->view.coli = COLUMN_FROM_XPIXEL(matrix, matrix->matrix_window_width);

	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		matrix->view.row0 = ROW_FROM_YPIXEL(matrix, 1);

	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, 1);

	children = matrix->children;
	while (children) {
		child = children->data;
		children = children->next;

		gtk_widget_size_request(child->widget, &child_requisition);
	}
}


static void gtk_matrix_size_allocate(GtkWidget * widget,
				    GtkAllocation * allocation)
{
	GtkMatrix *matrix;
	GtkAllocation matrix_allocation;
	gint border_width;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GTK_IS_MATRIX(widget));
	g_return_if_fail(allocation != NULL);

	matrix = GTK_MATRIX(widget);
	widget->allocation = *allocation;
	border_width = GTK_CONTAINER(widget)->border_width;
	if (GTK_WIDGET_REALIZED(widget)) {
		gdk_window_move_resize(widget->window,
				       allocation->x + border_width,
				       allocation->y + border_width,
				       allocation->width - 2 * border_width,
				       allocation->height - 2 * border_width);
	}
	/* use internal allocation structure for all the math
	 * because it's easier than always subtracting the container
	 * border width */
	matrix->internal_allocation.x = 0;
	matrix->internal_allocation.y = 0;
	matrix->internal_allocation.width = allocation->width - 2 * border_width;
	matrix->internal_allocation.height = allocation->height - 2 * border_width;

	matrix_allocation.x = 0;
	matrix_allocation.y = 0;
	matrix_allocation.width = allocation->width - 2 * border_width;
	matrix_allocation.height = allocation->height - 2 * border_width;

	matrix->matrix_window_width = matrix_allocation.width;
	matrix->matrix_window_height = matrix_allocation.height;

	if (GTK_WIDGET_REALIZED(widget))
		gdk_window_move_resize(matrix->matrix_window,
				       matrix_allocation.x,
				       matrix_allocation.y,
				       matrix_allocation.width,
				       matrix_allocation.height);

	/* position the window which holds the column title buttons */
	matrix->column_title_area.x = 0;
	matrix->column_title_area.y = 0;
	
	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		matrix->column_title_area.x = matrix->row_title_area.width;
	
	matrix->column_title_area.width = matrix_allocation.width -
		matrix->column_title_area.x;
	if (GTK_WIDGET_REALIZED(widget) && GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		gdk_window_move_resize(matrix->column_title_window,
				       matrix->column_title_area.x,
				       matrix->column_title_area.y,
				       matrix->column_title_area.width,
				       MATRIX_COL_TITLE_HEIGHT(matrix));

	matrix->matrix_window_width = matrix_allocation.width;
	matrix->matrix_window_height = matrix_allocation.height;

	/* column button allocation */
	size_allocate_column_title_buttons(matrix);

	/* position the window which holds the row title buttons */
	matrix->row_title_area.x = 0;
	matrix->row_title_area.y = 0;
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		matrix->row_title_area.y = MATRIX_COL_TITLE_HEIGHT(matrix);
	matrix->row_title_area.height = matrix_allocation.height -
	    matrix->row_title_area.y;

	if (GTK_WIDGET_REALIZED(widget) && GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		gdk_window_move_resize(matrix->row_title_window,
				       matrix->row_title_area.x,
				       matrix->row_title_area.y,
				       matrix->row_title_area.width,
				       matrix->row_title_area.height);

	/* row button allocation */
	size_allocate_row_title_buttons(matrix);

	/* position the pivot window */
	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		matrix->row_title_area.y = MATRIX_COL_TITLE_HEIGHT(matrix);
	matrix->row_title_area.height = matrix_allocation.height -
	    matrix->row_title_area.y;

	if (GTK_WIDGET_REALIZED(widget) && 
	    GTK_MATRIX_ROW_TITLES_VISIBLE(matrix) && 
	    GTK_MATRIX_COL_TITLES_VISIBLE(matrix))       
		size_allocate_pivot_window(matrix);
	

	matrix->view.row0 = ROW_FROM_YPIXEL(matrix, MATRIX_COL_TITLE_HEIGHT(matrix) + 1);
	matrix->view.rowi = ROW_FROM_YPIXEL(matrix, matrix->matrix_window_height - 1);
	matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, matrix->row_title_area.width + 1);
	matrix->view.coli = COLUMN_FROM_XPIXEL(matrix, matrix->matrix_window_width);

	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		matrix->view.row0 = ROW_FROM_YPIXEL(matrix, 1);

	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, 1);

	/* set the scrollbars adjustments */
	adjust_scrollbars(matrix);

	/* re-scale backing pixmap */
	if (GTK_WIDGET_REALIZED(matrix)) {
		gtk_matrix_make_backing_pixmap(matrix, 0, 0);
		gtk_matrix_position_children(matrix);
	}
}



static void size_allocate_column_title_buttons(GtkMatrix * matrix)
{
	gint i;
	gint x, width;

	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		return;
	if (!GTK_WIDGET_REALIZED(matrix))
		return;

	width = matrix->matrix_window_width;
	x = 0;

	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix)) {
		width -= matrix->row_title_area.width;
		x = matrix->row_title_area.width;
	}
	if (matrix->column_title_area.width != width || matrix->column_title_area.x != x) {
		matrix->column_title_area.width = width;
		matrix->column_title_area.x = x;
		gdk_window_move_resize(matrix->column_title_window,
				       matrix->column_title_area.x,
				       matrix->column_title_area.y,
				       matrix->column_title_area.width,
				       MATRIX_COL_TITLE_HEIGHT(matrix));
	}
	if (MAX_VISIBLE_COLUMN(matrix) == matrix->maxcol)
		gdk_window_clear_area(matrix->column_title_window,
				      0, 0,
				      matrix->column_title_area.width,
				      MATRIX_COL_TITLE_HEIGHT(matrix));

	for (i = MIN_VISIBLE_COLUMN(matrix); i <= MAX_VISIBLE_COLUMN(matrix); i++)
		gtk_matrix_button_draw(matrix, -1, i);

}







static void size_allocate_row_title_buttons(GtkMatrix * matrix)
{
	gint i;
	gint y, height;

	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		return;
	if (!GTK_WIDGET_REALIZED(matrix))
		return;


	height = matrix->matrix_window_height;
	y = 0;

	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix)) {
		height -= MATRIX_COL_TITLE_HEIGHT(matrix);
		y = MATRIX_COL_TITLE_HEIGHT(matrix);
	}
	if (matrix->row_title_area.height != height || matrix->row_title_area.y != y) {
		matrix->row_title_area.y = y;
		matrix->row_title_area.height = height;
		gdk_window_move_resize(matrix->row_title_window,
				       matrix->row_title_area.x,
				       matrix->row_title_area.y,
				       matrix->row_title_area.width,
				       matrix->row_title_area.height);
	}
	if (MAX_VISIBLE_ROW(matrix) == matrix->maxrow)
		gdk_window_clear_area(matrix->row_title_window,
				      0, 0,
				      matrix->row_title_area.width,
				      matrix->row_title_area.height);

	for (i = MIN_VISIBLE_ROW(matrix); i <= MAX_VISIBLE_ROW(matrix); i++)
		gtk_matrix_button_draw(matrix, i, -1);

}


static void 
size_allocate_pivot_window(GtkMatrix *matrix)
{
	gint width, height ,new_width, new_height;
	
	
	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		return;
	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		return;
	if (!GTK_WIDGET_REALIZED(matrix))
		return;
	
	width  = matrix->pivot_area.width;
	height = matrix->pivot_area.height;
	new_height = MATRIX_COL_TITLE_HEIGHT(matrix);
	new_width = matrix->row_title_area.width;

	if ( height != new_height || width != new_width ){
		
		matrix->pivot_area.width = new_width;
		matrix->pivot_area.height = new_height;
		gdk_window_move_resize(matrix->pivot_window,
				       0,
				       0,
				       matrix->pivot_area.width,
				       matrix->pivot_area.height);
	}
}




static void 
gtk_matrix_recalc_top_ypixels(GtkMatrix *matrix)
{
	gint i, cy;

	cy = MATRIX_COL_TITLE_HEIGHT(matrix);
	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		cy = 0;
	for (i = 0; i <= matrix->maxrow; i++) {
		matrix->row[i].top_ypixel = cy;
		if (matrix->row[i].is_visible)
			cy += MATRIX_ROW_HEIGHT(matrix,i);
	}
}

static void 
gtk_matrix_recalc_left_xpixels(GtkMatrix *matrix)
{
	gint i, cx;

	cx = matrix->row_title_area.width;
	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		cx = 0;
	for (i = 0; i <= matrix->maxcol; i++) {
		matrix->column[i].left_xpixel = cx;
		if (matrix->column[i].is_visible)
			cx += matrix->column[i].width;
	}

}



static void 
gtk_matrix_size_allocate_entry(GtkMatrix *matrix)
{
	GtkWidget     *widget;
	GtkAllocation  shentry_allocation;
	GtkEntry      *matrix_entry;
	GtkStyle      *style;
	gint           row, col;
	gint           size, max_size, text_size, column_width;
	
	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)))
		return;
	
	widget = GTK_WIDGET(matrix);
	
	matrix_entry = GTK_ENTRY(gtk_matrix_get_entry(matrix));
	
	if (GTK_WIDGET_REALIZED(matrix->matrix_entry)) {
				
		style = gtk_style_copy(GTK_WIDGET(matrix)->style);
		style->bg[GTK_STATE_NORMAL] = style->white; 
		gtk_widget_set_style(GTK_WIDGET(matrix_entry),style);
		gtk_widget_size_request(matrix->matrix_entry, NULL);

		gdk_window_set_background(GTK_ENTRY(matrix_entry)->text_area,
					  &(GTK_WIDGET(matrix)->style->white));
		
		if (matrix->matrix_entry_window)
			gdk_window_set_background(matrix->matrix_entry_window,
						  &(GTK_WIDGET(matrix)->style->white));
		else
			gdk_window_set_background(GTK_WIDGET(matrix_entry)->window,
						  &(GTK_WIDGET(matrix)->style->white));
		
	}
	
	max_size = 0;
	
	text_size = GTK_ENTRY(matrix_entry)->text == NULL ? 0 :
		gdk_string_width(MATRIX_FONT(matrix),
				 gtk_entry_get_text(GTK_ENTRY(matrix_entry)));
	column_width = matrix->column[matrix->active_cell.col].width;
	
	size = MIN(text_size, max_size);
	size = MAX(size, column_width);

	row = matrix->active_cell.row;
	col = matrix->active_cell.col;

	shentry_allocation.x = COLUMN_LEFT_XPIXEL(matrix, matrix->active_cell.col);
	shentry_allocation.y = ROW_TOP_YPIXEL(matrix, matrix->active_cell.row);
	shentry_allocation.width = column_width;
	shentry_allocation.height =  MATRIX_ROW_HEIGHT(matrix,matrix->active_cell.row);
	
	
	if (matrix->matrix_entry_window) {
		gdk_window_move_resize(matrix->matrix_entry_window,
				       shentry_allocation.x,
				       shentry_allocation.y,
				       shentry_allocation.width,
				       shentry_allocation.height);
		shentry_allocation.x = 0;
		shentry_allocation.y = 0;
		gtk_widget_size_allocate(matrix->matrix_entry, &shentry_allocation);
	} else
		gtk_widget_size_allocate(matrix->matrix_entry, &shentry_allocation);
	
}



/**
 * gtk_matrix_set_entry
 * @matrix     : A GtkMatrix
 * @new_entry : The new editing widget that the GtkMatrix will use, 
 *              can be a GtkEntry, GtkIentry or GtkCombo 
 * 
 * Set a new widget for editing the matrix.
 */
static void 
gtk_matrix_set_entry(GtkMatrix * matrix, GtkWidget *new_entry)
{
	GtkWidget  *widget;
	GtkWidget  *entry;
	
	widget = GTK_WIDGET(matrix);
	
	if (matrix->matrix_entry){
		gtk_widget_hide(matrix->matrix_entry);
		gtk_widget_ref(matrix->matrix_entry);
		gtk_widget_unparent(matrix->matrix_entry);
		
		if (matrix->matrix_entry_window){
			matrix->matrix_entry_window->user_data = NULL;
			matrix->matrix_entry_window = NULL;
		}
	}
	
	matrix->matrix_entry = new_entry;
	matrix->entry_type = GTK_WIDGET_TYPE(new_entry);
	entry = gtk_matrix_get_entry(matrix);
	
	
	if (GTK_WIDGET_REALIZED(matrix) && GTK_WIDGET_NO_WINDOW(matrix->matrix_entry)) {
		GdkWindowAttr attributes;
		gint attributes_mask;
		
		attributes.window_type = GDK_WINDOW_CHILD;
		attributes.x = 0;
		attributes.y = 0;
		attributes.width = matrix->matrix_entry->requisition.width;
		attributes.height = matrix->matrix_entry->requisition.height;
		attributes.wclass = GDK_INPUT_OUTPUT;
		attributes.visual = gtk_widget_get_visual(widget);
		attributes.colormap = gtk_widget_get_colormap(widget);
		attributes.event_mask = GDK_EXPOSURE_MASK;

		attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
		matrix->matrix_entry_window = gdk_window_new(matrix->matrix_window,
							   &attributes, attributes_mask);
		gdk_window_set_user_data(matrix->matrix_entry_window, widget);

		if (matrix->matrix_entry_window)
			gtk_style_set_background(widget->style,
						 matrix->matrix_entry_window,
						 GTK_STATE_NORMAL);
	}
	
	if (GTK_WIDGET_REALIZED(matrix)) {
		gtk_widget_set_parent(matrix->matrix_entry, GTK_WIDGET(matrix));
		gtk_widget_set_parent_window(matrix->matrix_entry,
					     matrix->matrix_entry_window ?
					     matrix->matrix_entry_window :
					     matrix->matrix_window);
		gtk_widget_realize(matrix->matrix_entry);
	}
	
	gtk_signal_connect_object(GTK_OBJECT(entry), "key_press_event",
				  (GtkSignalFunc) gtk_matrix_entry_key_press,
				  GTK_OBJECT(matrix));
	
	gtk_widget_show(matrix->matrix_entry);
}





/**
 * gtk_matrix_get_entry
 * @matrix : A GtkMatrix
 *
 * Return the entry of the editing widget used in the matrix.
 */
GtkWidget *
gtk_matrix_get_entry(GtkMatrix *matrix)
{
	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);
	g_return_val_if_fail(matrix->matrix_entry != NULL, NULL);

	if (GTK_IS_ENTRY(matrix->matrix_entry))
		return (matrix->matrix_entry);
	
	if (GTK_IS_COMBO(matrix->matrix_entry))
		return (GTK_COMBO(matrix->matrix_entry)->entry);
	
	return NULL;
}


/* BUTTONS */
static void row_button_set(GtkMatrix * matrix, gint row)
{
	if (matrix->row[row].button.state == GTK_STATE_ACTIVE)
		return;

	matrix->row[row].button.state = GTK_STATE_ACTIVE;
	gtk_matrix_button_draw(matrix, row, -1);

}

static void column_button_set(GtkMatrix * matrix, gint column)
{
	if (matrix->column[column].button.state == GTK_STATE_ACTIVE)
		return;

	matrix->column[column].button.state = GTK_STATE_ACTIVE;
	gtk_matrix_button_draw(matrix, -1, column);

}

static void row_button_release(GtkMatrix * matrix, gint row)
{
	if (matrix->row[row].button.state == GTK_STATE_NORMAL)
		return;

	matrix->row[row].button.state = GTK_STATE_NORMAL;
	gtk_matrix_button_draw(matrix, row, -1);
}

static void column_button_release(GtkMatrix * matrix, gint column)
{
	if (matrix->column[column].button.state == GTK_STATE_NORMAL)
		return;

	matrix->column[column].button.state = GTK_STATE_NORMAL;
	gtk_matrix_button_draw(matrix, -1, column);
}



static void
gtk_matrix_button_draw (GtkMatrix *matrix, gint row, gint column)
{
	GdkWindow *window = NULL;
	GtkShadowType shadow_type;
	gint width = 0, height = 0;
	gint x = 0, y = 0;
	gint index = 0;
	int text_width, text_height;
	GtkMatrixButton *button = NULL;
	GdkRectangle allocation;
	gint is_sensitive = FALSE;
	gint state;
	gint len;
	gchar *line;
	gchar *words;
	gchar label[10];
	
	if(!GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) return;
	
	if(row >= 0 && !matrix->row[row].is_visible) return;
	if(column >= 0 && !matrix->column[column].is_visible) return;
	if(row >= 0 && !GTK_MATRIX_ROW_TITLES_VISIBLE(matrix)) return;
	if(column >= 0 && !GTK_MATRIX_COL_TITLES_VISIBLE(matrix)) return;
	if(column>=0 && column <MIN_VISIBLE_COLUMN(matrix)) return;
	if(column>=0 && column >MAX_VISIBLE_COLUMN(matrix)) return;
	if(row>=0 && row <MIN_VISIBLE_ROW(matrix)) return;
	if(row>=0 && row >MAX_VISIBLE_ROW(matrix)) return;
	
	if(row==-1){
		window=matrix->column_title_window;
		button=&matrix->column[column].button;
		index=column;
		x = COLUMN_LEFT_XPIXEL(matrix, column)+CELL_SPACING;
		if(GTK_MATRIX_ROW_TITLES_VISIBLE(matrix)) x -= matrix->row_title_area.width;
		y = 0;
		width = matrix->column[column].width;
		height = MATRIX_COL_TITLE_HEIGHT(matrix);
		is_sensitive = matrix->column[column].is_sensitive;
	}
	
	if(column==-1){
		window=matrix->row_title_window;
		button = &matrix->row[row].button;
		index=row;
		x = 0;
		y = ROW_TOP_YPIXEL(matrix, row)+CELL_SPACING;
		if(GTK_MATRIX_COL_TITLES_VISIBLE(matrix)) 
			y-= MATRIX_COL_TITLE_HEIGHT(matrix);
		width = matrix->row_title_area.width;
		height = MATRIX_ROW_HEIGHT(matrix,row);
		is_sensitive = matrix->row[row].is_sensitive;
	}
	
	allocation.x = x;
	allocation.y = y;
	allocation.width = width;
	allocation.height = height;
	
	gdk_window_clear_area (window,
			       x, y,
			       width, height);
	
	gtk_paint_box (GTK_WIDGET(matrix)->style, window,
		       GTK_STATE_NORMAL, GTK_SHADOW_OUT, 
		       &allocation, GTK_WIDGET(matrix),
		       "buttondefault", x, y, width, height);
	
	state = button->state;
	if(!is_sensitive) state = GTK_STATE_INSENSITIVE;
	
	if (state == GTK_STATE_ACTIVE)
		shadow_type = GTK_SHADOW_IN;
	else 
		shadow_type = GTK_SHADOW_OUT;
	
	if(state != GTK_STATE_NORMAL && state != GTK_STATE_INSENSITIVE)
		gtk_paint_box (GTK_WIDGET(matrix)->style, window,
			       button->state, shadow_type, 
			       &allocation, GTK_WIDGET(matrix),
			       "button", x, y, width, height);
	
	text_height=GTK_WIDGET(matrix)->style->font->ascent + 
		GTK_WIDGET(matrix)->style->font->descent;
	
	y += DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix))/2+
		GTK_WIDGET(matrix)->style->font->ascent/2; 
	
	gdk_gc_set_clip_rectangle(GTK_WIDGET(matrix)->style->fg_gc[button->state], 
				  &allocation);
	gdk_gc_set_clip_rectangle(GTK_WIDGET(matrix)->style->white_gc, &allocation);
	
	if(button->label && strlen(button->label)>0){
		words=button->label;
		line = g_new(gchar, 1);
		line[0]='\0';
		
		while(words && *words != '\0'){
			if(*words != '\n'){
				len=strlen(line);
				line=g_realloc(line, len+2);
				line[len]=*words;
				line[len+1]='\0';
			}
			if(*words == '\n' || *(words+1) == '\0'){
				text_width = gdk_string_width (GTK_WIDGET (matrix)->style->font,
							       line);
				
				gtk_paint_string (GTK_WIDGET(matrix)->style, window, state,
						  &allocation, GTK_WIDGET(matrix), "label",
						  x + (width - text_width) /2, y,
						  line);
				
				
				y += GTK_WIDGET(matrix)->style->font->ascent+
					GTK_WIDGET(matrix)->style->font->descent + 2;
				
				g_free(line);
				line = g_new(gchar, 1);
				line[0]='\0';
			}
			words++;
		}
		g_free(line);
	}else{
		sprintf(label,"%d",index);
		text_width = gdk_string_width (GTK_WIDGET (matrix)->style->font,
					       label);
		
		gtk_paint_string (GTK_WIDGET(matrix)->style, window, state,
				  &allocation, GTK_WIDGET(matrix), "label",
				  x + (width - text_width) /2, y,
				  label);
	}
	
	gdk_gc_set_clip_rectangle(GTK_WIDGET(matrix)->style->fg_gc[button->state],
				  NULL);
	gdk_gc_set_clip_rectangle(GTK_WIDGET(matrix)->style->white_gc, NULL);
	
}



/* SCROLLBARS

 * functions:
 *   adjust_scrollbars
 *   vadjustment_changed
 *   hadjustment_changed
 *   vadjustment_value_changed
 *   hadjustment_value_changed */

static void adjust_scrollbars(GtkMatrix * matrix)
{
	if (matrix->vadjustment) {
		matrix->vadjustment->page_size = matrix->matrix_window_height;
		matrix->vadjustment->page_increment = matrix->matrix_window_height / 2;
		matrix->vadjustment->step_increment = DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix));
		matrix->vadjustment->lower = 0;
		matrix->vadjustment->upper = MATRIX_HEIGHT(matrix) + 2;
		
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment), "changed");
	}
	
	if (matrix->hadjustment) {
		matrix->hadjustment->page_size = matrix->matrix_window_width;
		matrix->hadjustment->page_increment = matrix->matrix_window_width / 2;
		matrix->hadjustment->step_increment = DEFAULT_COLUMN_WIDTH;
		matrix->hadjustment->lower = 0;
		matrix->hadjustment->upper = MATRIX_WIDTH(matrix) + 2;
		
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment), "changed");
		
	}
}


static void vadjustment_changed(GtkAdjustment * adjustment,
				gpointer data)
{
	GtkMatrix *matrix;

	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);

	matrix = GTK_MATRIX(data);

}

static void hadjustment_changed(GtkAdjustment * adjustment,
				gpointer data)
{
	GtkMatrix *matrix;

	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);

	matrix = GTK_MATRIX(data);
	
}


static void 
vadjustment_value_changed(GtkAdjustment *adjustment,
			  gpointer       data)
{
	GtkMatrix   *matrix;
	gint        i, y = 0;
	
	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);
	g_return_if_fail(GTK_IS_MATRIX(data));

	matrix = GTK_MATRIX(data);
	
	if (adjustment->value == matrix->old_vadjustment)
		return;
	
	
	for (i = 0; i <= matrix->maxrow; i++) {
		if (matrix->row[i].is_visible)
			y += MATRIX_ROW_HEIGHT(matrix,i);
		if (y > adjustment->value)
			break;
	}
	
	if (i == 0)
		matrix->vadjustment->page_increment = MATRIX_ROW_HEIGHT(matrix,0);
	else 
		matrix->vadjustment->page_increment =
			MIN(MATRIX_ROW_HEIGHT(matrix,i), 
			    MATRIX_ROW_HEIGHT(matrix,i-1));
	
	
	matrix->vadjustment->value = adjustment->value;
       	matrix->voffset = -adjustment->value;
	
	matrix->view.row0 = ROW_FROM_YPIXEL(matrix, MATRIX_COL_TITLE_HEIGHT(matrix) + 1);
	matrix->view.rowi = ROW_FROM_YPIXEL(matrix, matrix->matrix_window_height - 1);
	if (!GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		matrix->view.row0 = ROW_FROM_YPIXEL(matrix, 1);
	
	
	if (GTK_WIDGET_REALIZED(matrix->matrix_entry) &&
	    matrix->active_cell.row >= 0 && matrix->active_cell.col >= 0 &&
	    !gtk_matrix_cell_isvisible(matrix, matrix->active_cell.row,
				      matrix->active_cell.col)) {
		gchar *text;
		
		text = gtk_entry_get_text(GTK_ENTRY(gtk_matrix_get_entry(matrix)));
		if (!text || strlen(text) == 0)
			gtk_matrix_cell_clear(matrix,
					     matrix->active_cell.row,
					     matrix->active_cell.col);
		if (matrix->matrix_entry_window)
			gdk_window_hide(matrix->matrix_entry_window);
		else
			gdk_window_hide(matrix->matrix_entry->window);
	}
	
	gtk_matrix_position_children(matrix);
	gtk_matrix_range_draw(matrix, NULL);
	size_allocate_row_title_buttons(matrix);
	
	matrix->old_vadjustment = adjustment->value;
}





static void 
hadjustment_value_changed(GtkAdjustment *adjustment,
			  gpointer       data)
{
	GtkMatrix *matrix;
	gint      i, x = 0;
	
	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);
	g_return_if_fail(GTK_IS_MATRIX(data));
	matrix = GTK_MATRIX(data);
	
	
	if (adjustment->value == matrix->old_hadjustment)
		return;
	
	for (i = 0; i <= matrix->maxcol; i++) {
		if (matrix->column[i].is_visible)
			x += matrix->column[i].width;
		if (x > adjustment->value)
			break;
	}
	

	if (i == 0) {
		matrix->hadjustment->page_increment = matrix->column[0].width;
	} else {
		matrix->hadjustment->page_increment =
			MIN(matrix->column[i].width,matrix->column[i-1].width);
	}  
	
	
	matrix->hadjustment->value = adjustment->value;
	matrix->hoffset = -adjustment->value;
	
	
	matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix,matrix->row_title_area.width+1);
	matrix->view.coli = COLUMN_FROM_XPIXEL(matrix,matrix->matrix_window_width);
	if (!GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		matrix->view.col0 = COLUMN_FROM_XPIXEL(matrix, 1);
	
	
	if (GTK_WIDGET_REALIZED(matrix->matrix_entry) &&
	    matrix->active_cell.row >= 0 && matrix->active_cell.col >= 0 &&
	    !gtk_matrix_cell_isvisible(matrix, matrix->active_cell.row,
				      matrix->active_cell.col)) {
		
		gchar *text;
		
		text = gtk_entry_get_text(GTK_ENTRY(gtk_matrix_get_entry(matrix)));
		if (!text || strlen(text) == 0)
			gtk_matrix_cell_clear(matrix,
					     matrix->active_cell.row,
					     matrix->active_cell.col);
		if (matrix->matrix_entry_window)
			gdk_window_hide(matrix->matrix_entry_window);
		else
			gdk_window_hide(matrix->matrix_entry->window);
	}
	
	
	gtk_matrix_position_children(matrix);
	gtk_matrix_range_draw(matrix, NULL);
	size_allocate_column_title_buttons(matrix);
	
	matrix->old_hadjustment = adjustment->value;
}



/* COLUMN RESIZING */
static void draw_xor_vline(GtkMatrix * matrix)
{
	GtkWidget *widget;

	g_return_if_fail(matrix != NULL);

	widget = GTK_WIDGET(matrix);

	gdk_draw_line(widget->window, matrix->xor_gc,
		      matrix->x_drag,
		      MATRIX_COL_TITLE_HEIGHT(matrix),
		      matrix->x_drag,
		      matrix->matrix_window_height + 1);
}



/* ROW RESIZING */
static void draw_xor_hline(GtkMatrix * matrix)
{
	GtkWidget *widget;

	g_return_if_fail(matrix != NULL);

	widget = GTK_WIDGET(matrix);

	gdk_draw_line(widget->window, matrix->xor_gc,
		      matrix->row_title_area.width,
		      matrix->y_drag,

		      matrix->matrix_window_width + 1,
		      matrix->y_drag);
}




/* this function returns the new width of the column being resized given
 * the column and x position of the cursor; the x cursor position is passed
 * in as a pointer and automagicly corrected if it's beyond min/max limits */
static gint
new_column_width(GtkMatrix * matrix,
		 gint column,
		 gint * x)
{
	gint cx, width;

	cx = *x;

	/* you can't shrink a column to less than its minimum width */
	if (cx < (COLUMN_LEFT_XPIXEL(matrix, column) + COLUMN_MIN_WIDTH)) {
		*x = cx = COLUMN_LEFT_XPIXEL(matrix, column) + COLUMN_MIN_WIDTH;
	}
	/* don't grow past the end of the window */
	/*
	   if (cx > matrix->matrix_window_width)
	   {
	   *x = cx = matrix->matrix_window_width;
	   }
	 */
	/* calculate new column width making sure it doesn't end up
	 * less than the minimum width */
	width = (cx - COLUMN_LEFT_XPIXEL(matrix, column));
	if (width < COLUMN_MIN_WIDTH)
		width = COLUMN_MIN_WIDTH;

	matrix->column[column].width = width;
	gtk_matrix_recalc_left_xpixels(matrix);
	matrix->view.coli = COLUMN_FROM_XPIXEL(matrix, matrix->matrix_window_width);
	size_allocate_column_title_buttons(matrix);

	return width;
}



/* this function returns the new height of the row being resized given
 * the row and y position of the cursor; the y cursor position is passed
 * in as a pointer and automagicly corrected if it's beyond min/max limits */
static gint
new_row_height(GtkMatrix * matrix,
	       gint row,
	       gint * y)
{
	gint cy, height;
	
	cy = *y;
	
	/* you can't shrink a row to less than its minimum width */
	if (cy < (ROW_TOP_YPIXEL(matrix, row) +
		  DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix)))) {
		*y = cy = ROW_TOP_YPIXEL(matrix, row) + DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix));
	}

	/* calculate new row height making sure it doesn't end up
	 * less than the minimum height */
	height = (cy - ROW_TOP_YPIXEL(matrix, row));
	if (height < DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix)))
		height = DEFAULT_ROW_HEIGHT(GTK_WIDGET(matrix));
	
	matrix->row[row].height = height;
	gtk_matrix_recalc_top_ypixels(matrix);
	matrix->view.rowi = ROW_FROM_YPIXEL(matrix, matrix->matrix_window_height - 1);
	size_allocate_row_title_buttons(matrix);

	return height;
}




void 
gtk_matrix_set_column_width(GtkMatrix *matrix,
			   gint      column,
			   gint      width)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (column < 0 || column > matrix->maxcol)
		return;

	matrix->column[column].width = width;

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)) && !GTK_MATRIX_IS_FROZEN(matrix)) {
		size_allocate_column_title_buttons(matrix);
		adjust_scrollbars(matrix);
		gtk_matrix_size_allocate_entry(matrix);
		gtk_matrix_range_draw(matrix, NULL);
	}
	
	gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[NEW_COL_WIDTH], column, width);
}




void gtk_matrix_set_row_height(GtkMatrix * matrix,
			      gint row,
			      gint height)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	if (row < 0 || row > matrix->maxrow)
		return;

	matrix->row[row].height = height;

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)) && !GTK_MATRIX_IS_FROZEN(matrix)) {
		size_allocate_row_title_buttons(matrix);
		adjust_scrollbars(matrix);
		gtk_matrix_size_allocate_entry(matrix);
		gtk_matrix_range_draw(matrix, NULL);
	}
	gtk_signal_emit(GTK_OBJECT(matrix), matrix_signals[NEW_ROW_HEIGHT], row, height);
}




void gtk_matrix_add_column(GtkMatrix * matrix, gint ncols)
{

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	AddColumn(matrix, ncols);

	if (!GTK_WIDGET_REALIZED(matrix))
		return;

	adjust_scrollbars(matrix);
	
	matrix->old_hadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");
}

void 
gtk_matrix_add_row(GtkMatrix *matrix, 
		  gint      nrows)
{

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	AddRow(matrix, nrows);
	
	if (!GTK_WIDGET_REALIZED(matrix))
		return;

	adjust_scrollbars(matrix);
	
	matrix->old_vadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");
}


void gtk_matrix_insert_rows(GtkMatrix * matrix, gint row, gint nrows)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	InsertRow(matrix, row, nrows);

	if (!GTK_WIDGET_REALIZED(matrix))
		return;

	adjust_scrollbars(matrix);

	matrix->old_vadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");

}

void gtk_matrix_insert_columns(GtkMatrix * matrix, gint col, gint ncols)
{
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	
	InsertColumn(matrix, col, ncols);

	if (!GTK_WIDGET_REALIZED(matrix))
		return;

	adjust_scrollbars(matrix);

	matrix->old_hadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");

}

void gtk_matrix_delete_rows(GtkMatrix * matrix, gint row, gint nrows)
{
	gint     act_row, act_col;
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));
	
	act_col = matrix->active_cell.col;
	act_row = matrix->active_cell.row;
	if (act_row >= row){
		act_row -= nrows;
		if (act_row < 0)
			act_row = 0;
	}
	
	gtk_matrix_deactivate_cell(matrix);
	
	
	DeleteRow(matrix, row, nrows);
	
	if (!GTK_WIDGET_REALIZED(matrix))
		return;
	
	gtk_matrix_activate_cell(matrix,act_row,act_col);
	//gtk_matrix_click_cell(matrix,matrix->active_cell.row,
	//matrix->active_cell.col,&veto);
	
	adjust_scrollbars(matrix);
	
	matrix->old_vadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->vadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->vadjustment),
					"value_changed");

}


void gtk_matrix_delete_columns(GtkMatrix * matrix, gint col, gint ncols)
{
	gboolean veto;
	
	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	DeleteColumn(matrix, col, ncols);

	if (!GTK_WIDGET_REALIZED(matrix))
		return;
	
	gtk_matrix_click_cell(matrix, matrix->active_cell.row,
			      matrix->active_cell.col, &veto);
	adjust_scrollbars(matrix);
	
	matrix->old_hadjustment = -1.;
	if (!GTK_MATRIX_IS_FROZEN(matrix) && matrix->hadjustment)
		gtk_signal_emit_by_name(GTK_OBJECT(matrix->hadjustment),
					"value_changed");

}


/**********************************************************************
 * Memory allocation routines: 
 * AddRow & AddColumn allocate memory for GtkMatrixColumn & GtkMatrixRow structs.
 * InsertRow 
 * InsertColumn
 * DeleteRow
 * DeleteColumn
 * GrowMatrix allocates memory for the matrix cells contents using an array of 
 * pointers. Alternative to this could be a linked list or a hash table.
 * CheckBounds checks whether the given cell is currently allocated or not. 
 * If not, it calls to GrowMatrix.
 **********************************************************************/

static gint
AddColumn(GtkMatrix * tbl, gint ncols)
{
	gint i;
	
	if (ncols == -1 && tbl->maxcol == 0) {
		ncols = 1;
	} else {
		tbl->maxcol += ncols;
		tbl->column = (GtkMatrixColumn *) g_realloc(tbl->column, (tbl->maxcol + 1) *
						 sizeof(GtkMatrixColumn));
	}

	for (i = tbl->maxcol - ncols + 1; i <= tbl->maxcol; i++) {
		tbl->column[i].width = DEFAULT_COLUMN_WIDTH;
		tbl->column[i].button.label = NULL;
		tbl->column[i].button.state = GTK_STATE_NORMAL;
		tbl->column[i].name = NULL;
		tbl->column[i].is_visible = TRUE;
		tbl->column[i].is_sensitive = TRUE;
		tbl->column[i].left_text_column = i;
		tbl->column[i].right_text_column = i;
	}
	return TRUE;
}


static gint
AddRow(GtkMatrix * tbl, gint nrows)
{
	gint i;

	if (nrows == -1 && tbl->maxrow == 0) {
		nrows = 1;
	} else {
		tbl->maxrow += nrows;
		tbl->row = (GtkMatrixRow *) g_realloc(tbl->row, (tbl->maxrow + 1) *
						     sizeof(GtkMatrixRow));
	}

	for (i = tbl->maxrow - nrows + 1; i <= tbl->maxrow; i++) {
		tbl->row[i].height = 0;
		tbl->row[i].button.label = NULL;
		tbl->row[i].button.state = GTK_STATE_NORMAL;
		tbl->row[i].name = NULL;
		tbl->row[i].is_visible = TRUE;
		tbl->row[i].is_sensitive = TRUE;
		
	}
	return TRUE;
}

static gint
InsertRow(GtkMatrix * tbl, gint row, gint nrows)
{
	GtkMatrixCell **pp;
	gint i, j;
	GtkMatrixCell **auxdata;
	GtkMatrixRow auxrow;

	AddRow(tbl, nrows);

	for (i = tbl->maxrow; i >= row + nrows; i--) {
		auxrow = tbl->row[i];
		tbl->row[i] = tbl->row[i - nrows];
		tbl->row[i].is_visible = tbl->row[i - nrows].is_visible;
		tbl->row[i].is_sensitive = tbl->row[i - nrows].is_sensitive;
		tbl->row[i - nrows] = auxrow;
	}

	if (row <= tbl->maxallocrow) {

		GrowMatrix(tbl, nrows, 0);

		for (i = tbl->maxallocrow; i >= row + nrows; i--) {
			auxdata = tbl->data[i];
			tbl->data[i] = tbl->data[i - nrows];

			pp = tbl->data[i];
			for (j = 0; j <= tbl->maxalloccol; j++, pp++) {
				if (*pp != (GtkMatrixCell *) NULL)
					(*pp)->row = i;

			}
			tbl->data[i - nrows] = auxdata;
		}
	}
	return TRUE;
}

static gint
InsertColumn(GtkMatrix * tbl, gint col, gint ncols)
{
	gint i, j;
	GtkMatrixColumn auxcol;

	AddColumn(tbl, ncols);

	for (i = tbl->maxcol; i >= col + ncols; i--) {
		auxcol = tbl->column[i];
		tbl->column[i] = tbl->column[i - ncols];
		tbl->column[i].is_visible = tbl->column[i - ncols].is_visible;
		tbl->column[i].is_sensitive = tbl->column[i - ncols].is_sensitive;
		tbl->column[i].left_text_column = tbl->column[i - ncols].left_text_column;
		tbl->column[i].right_text_column = tbl->column[i - ncols].right_text_column;
		tbl->column[i - ncols] = auxcol;
	}

	if (col <= tbl->maxalloccol) {

		GrowMatrix(tbl, 0, ncols);

		for (i = 0; i <= tbl->maxallocrow; i++) {
			for (j = tbl->maxalloccol; j >= col + ncols; j--) {
				tbl->data[i][j] = tbl->data[i][j - ncols];
				if (tbl->data[i][j])
					tbl->data[i][j]->col = j;
				tbl->data[i][j - ncols] = NULL;
			}
		}
	}
	
	return TRUE;
}

static gint
DeleteRow(GtkMatrix * tbl, gint row, gint nrows)
{
	GtkMatrixCell **pp;
	GtkMatrixCell **auxdata = NULL;
	gint i, j;

	nrows = MIN(nrows, tbl->maxrow - row + 1);

	for (i = row; i <= tbl->maxrow - nrows; i++) {
		tbl->row[i] = tbl->row[i + nrows];
		tbl->row[i].is_visible = tbl->row[i + nrows].is_visible;
		tbl->row[i].is_sensitive = tbl->row[i + nrows].is_sensitive;
	}

	if (row <= tbl->maxallocrow) {

		for (i = row; i <= tbl->maxrow - nrows; i++) {
			if (i <= tbl->maxallocrow) {
				auxdata = tbl->data[i];
				for (j = 0; j <= tbl->maxalloccol; j++) {
					if (tbl->data[i][j] && tbl->data[i][j]->text) {
						g_free(tbl->data[i][j]->text);
						tbl->data[i][j]->text = NULL;
					}
					tbl->data[i][j] = NULL;
				}
			}
			if (i + nrows <= tbl->maxallocrow) {
				tbl->data[i] = tbl->data[i + nrows];
				tbl->data[i + nrows] = auxdata;
				pp = tbl->data[i];
				for (j = 0; j <= tbl->maxalloccol; j++, pp++) {
					if (*pp != (GtkMatrixCell *) NULL)
						(*pp)->row = i;
				}
			}
		}

		tbl->maxallocrow -= MIN(nrows, tbl->maxallocrow - row);

	}
	tbl->maxrow -= nrows;
	return TRUE;
}


static gint
DeleteColumn(GtkMatrix * tbl, gint column, gint ncols)
{
	gint i, j;
	GtkMatrixColumn auxcol;

	ncols = MIN(ncols, tbl->maxcol - column + 1);

	for (i = column; i <= tbl->maxcol - ncols; i++) {
		auxcol = tbl->column[i];
		tbl->column[i] = tbl->column[i + ncols];
		tbl->column[i].is_visible = tbl->column[i + ncols].is_visible;
		tbl->column[i].is_sensitive = tbl->column[i + ncols].is_sensitive;
		tbl->column[i].left_text_column = tbl->column[i + ncols].left_text_column;
		tbl->column[i].right_text_column = tbl->column[i + ncols].right_text_column;
	
	}

	if (column <= tbl->maxalloccol) {

		for (i = column; i <= tbl->maxcol - ncols; i++) {
			if (i <= tbl->maxalloccol) {
				for (j = 0; j <= tbl->maxallocrow; j++)
					if (tbl->data[j][i])
						tbl->data[j][i] = NULL;
			}
			if (i + ncols <= tbl->maxalloccol) {
				for (j = 0; j <= tbl->maxallocrow; j++) {
					tbl->data[j][i] = tbl->data[j][i + ncols];
					if (tbl->data[j][i])
						tbl->data[j][i]->col = i;
				}
			}
		}

		tbl->maxalloccol -= MIN(ncols, tbl->maxalloccol - column);
	}
	tbl->maxcol -= ncols;
	return TRUE;
}

static gint
GrowMatrix(GtkMatrix * tbl, gint newrows, gint newcols)
{
	gint i, j;
	gint inirow, inicol;

	tbl->maxalloccol = tbl->maxalloccol == 0 ? newcols : tbl->maxalloccol + newcols;
	tbl->maxallocrow = tbl->maxallocrow == 0 ? newrows : tbl->maxallocrow + newrows;

	inirow = tbl->maxallocrow == newrows ? 0 : tbl->maxallocrow - newrows + 1;
	inicol = tbl->maxalloccol == newcols ? 0 : tbl->maxalloccol - newcols + 1;

	if (newrows > 0) {
		tbl->data = (GtkMatrixCell ***)
		    g_realloc(tbl->data, (tbl->maxallocrow + 1) * sizeof(GtkMatrixCell **) + sizeof(double));

		for (i = inirow; i <= tbl->maxallocrow; i++) {
			tbl->data[i] = (GtkMatrixCell **) \
			    g_malloc((tbl->maxcol + 1) * sizeof(GtkMatrixCell *) + sizeof(double));
			for (j = 0; j < inicol; j++) {
				tbl->data[i][j] = NULL;
			}
		}

	}
	if (newcols > 0) {
		for (i = 0; i <= tbl->maxallocrow; i++) {
			tbl->data[i] = (GtkMatrixCell **) \
			    g_realloc(tbl->data[i], (tbl->maxalloccol + 1) * sizeof(GtkMatrixCell *) + sizeof(double));
			for (j = inicol; j <= tbl->maxalloccol; j++) {
				tbl->data[i][j] = NULL;
			}
		}
	}
	return (0);
}

static gint
CheckBounds(GtkMatrix * tbl, gint row, gint col)
{
	gint newrows = 0, newcols = 0;

	if (col > tbl->maxalloccol)
		newcols = col - tbl->maxalloccol;
	if (row > tbl->maxallocrow)
		newrows = row - tbl->maxallocrow;
	if (newrows > 0 || newcols > 0)
		GrowMatrix(tbl, newrows, newcols);
	return (0);
}

/********************************************************************
 * Container Functions:
 * gtk_matrix_add
 * gtk_matrix_put
 * gtk_matrix_attach
 * gtk_matrix_remove
 * gtk_matrix_move_child
 * gtk_matrix_position_child
 * gtk_matrix_position_children 
 * gtk_matrix_realize_child
 ********************************************************************/

GtkMatrixChild *
gtk_matrix_put(GtkMatrix * matrix, GtkWidget * child, gint x, gint y)
{
	GtkRequisition child_requisition;
	GtkMatrixChild *child_info;

	g_return_val_if_fail(matrix != NULL, NULL);
	g_return_val_if_fail(GTK_IS_MATRIX(matrix), NULL);
	g_return_val_if_fail(child != NULL, NULL);
	g_return_val_if_fail(child->parent == NULL, NULL);

	child_info = g_new(GtkMatrixChild, 1);
	child_info->widget = child;
	child_info->x = x;
	child_info->y = y;
	child_info->window = NULL;
	child_info->attached_to_cell = FALSE;

	matrix->children = g_list_append(matrix->children, child_info);

	gtk_widget_set_parent(child, GTK_WIDGET(matrix));

	gtk_widget_size_request(child, &child_requisition);

	if (GTK_WIDGET_VISIBLE(GTK_WIDGET(matrix))) {

		if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)) &&
		    !GTK_WIDGET_REALIZED(child))
			gtk_matrix_realize_child(matrix, child_info);

		if (GTK_WIDGET_MAPPED(GTK_WIDGET(matrix)) &&
		    !GTK_WIDGET_MAPPED(child))
			gtk_widget_map(child);
	}
	gtk_matrix_position_child(matrix, child_info);

/* This will avoid drawing on the titles */

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix))) {
		if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			gdk_window_show(matrix->row_title_window);
		if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
			gdk_window_show(matrix->column_title_window);
	}
	return (child_info);
}



void gtk_matrix_attach(GtkMatrix * matrix, GtkWidget * widget, gint row, gint col,
		      gfloat x_align, gfloat y_align)
{
	GdkRectangle area;
	GtkMatrixChild *child;

	gtk_matrix_get_cell_area(matrix, row, col, &area);

	child = gtk_matrix_put(matrix, widget, area.x, area.y);

	child->attached_to_cell = TRUE;
	child->row = row;
	child->col = col;
	child->x_align = x_align;
	child->y_align = y_align;
}



void gtk_matrix_move_child(GtkMatrix * matrix, GtkWidget * widget, gint x, gint y)
{
	GtkMatrixChild *child;
	GList *children;

	g_return_if_fail(matrix != NULL);
	g_return_if_fail(GTK_IS_MATRIX(matrix));

	children = matrix->children;
	while (children) {
		child = children->data;
		children = children->next;

		if (child->widget == widget) {
			child->x = x;
			child->y = y;
			child->row = ROW_FROM_YPIXEL(matrix, y);
			child->col = COLUMN_FROM_XPIXEL(matrix, x);
			gtk_matrix_position_child(matrix, child);
			return;
		}
	}

	g_warning("Widget must be a GtkMatrix child");

}

static void gtk_matrix_position_child(GtkMatrix * matrix, GtkMatrixChild * child)
{
	GtkRequisition child_requisition;
	gint xoffset = 0;
	gint yoffset = 0;
	gint x, y;

	gtk_widget_get_child_requisition(child->widget, &child_requisition);

	if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
		yoffset = MATRIX_COL_TITLE_HEIGHT(matrix);

	if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
		xoffset = matrix->row_title_area.width;

	if (child->attached_to_cell) {
		child->x = COLUMN_LEFT_XPIXEL(matrix, child->col);
		child->y = ROW_TOP_YPIXEL(matrix, child->row);
		if (GTK_MATRIX_ROW_TITLES_VISIBLE(matrix))
			child->x -= matrix->row_title_area.width;
		if (GTK_MATRIX_COL_TITLES_VISIBLE(matrix))
			child->y -= MATRIX_COL_TITLE_HEIGHT(matrix);
		child->x += (matrix->column[child->col].width -
			     child_requisition.width) * child->x_align;
		child->y += ( MATRIX_ROW_HEIGHT(matrix,child->row) -
			      child_requisition.height) * child->y_align;
		x = child->widget->allocation.x = child->x + xoffset;
		y = child->widget->allocation.y = child->y + yoffset;
	} else {
		x = child->widget->allocation.x = child->x + matrix->hoffset + xoffset;
		y = child->widget->allocation.y = child->y + matrix->voffset + yoffset;
	}

	child->widget->allocation.width = child_requisition.width;
	child->widget->allocation.height = child_requisition.height;

	if (GTK_WIDGET_NO_WINDOW(child->widget)) {
		child->widget->allocation.x = 0;
		child->widget->allocation.y = 0;
	}
	if (GTK_WIDGET_REALIZED(GTK_WIDGET(matrix)) &&
	    GTK_WIDGET_MAPPED(child->widget)) {
		gtk_widget_size_allocate(child->widget,
					 &child->widget->allocation);
		if (GTK_WIDGET_NO_WINDOW(child->widget) && child->window) {
			gdk_window_move_resize(child->window,
					       x, y,
					 child->widget->allocation.width,
				       child->widget->allocation.height);
			gtk_widget_queue_draw(child->widget);
		}
	}
}

static void gtk_matrix_position_children(GtkMatrix * matrix)
{
	GList *children;
	GtkMatrixChild *child;

	children = matrix->children;

	while (children) {
		child = children->data;
		children = children->next;

		gtk_matrix_position_child(matrix, child);

	}

}

static void gtk_matrix_remove(GtkContainer * container, GtkWidget * widget)
{
	GtkMatrix *matrix;
	GList *children;
	GtkMatrixChild *child;

	g_return_if_fail(container != NULL);
	g_return_if_fail(GTK_IS_MATRIX(container));

	matrix = GTK_MATRIX(container);

	children = matrix->children;

	while (children) {
		child = children->data;

		if (child->widget == widget)
			break;

		children = children->next;
	}

	if (children) {
		if (child->window)
			gdk_window_destroy(child->window);

		gtk_widget_unparent(widget);
		child->widget = NULL;

		matrix->children = g_list_remove_link(matrix->children, children);
		g_list_free_1(children);
	}
}




static void 
gtk_matrix_realize_child(GtkMatrix * matrix, GtkMatrixChild * child)
{
	gint attributes_mask;
	GtkWidget *widget;

	widget = GTK_WIDGET(matrix);

	if (GTK_WIDGET_NO_WINDOW(child->widget)) {
		GdkWindowAttr attributes;

		gint x = child->x - matrix->hoffset;
		gint y = child->y - matrix->voffset;

		attributes.window_type = GDK_WINDOW_CHILD;
		attributes.x = x;
		attributes.y = y;
		attributes.width = child->widget->requisition.width;
		attributes.height = child->widget->requisition.height;
		attributes.wclass = GDK_INPUT_OUTPUT;
		attributes.visual = gtk_widget_get_visual(widget);
		attributes.colormap = gtk_widget_get_colormap(widget);
		attributes.event_mask = GDK_EXPOSURE_MASK;

		attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
		child->window = gdk_window_new(widget->window,
					   &attributes, attributes_mask);
		gdk_window_set_user_data(child->window, widget);

		if (child->window)
			gtk_style_set_background(widget->style,
						 child->window,
						 GTK_STATE_NORMAL);
		
		gtk_widget_set_parent_window(child->widget, child->window);
		gdk_window_show(child->window);
		
	}
	gtk_widget_realize(child->widget);

}





/***************************************
 * Desde aqui                          *
 ***************************************/



/*
 * gtk_matrix_create
 *
 * Create a new GtkMatrix Widget, and initialize
 * the rows, columns, cell memory mangement.
 */
GtkWidget *
gtk_matrix_create()
{
	GtkMatrix *matrix;
	
	matrix = gtk_type_new(gtk_matrix_get_type());
	
	matrix->row = (GtkMatrixRow *) g_malloc(sizeof(GtkMatrixRow));
	matrix->column = (GtkMatrixColumn *) g_malloc(sizeof(GtkMatrixColumn));
	matrix->data = (GtkMatrixCell ***) g_malloc(sizeof(GtkMatrixCell **));
	
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_ROW_TITLES_VISIBLE);
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_COL_TITLES_VISIBLE);
	
	/* set number of rows and columns */
	GrowMatrix(matrix, MINROWS, MINCOLS);
	
	/* Init row an column zero */
	AddRow(matrix, -1);
	AddColumn(matrix, -1);
	
	return GTK_WIDGET(matrix);
}





/*
 * gtk_matrix_new
 * @rows    : Number of rows
 * @colums  : Number of cols
 *
 * Create a new GtkMatrix Widget, and add
 * to it @rows and @columns.
 */
GtkWidget *
gtk_matrix_new(gint   rows,
	       gint   columns)
{
	GtkWidget *matrix;
	
	g_return_val_if_fail(columns >= MINCOLS, NULL);
	g_return_val_if_fail(rows >= MINROWS, NULL);
	
	matrix = gtk_matrix_create();
	gtk_matrix_set_entry(GTK_MATRIX(matrix),gtk_entry_new());
	
	/* Initialize the rows and columns */
	AddRow(GTK_MATRIX(matrix),rows-1);
	AddColumn(GTK_MATRIX(matrix),columns-1);
		
	return matrix;
}




/*
 * gtk_matrix_new_browser
 * @rows    : Number of rows
 * @colums  : Number of cols
 *
 * Create a new GtkMatrix Widget, add to it @rows and @columns, 
 * and set it to be not ediable (LOCKED).
 */
GtkWidget *
gtk_matrix_new_browser(gint   rows, 
		       gint   columns)
{
	GtkWidget *matrix;
	GtkWidget *entry;

	g_return_val_if_fail(columns >= MINCOLS, NULL);
	g_return_val_if_fail(rows >= MINROWS, NULL);
	
	matrix = gtk_matrix_create();
	entry = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(entry),FALSE);
	gtk_matrix_set_entry(GTK_MATRIX(matrix),entry);
			    
	AddRow(GTK_MATRIX(matrix),rows-1);
	AddColumn(GTK_MATRIX(matrix),columns-1);
	
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_IS_LOCKED);
	GTK_MATRIX_SET_FLAGS(matrix, GTK_MATRIX_AUTORESIZE);

	return matrix;
}




/*
 * gtk_matrix_new_with_custom_entry
 * @rows    : Number of rows
 * @colums  : Number of cols
 *
 * Create a new GtkMatrix Widget, add to it @rows and @columns, 
 * and set it to be not ediable (LOCKED).
 */
GtkWidget *
gtk_matrix_new_with_custom_entry(gint       rows, 
				 gint       columns,
				 GtkWidget *entry)
{
	GtkWidget *matrix;
	
	g_return_val_if_fail(columns >= MINCOLS, NULL);
	g_return_val_if_fail(rows >= MINROWS, NULL);
	g_return_val_if_fail(entry != NULL, NULL);
	
	matrix = gtk_matrix_create();
	gtk_matrix_set_entry(GTK_MATRIX(matrix),entry);
	
	AddRow(GTK_MATRIX(matrix),rows-1);
	AddColumn(GTK_MATRIX(matrix),columns-1);
	
	return matrix;
}



