/*
 * Run test scripts in parallel
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "ProcessRunner.h"

#include "Thread.h"
#include "Mutex.h"
#include "Exec.h"
#include "ChildProcess.h"
#include "Environment.h"

#include <sys/types.h>	// pid_t
#include <sys/wait.h>	// wait
#include <errno.h>

#include <map>
#include <queue>

using namespace std;
using namespace stringf;

//#define DEBUG(args...) fprintf(stderr, ##args)
#define DEBUG(args...) do {} while(0)


class Script : public Process
{
protected:
	string tag;
	string iface;
	string cmdline;

public:
	Script(const std::string& tag, const std::string& iface, const std::string& cmdline) throw ()
		: tag(tag), iface(iface), cmdline(cmdline) {}
	virtual ~Script() throw () {}

	virtual int main() throw ();
};

int Script::main() throw ()
{
	try {
		Exec runner("/bin/sh");

		runner.addEnv("NAME=" + tag);
		runner.addEnv("IFACE=" + iface);
		runner.addEnv("GUESSNET=true");

		runner.addArg("/bin/sh");
		runner.addArg("-c");
		runner.addArg(cmdline);

		DEBUG("SCRIPT MAIN Running %.*s\n", PFSTR(cmdline));

		runner.exec();
	} catch (Exception& e) {
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
	}
	return 1;
}

class ProcessRunnerImpl : public Thread
{
protected:
	int _ref;

	Mutex listenersMutex;
	Condition listenersCond;

	struct ProcData
	{
		string tag;
		Script* script;
		ChildProcess* proc;
		ProcessListener* listener;

		ProcData(const string& tag, const string& cmdline, ProcessListener* listener) throw ();
		void run() throw (SystemException);
	};

	queue<ProcData> queuedForRunning;
	bool requested_shutdown;
	bool running;

	virtual void* main() throw ();

public:
	ProcessRunnerImpl() throw () : _ref(0), requested_shutdown(false), running(true) {}
	~ProcessRunnerImpl() throw () { shutdown(); }

	/// Increment the reference count for this object
	void ref() throw () { ++_ref; }

	/// Decrement the reference count for this object, returning true when it
	/// reaches 0
	bool unref() throw () { return --_ref == 0; }

	void shutdown() throw (SystemException)
	{
		if (running)
		{
			{
				MutexLock lock(listenersMutex);
				requested_shutdown = true;
				DEBUG("PRI asked for shutdown\n");
			}
			// FIXME: if someone has a cleaner ideas...
			//kill(SIGCHLD);
			/*
			cancel();
			join();
			*/
			running = false;
		}
	}

	void addProcess(const std::string& tag, const std::string& cmdline, ProcessListener* pl) throw ();
};

ProcessRunnerImpl::ProcData::ProcData(const string& tag, const string& cmdline, ProcessListener* listener)
	throw () : tag(tag), listener(listener) 
{
	script = new Script(tag, Environment::get().iface(), cmdline);
	proc = new ChildProcess(script);
}

void ProcessRunnerImpl::ProcData::run() throw (SystemException)
{
	proc->fork();
	delete script;
	script = 0;
}

void ProcessRunnerImpl::addProcess(
						const std::string& tag,
						const std::string& cmdline,
						ProcessListener* pl) throw ()
{
	{
		MutexLock lock(listenersMutex);
		queuedForRunning.push(ProcData(tag, cmdline, pl));
		listenersCond.broadcast();
		DEBUG("PRI added process %.*s: %.*s\n", PFSTR(tag), PFSTR(cmdline));
	}
	// FIXME: if someone has a cleaner ideas...
	//kill(SIGCHLD);
}

void* ProcessRunnerImpl::main() throw ()
{
	// FIXME: Whatever the man pages tell about wait, it seems that  all
	// process handling needs to be done in the same thread.  With the bad
	// consequence that we can't use wait to sleep, since we have to keep track
	// of other things happening >:-(((
	// The wait(2) manpage states that wait should also wait for the children
	// of other threads with 2.4 kernels, but I have 2.4.20 and it didn't seem
	// to work (or I had other problems and made a wrong observation).


	// Let the signals be caught by some other process
	sigset_t sigs, oldsigs;
	sigfillset(&sigs);
	sigdelset(&sigs, SIGFPE);
	sigdelset(&sigs, SIGILL);
	sigdelset(&sigs, SIGSEGV);
	sigdelset(&sigs, SIGBUS);
	sigdelset(&sigs, SIGABRT);
	sigdelset(&sigs, SIGIOT);
	sigdelset(&sigs, SIGTRAP);
	sigdelset(&sigs, SIGSYS);
	// Don't block sigchld: we need it
	//sigdelset(&sigs, SIGCHLD);
	pthread_sigmask(SIG_SETMASK, &sigs, &oldsigs);

	map<pid_t, ProcData> proclist;

	while (true)
	{
		DEBUG("PRI Main Loop\n");
		bool want_shutdown = false;
		try {
			{
				// Wait for some child process to be run
				MutexLock lock(listenersMutex);

				DEBUG("PRI Check Requested Shutdown\n");

				if (requested_shutdown)
				{
					DEBUG("PRI Requested Shutdown\n");
					want_shutdown = true;
					requested_shutdown = false;
				} else {
					DEBUG("PRI Run queued processes\n");
					while (!queuedForRunning.empty())
					{
						DEBUG("PRI Run a queued process\n");
						ProcData d = queuedForRunning.front();
						queuedForRunning.pop();
						d.run();
						proclist.insert(make_pair(d.proc->pid(), d));
						debug("run process %.*s pid: %d\n", PFSTR(d.tag), d.proc->pid());
					}
				}
			}

			if (want_shutdown)
			{
				DEBUG("PRI Perform shutdown\n");
				debug("Requested shutdown\n");
				for (map<pid_t, ProcData>::iterator i = proclist.begin();
						i != proclist.end(); i++)
				{
					DEBUG("PRI Perform shutdown of %.*s\n", PFSTR(i->second.tag));
					debug("still running: %.*s (%d)\n", PFSTR(i->second.tag), i->second.proc->pid());
					i->second.proc->kill(9);
					i->second.proc->wait();
					delete i->second.proc;
					DEBUG("PRI Performed shutdown of %.*s\n", PFSTR(i->second.tag));
				}
				proclist.clear();
				DEBUG("PRI Performed shutdown\n");
				return 0;
			}

			//fprintf(stderr, "There are %d processes\n", proclist.size());
			
			DEBUG("PRI Checking Children\n");

			pid_t pid = 0;
			int status;

			if (!proclist.empty())
				pid = waitpid(-1, &status, WNOHANG);

			debug("Wait gave %d\n", pid);

			if (pid == -1)
				throw SystemException(errno, "Waiting for a child to terminate");
			if (pid == 0)
			{
				/*
				sigset_t wsigs;
				sigemptyset(&wsigs);
				sigaddset(&wsigs, SIGCHLD);
				int sig;
				debug("sigwaiting\n");
				sigwait(&wsigs, &sig);
				debug("sigwaited\n");
				*/
				
				struct timespec sleep_time = { 0, 100000000 };
				nanosleep(&sleep_time, 0);
				
			}
			else
			{
				debug("Process %d has terminated\n", pid);

				map<pid_t, ProcData>::iterator i = proclist.find(pid);
				if (i == proclist.end())
					throw ConsistencyCheckException("Child pid " + fmt(pid) + " exited, but has not been found in the child pid list");

				debug("Notifying termination of %.*s\n", PFSTR(i->second.tag));

				string tag = i->second.tag;
				
				// Remove the process from the proclist
				delete i->second.proc;
				proclist.erase(i);

				// Notify the listener
				i->second.listener->handleTermination(tag, status);
			}
		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		}
	}
}



ProcessRunner::ProcessRunner() throw (SystemException)
{
	impl = new ProcessRunnerImpl();
	impl->ref();
	impl->start();
}

ProcessRunner::ProcessRunner(const ProcessRunner& ns) throw ()
{
	if (ns.impl)
		ns.impl->ref();
	impl = ns.impl;
}

ProcessRunner::~ProcessRunner() throw (SystemException)
{
	if (impl && impl->unref())
		shutdown();
}

ProcessRunner& ProcessRunner::operator=(const ProcessRunner& ns) throw (SystemException)
{
	if (ns.impl)
		ns.impl->ref();  // Do it early to correctly handle the case of x = x;
	if (impl && impl->unref())
		delete impl;
	impl = ns.impl;
	return *this;
}

void ProcessRunner::addProcess(const std::string& tag, const std::string& cmdline, ProcessListener* pl)
	throw ()
{
	return impl->addProcess(tag, cmdline, pl);
}

void ProcessRunner::shutdown() throw (SystemException)
{
	impl->shutdown();
}

// vim:set ts=4 sw=4:
