// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Different message boxes we need all the time
// ****************************************************************************

#include "dlgmessage.h"

// -----------------------------
//           Constants
// -----------------------------

const int DLGMESSAGE_MAX_SCREEN_PERCENTAGE = 80;  // When opening the dialog, do not use more than 80% of the screen width / height

// -----------------------------
//           Classes
// -----------------------------

class t_DlgMessageLocal
{
   public:
      QVBoxLayout *pLayout;
      QTextEdit   *pTextBox;
      QPushButton *pButtonClose;
};

t_DlgMessage::t_DlgMessage ()
{
   CHK_EXIT (ERROR_DLGMESSAGE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 pOwn not initialised

t_DlgMessage::t_DlgMessage (const QString &Title, const QString &Message, bool Monospaced, QWidget *pParent, Qt::WFlags Flags)
   :QDialog (pParent, Flags)
{
   static bool Initialised = false;
   QSize       MaxSize;

   if (!Initialised)
   {
      Initialised = true;
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGMESSAGE_CONSTRUCTOR_NOT_SUPPORTED))
   }

   pOwn = new t_DlgMessageLocal;

   pOwn->pLayout      = new QVBoxLayout ();
   if (Monospaced)
        pOwn->pTextBox = new QTextEdit ("<pre>" + Message + "</pre>", this); // Use "pre" in order to have the text displayed in monospaced font and with CRLF interpreted the way we want
   else pOwn->pTextBox = new QTextEdit (          Message           , this);
   pOwn->pButtonClose = new QPushButton (tr("Close"), this);

   pOwn->pLayout->addWidget (pOwn->pTextBox    );
   pOwn->pLayout->addWidget (pOwn->pButtonClose);

   pOwn->pTextBox->setReadOnly (true);
   pOwn->pTextBox->zoomIn ();

   setLayout (pOwn->pLayout);
   setWindowTitle (Title);

   CHK_QT_EXIT (connect (pOwn->pButtonClose, SIGNAL (released()), this, SLOT(accept())))
}

void t_DlgMessage::AdjustGeometry (void)
{
   int X, Y;
   int Dx, Dy;

   // Trying to adjust the size of the dialog to size of the text inside. Not easy, as Qt doesn't provide the right functions for
   // doing so. Important: AdjustSize must be called after the dialog has been showed in order have QTextBox calculated all scroll
   // bar values.
   Dx  = pOwn->pTextBox->horizontalScrollBar()->maximum();     // See source code of QAbstractScrollArea: The scroll bar range is the full
   Dy  = pOwn->pTextBox->verticalScrollBar  ()->maximum();     // scroll area width minus the window area displayed. So, we, take the
   Dx += pOwn->pTextBox->viewport()->width ();                 // scroll bar range and add the viewport size.
   Dy += pOwn->pTextBox->viewport()->height();
   Dx += pOwn->pTextBox->verticalScrollBar  ()->height();      // Scrollbars need some place as well...
   Dy += pOwn->pTextBox->horizontalScrollBar()->height();
   Dx += frameGeometry().width () - pOwn->pTextBox->width ();  // Add the space surrounding the Textbox required by the dialog
   Dy += frameGeometry().height() - pOwn->pTextBox->height();

   Dx = std::min (Dx, (DLGMESSAGE_MAX_SCREEN_PERCENTAGE * QApplication::desktop()->width ()) / 100);  // Limit to a certain amount
   Dy = std::min (Dy, (DLGMESSAGE_MAX_SCREEN_PERCENTAGE * QApplication::desktop()->height()) / 100);  // of the available screen space

   X = (QApplication::desktop()->width  () - Dx) / 2;
   Y = (QApplication::desktop()->height () - Dy) / 2;
   setGeometry (X, Y, Dx, Dy);
}

t_DlgMessage::~t_DlgMessage ()
{
   delete pOwn;
}

APIRET t_DlgMessage::Show (const QString &Title, const QString &Message, bool Monospaced)
{
   t_DlgMessage *pDlg;

   pDlg = new t_DlgMessage (Title, Message, Monospaced);
   pDlg->setModal      (true);
   pDlg->show          ();
   pDlg->AdjustGeometry();
   pDlg->exec          ();
   delete pDlg;

   return NO_ERROR;
}

