/*  -*-objc-*-
 *  SmallIconsViewer.m: Implementation of the SmallIconsViewer Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2002 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: February 2002
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "SmallIconsViewer.h"
#include "Banner.h"
#include "PathsPopUp.h"
#include "SmallIconsPanel.h"
#include "Functions.h"
#include "GNUstep.h"

/* Notifications */
#ifndef GNUSTEP
	NSString *GWFileSystemWillChangeNotification = @"GWFileSystemWillChangeNotification";
	NSString *GWFileSystemDidChangeNotification = @"GWFileSystemDidChangeNotification";
	NSString *GWDidSetFileAttributesNotification = @"GWDidSetFileAttributesNotification";
	NSString *GWSortTypeDidChangeNotification = @"GWSortTypeDidChangeNotification";
#else
	extern NSString *GWFileSystemWillChangeNotification;
	extern NSString *GWFileSystemDidChangeNotification;
	extern NSString *GWDidSetFileAttributesNotification;
	extern NSString *GWSortTypeDidChangeNotification;
#endif

/* File Operations */
#ifndef GNUSTEP
	NSString *GWorkspaceRenameOperation = @"GWorkspaceRenameOperation";
	NSString *GWorkspaceCreateFileOperation = @"GWorkspaceCreateFileOperation";
	NSString *GWorkspaceCreateDirOperation = @"GWorkspaceCreateDirOperation";
	NSString *GWorkspaceRecycleOutOperation = @"GWorkspaceRecycleOutOperation";
	NSString *GWorkspaceEmptyRecyclerOperation = @"GWorkspaceEmptyRecyclerOperation";
#else
	extern NSString *GWorkspaceRenameOperation;
	extern NSString *GWorkspaceCreateFileOperation;
	extern NSString *GWorkspaceCreateDirOperation;
	extern NSString *GWorkspaceRecycleOutOperation;
	extern NSString *GWorkspaceEmptyRecyclerOperation;
#endif

/* File Watcher Strings */
#ifndef GNUSTEP
	NSString *GWFileWatcherFileDidChangeNotification = @"GWFileWatcherFileDidChangeNotification";
	NSString *GWWatchedDirectoryDeleted = @"GWWatchedDirectoryDeleted";
	NSString *GWFileDeletedInWatchedDirectory = @"GWFileDeletedInWatchedDirectory";
	NSString *GWFileCreatedInWatchedDirectory = @"GWFileCreatedInWatchedDirectory";
#else
	extern NSString *GWFileWatcherFileDidChangeNotification;
	extern NSString *GWWatchedDirectoryDeleted;
	extern NSString *GWFileDeletedInWatchedDirectory;
	extern NSString *GWFileCreatedInWatchedDirectory;
#endif

@implementation SmallIconsViewer

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver: self];
  TEST_RELEASE (currentPath);
  TEST_RELEASE (selectedPaths);
  TEST_RELEASE (watchedPaths);
	TEST_RELEASE (banner);
  TEST_RELEASE (panelScroll);  
  TEST_RELEASE (panel);
  [super dealloc];
}

- (id)init
{
	self = [super initWithFrame: NSZeroRect];
  currentPath = nil;
  selectedPaths = nil;
  watchedPaths = nil;
  panelScroll = nil;  
	banner = nil;
  panel = nil;	
	return self;
}

- (void)makePopUp:(NSArray *)pathComps
{
	NSArray *titles = [pathsPopUp itemTitles];
	int i;		
		
	if ((titles != nil) && ([titles count] != 0)) {
		if ([titles isEqualToArray: pathComps]) {
		  return;
		} else {
			[pathsPopUp removeAllItems];
		}
	}

	for (i = [pathComps count] -1; i >= 0; i--) {
		[pathsPopUp addItemWithTitle: [pathComps objectAtIndex: i]]; 	
	}
}

- (void)popUpAction:(id)sender
{
	NSArray *titles = [sender itemTitles];
	NSString *title = [sender titleOfSelectedItem];
	NSString *path = @"/";
	int i = [titles count] -1;
	
	while (1) {
		NSString *s = [titles objectAtIndex: i];
		path = [path stringByAppendingPathComponent: s];
		if ([s isEqualToString: title]) {
			break;
		}		
		i--;
	}
	
	if ([currentPath isEqualToString: path] == NO) {
		BOOL newview = [pathsPopUp newViewer];
		[self openCurrentSelection: [NSArray arrayWithObject: path] 
										 newViewer: newview];
		if (newview) {
			[self makePopUp: [currentPath pathComponents]];
		}								 
	}
}

- (void)fileSystemWillChange:(NSNotification *)notification
{
  NSDictionary *dict;
  NSString *operation, *source, *destination;
  NSArray *files;
     
  dict = (NSDictionary *)[notification object];
  operation = [dict objectForKey: @"operation"];
  source = [dict objectForKey: @"source"];
  destination = [dict objectForKey: @"destination"];
  files = [dict objectForKey: @"files"];

  if ([operation isEqualToString: NSWorkspaceMoveOperation] 
      	|| [operation isEqualToString: NSWorkspaceDestroyOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
				|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
				|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) { 
    
    if ([[panel currentPath] isEqualToString: source]) {
      [panel removeIconsWithNames: files];
      [self setCurrentSelection: [NSArray arrayWithObject: source]];
      
    } else {
      if (isSubPath(source, currentPath) == NO) {     	// E' SBAGLIATO ????
        NSArray *components = [[panel currentPath] pathComponents];  
        NSMutableArray *paths = [NSMutableArray arrayWithCapacity: 1];  
        NSString *s = [NSString string];        
        int i;

        for (i = 0; i < [components count]; i++) {
          s = [s stringByAppendingPathComponent: [components objectAtIndex: i]];
          if (isSubPath(s, currentPath) == NO) {
            [paths addObject: s];  
          }         
        }  

        for (i = 0; i < [files count]; i++) {  
          NSString *fname = [files objectAtIndex: i];
          NSString *fpath = [source stringByAppendingPathComponent: fname];

          if ([paths containsObject: fpath]) {
            fpath = [fpath stringByDeletingLastPathComponent];
            [self setCurrentSelection: [NSArray arrayWithObject: fpath]];
            [panel removeIconsWithNames: files];
            break;
          }
        }
      }
    }
  }
  
  [self validateCurrentPathAfterOperation: dict];
}

- (void)fileSystemDidChange:(NSNotification *)notification
{
  NSDictionary *dict;
  NSString *operation, *source, *destination;
  NSArray *files;
	  
  dict = (NSDictionary *)[notification object];    
  operation = [dict objectForKey: @"operation"];
  source = [dict objectForKey: @"source"];
  destination = [dict objectForKey: @"destination"];
  files = [dict objectForKey: @"files"];
    
  if ([operation isEqualToString: GWorkspaceRenameOperation]) {
    destination = [destination stringByDeletingLastPathComponent];
  }  

  if (isSubPath(destination, currentPath) == NO) {
    if ([[panel currentPath] isEqualToString: destination] == YES) {
      [panel makeFileIcons];
			[panel resizeWithOldSuperviewSize: [panel frame].size];  
      [self setCurrentSelection: [NSArray arrayWithObject: destination]];
    }

    if ([operation isEqualToString: GWorkspaceCreateFileOperation] 
          || [operation isEqualToString: GWorkspaceCreateDirOperation]) { 
      NSString *filePath = [destination stringByAppendingPathComponent: [files objectAtIndex: 0]];
      [panel selectIconsWithPaths: [NSArray arrayWithObject: filePath]];
    }
  }    
  
  [self validateCurrentPathAfterOperation: dict];  
  [self updateDiskInfo];
}

- (void)sortTypeDidChange:(NSNotification *)notification
{
	NSString *notifPath = [notification object];

	if (notifPath != nil) {
		if ([currentPath isEqualToString: notifPath]) {
			[panel makeFileIcons];
			[panel resizeWithOldSuperviewSize: [panel frame].size];  
		}
	} else {
		[panel makeFileIcons];
		[panel resizeWithOldSuperviewSize: [panel frame].size];  
	}
}

- (void)validateCurrentPathAfterOperation:(NSDictionary *)opdict
{
  if ([currentPath isEqualToString: @"/"] == YES) {
    return;
  } else {
    NSString *operation = [opdict objectForKey: @"operation"];
    NSString *source = [opdict objectForKey: @"source"];
    NSArray *files = [opdict objectForKey: @"files"];
    int i;
    
    if ([operation isEqualToString: NSWorkspaceMoveOperation] 
        || [operation isEqualToString: NSWorkspaceDestroyOperation]
        || [operation isEqualToString: GWorkspaceRenameOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
				|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
				|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) { 

      if ([operation isEqualToString: GWorkspaceRenameOperation]) {      
        files = [NSArray arrayWithObject: [source lastPathComponent]]; 
        source = [source stringByDeletingLastPathComponent];            
      } 
      
      for (i = 0; i < [files count]; i++) {
        NSString *fpath = [source stringByAppendingPathComponent: [files objectAtIndex: i]];

        if (isSubPath(fpath, currentPath)
                        	|| [fpath isEqualToString: currentPath]) {  
          [self closeNicely];      
          break;
        }
      }
    }
  }  
}

- (void)watcherNotification:(NSNotification *)notification
{
  NSDictionary *notifdict = (NSDictionary *)[notification object];
  NSString *path = [notifdict objectForKey: @"path"];
	
  if ([watchedPaths containsObject: path] == NO) {
    return;    

  } else {
    NSString *event = [notifdict objectForKey: @"event"];
		int i;
		
    if (event == GWWatchedDirectoryDeleted) {
      if ((isSubPath(path, currentPath)) || ([path isEqualToString: currentPath])) {  
        [self closeNicely];      
        return;
      } else {
        NSString *s = [path stringByDeletingLastPathComponent];
        [self unsetWatcherForPath: path];        
        [self setCurrentSelection: [NSArray arrayWithObject: s]];
        return;
      }
    }

    if (event == GWFileDeletedInWatchedDirectory) {
      if ([path isEqualToString: [panel currentPath]] == NO) {
        return;       
      } else {
				NSArray *files;
				int count;
			
				files = [notifdict objectForKey: @"files"];
        [panel removeIconsWithNames: files];
				
				count = 0;
				if (selectedPaths != nil) {
					count = [selectedPaths count];
				}
				
				if (count > 0) {
					NSMutableArray *selpaths = [selectedPaths mutableCopy];
								
					for (i = 0; i < count; i++) {
						NSString *fname = [[selpaths objectAtIndex: i] lastPathComponent];

						if ([files containsObject: fname] == YES) {
  						[selpaths removeObjectAtIndex: i];
  						count--;
  						i--;
						}
					}
				
					if ([selpaths count] > 0) {
						[self setCurrentSelection: selpaths];
					} else {
						[self setCurrentSelection: [NSArray arrayWithObject: [panel currentPath]]];
					}
				
					RELEASE (selpaths);
				} else {
					[self setCurrentSelection: [NSArray arrayWithObject: [panel currentPath]]];
				}
				
        return;
      }
    }
    
    if (event == GWFileCreatedInWatchedDirectory) {
      if ([path isEqualToString: [panel currentPath]] == NO) {
        return; 
      } else {
        [panel makeFileIcons];
				[panel resizeWithOldSuperviewSize: [panel frame].size];  
      }
    }

  }
  
  [self updateDiskInfo];
}

- (void)setWatchers
{
  int i;
  	
  for (i = 0; i < [watchedPaths count]; i++) {
    [self setWatcherForPath: [watchedPaths objectAtIndex: i]];  
  }

  [[NSNotificationCenter defaultCenter] addObserver: self 
                		selector: @selector(watcherNotification:) 
                				name: GWFileWatcherFileDidChangeNotification
                			object: nil];
}

- (void)setWatcherForPath:(NSString *)path
{
	[delegate addWatcherForPath: path];
}

- (void)unsetWatcherForPath:(NSString *)path
{
	[delegate removeWatcherForPath: path];
}

- (void)updateDiskInfo
{
  NSDictionary *attributes;
	NSNumber *freeFs;
  NSString *infoString;
  NSDictionary *infoDict;
	
	attributes = [fm fileSystemAttributesAtPath: currentPath];
	freeFs = [attributes objectForKey: NSFileSystemFreeSize];
  
	if(freeFs == nil) {  
		infoString = [NSString stringWithString: @"unknown size"];    
	} else {
		infoString = [NSString stringWithFormat: @"%@ available", 
                        fileSizeDesc([freeFs unsignedLongLongValue])];
	}
  	
	infoDict = [NSDictionary dictionaryWithObject: infoString 
																				 forKey: @"freespace"];
  [banner updateInfo: infoDict];
}

- (void)setSelectedIconsPaths:(NSArray *)paths
{
  if (([paths count] == 0) || ([paths isEqualToArray: selectedPaths] == YES)) {
    return;
  }

  ASSIGN (selectedPaths, paths);
	[delegate setTheSelectedPaths: selectedPaths];
}

- (void)openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
  NSString *defApp, *type;
  NSMutableArray *files, *dirs, *apps;
  int i;
  
	if (newv == NO) {
  	[self setSelectedPaths: paths];
  }
	  
  files = [NSMutableArray arrayWithCapacity: 1]; 
  apps = [NSMutableArray arrayWithCapacity: 1]; 
  dirs = [NSMutableArray arrayWithCapacity: 1]; 
      
  for (i = 0; i < [paths count]; i++) {
    NSString *path = [paths objectAtIndex: i];
    
    [[NSWorkspace sharedWorkspace] getInfoForFile: path application: &defApp type: &type];     

    if ([type isEqualToString: NSPlainFileType]) {
      [files addObject: path];  
    } else if ([type isEqualToString: NSApplicationFileType]) {
      [apps addObject: path];      
    } else if (([type isEqualToString: NSDirectoryFileType])
                		|| ([type isEqualToString: NSFilesystemFileType])) { 
      [dirs addObject: path];      
    }
  }
        
  for (i = 0; i < [files count]; i++) {		
		[delegate openFile: [files objectAtIndex: i]];
  }

  for (i = 0; i < [apps count]; i++) {
    NSString *path = [apps objectAtIndex: i];
    
    if (newv == YES) {
			[delegate newViewerAtPath: path canViewApps: YES];
    } else {
      [[NSWorkspace sharedWorkspace] launchApplication: path];
    }
  }
    
  if (([dirs count] == 1) && ([paths count] == 1)) {  
    if (newv == YES) {    		
			[delegate newViewerAtPath: [dirs objectAtIndex: 0] canViewApps: NO];
    } else {
			[delegate addPathToHistory: dirs];
      [panel setPath: [dirs objectAtIndex: 0]];
    }
  }
}

- (void)closeNicely
{
  NSTimer *t;
  
  [self unsetWatchers]; 
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  t = [NSTimer timerWithTimeInterval: 1 target: self 
          selector: @selector(close:) userInfo: nil repeats: NO];                                             
  [[NSRunLoop currentRunLoop] addTimer: t forMode: NSDefaultRunLoopMode];
}

- (void)close:(id)sender
{
  [[self window] performClose: nil];
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  NSRect r = [self frame];
	float w = r.size.width;
	float h = r.size.height;
	int winwidth = [delegate getWindowFrameWidth];
	
	[banner setFrame: NSMakeRect(0, h - 30, w, 30)];
	[banner resizeWithOldSuperviewSize: [banner frame].size];
	
  [panelScroll setFrame: NSMakeRect(0, 0, w, h - 30)];
				
  columns = (int)winwidth / resizeIncrement;
  columnsWidth = (winwidth - 16) / columns;	
}

//
// ViewersProtocol
//
- (void)setPath:(NSString *)path 
       delegate:(id)adelegate
       viewApps:(BOOL)canview
{
	int colswidth, winwidth;

  fm = [NSFileManager defaultManager];

	[self setDelegate: adelegate];
  ASSIGN (currentPath, path);	
  viewsapps = canview;

	colswidth = [delegate getBrowserColumnsWidth];
	resizeIncrement = colswidth;			
	winwidth = [delegate getWindowFrameWidth];			
	columns = (int)winwidth / resizeIncrement;      
	columnsWidth = (winwidth - 16) / columns;		

  [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

	if (banner != nil) {
		[banner removeFromSuperview];
		RELEASE (banner);
	}

	banner = [Banner new];
	pathsPopUp = [banner pathsPopUp];
	[pathsPopUp setTarget: self];
	[pathsPopUp setAction: @selector(popUpAction:)];
	[self addSubview: banner]; 

	TEST_RELEASE (panel);

	if (panelScroll != nil) {
		[panelScroll removeFromSuperview];
		RELEASE (panelScroll);
	}
  panelScroll = [NSScrollView new];
  [panelScroll setHasHorizontalScroller: YES];
  [panelScroll setHasVerticalScroller: YES]; 
  [panelScroll setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];      

  panel = [[SmallIconsPanel alloc] initAtPath: currentPath delegate: self];

  [panelScroll setDocumentView: panel];	
  [self addSubview: panelScroll]; 

	[[NSNotificationCenter defaultCenter] removeObserver: self];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(fileSystemWillChange:) 
                					  name: GWFileSystemWillChangeNotification
                					object: nil];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(fileSystemDidChange:) 
                					  name: GWFileSystemDidChangeNotification
                					object: nil];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(sortTypeDidChange:) 
                					  name: GWSortTypeDidChangeNotification
                					object: nil];

	if (watchedPaths != nil) {
		[self unsetWatchers];
		RELEASE (watchedPaths);
	}
	watchedPaths = nil;

  [self setSelectedPaths: [NSArray arrayWithObject: currentPath]]; 
	[self resizeWithOldSuperviewSize: [self frame].size];
	[self updateDiskInfo];
}

- (NSString *)menuName
{
	return @"Small Icon";
}

- (NSString *)shortCut
{
	return @"k";
}

- (BOOL)usesShelf
{
	return NO;
}

- (NSImage *)preview
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"preview" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];	
	return AUTORELEASE (img);
}

- (NSImage *)miniicon
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"miniwindow" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];	
	return AUTORELEASE (img);
}

- (void)setSelectedPaths:(NSArray *)paths
{
  NSString *fullPath;
  NSArray *components;
  NSMutableArray *wpaths;
  NSString *s;
	NSString *defApp, *type;
  BOOL isDir;
  int i, j;			
			
  if ([paths count] == 0) {
    return;
  }

  ASSIGN (selectedPaths, paths);
    
  fullPath = [paths objectAtIndex: 0]; 
  [fm fileExistsAtPath: fullPath isDirectory: &isDir]; 
  if ((isDir == NO) || ([paths count] > 1)) {
    fullPath = [fullPath stringByDeletingLastPathComponent];
  } else {
		[[NSWorkspace sharedWorkspace] getInfoForFile: fullPath application: &defApp type: &type]; 
		if ((type == NSApplicationFileType) && (viewsapps == NO)) {
			fullPath = [fullPath stringByDeletingLastPathComponent];
		}
	}
	
	ASSIGN (currentPath, fullPath);
	
	[delegate setTitleAndPath: currentPath selectedPaths: selectedPaths];

  components = [currentPath pathComponents];  
	[self makePopUp: components];
	
  wpaths = [NSMutableArray arrayWithCapacity: 1];  
  s = [NSString string];        
   
  for (i = 0; i < [components count]; i++) {  
    s = [s stringByAppendingPathComponent: [components objectAtIndex: i]];
    [wpaths addObject: s];            
  }  
		
  if (watchedPaths == nil) {          
    ASSIGN (watchedPaths, wpaths);  
    [self setWatchers];

  } else {  
    int count = [wpaths count];
  
    for (i = 0; i < [watchedPaths count]; i++) {
      NSString *s1, *s2;

      s1 = [watchedPaths objectAtIndex: i];

      if (count > i) {
        s2 = [wpaths objectAtIndex: i];  
      } else {
        i = count;
        break;
      }

      if ([s1 isEqualToString: s2] == NO) {
        break;
      }    
    }

    for (j = i; j < [watchedPaths count]; j++) {  
      [self unsetWatcherForPath: [watchedPaths objectAtIndex: j]];
    }

    for (j = i; j < [wpaths count]; j++) {  
      [self setWatcherForPath: [wpaths objectAtIndex: j]];
    }

    ASSIGN (watchedPaths, wpaths);
  }
}

- (NSArray *)selectedPaths
{
  return selectedPaths;
}

- (NSPoint)locationOfIconForPath:(NSString *)path
{	
	NSString *name = [path lastPathComponent];
	NSPoint p = [panel locationOfIconWithName: name];
	
	if ((p.x == 0) && (p.y == 0)) {
		return p;
		
	} else { 
		NSView *sview = [[self window] contentView];
		NSRect r = [self visibleRect];
		NSPoint lim1 = r.origin;
		NSPoint lim2 = NSMakePoint(lim1.x + r.size.width, lim1.y + r.size.height);

		p = [sview convertPoint: p fromView: panel];
		lim1 = [sview convertPoint: lim1 fromView: self];
		lim2 = [sview convertPoint: lim2 fromView: self];

		if (p.x < lim1.x) p.x = lim1.x;
		if (p.y < lim1.y) p.y = lim1.y;

		if (p.x >= lim2.x) p.x = lim2.x - 60;
		if (p.y >= lim2.y) p.y = lim2.y - 60;

		if (p.x == 0) p.x = 1;
		if (p.y == 0) p.y = 1;
	}
	
	return p;
}

- (void)setCurrentSelection:(NSArray *)paths
{
  NSString *path, *defApp, *type;
  NSArray *selection;
	
  if (([paths count] == 0) || ([paths isEqualToArray: selectedPaths] == YES)) {
    return;
  }
  
  [self setSelectedPaths: paths];
  
  path = [paths objectAtIndex: 0];
  
  if ([paths count] > 1) {
    path = [path stringByDeletingLastPathComponent];
    selection = [NSArray arrayWithArray: selectedPaths];
  } else {
    BOOL isdir;
  
    [fm fileExistsAtPath: path isDirectory: &isdir];
  
    if (isdir == NO) {
      path = [path stringByDeletingLastPathComponent];
      selection = [NSArray arrayWithArray: selectedPaths];
    } else {
			[[NSWorkspace sharedWorkspace] getInfoForFile: path application: &defApp type: &type]; 
			if ((type == NSApplicationFileType) && (viewsapps == NO)) {
				path = [path stringByDeletingLastPathComponent];
			}
      selection = [NSArray array];
    }
  }
  
  [panel setPath: path];  
  [panel selectIconsWithPaths: selection];
  [self updateDiskInfo];
}

- (void)unsetWatchers
{
  int i;

  [[NSNotificationCenter defaultCenter] removeObserver: self 
                name: GWFileWatcherFileDidChangeNotification object: nil];

  for (i = 0; i < [watchedPaths count]; i++) {
    [self unsetWatcherForPath: [watchedPaths objectAtIndex: i]]; 
  }
}

- (void)setResizeIncrement:(int)increment
{
	resizeIncrement = increment;
}

- (void)setAutoSynchronize:(BOOL)value
{
	autoSynchronize = value;
}

- (id)viewerView
{
	return panel;
}

- (BOOL)viewsApps
{
	return viewsapps;
}

- (void)selectAll
{
	[panel selectAllIcons];
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)anObject
{	
  delegate = anObject;
}

@end


//
// SmallIconsPanel delegate methods
//

@implementation SmallIconsViewer (SmallIconsPanelDelegateMethods)

- (NSImage *)smallIconForFile:(NSString*)aPath
{
	return [delegate smallIconForFile: aPath];
}

- (NSImage *)smallIconForFiles:(NSArray*)pathArray
{
	return [delegate smallIconForFiles: pathArray];
}

- (void)setTheSelectedPaths:(id)paths
{
	[self setSelectedIconsPaths: paths];
}

- (void)openTheCurrentSelection:(id)paths newViewer:(BOOL)newv
{
	[self openCurrentSelection: paths newViewer: newv];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (int)sortTypeForDirectoryAtPath:(id)path
{
	return [delegate sortTypeForDirectoryAtPath: path];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

- (int)getIconViewCellsWidth
{	
	return [delegate getIconViewCellsWidth];
}

@end

