 /*
 *  Backgrounder.m: Implementation of the Backgrounder Class 
 *  of the backgrounder tool for the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "backgrounder.h"
#include "GNUstep.h"

#define VERIFYSTOP \
if (stopped) { \
result = [gwFileOp sendDidChangeNotification]; \
[gwFileOp endOperation]; \
exit(0); \
}

/* File Operations */
NSString *NSWorkspaceMoveOperation = @"NSWorkspaceMoveOperation";
NSString *NSWorkspaceCopyOperation = @"NSWorkspaceCopyOperation";
NSString *NSWorkspaceLinkOperation = @"NSWorkspaceLinkOperation";
NSString *NSWorkspaceDestroyOperation = @"NSWorkspaceDestroyOperation";
NSString *NSWorkspaceDuplicateOperation = @"NSWorkspaceDuplicateOperation";
NSString *NSWorkspaceRecycleOperation = @"NSWorkspaceRecycleOperation";
NSString *GWorkspaceRecycleOutOperation = @"GWorkspaceRecycleOutOperation";
NSString *GWorkspaceEmptyRecyclerOperation = @"GWorkspaceEmptyRecyclerOperation";

@implementation Backgrounder

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver: self];
  TEST_RELEASE (operation);
  TEST_RELEASE (source);
  TEST_RELEASE (destination);
  TEST_RELEASE (files);
	[super dealloc];
}

- (id)initWithArgc:(int)argc argv:(char **)argv
{  
	self = [super init];
  if(self) {
    NSString *connName;
    NSConnection *connection;
    id anObject;
    NSDictionary *dict;
    id dictEntry;
    int i, ref;
    
    fm = [NSFileManager defaultManager];
    
    ref = [[NSString stringWithCString: argv[1]] intValue];    
    connName = [NSString stringWithFormat: @"file_operation_with_ref_%i", ref];

    connection = [NSConnection connectionWithRegisteredName: connName host: @""];
    if (connection == nil) {
      NSLog(@"backgrounder - failed to get the connection - bye.");
	    exit(1);               
    }

    anObject = [connection rootProxy];
    
    if (anObject == nil) {
      NSLog(@"backgrounder - failed to contact gworkspace - bye.");
	    exit(1);           
    } 

    [anObject setProtocolForProxy: @protocol(FileOperationProtocol)];
    gwFileOp = (id <FileOperationProtocol>)anObject;

    [[NSNotificationCenter defaultCenter] addObserver: self
                            selector: @selector(connectionDidDie:)
                                name: NSConnectionDidDieNotification
                              object: connection];    
    
    dict = [gwFileOp operationDict];
    dictEntry = [dict objectForKey: @"operation"];
    if (dictEntry != nil) {
      ASSIGN (operation, [NSString stringWithString: dictEntry]);   
    }
    
    dictEntry = [dict objectForKey: @"source"];
    if (dictEntry != nil) {
      ASSIGN (source, [NSString stringWithString: dictEntry]);
    }  
      
    dictEntry = [dict objectForKey: @"destination"];
    if (dictEntry != nil) {
      ASSIGN (destination, [NSString stringWithString: dictEntry]);
    }  
      
    files = [[NSMutableArray alloc] initWithCapacity: 1];
    dictEntry = [dict objectForKey: @"files"];
    if (dictEntry != nil) {
      for (i = 0; i < [dictEntry count]; i++) {
        [files addObject: [dictEntry objectAtIndex: i]];
      }
    }		
		
    started = NO;
		stopped = NO;
    [self prepareOperation];
  }
    
	return self;
}

- (void)prepareOperation
{
	NSArray *dirContents;
	NSString *msg, *title = nil;
	int i, j, result;
    
	samename = NO;
	
	if (destination && [files count]) {
		dirContents = [fm directoryContentsAtPath: destination];
		for (i = 0; i < [files count]; i++) {
			for (j = 0; j < [dirContents count]; j++) {
				if ([[files objectAtIndex: i] isEqualToString: [dirContents objectAtIndex: j]]) {
					samename = YES;
					break;
				}
			}
		}
	}
	
	if (samename) {
		if ([operation isEqualToString: NSWorkspaceMoveOperation]) {	
			msg = @"Some items have the same name;\ndo you want to sobstitute them?";
			title = @"Move";
		
		} else if ([operation isEqualToString: NSWorkspaceCopyOperation]) {
			msg = @"Some items have the same name;\ndo you want to sobstitute them?";
			title = @"Copy";

		} else if([operation isEqualToString: NSWorkspaceLinkOperation]) {
			msg = @"Some items have the same name;\ndo you want to sobstitute them?";
			title = @"Link";

		} else if([operation isEqualToString: NSWorkspaceRecycleOperation]) {
			msg = @"Some items have the same name;\ndo you want to sobstitute them?";
			title = @"Recycle";

		} else if([operation isEqualToString: GWorkspaceRecycleOutOperation]) {
			msg = @"Some items have the same name;\ndo you want to sobstitute them?";
			title = @"Recycle";

		} else if ([operation isEqualToString: NSWorkspaceDestroyOperation]) {
			[self performOperation];
			return;	
			
		} else if ([operation isEqualToString: NSWorkspaceDuplicateOperation]) {
			[self performOperation];
			return;	
			
		} else if ([operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) {
			[self performOperation];
			return;	
		} 
		
    result = [gwFileOp requestUserConfirmationWithMessage: msg title: title];
    
		if(result == NSAlertDefaultReturn) {
			[self performOperation];
		} else {
      result = [gwFileOp sendDidChangeNotification];
      [gwFileOp endOperation];
      exit(0);
		}
    
	} else {	
		[self performOperation];
	}
}

- (void)performOperation
{
	NSString *fulldestpath;
  NSString *filename;
	NSString *newname;
	int i, count, result;
  
  started = YES;  
	fcount = 0;
  
	if ([operation isEqualToString: NSWorkspaceMoveOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
						|| [operation isEqualToString: GWorkspaceRecycleOutOperation]) {
    count = [files count];
	  for(i = 0; i < count; i++) {
      filename = [files objectAtIndex: i];       
			[self calculateNumFilesAtPath: [source stringByAppendingPathComponent: filename]];
		}
    [self sendFilesCountAndWait];
	
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];         
				if([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm movePath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		 		   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
      
		} else {    
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm movePath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		 		   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}
        
	} else if ([operation isEqualToString: NSWorkspaceCopyOperation]) {
    count = [files count];
	  for(i = 0; i < count; i++) {
      filename = [files objectAtIndex: i];       
			[self calculateNumFilesAtPath: [source stringByAppendingPathComponent: filename]];
		}
    [self sendFilesCountAndWait];
  
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];       
				if ([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm copyPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		   	   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
      
		} else {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm copyPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		   	   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}

	} else if([operation isEqualToString: NSWorkspaceLinkOperation]) {
    count = [files count];
	  for(i = 0; i < count; i++) {
      filename = [files objectAtIndex: i];       
			[self calculateNumFilesAtPath: [source stringByAppendingPathComponent: filename]];
		}
    [self sendFilesCountAndWait];
  
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];       
				if ([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm linkPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 	   	  	 handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }

		} else {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm linkPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 	   	  	 handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}

	} else if([operation isEqualToString: NSWorkspaceDestroyOperation]
					|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) {
    count = [files count];
	  for(i = 0; i < count; i++) {
      filename = [files objectAtIndex: i];       
			[self calculateNumFilesAtPath: [source stringByAppendingPathComponent: filename]];
		}
    [self sendFilesCountAndWait];
    
    while (count > 0) {
      filename = [files objectAtIndex: 0];    

			[fm removeFileAtPath: [destination stringByAppendingPathComponent: filename]
				           handler: self];
                   
			[files removeObject: filename];	
      count--;
			VERIFYSTOP;
    }

	} else if([operation isEqualToString: NSWorkspaceDuplicateOperation]) {
    count = [files count];
	  for(i = 0; i < count; i++) {
      filename = [files objectAtIndex: i];       
			[self calculateNumFilesAtPath: [source stringByAppendingPathComponent: filename]];
		}
    [self sendFilesCountAndWait];
   
		while (count > 0) {
      filename = [files objectAtIndex: 0];           
			newname = [NSString stringWithString: filename];
      
			while(1) {
				newname = [newname stringByAppendingString: @"_copy"];
				fulldestpath = [destination stringByAppendingPathComponent: newname];
                
				if (![fm fileExistsAtPath: fulldestpath]) {
					break;
        }
			}
      
			[fm copyPath: [destination stringByAppendingPathComponent: filename]
				   	toPath: fulldestpath
	 		     handler: self];
                              
			[files removeObject: filename];	
      count--;
			VERIFYSTOP;
		}
	}
  
  result = [gwFileOp sendDidChangeNotification];
  [gwFileOp endOperation];
  exit(0);
}	
	
- (void)continueOperation
{
	NSString *fulldestpath;
  NSString *filename;
	NSString *newname;
	int count, result;

  started = YES;
	fcount = 0;
  
	if ([operation isEqualToString: NSWorkspaceMoveOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
						|| [operation isEqualToString: GWorkspaceRecycleOutOperation]) {
    count = [files count];
  
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];         
				if([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm movePath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		 		   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
      
		} else {    
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm movePath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		 		   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}
    
	} else if ([operation isEqualToString: NSWorkspaceCopyOperation]) {
    count = [files count];
  
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];       
				if ([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm copyPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		   	   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
      
		} else {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm copyPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 		   	   handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}

	} else if([operation isEqualToString: NSWorkspaceLinkOperation]) {
    count = [files count];
  
		if (samename) {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				fulldestpath = [destination stringByAppendingPathComponent: filename];       
				if ([fm fileExistsAtPath: fulldestpath]) {
					[fm removeFileAtPath: fulldestpath handler: self];
        }

				[fm linkPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 	   	  	 handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }

		} else {
      while (count > 0) {
        filename = [files objectAtIndex: 0];    

				[fm linkPath: [source stringByAppendingPathComponent: filename]
				   	  toPath: [destination stringByAppendingPathComponent: filename]
	 	   	  	 handler: self];

			  [files removeObject: filename];	
        count--;
				VERIFYSTOP;
      }
		}

	} else if([operation isEqualToString: NSWorkspaceDestroyOperation]
					|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) {
    count = [files count];
    
    while (count > 0) {
      filename = [files objectAtIndex: 0];    

			[fm removeFileAtPath: [destination stringByAppendingPathComponent: filename]
				           handler: self];
                   
		  [files removeObject: filename];	
      count--;
			VERIFYSTOP;
    }

	} else if([operation isEqualToString: NSWorkspaceDuplicateOperation]) {
    count = [files count];
   
		while (count > 0) {
      filename = [files objectAtIndex: 0];           
			newname = [NSString stringWithString: filename];
			while(1) {
				newname = [newname stringByAppendingString: @"_copy"];
				fulldestpath = [destination stringByAppendingPathComponent: newname];
				if (![fm fileExistsAtPath: fulldestpath]) {
					break;
        }
			}
      
			[fm copyPath: [destination stringByAppendingPathComponent: filename]
				   	toPath: fulldestpath
	 		     handler: self];
                              
			[files removeObject: filename];	
      count--;
			VERIFYSTOP;
		}
	}

  result = [gwFileOp sendDidChangeNotification];
  [gwFileOp endOperation];
  exit(0);
}

- (void)sendFilesCountAndWait
{
  int result;
  result = [gwFileOp prepareAlertViewsWithFilesCount: fcount];	
}

- (void)calculateNumFilesAtPath:(NSString *)apath
{
	BOOL isDir;
  NSDirectoryEnumerator *enumerator;
  NSString *dirEntry;
	
	isDir = NO;
	[fm fileExistsAtPath: apath isDirectory: &isDir];
	if (isDir) {
    enumerator = [fm enumeratorAtPath: apath];
    while ((dirEntry = [enumerator nextObject])) {
			fcount++;
    }
	} else {
		fcount++;
	}	
}

- (void)connectionDidDie:(NSNotification *)notification
{
  NSLog(@"BACKGROUNDER: connection died!");
  exit(0);
}

- (BOOL)fileManager:(NSFileManager *)manager 
              shouldProceedAfterError:(NSDictionary *)errorDict
{  
  NSString *path, *msg;
  BOOL iserror = NO;
  int result;
  
  path = [[errorDict objectForKey: @"Path"] copy];
  
  msg = [NSString stringWithFormat: @"File operation error: %@\n"
  						@"with file: %@\n", [errorDict objectForKey: @"Error"], path];

  result = [gwFileOp requestUserConfirmationWithMessage: msg title: @"Error"];
    
	if(result != NSAlertDefaultReturn) {
    result = [gwFileOp sendDidChangeNotification];
    [gwFileOp endOperation];
    exit(0);
    
	} else {  
    NSString *fname = [path lastPathComponent];
    BOOL found = NO;
    
    while (1) {     
      if ([path isEqualToString: source] == YES) {
        break;      
      }    
     
      if ([files containsObject: fname] == YES) {
        [files removeObject: fname];
        found = YES;
        break;
      }
         
      path = [path stringByDeletingLastPathComponent];
      fname = [path lastPathComponent];
    }   
    
    if (found == YES) {
      if (started == NO) {
        [self performOperation]; 
      } else {
        [self continueOperation];
      }
      
    } else {
      result = [gwFileOp showErrorAlertWithMessage: @"File Operation Error!"];
      result = [gwFileOp sendDidChangeNotification];
      [gwFileOp endOperation];
      return NO;
      exit(0);
    }
  }
  
	return !iserror;
}

- (void)fileManager:(NSFileManager *)manager willProcessPath:(NSString *)path
{
  stopped = [gwFileOp updateAlertsForFileName: [path lastPathComponent]];
}
 
@end

int main(int argc, char** argv)
{
	NSAutoreleasePool	*pool;
	Backgrounder *backgrounder;
  
  pool = [NSAutoreleasePool new]; 
	backgrounder = [[Backgrounder alloc] initWithArgc: argc argv: argv];
  RELEASE (pool);
  
  if (backgrounder != nil) {
    [[NSRunLoop currentRunLoop] run];
  }

  exit(0);
}
