/* 
 * File: gmlplaylists.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2 DR2 or better
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"

extern xmmsc_connection_t *connection; /* available from gxmms2.c */

struct pl_params
{
	GMedialib *gml;
	gchar     name[256];
};

static void n_query_playlists(xmmsc_result_t *res, void *pt)
{
	struct pl_params *p = (struct pl_params *)pt;
	GtkTreeIter      iter;

	if (xmmsc_result_iserror(res)) {
		gml_set_statusbar_text(p->gml, (gchar *)xmmsc_result_get_error(res));
	} else {
		gint cnt = 0;
		xmmsc_result_get_dict_entry_int32(res, "cnt", &cnt);
		/* Hide all lists that start with _ */
		if (p->gml->playlists.ls != NULL && p->name[0] != '_') {
			gtk_list_store_append(p->gml->playlists.ls, &iter);
			gtk_list_store_set(p->gml->playlists.ls, &iter, 
			                   0, (gchar *)p->name,
			                   1, cnt,
			                   -1);
		}
	}
	g_free(p);
	xmmsc_result_unref(res);
}

static void n_update_playlists(xmmsc_result_t *res, void *pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	struct pl_params *p;

	if (xmmsc_result_iserror(res)) {
		gml_set_statusbar_text(gml, (gchar *)xmmsc_result_get_error(res));
	} else {
		if (gml->playlists.ls != NULL)
			gtk_list_store_clear(gml->playlists.ls);
		for (; xmmsc_result_list_valid(res); xmmsc_result_list_next(res)) {
			gchar          *name;
			gint32         id = 0;
			gchar          query[1024];
			xmmsc_result_t *res2;

			xmmsc_result_get_dict_entry_str(res,   "name", &name);
			xmmsc_result_get_dict_entry_int32(res, "id",   &id);

			p = g_malloc(sizeof(struct pl_params));
			p->gml = gml;
			g_strlcpy(p->name, name, sizeof(p->name));

			g_snprintf(query, sizeof(query), 
			           "SELECT entry, count(entry) AS cnt " \
			           "FROM Playlistentries WHERE playlist_id = %d",
			           id);
			res2 = xmmsc_medialib_select(connection, query);
			xmmsc_result_notifier_set(res2, n_query_playlists, p);
			xmmsc_result_unref(res2);
		}
		gml_set_statusbar_text(gml, "Ready.");
	}
	xmmsc_result_unref(res);
}

void gml_playlists_update_playlists(GMedialib *gml)
{
	xmmsc_result_t *res;
	gchar          *query = "SELECT name, id FROM Playlist ORDER BY name;";

	gml_set_statusbar_text(gml, "Loading playlists...");
	res = xmmsc_medialib_select(connection, query);
	xmmsc_result_notifier_set(res, n_update_playlists, gml);
	xmmsc_result_unref(res);
}

static void n_playlist_load(xmmsc_result_t *res, void *pt)
{
	GMedialib *gml = (GMedialib *)pt;

	gml_set_statusbar_text(gml, "Ready.");
	gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 4);
	xmmsc_result_unref(res);
}

static void cb_playlists_list_activated(GtkTreeView       *view,
                                        GtkTreePath       *path,
                                        GtkTreeViewColumn *col,
                                        gpointer          userdata)
{
	GtkTreeIter   iter;
	GtkTreeModel *model;
	GMedialib    *gml = (GMedialib *)userdata;

	model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gchar          *playlist = "Unknown";
		xmmsc_result_t *res;

		gtk_tree_model_get(model, &iter, 0, &playlist, -1);

		res = xmmsc_playlist_clear(connection);
		xmmsc_result_unref(res);

		res = xmmsc_medialib_playlist_load(connection, playlist);
		xmmsc_result_notifier_set(res, n_playlist_load, gml);
		xmmsc_result_unref(res);
		g_free(playlist);
	}
}

static void n_playlist_delete(xmmsc_result_t *res, void *pt)
{
	GMedialib *gml = (GMedialib *)pt;

	gml_playlists_update_playlists(gml);
	xmmsc_result_unref(res);
	gml_set_statusbar_text(gml, "Ready.");
}

static void cb_playlists_button_delete_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	GtkTreeSelection *sel;
	GtkTreeModel     *model;
	GtkTreeIter       selected_row;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlists.list));

	g_assert(gtk_tree_selection_get_mode(sel) == GTK_SELECTION_SINGLE);

	if (gtk_tree_selection_get_selected(sel, &model, &selected_row)) {
		gchar          *playlist;
		xmmsc_result_t *res;

		res = xmmsc_playlist_clear(connection);
		xmmsc_result_unref(res);

		gtk_tree_model_get(model, &selected_row, 0, &playlist, -1);

		res = xmmsc_medialib_playlist_remove(connection, playlist);
		xmmsc_result_notifier_set(res, n_playlist_delete, gml);
		xmmsc_result_unref(res);
		g_free(playlist);
	}
}

static void cb_playlists_button_open_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib			*gml = (GMedialib *)pt;
	GtkTreeSelection	*sel;
	GtkTreeModel		*model;
	GtkTreeIter			selected_row;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlists.list));

	g_assert(gtk_tree_selection_get_mode(sel) == GTK_SELECTION_SINGLE);

	if (gtk_tree_selection_get_selected(sel, &model, &selected_row)) {
		gchar          *playlist;
		xmmsc_result_t *res;

		res = xmmsc_playlist_clear(connection);
		xmmsc_result_unref(res);

		gtk_tree_model_get(model, &selected_row, 0, &playlist, -1);

		res = xmmsc_medialib_playlist_load(connection, playlist);
		xmmsc_result_notifier_set(res, n_playlist_load, gml);
		xmmsc_result_unref(res);
		g_free(playlist);
	}
}

static void cb_playlists_button_add_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib			*gml = (GMedialib *)pt;
	GtkTreeSelection	*sel;
	GtkTreeModel		*model;
	GtkTreeIter			selected_row;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlists.list));

	g_assert(gtk_tree_selection_get_mode(sel) == GTK_SELECTION_SINGLE);

	if (gtk_tree_selection_get_selected(sel, &model, &selected_row)) {
		gchar          *playlist;
		xmmsc_result_t *res;

		gtk_tree_model_get(model, &selected_row, 0, &playlist, -1);

		res = xmmsc_medialib_playlist_load(connection, playlist);
		xmmsc_result_notifier_set(res, n_playlist_load, gml);
		xmmsc_result_unref(res);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 3);
		g_free(playlist);
	}
}

static void cb_playlists_popup_refresh_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gml_playlists_update_playlists(gml);
}

static void playlists_view_popup_menu(GtkWidget      *treeview,
                                      GdkEventButton *event,
                                      gpointer       userdata)
{
	GtkWidget        *menu, *menuitem;
	GtkTreeSelection *selection;
	GMedialib        *gml = (GMedialib *)userdata;
	gboolean         multiple = FALSE, none = FALSE;

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlists.list));
	if (gtk_tree_selection_count_selected_rows(selection) > 1)
		multiple = TRUE;
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		none = TRUE;

	menu = gtk_menu_new();

	/* open playlist */
	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_OPEN, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_playlists_button_open_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	/* append to current playlist */
	menuitem = gtk_image_menu_item_new_with_mnemonic("_Add to current playlist");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_ADD, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none)gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_playlists_button_add_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	/* delete playlist */
	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_DELETE, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_playlists_button_delete_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);


	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	/* refresh playlists list */
	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_REFRESH, NULL);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_playlists_popup_refresh_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
	               (event != NULL) ? event->button : 0,
	               gdk_event_get_time((GdkEvent*)event));
}

static gboolean cb_playlists_selection_button_press(GtkWidget      *treeview,
                                                    GdkEventButton *event,
                                                    gpointer       userdata)
{
	gboolean result = FALSE;
	if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
		playlists_view_popup_menu(treeview, event, userdata);
		result = TRUE;
	}
	return result;
}

static gboolean cb_playlists_selection_popup_menu(GtkWidget *treeview, 
                                                  gpointer  userdata)
{
	playlists_view_popup_menu(treeview, NULL, userdata);
	return TRUE;
}

void gml_create_playlists(GMedialib *gml)
{
	GtkTreeViewColumn *column_playlist, *column_items;

	gml->playlists.frame = gtk_frame_new("Playlists");
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlists.frame), 2);

	/* table mlib browse */
	gml->playlists.table = gtk_table_new(2, 1, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlists.table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(gml->playlists.table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gml->playlists.table), 4);
	gtk_container_add(GTK_CONTAINER(gml->playlists.frame), 
	                  gml->playlists.table);

	/* list artist browse */
	gml->playlists.ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);

	gml->playlists.list = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(gml->playlists.ls));
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(gml->playlists.list), 
	                                TRUE);

	gml->playlists.scroll_widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(gml->playlists.scroll_widget),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(gml->playlists.scroll_widget),
	                  gml->playlists.list);

	gtk_table_attach(GTK_TABLE(gml->playlists.table),
		             gml->playlists.scroll_widget,
		             0, 1, 0, 1, 
		             GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

	column_playlist = gtk_tree_view_column_new_with_attributes("Playlist", 
	                                                           gml->renderer_text,
	                                                           "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlists.list),
	                            column_playlist);
	gtk_tree_view_column_set_resizable(column_playlist, TRUE);
	gtk_tree_view_column_set_cell_data_func(column_playlist, gml->renderer_text,
	                                        gml_cell_data_function, gml, NULL);


	column_items = gtk_tree_view_column_new_with_attributes("Items", 
	                                                         gml->renderer_text,
	                                                         "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlists.list),
	                            column_items);
	gtk_tree_view_column_set_resizable(column_items, TRUE);
	gtk_tree_view_column_set_cell_data_func(column_items, gml->renderer_text,
	                                        gml_cell_data_function, gml, NULL);

	/* load button */
	gml->playlists.button_load = gtk_button_new_from_stock(GTK_STOCK_OPEN);
	gtk_table_attach(GTK_TABLE(gml->playlists.table),
		             gml->playlists.button_load,
		             0, 1, 1, 2,
		             GTK_SHRINK, GTK_SHRINK, 0, 0);

	g_signal_connect(G_OBJECT(gml->playlists.list),      "row-activated",
	                 G_CALLBACK(cb_playlists_list_activated),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlists.button_load),     "clicked",
	                 G_CALLBACK(cb_playlists_button_open_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlists.list), "button-press-event",
                     G_CALLBACK(cb_playlists_selection_button_press),
                     (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlists.list),         "popup-menu",
                     G_CALLBACK(cb_playlists_selection_popup_menu),
                     (gpointer)gml);
}
