#include "hx_types.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include "screen.h"
#include "term.h"
#include "hxlib.h"
#include "xmalloc.h"
#include <errno.h>

struct screen main_scr, *curscr = &main_scr;

int
scr_init (int li, int co)
{
	term_set_scroll_region(0, li);
	term_goto_line(LI);

	main_scr.li = li;
	main_scr.co = co;
	main_scr.len = main_scr.start = 0;
	main_scr.buf = 0;
	main_scr.draw = default_status_draw;
	hist_init(&(main_scr.history));
	main_scr.history.fun = hx_input_fun;

	return 0;
}

static inline unsigned int
__liup (struct screen *scr)
{
	register unsigned int i, line = scr->start - scr->co;

	if ((signed)line < 0)
		line = 0;
	for (i = scr->start - 2; i > line; i--) {
		if (scr->buf[i] == '\n')
			break;
	}

	return scr->start - i;
}

static inline unsigned int
__lidn (struct screen *scr)
{
	register unsigned int i, line = scr->start + scr->co;

	if (line > scr->len)
		line = scr->len;
	for (i = scr->start; i < line; i++) {
		if (scr->buf[i] == '\n')
			break;
	}

	return i - scr->start;
}

static inline unsigned int
__pgup (struct screen *scr)
{
	register unsigned int i, li = scr->li, lines = 0, pglen = scr->start - (scr->li * scr->co);

	if ((signed)pglen < 0)
		pglen = 0;
	for (i = scr->start; i > pglen; i--) {
		if (scr->buf[i] == '\n' && lines++ == li)
			break;
	}

	return scr->start - i;
}

static inline unsigned int
__pgdn (struct screen *scr)
{
	register unsigned int i, li = scr->li, lines = 0, pglen = scr->start + (scr->li * scr->co);

	if (pglen > scr->len)
		pglen = scr->len;
	for (i = scr->start; i < pglen; i++) {
		if (scr->buf[i] == '\n' && lines++ == li)
			break;
	}

	return i - scr->start;
}

void
scr_line_up (struct screen *scr)
{
	if (scr->start == 1)
		return;
	scr->start -= __liup(scr) - 1;
	term_goto_line(0);
	term_scroll_up();
	write(1, &(scr->buf[scr->start]), __lidn(scr));
	term_move_cursor(LI, curscr->history.cur->pos % CO);
}

void
scr_line_down (struct screen *scr)
{
	if (scr->start + __pgdn(scr) == scr->len)
		return;
	scr->start += __pgdn(scr) + 1;
	term_goto_line(curscr->li);
	term_scroll_down();
	write(1, &(scr->buf[scr->start]), __lidn(scr));
	term_move_cursor(LI, curscr->history.cur->pos % CO);
	scr->start -= __pgup(scr) - 1;
}

void
scr_page_home (struct screen *scr)
{
	if (scr->start == 1)
		return;
	scr->start = 1;
	scr_draw(scr);
}

void
scr_page_end (struct screen *scr)
{
	if (scr->start + __pgdn(scr) == scr->len)
		return;
	scr->start = scr->len;
	scr->start -= __pgup(scr) - 1;
	scr_draw(scr);
}

void
scr_page_up (struct screen *scr)
{
	if (scr->start == 1)
		return;
	scr->start -= 2;
	scr->start -= __pgup(scr) - 1;
	scr_draw(scr);
}

void
scr_page_down (struct screen *scr)
{
	if (scr->start + __pgdn(scr) == scr->len)
		return;
	scr->start += __pgdn(scr) + 1;
	if (scr->start + __pgdn(scr) == scr->len) {
		scr->start = scr->len;
		scr->start -= __pgup(scr) - 1;
	}
	scr_draw(scr);
}

void
scr_draw (struct screen *scr)
{
	term_clear();
	scr->draw(scr);
	term_home();
	write(1, &(scr->buf[scr->start]), __pgdn(scr));
	input_puts(scr->history.cur);
}

void
scr_resize (struct screen *scr, int li, int co)
{
	scr->li = li;
	scr->co = co;
}

void
scr_resize_all (int li, int co)
{
	scr_resize(&main_scr, li, co);
	curscr->start = curscr->len;
	curscr->start -= __pgup(curscr) - 1;
	scr_draw(curscr);
}

int cmd_clear (void);
int cmd_save (int argc, char *const *argv);

int
cmd_clear (void)
{
	xfree(curscr->buf);
	curscr->buf = 0;
	curscr->len = curscr->start = 0;
	term_clear();
	curscr->draw(curscr);
	input_puts(curscr->history.cur);

	return 0;
}

int
cmd_save (int argc, char *const *argv)
{
	FILE *fp;
	char buf[4096];

	if (argc < 2) {
		curscr_printf("\nusage: %s <file>\n", argv[0]);
		return 1;
	}
	expand_tilde(buf, argv[1]);
	fp = fopen(buf, "a");
	if (!fp) {
		curscr_printf("\n%s: %s: %s", argv[0], buf, strerror(errno));
		return -1;
	}
	curscr_printf("\n%d bytes written to %s",
		fwrite(curscr->buf, sizeof *(curscr->buf), curscr->len, fp), buf);
	fclose(fp);

	return 0;
}

void
scr_puts (struct screen *scr, const char *buf, unsigned int len)
{
	register unsigned int i, c = 0, co = scr->co, bpos = scr->len;

	for (i = 0; i < len; i++) {
		if (c++ == co) {
			if (buf[i] != '\n') {
				scr->buf = xrealloc(scr->buf, 2 + bpos);
				scr->buf[bpos++] = '\n';
				scr->buf[bpos++] = buf[i];
				c = 1;
			} else {
				scr->buf = xrealloc(scr->buf, 1 + bpos);
				scr->buf[bpos++] = '\n';
				c = 0;
			}
		} else {
			scr->buf = xrealloc(scr->buf, 1 + bpos);
			scr->buf[bpos++] = buf[i];
			if (buf[i] == '\n')
				c = 0;
		}
	}
	scr->len = bpos;
}

void
curscr_puts (const char *buf, unsigned int len)
{
	if (__pgdn(curscr) + curscr->start == curscr->len) {
		term_goto_line(curscr->li);
		term_puts(buf, len);
		term_move_cursor(LI, curscr->history.cur->pos % CO);
		scr_puts(curscr, buf, len);
		curscr->start = curscr->len;
		curscr->start -= __pgup(curscr) - 1;
	} else {
		scr_puts(curscr, buf, len);
	}
}

void
curscr_printf (const char *fmt, ...)
{
	va_list ap;
	char buf[4096];
	int len;

	va_start(ap, fmt);
	if ((len = vsnprintf(buf, sizeof buf, fmt, ap)) == -1)
		len = sizeof buf;
	va_end(ap);
	if (__pgdn(curscr) + curscr->start == curscr->len) {
		term_goto_line(curscr->li);
		term_puts(buf, len);
		term_move_cursor(LI, curscr->history.cur->pos % CO);
		scr_puts(curscr, buf, len);
		curscr->start = curscr->len;
		curscr->start -= __pgup(curscr) - 1;
	} else {
		scr_puts(curscr, buf, len);
	}
}
