/*  IBP - show which beacon of the International Beacon Project is active now
 *
 *  Copyright (C) 2000, Pieter-Tjerk de Boer -- pa3fwm@amsat.org
 *
 *  Distributed on the conditions of the Gnu Public Licence, version 2.
 */

#define VERSION "0.20"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <curses.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include <ctype.h>

#ifndef noX11
  #include "xdisp.h"
#endif


typedef struct {
   char callsign[8];
   char location[40];
   int activeband[5];
   double lat, lon;
   double dist;
   double az_shortpath, az_longpath;
} Beacon;

Beacon beacons[]={
   {"4U1UN",  "United Nations   New York City",    {1,1,1,1,1},  40+ 45/60.,  -73- 58/60. },
   {"VE8AT",  "Canada           Eureka, Nunavut",  {1,1,1,1,1},  79+ 59/60.,  -85- 57/60. },
   {"W6WX",   "United States    Mt. Umunhum",      {1,1,1,1,1},  37+  9/60., -121- 54/60. },
   {"KH6WO",  "Hawaii           Laie",             {1,0,1,0,1},  21+ 38/60., -157- 55/60. },
   {"ZL6B",   "New Zealand      Masterton",        {1,1,1,1,1}, -41-  3/60.,  175+ 36/60. },
   {"VK6RBP", "Australia        Rolystone",        {1,1,1,1,1}, -32-  6/60.,  116+ 03/60. },
   {"JA2IGY", "Japan            Mt. Asama",        {1,1,1,1,1},  34+ 27/60.,  136+ 47/60. },
   {"RR9O",   "Russia           Novosibirsk",      {1,1,1,1,1},  54+ 59/60.,   82+ 54/60. },
   {"VR2HK",  "Hong Kong        Hong Kong",        {1,1,1,1,1},  22+ 16/60.,  114+ 11/60. },
   {"4S7B",   "Sri Lanka        Colombo",          {1,1,1,1,1},   6+ 54/60.,   79+ 52/60. },
   {"ZS6DN",  "South Africa     Pretoria",         {1,1,1,1,1}, -25- 54/60.,   28+ 16/60. },
   {"5Z4BN",  "Kenya            Kilifi",           {1,1,1,1,1},  -3- 37/60.,   39+ 50/60. },
   {"4X6TU",  "Israel           Tel Aviv",         {1,1,1,1,1},  32+  6/60.,   34+ 48/60. },
   {"OH2B",   "Finland          Espoo",            {1,1,1,1,1},  60+ 11/60.,   24+ 50/60. },
   {"CS3B",   "Madeira          Santo da Serra",   {1,1,1,1,1},  32+ 43/60.,  -16- 48/60. },
   {"LU4AA",  "Argentina        Buenos Aires",     {1,1,1,1,1}, -34- 37/60.,  -58- 21/60. },
   {"OA4B",   "Peru             Lima",             {1,1,1,1,1}, -12-  4/60.,  -76- 57/60. },
   {"YV5B",   "Venezuela        Caracas",          {1,1,1,1,1},  10+ 25/60.,  -66- 51/60. },
};


char frequencies[][12]={
   "14.100 MHz",
   "18.110 MHz",
   "21.150 MHz",
   "24.930 MHz",
   "28.200 MHz"
};


int multiband=0;     /* flag for multiband display */
int colour=1;        /* flag for colour display */
int band=0;          /* number of selected band, 0...4 = 14 ... 28 MHz */
int morse=0;         /* flag for showing the morse code of the callsign */
#ifndef noX11
int dox=1;           /* flag for opening the X11 window */
int xfd;             /* filedescriptor for X11 */
#endif

char mylocator[8]="";   /* listener's maidenhead locator given on command line */
double mylat, mylon;    /* corresponding latitude and longitude */
int myloc_ok=0;         /* flag; if zero, distance and azimuth are not displayed */



char *makemorse(char *s)
/* returns a static string containing the morse code representation of s */
{
   static char result[80];
   static char alphabet[][8]={
      ".- ", "-... ", "-.-. ", "-.. ", ". ", "..-. ", "-.. ", ".... ", ".. ", ".--- ",
      "-.- ", ".-.. ", "-- ", "-. ", "--- ", ".--. ", "--.- ", ".-. ", "... ", "- ",
      "..- ", "...- ", ".-- ", "-..- ", "-.-- ", "--.. " };
   static char digits[][8]={
      "-----", ".---- ", "..--- ", "...-- ", "....- ",
      "..... ", "-.... ", "--... ", "---.. ", "----. " };

   result[0]=0;
   while (*s) {
      if (isalpha(*s)) strcat(result, alphabet[toupper(*s)-'A']);
      else if (isdigit(*s)) strcat(result, digits[*s-'0']);
      s++;
   }
   return result;
}


void mainloop(void)
/* the main loop: display the beacons, time, etc., and process key presses */
{
   int attrs[5];
   int previ=-1;
   time_t prevt=-1;

   /* prepare for using the curses library and draw the static parts of the screen */
   initscr();
   cbreak(); noecho();
   leaveok(stdscr,TRUE); curs_set(0); nonl();
   if (has_colors()) {
      start_color();
      init_pair(COLOR_BLACK,   COLOR_BLACK,   COLOR_BLACK);
      init_pair(COLOR_GREEN,   COLOR_GREEN,   COLOR_BLACK);
      init_pair(COLOR_RED,     COLOR_RED,     COLOR_WHITE);
      init_pair(COLOR_CYAN,    COLOR_CYAN,    COLOR_BLACK);
      init_pair(COLOR_WHITE,   COLOR_WHITE,   COLOR_BLACK);
      init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_WHITE);
      init_pair(COLOR_BLUE,    COLOR_BLUE,    COLOR_WHITE);
      init_pair(COLOR_YELLOW,  COLOR_YELLOW,  COLOR_BLACK);

      attrs[0]=A_REVERSE|COLOR_PAIR(COLOR_RED);
      attrs[1]=A_REVERSE|COLOR_PAIR(COLOR_YELLOW);
      attrs[2]=A_REVERSE|COLOR_PAIR(COLOR_GREEN);
      attrs[3]=A_REVERSE|COLOR_PAIR(COLOR_CYAN);
      attrs[4]=A_REVERSE|COLOR_PAIR(COLOR_BLUE);
   } else {
      int i;
      for (i=0;i<5;i++) attrs[i]=A_REVERSE;
   }
   attron(A_REVERSE); 
   move(21,0); 
   if (multiband) addstr(" Keys:   1...5: select band   M: toggle multi-band                     Q: quit "); 
   else addstr(" Keys:  1..5: select band  M: toggle multi-band  C: toggle morse code  Q: quit "); 
   attroff(A_REVERSE);

   for(;;) {
      time_t t;
      int i,j,mfd;
      char s[100];
      fd_set readfds;
      struct timeval timeout;
      int redraw=0;

      FD_ZERO(&readfds);
      FD_SET(0,&readfds);
      mfd=1;
#ifndef noX11
      FD_SET(xfd,&readfds);
      mfd=xfd+1;
#endif
      timeout.tv_sec=0;
      timeout.tv_usec=10000;
      i=select(mfd,&readfds,NULL,NULL,&timeout);

      /* process key press, if any */
      if (i>0) {
         int c;
         if (FD_ISSET(0,&readfds)) c=getchar();
#ifndef noX11
         else c=x_getkeypress();
#endif
         switch (c) {
            case 'C': case 'c':
               morse=!morse;
               if (!multiband) redraw=1;
               else if (morse) {
                  move(23,0);
                  addstr(" (no morse code display in multi-band mode)");
                  refresh();
               }
               break;
            case 'M': case 'm':
               multiband=!multiband;
               redraw=1;
               break;
            case 'Q': case 'q':
               endwin();
               return;
            case '1': 
            case '2':
            case '3':
            case '4':
            case '5':
               band=c-'1';
               multiband=0;
               redraw=1;
               break;
         }
      }

      time(&t);
      if (t!=prevt || redraw) {
         int sec;
         struct tm *tp;

         /* draw the screen's top line, containing the time etc. */
         prevt=t;
         tp=gmtime(&t);
         sec = 60*(tp->tm_min%3) + tp->tm_sec;
         sprintf(s," %02i:%02i:%02iz    %-10s    %6s                                   IBP v" VERSION " \n",
            tp->tm_hour, tp->tm_min, tp->tm_sec,
            (!multiband)?frequencies[band]:"",
            mylocator);
         attron(A_REVERSE); move(0,0); addstr(s); attroff(A_REVERSE);

         /* calculate the phase within the 3 minute cycle */
         i = (sec/10) % 18;
         if (i!=previ) redraw=1;
         else refresh();
         previ=i;
      }

      if (redraw) {
         /* write the beacon list, with active beacon(s) marked */
         int act[5];
         for (j=0;j<5;j++) act[j]=-1;
         for (j=0;j<18;j++) {
            int b,active;
            char *sp;
            move(j+2,0);
            b = (i-j+18)%18;
            active = b<5 && beacons[j].activeband[b] && (multiband || b==band);
            if (active) {
               if (colour) attrset(attrs[b]);
               else attrset(A_REVERSE);
               act[b]=j;
            }
            sp=s+sprintf(s,"  %10s   %-6s    %-33s", 
                    active ? frequencies[b] : "",
                    beacons[j].callsign, beacons[j].location
                 );
            if (myloc_ok)
               sprintf(sp,"  %5ikm %3i/%3ideg \n",
                  (int)(0.5+beacons[j].dist),
                  (int)(0.5+beacons[j].az_shortpath),
                  (int)(0.5+beacons[j].az_longpath)
               );
            else strcpy(sp,"                     \n");
            addstr(s);
            attrset(A_NORMAL);
         }
         /* write the bottom line with the morse code, if enabled: */
         j=(i-band+18)%18;
         if (morse && !multiband && beacons[j].activeband[band]) 
            sprintf(s," %-70s        ", makemorse(beacons[j].callsign));
         else
            strcpy(s,"                                                                               ");
         move(23,0);
         addstr(s);
         refresh();
#ifndef noX11
         if (dox) x_update(act);
#endif
      }
   }
}


int is_locator(char *s)
/* returns 1 if and only if s looks like a Maidenhead locator */
{
   if (!isalpha(s[0])) return 0;
   if (!isalpha(s[1])) return 0;
   if (!isdigit(s[2])) return 0;
   if (!isdigit(s[3])) return 0;
   if (!isalpha(s[4])) return 0;
   if (!isalpha(s[5])) return 0;
   if (isgraph(s[6])) return 0;
   return 1;
}


void calc_dist_and_az(void)
/* calculate distance and azimuth for all beacons w.r.t. the location given by mylocator[] */
/* results are stored in the beacons[] array */
{
   int i;
   mylon= -180 + (toupper(mylocator[0])-'A')*20
               + (mylocator[2]-'0')*2
               + (toupper(mylocator[4])-'A')/12. + 1/24. ;
   mylat= -90 + (toupper(mylocator[1])-'A')*10
              + (mylocator[3]-'0')*1
              + (toupper(mylocator[5])-'A')/24. + 1/48. ;
   mylon *= M_PI/180;
   mylat *= M_PI/180;
   for (i=0;i<18;i++) {
      double blon,blat;
      double tmp,arc;
      blon = beacons[i].lon * M_PI/180;
      blat = beacons[i].lat * M_PI/180;
      tmp = sin(mylat)*sin(blat) + cos(mylat)*cos(blat)*cos(blon-mylon);
      arc = acos(tmp);
      beacons[i].dist = arc*180/M_PI*111.2;
      tmp = (sin(blat)-sin(mylat)*tmp)/(sin(arc)*cos(mylat));
      beacons[i].az_shortpath = acos(tmp)*180/M_PI;
      beacons[i].az_longpath = beacons[i].az_shortpath+180;
      if (sin(blon-mylon)<0) {
         beacons[i].az_shortpath = 360 - beacons[i].az_shortpath;
         beacons[i].az_longpath = beacons[i].az_shortpath-180;
      }
   }
}



int main(int argc,char **argv)
{
   int err;

   myloc_ok=0;
   band=-1;
   multiband=0;
   err=0;

   /* parse the command-line options */
   while (argc>1 && !err) {
      int i;
      argc--;
      if (argv[argc][0]=='-') {
         switch (argv[argc][1]) {
            case 'c': colour=0; break;
            case 'm': morse=1; break;
#ifndef noX11
            case 'x': dox=0; break;
#endif
            case '-':
               if (!strcmp(argv[argc],"--nocolour")) colour=0;
               else if (!strcmp(argv[argc],"--morse")) morse=1;
#ifndef noX11
               else if (!strcmp(argv[argc],"--nograph")) dox=0;
#endif
               else err=1;
               break;
            default:
               err=1;
         }
      } else {
         i=atoi(argv[argc]);
         if (i>0) {
            if (band>=0) err=1;
            switch (i) {
               case 14: case 20: band=0; break;
               case 18: case 17: band=1; break;
               case 21: case 15: band=2; break;
               case 24: case 12: band=3; break;
               case 28: case 10: band=4; break;
               default: err=1;
            }
         } else if (is_locator(argv[argc])) {
            if (myloc_ok) err=1;
            else {
               strncpy(mylocator,argv[argc],6);
               mylocator[6]=0;
               myloc_ok=1;
            }
         } else {
            err=1;
         }
      }
   }
   if (band<0) { band=0; multiband=1; }


   if (err) {
      printf("Usage: %s [arguments]\n"
             "shows which of the International Beacon Project beacons is now transmitting.\n"
             "\n"
             "The following arguments can be used:\n"
             "-c or --nocolour: don't use colour, even on a colour terminal.\n"
             "-m or --morse:    display callsign of active beacon in morse code (not in\n"
             "                  multi-band mode).\n"
#ifndef noX11
             "-x or --nograph:  don't open the X11 graphic window, even if DISPLAY is set.\n"
#endif
             "a band expressed either in MHz (14, 18, 21, 24 or 28) or in meters (20, 17, 15,\n"
             "    12 or 10); if no band is specified, the program starts in multi-band mode.\n"
             "a Maidenhead locator (like JO32KF), to be used for the calculation of distance\n"
             "    and azimuth (both short and long path) to each beacon.\n"
             "\n"
             "Note: since each beacon transmission lasts only 10 seconds, your computer's\n"
             "clock should be set reasonably accurately for the results to be valid.\n" 
             "\n"
             "More information about the International Beacon Project can be found at\n"
             "http://www.ncdxf.org/beacon.htm\n",
             argv[0]);
      return 1;
   }

   if (myloc_ok) calc_dist_and_az();

#ifndef noX11
   if (!getenv("DISPLAY")) dox=0;
   if (dox) {
      double lat[18],lon[18];
      int i;
      xfd = x_init();
      for (i=0;i<18;i++) {
         lat[i]=beacons[i].lat;
         lon[i]=beacons[i].lon;
      }
      x_storeBeacons(myloc_ok,mylat*180/M_PI,mylon*180/M_PI,lat,lon);
   }
#endif

   mainloop();

   return 0;
}

