/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ihistogrammaker.h"


#include "icommoneventobservers.h"
#include "idatalimits.h"
#include "ierror.h"
#include "ihistogram.h"
#include "imath.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include <vtkDataSetSource.h>
#include <vtkFloatArray.h>
#include <vtkIdTypeArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>


//------------------------------------------------------------------------------
iHistogramMaker* iHistogramMaker::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iHistogramMaker(vm);
}


// Construct object to extract all of the input data.
iHistogramMaker::iHistogramMaker(iViewModule *vm) : iParallelWorker(vm->GetParallelManager()), mViewModule(vm), mLowResBins(256), mLowResSize(mLowResBins*mLowResBins)
{
	mHisto = iHistogram::New(this->GetViewModule(),this); IERROR_ASSERT(mHisto);

	mNumProcs = mNumBins = 0;

	mDataRank = 0U;
	mFullResolution = mSupportsFullResolution = false;

	mRanges = 0;
	mValues = 0;
	mComponent = -1;

	mAttached = true;

#ifdef IVTK_4
	this->SetNumberOfInputs(1);
#else
	this->SetNumberOfInputConnections(0,1);
#endif

	this->AddObserver(vtkCommand::StartEvent,vm->GetProgressEventObserver());
	this->AddObserver(vtkCommand::ProgressEvent,vm->GetProgressEventObserver());
	this->AddObserver(vtkCommand::EndEvent,vm->GetProgressEventObserver());
}


iHistogramMaker::~iHistogramMaker()
{
	if(mAttached) mHisto->Delete();;
	if(mRanges != 0) 
	{
		delete [] mRanges;
	}
	if(mValues != 0)
	{
		for(int i=0; i<mNumProcs; i++)
		{
			if(mValues[i] != 0) delete [] mValues[i];
		}
		delete [] mValues;
	}
}


void iHistogramMaker::AttachHistogram(bool s)
{
	mAttached = s;
}


void iHistogramMaker::SetInput(vtkDataSet *input, int c)
{
	this->SetNthInput(0,input);
	this->SetInputComponent(c);
}


void iHistogramMaker::SetInputComponent(int c)
{
	if(c != mComponent)
	{
		mComponent = c;
		this->Modified();
	}
}


void iHistogramMaker::OnStretchChanged()
{
	this->Modified();
}


vtkDataSet* iHistogramMaker::GetInput()
{
	if(this->Inputs == 0) return 0; else return vtkDataSet::SafeDownCast(this->Inputs[0]);
}


void iHistogramMaker::SetDataRank(unsigned int r)
{
	if(r<=2U && r!=mDataRank)
	{
		mDataRank = r;
		this->Modified();
	}
}


void iHistogramMaker::SetFullResolution(bool s)
{
	if(mFullResolution != s)
	{
		mFullResolution = s;
		this->Modified();
	}
}


int iHistogramMaker::GetEstimatedNumberOfBins()
{ 
	if(this->GetInput() == 0)
	{
		mSupportsFullResolution = false;
		return 0;
	}
	else
	{
		int nb = round(1.0+sqrt((float)this->GetNumberOfCells()));
		if(nb > mLowResBins)
		{
			mSupportsFullResolution = true;
			if(!mFullResolution) nb = mLowResBins;
		}
		else mSupportsFullResolution = false;
		return nb;
	}
}


void iHistogramMaker::Update()
{
	//
	//  Force to recompute if needed
	//
	this->GetHistogram();
}


bool iHistogramMaker::ComputeRange()
{
	if(!this->IsDataPresent()) return false;

	//
	//  Do we need to recompute?
	//
	if(this->GetInput()->GetMTime() > mComputeTime) 
	{
		//	
		//  Recompute the limits
		//
		int i;
		
		if(mNumProcs != this->GetManager()->GetNumberOfProcessors())
		{
			if(mRanges != 0) delete [] mRanges;
			mNumProcs = this->GetManager()->GetNumberOfProcessors();
			mRanges = new Range[mNumProcs]; 
			if(mRanges == 0) 
			{
				return false;
			}
		}

		for(i=0; i<mNumProcs; i++)
		{
			mRanges[i].ValMax = mRanges[i].ValMin = 0.0;
			mRanges[i].CellMax = mRanges[i].CellMin = -1;
		}

		if(!this->PrepareForPass() || this->ParallelExecute(1)!=0) 
		{
			return false;
		}
		
		mHisto->mValMin = mRanges[0].ValMin;
		mHisto->mValMax = mRanges[0].ValMax;
		mHisto->mCellMin = mRanges[0].CellMin;
		mHisto->mCellMax = mRanges[0].CellMax;
		for(i=1; i<mNumProcs; i++)
		{
			if(mHisto->mValMin > mRanges[i].ValMin) 
			{
				mHisto->mValMin = mRanges[i].ValMin;
				mHisto->mCellMin = mRanges[i].CellMin;
			}
			if(mHisto->mValMax < mRanges[i].ValMax) 
			{
				mHisto->mValMax = mRanges[i].ValMax;
				mHisto->mCellMax = mRanges[i].CellMax;
			}
		}
		
		this->FindPositionForCell(mHisto->mCellMin,mHisto->mPosMin);
		this->FindPositionForCell(mHisto->mCellMax,mHisto->mPosMax);
		
		//
		//  Spread values a little bit if needed
		//
		if(mHisto->mValMax-mHisto->mValMin < 0.01*(fabs(mHisto->mValMax)+fabs(mHisto->mValMin)))
		{
			if(mHisto->mValMin < 0) mHisto->mValMin *= 1.001; else mHisto->mValMin *= 0.999;
			if(mHisto->mValMax < 0) mHisto->mValMax *= 0.999; else mHisto->mValMax *= 1.001;
		}
	}

	return true;
}


void iHistogramMaker::GetHistogramRange(float &min, float &max)
{
	if(this->ComputeRange())
	{
		min = mHisto->mValMin;
		max = mHisto->mValMax;
	}
}


void iHistogramMaker::GetHistogramRange(float &valMin, vtkIdType &cellMin, const iPosition* &posMin, float &valMax, vtkIdType &cellMax, const iPosition* &posMax)
{
	if(this->ComputeRange())
	{
		valMin = mHisto->mValMin;
		posMin = &mHisto->mPosMin;
		cellMin = mHisto->mCellMin;
		valMax = mHisto->mValMax;
		posMax = &mHisto->mPosMax;
		cellMax = mHisto->mCellMax;
	}
}


const iHistogram* iHistogramMaker::GetHistogram(int nBins)
{
	if(this->ComputeRange())
	{
		return this->GetHistogram(nBins,mHisto->mValMin,mHisto->mValMax);
	}
	else
	{
		return 0;
	}
}


const iHistogram* iHistogramMaker::GetHistogram(int nBins, float vMin, float vMax)
{
	int i, j;

	if(!this->IsDataPresent()) return 0;

	if(nBins<-1 || nBins>1000000 || vMin>vMax || this->GetInput()==0) return 0;

	//
	//  No need to recompute
	//
	if(nBins == 0) nBins = this->GetEstimatedNumberOfBins();

	if(nBins==mNumBins && vMin==mHisto->mValMin && vMax==mHisto->mValMax && this->GetInput()->GetMTime()<mComputeTime && this->GetMTime()<mComputeTime) 
	{
		return mHisto;
	}

	//
	//  Must recompute
	//
	mComputeTime.Modified();

	//
	//  Check if we need to re-create the helper data 
	//
	if(mNumProcs!=this->GetManager()->GetNumberOfProcessors() || nBins!=mNumBins)
	{
		int i;
		if(mValues != 0)
		{
			for(i=0; i<mNumProcs; i++)
			{
				if(mValues[i] != 0) delete [] mValues[i];
			}
			delete [] mValues;
		}
		mNumProcs = this->GetManager()->GetNumberOfProcessors();
		mValues = new float*[mNumProcs]; 
		if(mValues == 0) 
		{
			return 0;
		}
		memset(mValues,0,mNumProcs*sizeof(float *));
		for(i=0; i<mNumProcs; i++)
		{
			mValues[i] = new float[nBins]; 
			if(mValues[i] == 0) return 0;
		}
	}

	mNumBins = nBins;
	mHisto->mValMin = vMin;
	mHisto->mValMax = vMax;
	mHisto->mStretch = this->GetStretch();

	if(!this->PrepareForPass()) return 0;
	this->ParallelExecute(2);

	for(i=0; i<mNumBins; i++) 
	{
		for(j=1; j<mNumProcs; j++) mValues[0][i] += mValues[j][i];
	}
	mHisto->SetData(mNumBins,mValues[0]);

	return mHisto;
}


int iHistogramMaker::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	if(p.NumProcs != mNumProcs) 
	{
		IERROR_LOW("Error calling iHistogramMaker in parallel.");
		return -1;
	}

	vtkIdType ntup = this->GetNumberOfCells();

	vtkIdType l, lbeg, lend, lstp, linc = 1;
	float v, vmin, vmax;
	int ret = 999;

	//
	//  Distribute data
	//
	iParallel::SplitRange(p,ntup,lbeg,lend,lstp);
	if(!mFullResolution) linc = (lend-lbeg+mLowResSize-1)/mLowResSize;

	if(mFullResolution && this->IsMaster(p))
	{
		this->GetViewModule()->GetProgressEventObserver()->SetMode(iProgressEventObserver::_Computing);
		this->InvokeEvent(vtkCommand::StartEvent);
	}

	//	
	//  Recompute the limits
	//
	if(step == 1)
	{
		vtkIdType lmin, lmax;
		
		for(l=lbeg; l<lend; l+=linc) if(this->IsValidIndex(l))
		{
			vmin = vmax = this->GetValue(l);
			lmin = lmax = l;
			break;
		}

		l += linc;

		for(; l<lend; l+=linc) if(this->IsValidIndex(l))
		{
			if(l%1000 == 0)
			{
				if(mFullResolution && this->IsMaster(p))
				{
					double p = double(l-lbeg)/(lend-lbeg);
					this->InvokeEvent(vtkCommand::ProgressEvent,&p);
				}
				if(this->GetAbortExecute()) break;
			}

			v = this->GetValue(l);
			if(v < vmin)
			{
				vmin = v;
				lmin = l;
			}
			if(v > vmax) 
			{
				vmax = v;
				lmax = l;
			}
		}
		mRanges[p.ThisProc].ValMin = vmin;
		mRanges[p.ThisProc].ValMax = vmax;
		mRanges[p.ThisProc].CellMin = lmin;
		mRanges[p.ThisProc].CellMax = lmax;
		
		ret = 0;
	}

	//	
	//  Compute the histogram
	//
	if(step == 2)
	{
		int iv;
		float vstp;

		vmin = this->ApplyStretch(mHisto->mValMin,false);
		vmax = this->ApplyStretch(mHisto->mValMax,true);
		vstp = 1.0001*(vmax-vmin)/mNumBins;

		float *d = mValues[p.ThisProc];
		memset(d,0,sizeof(float)*mNumBins);

		for(l=lbeg; l<lend; l+=linc) if(this->IsValidIndex(l))
		{
			if(l%1000 == 0)
			{
				if(mFullResolution && this->IsMaster(p))
				{
					double p = double(l-lbeg)/(lend-lbeg);
					this->InvokeEvent(vtkCommand::ProgressEvent,&p);
				}
				if(this->GetAbortExecute()) break;
			}
			
			v = this->ApplyStretch(this->GetValue(l),false);

			iv = round((v-vmin)/vstp-0.4999);
			if(iv>=0 && iv<mNumBins)
			{
				d[iv] += 1.0;
			} 
		}

		ret = 0;
	}

	if(mFullResolution && this->IsMaster(p)) this->InvokeEvent(vtkCommand::EndEvent);

	return ret;
}


vtkIdType iHistogramMaker::GetNumberOfCells()
{
	vtkDataArray *d = 0;
	switch(mDataRank)
	{
	case 0U:
		{
			d = this->GetInput()->GetPointData()->GetScalars();
			break;
		}
	case 1U:
		{
			d = this->GetInput()->GetPointData()->GetVectors();
			break;
		}
	case 2U:
		{
			d = this->GetInput()->GetPointData()->GetTensors();
			break;
		}
	}
	if(d == 0) return 0; else return d->GetNumberOfTuples();
}


void iHistogramMaker::FindPositionForCell(long cell, iPosition &pos)
{
	int i;

	vtkImageData *id = vtkImageData::SafeDownCast(this->GetInput());
	if(id != 0)
	{
		int dims[3], ijk[3];
		double spa[3], org[3];
		id->GetDimensions(dims);
		id->GetSpacing(spa);
		id->GetOrigin(org);
		
		ijk[2] = cell/(dims[0]*dims[1]);
		ijk[1] = (cell-dims[0]*dims[1]*ijk[2])/dims[0];
		ijk[0] = cell % dims[0];
		for(i=0; i<3; i++) pos[i] = org[i] + spa[i]*ijk[i];
	}
	else
	{
		for(i=0; i<3; i++) pos[i] = 0.0;
	}
}


bool iHistogramMaker::IsDataPresent()
{
	vtkDataSet *input = this->GetInput();
	if(input == 0) return false;
	input->Update();
	if(input->GetPointData() == 0) return false;
	
	switch(mDataRank)
	{
	case 0U:
		{
			if(input->GetPointData()->GetScalars() == 0) return false;
			break;
		}
	case 1U:
		{
			if(input->GetPointData()->GetVectors() == 0) return false;
			break;
		}
	case 2U:
		{
			if(input->GetPointData()->GetTensors() == 0) return false;
			break;
		}
	default:
		{
			return false;
		}
	}

	return true;
}


bool iHistogramMaker::PrepareForPass()
{
	vtkDataSet *input = this->GetInput();
	if(input==0 || input->GetPointData()==0) return false;

	wDataPtr = 0;
	switch(mDataRank)
	{
	case 0U:
		{
			if(input->GetPointData()->GetScalars() == 0) return false;
			//
			//  Access the data for both vtkPolyData and vtkStructuredPoints
			//
			vtkFloatArray *arr = vtkFloatArray::SafeDownCast(input->GetPointData()->GetScalars());
			if(arr == 0) return false;
			wDataPtr = arr->GetPointer(0);
			wNumCom = input->GetPointData()->GetScalars()->GetNumberOfComponents();
			if(mComponent<0 || mComponent>=wNumCom) return false;
			break;
		}
	case 1U:
		{
			if(input->GetPointData()->GetVectors() == 0) return false;
			//
			//  Access the data for both vtkPolyData and vtkStructuredPoints
			//
			vtkFloatArray *arr = vtkFloatArray::SafeDownCast(input->GetPointData()->GetVectors());
			if(arr == 0) return false;
			wDataPtr = arr->GetPointer(0);
			wNumCom = input->GetPointData()->GetVectors()->GetNumberOfComponents();
			if(mComponent!=0 || wNumCom!=3) return false;
			break;
		}
	case 2U:
		{
			if(input->GetPointData()->GetTensors() == 0) return false;
			//
			//  Access the data for both vtkPolyData and vtkStructuredPoints
			//
			vtkFloatArray *arr = vtkFloatArray::SafeDownCast(input->GetPointData()->GetTensors());
			if(arr == 0) return false;
			wDataPtr = arr->GetPointer(0);
			wNumCom = input->GetPointData()->GetTensors()->GetNumberOfComponents();
			if(mComponent<0 || mComponent>2 || wNumCom!=9) return false;
			break;
		}
	default:
		{
			return false;
		}
	}

	if(wDataPtr == 0) return false;

	if(mDataRank == 0U) wDataPtr += mComponent;

	return true;
}


bool iHistogramMaker::IsValidIndex(vtkIdType l) const
{
	return true; // no holes in data
}


float iHistogramMaker::GetScalarValue(vtkIdType l) const
{
	return wDataPtr[l*wNumCom];
}


float iHistogramMaker::GetVectorValue(vtkIdType l) const
{
	float *p = wDataPtr + l*wNumCom;
	return sqrt(p[0]*p[0]+p[1]*p[1]+p[2]*p[2]);
}


float iHistogramMaker::GetTensorValue(vtkIdType l) const
{
	int i;
	float *p = wDataPtr + l*wNumCom;
	float *m[3], e[3], *v[3], store[9];
	
	for(i=0; i<3; i++)
	{
		v[i] = store + 3*i;
		m[i] = p + 3*i;
	}
	vtkMath::Jacobi(m,e,v);
	return e[mComponent];
}

