/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iwriter.h"


#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "ipostscriptwriter.h"
#include "ishell.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include <vtkBMPWriter.h>
#include <vtkImageData.h>
#include <vtkJPEGWriter.h>
#include <vtkPNGWriter.h>
#include <vtkPNMWriter.h>
#include <vtkTIFFWriter.h>

#ifndef IVTK_4

#include "vtkGenericMovieWriter.h"

#ifdef IVTK_SUPPORTS_AVI
#include <vtkAVIWriter.h>
#endif

#ifdef IVTK_SUPPORTS_MPEG
#include <vtkMPEG2Writer.h>
#endif

#else

class vtkGenericMovieWriter;

#endif

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


//
//  Declare static members
//
int iWriter::mCurrentSnapshotImageIndex = 1;
int iWriter::mCurrentAnimationImageIndex = 1;
iString iWriter::mCurrentSnapshotImageFilePrefix = "ifrit-im";
iString iWriter::mCurrentAnimationImageFilePrefix = "ifrit-an";


//
//  Helper class
//
class iGenericMultiImageMovieWriter : public vtkObjectBase
{

	IPOINTER_AS_PART(ViewModule);

public:

	vtkTypeMacro(iGenericMultiImageMovieWriter,vtkObjectBase);

	void Start()
	{
		mStarted = true;
	}

	void WriteFrame(const iStereoImageArray& images, const iString &path, const iString &root);

	void Finish()
	{
#ifndef IVTK_4
		int i;

		for(i=0; i<mImageWriters.Size(); i++) if(mImageWriters[i]->GetInput() != 0)
		{
			mImageWriters[i]->End();
			if(mImageWriters[i]->GetError() != 0)
			{
				IERROR_LOW("Unable to write a movie file: different frames have varied number of images");
				return;
			}
		}
#endif
		mStarted = false;
	}

protected:

	iGenericMultiImageMovieWriter(iViewModule *vm) : mViewModule(vm)
	{
		IERROR_ASSERT(vm);
		mStarted = false;
	}

	virtual ~iGenericMultiImageMovieWriter()
	{
#ifndef IVTK_4
		while(mImageWriters.Size() > 0) mImageWriters.RemoveLast()->Delete();
#endif
	}

	virtual vtkGenericMovieWriter* CreateWriter() const = 0;  // factory method
	virtual const iString& GetFileSuffix() const = 0;

	bool mStarted;
	iArray<vtkGenericMovieWriter*> mImageWriters;
};


namespace iWriter_Private
{
	template <class T>
	class MultiImageMovieWriter : public iGenericMultiImageMovieWriter
	{

	public:

		static MultiImageMovieWriter<T>* New(iViewModule *vm, const iString &suf)
		{
			return new MultiImageMovieWriter<T>(vm,suf);
		}

	protected:

		MultiImageMovieWriter(iViewModule *vm, const iString &suf) : iGenericMultiImageMovieWriter(vm)
		{
			mSuffix = suf;
		}

		virtual vtkGenericMovieWriter* CreateWriter() const
		{
			return T::New();
		}

		virtual const iString& GetFileSuffix() const
		{
			return mSuffix;
		}

		iString mSuffix;
	};


	iString FrameFileName(int i, const iString &tag, const iString &root)
	{
		if(tag.IsEmpty())
		{
			if(i == 0)
			{
				//
				//  Main view
				//
				return root;
			}
			else
			{
				//
				//  Other views
				//
				return "v" + iString::FromNumber(i+1,(i<99)?"%02d":"%d") + "-" + root;
			}
		}
		else
		{
			return tag + "-" + root;
		}
	}


	iString RightEyeFrameFileName(const iString &name)
	{
		return "re-" + name;
	}
};


using namespace iWriter_Private;


void iGenericMultiImageMovieWriter::WriteFrame(const iStereoImageArray& images, const iString &path, const iString &root)
{
	if(mStarted)
	{
#ifndef IVTK_4
		while(2*images.Size() < mImageWriters.Size()) mImageWriters.RemoveLast()->Delete();
		while(2*images.Size() > mImageWriters.Size())
		{
			vtkGenericMovieWriter *tmp = this->CreateWriter(); IERROR_ASSERT(tmp);
			tmp->AddObserver(vtkCommand::StartEvent,this->GetViewModule()->GetProgressEventObserver());
			tmp->AddObserver(vtkCommand::ProgressEvent,this->GetViewModule()->GetProgressEventObserver());
			tmp->AddObserver(vtkCommand::EndEvent,this->GetViewModule()->GetProgressEventObserver());
			mImageWriters.Add(tmp);
		}

		int i;
		iString name;
		for(i=0; i<mImageWriters.Size(); i+=2)
		{
			name = FrameFileName(i/2,images[i/2].GetTag(),root);
			mImageWriters[i]->SetFileName((path+name+"."+this->GetFileSuffix()).ToCharPointer());
			mImageWriters[i]->SetInput(images[i/2].LeftEye().DataObject());
			mImageWriters[i]->Start();
			//
			//  Right eye
			//
			if(images[i/2].IsStereo())
			{
				mImageWriters[i+1]->SetInput(images[i/2].RightEye().DataObject());
				mImageWriters[i+1]->SetFileName((path+RightEyeFrameFileName(name)+"."+this->GetFileSuffix()).ToCharPointer());
				mImageWriters[i+1]->Start();
			}
			else
			{
				mImageWriters[i+1]->SetInput(0);
			}
		}
#endif
		mStarted = false;
	}

#ifndef IVTK_4
	if(!mStarted && 2*images.Size()!=mImageWriters.Size())
	{
		IERROR_LOW("Unable to write a movie file: different frames have varied number of images.");
		return;
	}

	int i;
	for(i=0; i<mImageWriters.Size(); i+=2)
	{
		mImageWriters[i]->SetInput(images[i/2].LeftEye().DataObject());
		mImageWriters[i]->Write();
		mImageWriters[i]->SetInput(0);
		if(mImageWriters[i]->GetError() != 0)
		{
			IERROR_LOW("Unable to write a movie file: VTK reported an unidentified error.");
			return;
		}
		if(images[i/2].IsStereo())
		{
			mImageWriters[i+1]->SetInput(images[i/2].RightEye().DataObject());
			mImageWriters[i+1]->Write();
			mImageWriters[i+1]->SetInput(0);
			if(mImageWriters[i+1]->GetError() != 0)
			{
				IERROR_LOW("Unable to write a movie file: VTK reported an unidentified error.");
				return;
			}
		}
	}
#endif
}


//
//  Main class
//
iWriter* iWriter::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iWriter(vm);
}


iWriter::iWriter(iViewModule *vm) : mViewModule(vm)
{
	mPostScriptPaperFormat = 9;
	mPostScriptOrientation = 0;

	mImageWriter = 0;
	mMovieWriter = 0;

	mImageFormat = -1;
	mAnimationOutput = -1;
}


iWriter::~iWriter()
{
	if(mImageWriter != 0) mImageWriter->Delete();
	if(mMovieWriter != 0) mMovieWriter->Delete();
}




void iWriter::SetImageFormat(int i)
{
	if(i>=0 && i<__NumImageFormats && i!=mImageFormat)
	{
		mImageFormat = i;
		if(mImageWriter != 0) mImageWriter->Delete();
		switch(mImageFormat) 
		{
		case _ImageFormatJPG:
			{
				mImageWriter = vtkJPEGWriter::New(); IERROR_ASSERT(mImageWriter); 
				break;
			}
		case _ImageFormatPNM:
			{
				mImageWriter = vtkPNMWriter::New(); IERROR_ASSERT(mImageWriter); 
				break;
			}
		case _ImageFormatBMP:
			{
				mImageWriter = vtkBMPWriter::New(); IERROR_ASSERT(mImageWriter); 
				break;
			}
		case _ImageFormatTIF:
			{
				mImageWriter = vtkTIFFWriter::New(); IERROR_ASSERT(mImageWriter); 
				break;
			}
		case _ImageFormatEPS:
			{
				iPostScriptWriter *p = iPostScriptWriter::New(); IERROR_ASSERT(p);
				p->SetPaperFormat(mPostScriptPaperFormat);
				p->SetOrientation(mPostScriptOrientation);
				mImageWriter = p;
				break;
			}
		default:
			{
				mImageWriter = vtkPNGWriter::New(); IERROR_ASSERT(mImageWriter); 
				break;
			}
		}

		mImageWriter->AddObserver(vtkCommand::StartEvent,this->GetViewModule()->GetProgressEventObserver());
		mImageWriter->AddObserver(vtkCommand::ProgressEvent,this->GetViewModule()->GetProgressEventObserver());
		mImageWriter->AddObserver(vtkCommand::EndEvent,this->GetViewModule()->GetProgressEventObserver());
		this->GetViewModule()->ClearCache();
	}
}


void iWriter::SetAnimationOutput(int i)
{
	if(i>=0 && i<__NumAnimationOutputs && i!=mAnimationOutput)
	{
#ifdef IVTK_4
		mAnimationOutput = 0;
#else
		mAnimationOutput = i;
		if(mMovieWriter != 0)
		{
			mMovieWriter->Delete();
			mMovieWriter = 0;
		}
		switch(mAnimationOutput)
		{
#ifdef IVTK_SUPPORTS_MPEG
		case _AnimationOutputMPEG2:
			{
				mMovieWriter = MultiImageMovieWriter<vtkMPEG2Writer>::New(this->GetViewModule(),"mpg"); IERROR_ASSERT(mMovieWriter);
				break;
			}
#endif
#ifdef IVTK_SUPPORTS_AVI
		case _AnimationOutputAVI:
			{
				mMovieWriter = MultiImageMovieWriter<vtkAVIWriter>::New(this->GetViewModule(),"avi"); IERROR_ASSERT(mMovieWriter);
				break;
			}
#endif
		default:
			{
				//
				//  Did not find a movie writer
				//
				mAnimationOutput = 0;
			}
		}
#endif
		this->GetViewModule()->ClearCache();
	}
}


void iWriter::SetPostScriptPaperFormat(int i)
{
	if(i>=0 && i!=mPostScriptPaperFormat)
	{
		mPostScriptPaperFormat = i;
		this->GetViewModule()->ClearCache();
		if(mImageFormat == _ImageFormatEPS) iRequiredCast<iPostScriptWriter>(INFO,mImageWriter)->SetPaperFormat(i);
	}
}


void iWriter::SetPostScriptOrientation(int i)
{
	if(i>=0 && i!=mPostScriptOrientation)
	{
		mPostScriptOrientation = i;	
		this->GetViewModule()->ClearCache();
		if(mImageFormat == _ImageFormatEPS) iRequiredCast<iPostScriptWriter>(INFO,mImageWriter)->SetOrientation(i);
	}
}


void iWriter::Start()
{
	if(mMovieWriter != 0)
	{
		mMovieWriter->Start();
	}
}


void iWriter::Finish()
{
	if(mMovieWriter != 0)
	{
		mMovieWriter->Finish();
		mCurrentAnimationImageIndex++;
	}
}


void iWriter::WriteFrame(const iStereoImageArray& images, int mode)
{
	iString path, root;
	int *cur;
	
	switch(mode) 
	{
	case _ImageTypeSnapshot:
		{
			root = mCurrentSnapshotImageFilePrefix;
			cur = &mCurrentSnapshotImageIndex;
			break;
		}
	case _ImageTypeAnimationFrame:
		{
			root = mCurrentAnimationImageFilePrefix;
			cur = &mCurrentAnimationImageIndex;
			break;
		}
	default:
		{
			IERROR_LOW("Invalid argument to iWriter::WriteFrame.");
			return;
		}
	}

	path = this->GetViewModule()->GetControlModule()->GetShell()->GetEnvironment(_EnvironmentImage);
	root += iString::FromNumber(*cur,"-%05d");

	if(mode==_ImageTypeSnapshot || mMovieWriter==0)
	{
		this->WriteImageFrame(images,path,root);
		(*cur)++;
	}
	else
	{
		this->WriteMovieFrame(images,path,root);
	}
}


void iWriter::WriteImageFrame(const iStereoImageArray& images, const iString &path, const iString &root)
{
	int i;
	iString suf, name;

	switch(mImageFormat) 
	{
	case _ImageFormatJPG: { suf = ".jpg"; break; }
	case _ImageFormatPNM: { suf = ".ppm"; break; }
	case _ImageFormatBMP: { suf = ".bmp"; break; }
	case _ImageFormatTIF: { suf = ".tif"; break; }
	case _ImageFormatEPS: { suf = ".eps"; break; }
	default:			  { suf = ".png"; }
	}
	
	//
	// Create and write files
	//
	for(i=0; i<images.Size(); i++)
	{
		name = FrameFileName(i,images[i].GetTag(),root);

		mImageWriter->SetInput(images[i].LeftEye().DataObject());
		mImageWriter->SetFileName((path+name+suf).ToCharPointer());
		mImageWriter->Write();
		//
		//  Right eye
		//
		if(images[i].IsStereo())
		{
			mImageWriter->SetInput(images[i].RightEye().DataObject());
			mImageWriter->SetFileName((path+RightEyeFrameFileName(name)+suf).ToCharPointer());
			mImageWriter->Write();
		}
	}
}


void iWriter::WriteMovieFrame(const iStereoImageArray& images, const iString &path, const iString &root)
{
	if(mMovieWriter != 0)
	{
		mMovieWriter->WriteFrame(images,path,root);
	}
	else
	{
		IERROR_LOW("Cannot write a movie file: a MovieWriter has not been created.");
	}
}

