/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggwidgetkeyhandler.h"


#include "icontrolmodule.h"
#include "ierror.h"

#include "iggdatatypeprovider.h"
#include "iggframe.h"

//
//  templates
//
#include "iarraytemplate.h"
#include "iggwidgetkeyhandlertemplate.h"


using namespace iParameter;
using namespace iParameter;


int iggWidgetKeyHandlerBase::mGlobalRenderMode = RenderMode::Delayed;
#ifdef I_CHECK1
bool iggWidgetKeyHandlerBase::mCheckNullIndexKeys =  false;
#endif

//
//  Base class
//
iggWidgetKeyHandlerBase::iggWidgetKeyHandlerBase(int type, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidget(parent), mType(type), mKey(key), mIndexKey(indkey), mProvider(0)
{
	int i;

	mIndex = index;
	if(mIndex<-1 || (mKey.Dimension()>0 && mIndex>=mKey.Dimension()))
	{
		IERROR_LOW("Incorrect index value.");
		mIndex = -1;
	}

	if(mIndexKey!=0 && (mIndexKey->Argument()!=iObjectKey::_OffsetInt || mIndexKey->Dimension()!=1))
	{
		IERROR_LOW("Index key must be a scalar OffsetInt-valued key.");
		mIndexKey = 0;
	}

#ifdef I_CHECK1
	if(mCheckNullIndexKeys && mIndexKey==0)
	{
		IERROR_LOW("Null index key.");
	}
#endif

	mSelfChecking = false;

	mRenderMode = rm;

	if(mParent == 0) IERROR_FATAL("iggWidgetKeyHandlerBase must have a parent.");

	//
	//  Find all my buddies
	//
	iPointerArray<iggWidgetKeyHandlerBase> &list = iggWidgetKeyHandlerBase::List();
	for(i=0; i<list.Size(); i++)
	{
		if(list[i]->mKey == mKey) list[i]->AddBuddy(this);
	}
	//
	//  Add myself to the list
	//
	list.AddUnique(this);

	mHelpIsSet = false;
}


iggWidgetKeyHandlerBase::~iggWidgetKeyHandlerBase()
{
	iggWidgetKeyHandlerBase::List().Remove(this);

	if(mNeedsBaloonHelp && !mHelpIsSet)
	{
		//
		//  We were never updated, so the help has never been set. Then we do not need it at all.
		//
		mNeedsBaloonHelp = false;
	}
}


void iggWidgetKeyHandlerBase::UpdateHelp()
{
	if(mNeedsBaloonHelp && !mHelpIsSet)
	{
		mHelpIsSet = true;
		this->SetBaloonHelp("Controls the "+this->GetKey().PrefixedFullName()+" property. Press Shift+F1 for more help.",(this->GetKey().GetHelp() == 0)?"No help is set for this property.":this->GetKey().GetHelp()->GetHTML());
	}
}


void iggWidgetKeyHandlerBase::SetSelfChecking(bool s)
{
	mSelfChecking = s;
}


void iggWidgetKeyHandlerBase::SetIndex(int i)
{
	if(mIndexKey != 0)
	{
		IERROR_LOW("Cannot set an index if the IndexKey is defined.");
		return;
	}

	if(i<-1 || (mKey.Dimension()>0 && i>=mKey.Dimension())) // zero-dimension key has no range checking
	{
		IERROR_LOW("Incorrect index value.");
		i = -1;
	}
	mIndex = i;
}


void iggWidgetKeyHandlerBase::SetRenderMode(int m)
{
	mRenderMode = m;
}


iPointerArray<iggWidgetKeyHandlerBase>& iggWidgetKeyHandlerBase::List()
{
	static iPointerArray<iggWidgetKeyHandlerBase> list(100);
	return list;
}


iPointerArray<iggWidgetKeyHandlerBase>& iggWidgetKeyHandlerBase::PositionList()
{
	static iPointerArray<iggWidgetKeyHandlerBase> list;
#ifdef I_CHECK2
	//
	//  Check that all positions are GL ones
	//
	int i;
	for(i=0; i<list.Size(); i++)
	{
		const iObjectKey& key = list[i]->mKey;
		if(key.PrefixedFullName().Part(-2) != "GL")
		{
			IERROR_LOW("Incorrect position key");
		}
	}
#endif
	return list;
}


iPointerArray<iggWidgetKeyHandlerBase>& iggWidgetKeyHandlerBase::VariableLimitsList()
{
	static iPointerArray<iggWidgetKeyHandlerBase> list;
	return list;
}


void iggWidgetKeyHandlerBase::SetGlobalRenderMode(int m)
{
	if(m != RenderMode::UseGlobal)
	{
		mGlobalRenderMode = m;
	}
}


int iggWidgetKeyHandlerBase::GetRenderMode() const
{
	if(mRenderMode != RenderMode::UseGlobal)
	{
		return mRenderMode;
	}
	else
	{
		switch(mType)
		{
		case WidgetType::Entry:
			{
				return mGlobalRenderMode;
			}
		default:
			{
				return RenderMode::Immediate;
			}
		}
	}
}


bool iggWidgetKeyHandlerBase::CheckBaloonHelpStatus()
{
	return !mHelpIsSet || this->iggWidget::CheckBaloonHelpStatus();
}


void iggWidgetKeyHandlerBase::AttachDataTypeProvider(const iggDataTypeProvider *provider)
{
	mProvider = provider;
}


const iObjectKey& iggWidgetKeyHandlerBase::Translate(const iObjectKey &key) const
{
	if(mProvider == 0)
	{
		return key;
	}
	else
	{
		return mProvider->Translate(key);
	}
}


const iObjectKey& iggWidgetKeyHandlerBase::GetKey() const
{
	return this->Translate(mKey);
}


#ifdef I_CHECK1
//
//  For internal checking
//
void iggWidgetKeyHandlerBase::SetCheckNullIndexKeys(bool s)
{
	mCheckNullIndexKeys = s;
}
#endif

#endif
