/*
 * constraint-scaling.c : routines to scale image following a boundary box
 *			  and to compute the optimal font for a given 
 *			  boundary box.
 *
 * Copyright (C) 2003 Sun Microsystems, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Authors:
 *   Erwann Chenede - <erwann.chenede@sun.com>
 *
 *   scaling inspired from the pixbuf gtk theme constrained scaling 
 *   routines from :
 *	  Owen Taylor <otaylor@redhat.com> 
 *	  Carsten Haitzler <raster@rasterman.com>
 */

#include "constraint-scaling.h"

static GdkPixbuf *
replicate_single (GdkPixbuf    *src,
		  gint          src_x,
		  gint          src_y,
		  gint          width,
		  gint          height)
{
  guint n_channels = gdk_pixbuf_get_n_channels (src);
  guchar *pixels = (gdk_pixbuf_get_pixels (src) +
		    src_y * gdk_pixbuf_get_rowstride (src) +
		    src_x * n_channels);
  guchar r = *(pixels++);
  guchar g = *(pixels++);
  guchar b = *(pixels++);
  guint dest_rowstride;
  guchar *dest_pixels;
  guchar a = 0;
  GdkPixbuf *result;
  int i, j;

  if (n_channels == 4)
    a = *(pixels++);

  result = gdk_pixbuf_new (GDK_COLORSPACE_RGB, n_channels == 4, 8,
			   width, height);
  dest_rowstride = gdk_pixbuf_get_rowstride (result);
  dest_pixels = gdk_pixbuf_get_pixels (result);
  
  for (i = 0; i < height; i++)
    {
      guchar *p = dest_pixels + dest_rowstride *i;

      for (j = 0; j < width; j++)
	{
	  *(p++) = r;
	  *(p++) = g;
	  *(p++) = b;

	  if (n_channels == 4)
	    *(p++) = a;
	}
    }

  return result;
}

static GdkPixbuf *
replicate_rows (GdkPixbuf    *src,
		gint          src_x,
		gint          src_y,
		gint          width,
		gint          height)
{
  guint n_channels = gdk_pixbuf_get_n_channels (src);
  guint src_rowstride = gdk_pixbuf_get_rowstride (src);
  guchar *pixels = (gdk_pixbuf_get_pixels (src) + src_y * src_rowstride + src_x * n_channels);
  guchar *dest_pixels;
  GdkPixbuf *result;
  guint dest_rowstride;
  int i;

  result = gdk_pixbuf_new (GDK_COLORSPACE_RGB, n_channels == 4, 8,
			   width, height);
  dest_rowstride = gdk_pixbuf_get_rowstride (result);
  dest_pixels = gdk_pixbuf_get_pixels (result);

  for (i = 0; i < height; i++)
    memcpy (dest_pixels + dest_rowstride * i, pixels, n_channels * width);

  return result;
}

static GdkPixbuf *
replicate_cols (GdkPixbuf    *src,
		gint          src_x,
		gint          src_y,
		gint          width,
		gint          height)
{
  guint n_channels = gdk_pixbuf_get_n_channels (src);
  guint src_rowstride = gdk_pixbuf_get_rowstride (src);
  guchar *pixels = (gdk_pixbuf_get_pixels (src) + src_y * src_rowstride + src_x * n_channels);
  guchar *dest_pixels;
  GdkPixbuf *result;
  guint dest_rowstride;
  int i, j;

  result = gdk_pixbuf_new (GDK_COLORSPACE_RGB, n_channels == 4, 8,
			   width, height);
  dest_rowstride = gdk_pixbuf_get_rowstride (result);
  dest_pixels = gdk_pixbuf_get_pixels (result);

  for (i = 0; i < height; i++)
    {
      guchar *p = dest_pixels + dest_rowstride * i;
      guchar *q = pixels + src_rowstride * i;

      guchar r = *(q++);
      guchar g = *(q++);
      guchar b = *(q++);
      guchar a = 0;
      
      if (n_channels == 4)
	a = *(q++);

      for (j = 0; j < width; j++)
	{
	  *(p++) = r;
	  *(p++) = g;
	  *(p++) = b;

	  if (n_channels == 4)
	    *(p++) = a;
	}
    }

  return result;
}

/* Scale the rectangle (src_x, src_y, src_width, src_height)
 * onto the rectangle (dest_x, dest_y, dest_width, dest_height)
 * of the destination 
 */
static void
partial_render (GdkPixbuf    *src,
		guint         hints,
		GdkPixbuf    *scaled,
		gint          src_x,
		gint          src_y,
		gint          src_width,
		gint          src_height,
		gint          dest_x,
		gint          dest_y,
		gint          dest_width,
		gint          dest_height)
{
  GdkPixbuf *tmp_pixbuf;
  GdkRectangle rect;
  int x_offset, y_offset;
  gboolean has_alpha = gdk_pixbuf_get_has_alpha (src);
  gint src_rowstride = gdk_pixbuf_get_rowstride (src);
  gint src_n_channels = gdk_pixbuf_get_n_channels (src);

  if (dest_width <= 0 || dest_height <= 0)
    return;

  rect.x = dest_x;
  rect.y = dest_y;
  rect.width = dest_width;
  rect.height = dest_height;

  if (hints & MISSING)
    return;

  if (dest_width == src_width && dest_height == src_height)
    {
      tmp_pixbuf = g_object_ref (src);

      x_offset = src_x + rect.x - dest_x;
      y_offset = src_y + rect.y - dest_y;
    }
  else if ((hints & CONSTANT_COLS) && (hints & CONSTANT_ROWS))
    {
      tmp_pixbuf = replicate_single (src, src_x, src_y, dest_width, dest_height);

      x_offset = rect.x - dest_x;
      y_offset = rect.y - dest_y;
    }
  else if (dest_width == src_width && (hints & CONSTANT_COLS))
    {
      tmp_pixbuf = replicate_rows (src, src_x, src_y, dest_width, dest_height);

      x_offset = rect.x - dest_x;
      y_offset = rect.y - dest_y;
    }
  else if (dest_height == src_height && (hints & CONSTANT_ROWS))
    {
      tmp_pixbuf = replicate_cols (src, src_x, src_y, dest_width, dest_height);

      x_offset = rect.x - dest_x;
      y_offset = rect.y - dest_y;
    }
  else 
    {
      double x_scale = (double)dest_width / src_width;
      double y_scale = (double)dest_height / src_height;
      guchar *pixels;
      GdkPixbuf *partial_src;
      
      pixels = (gdk_pixbuf_get_pixels (src)
		+ src_y * src_rowstride
		+ src_x * src_n_channels);

      partial_src = gdk_pixbuf_new_from_data (pixels, GDK_COLORSPACE_RGB,
					      has_alpha,
					      8, src_width, src_height,
					      src_rowstride,
					      NULL, NULL);
						  
      tmp_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
				   has_alpha, 8,
				   rect.width, rect.height);

      gdk_pixbuf_scale (partial_src, tmp_pixbuf,
			0, 0, rect.width, rect.height,
			dest_x - rect.x, dest_y - rect.y, 
			x_scale, y_scale,
			GDK_INTERP_BILINEAR);

      gdk_pixbuf_unref (partial_src);

      x_offset = 0;
      y_offset = 0;
    }

    gdk_pixbuf_copy_area (tmp_pixbuf,
			  x_offset, y_offset,
			  rect.width, rect.height,
			  scaled,
			  rect.x,
			  rect.y);

  gdk_pixbuf_unref (tmp_pixbuf);
}


void
constrained_pixbuf_render (ConstrainedPixbuf *cp,
			   gint		   height)
{
  gint src_x[4], src_y[4], dest_x[4], dest_y[4];
  gint x = 0, y = 0, width, pixbuf_width, pixbuf_height, tmp_width;
  GdkPixbuf *tmp_scaled;
  
  tmp_width = (gdk_pixbuf_get_width (cp->pixbuf) * height) / gdk_pixbuf_get_height (cp->pixbuf);
  cp->border_left_scaled = (cp->border_left * height) / gdk_pixbuf_get_height (cp->pixbuf);
  cp->border_right_scaled = (cp->border_right * height) / gdk_pixbuf_get_height (cp->pixbuf);
  
  width = cp->border_left_scaled + cp->border_right_scaled + cp->resulting_width;
  
  tmp_scaled = gdk_pixbuf_scale_simple (cp->pixbuf, tmp_width, height, 
					GDK_INTERP_BILINEAR);
  
  pixbuf_width = gdk_pixbuf_get_width (tmp_scaled);
  pixbuf_height = gdk_pixbuf_get_height (tmp_scaled);
  

  if (width == pixbuf_width && height == pixbuf_height)
    return;

  if (cp->scaled)
    {
      g_object_unref (cp->scaled);
      cp->scaled = NULL;
    }
  cp->scaled = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (tmp_scaled),
			       gdk_pixbuf_get_has_alpha (tmp_scaled),
			       gdk_pixbuf_get_bits_per_sample (tmp_scaled),
			       width, 
			       height);
  gdk_pixbuf_fill (cp->scaled, 0xff0000ff);

  src_x[0] = 0;
  src_x[1] = cp->border_left_scaled;
  src_x[2] = pixbuf_width - cp->border_right_scaled;
  src_x[3] = pixbuf_width;
  
  src_y[0] = 0;
  src_y[1] = cp->border_top;
  src_y[2] = pixbuf_height - cp->border_bottom;
  src_y[3] = pixbuf_height;
  
  dest_x[0] = x;
  dest_x[1] = x + cp->border_left_scaled;
  dest_x[2] = x + width - cp->border_right_scaled;
  dest_x[3] = x + width;

  dest_y[0] = y;
  dest_y[1] = y + cp->border_top;
  dest_y[2] = y + height - cp->border_bottom;
  dest_y[3] = y + height;

  if (cp->component_mask & COMPONENT_ALL)
    cp->component_mask = (COMPONENT_ALL - 1) & ~cp->component_mask;

#define RENDER_COMPONENT(X1,X2,Y1,Y2)				         \
        partial_render   (tmp_scaled, cp->hints[Y1][X1],			 \
			  cp->scaled,						 \
	 	          src_x[X1], src_y[Y1],				         \
			  src_x[X2] - src_x[X1], src_y[Y2] - src_y[Y1],	         \
			  dest_x[X1], dest_y[Y1],				 \
			  dest_x[X2] - dest_x[X1], dest_y[Y2] - dest_y[Y1]);
    
  if (cp->component_mask & COMPONENT_NORTH_WEST)
      RENDER_COMPONENT (0, 1, 0, 1);

  if (cp->component_mask & COMPONENT_NORTH)
      RENDER_COMPONENT (1, 2, 0, 1);

  if (cp->component_mask & COMPONENT_NORTH_EAST)
      RENDER_COMPONENT (2, 3, 0, 1);

  if (cp->component_mask & COMPONENT_WEST)
      RENDER_COMPONENT (0, 1, 1, 2);

  if (cp->component_mask & COMPONENT_CENTER)
      RENDER_COMPONENT (1, 2, 1, 2);

  if (cp->component_mask & COMPONENT_EAST)
      RENDER_COMPONENT (2, 3, 1, 2);

  if (cp->component_mask & COMPONENT_SOUTH_WEST)
      RENDER_COMPONENT (0, 1, 2, 3);

  if (cp->component_mask & COMPONENT_SOUTH)
      RENDER_COMPONENT (1, 2, 2, 3);

  if (cp->component_mask & COMPONENT_SOUTH_EAST)
      RENDER_COMPONENT (2, 3, 2, 3);

 g_object_unref (tmp_scaled); 
}

static guint
compute_hint (GdkPixbuf *pixbuf,
	      gint       x0,
	      gint       x1,
	      gint       y0,
	      gint       y1)
{
  int i, j;
  int hints = CONSTANT_ROWS | CONSTANT_COLS | MISSING;
  int n_channels = gdk_pixbuf_get_n_channels (pixbuf);
  
  guchar *data = gdk_pixbuf_get_pixels (pixbuf);
  int rowstride = gdk_pixbuf_get_rowstride (pixbuf);

  if (x0 == x1 || y0 == y1)
    return 0;

  for (i = y0; i < y1; i++)
    {
      guchar *p = data + i * rowstride + x0 * n_channels;
      guchar r = p[0];
      guchar g = p[1];
      guchar b = p[2];
      guchar a = 0;
      
      if (n_channels == 4)
	a = p[3];

      for (j = x0; j < x1 ; j++)
	{
	  if (n_channels != 4 || p[3] != 0)
	    {
	      hints &= ~MISSING;
	      if (!(hints & CONSTANT_ROWS))
		goto cols;
	    }
	  
	  if (r != *(p++) ||
	      g != *(p++) ||
	      b != *(p++) ||
	      (n_channels != 4 && a != *(p++)))
	    {
	      hints &= ~CONSTANT_ROWS;
	      if (!(hints & MISSING))
		goto cols;
	    }
	}
    }

 cols:
  for (i = y0 + 1; i < y1; i++)
    {
      guchar *base = data + y0 * rowstride + x0 * n_channels;
      guchar *p = data + i * rowstride + x0 * n_channels;

      if (memcmp (p, base, n_channels * (x1 - x0)) != 0)
	{
	  hints &= ~CONSTANT_COLS;
	  return hints;
	}
    }

  return hints;
}

static gboolean
compute_hints (ConstrainedPixbuf *const_pixbuf)
{
  int i, j, k = 0;
  gint width = gdk_pixbuf_get_width (const_pixbuf->pixbuf);
  gint height = gdk_pixbuf_get_height (const_pixbuf->pixbuf);

  if (const_pixbuf->border_left + const_pixbuf->border_right > width ||
      const_pixbuf->border_top + const_pixbuf->border_bottom > height)
    return FALSE;
  
  for (i = 0; i < 3; i++)
    {
      gint y0, y1;

      switch (i)
	{
	case 0:
	  y0 = 0;
	  y1 = const_pixbuf->border_top;
	  break;
	case 1:
	  y0 = const_pixbuf->border_top;
	  y1 = height - const_pixbuf->border_bottom;
	  break;
	default:
	  y0 = height - const_pixbuf->border_bottom;
	  y1 = height;
	  break;
	}
      
      for (j = 0; j < 3; j++)
	{
	  gint x0, x1;
	  
	  
	  switch (j)
	    {
	    case 0:
	      x0 = 0;
	      x1 = const_pixbuf->border_left;
	      break;
	    case 1:
	      x0 = const_pixbuf->border_left;
	      x1 = width - const_pixbuf->border_right;
	      break;
	    default:
	      x0 = width - const_pixbuf->border_right;
	      x1 = width;
	      break;
	    }

	  const_pixbuf->hints[i][j] = compute_hint (const_pixbuf->pixbuf, x0, x1, y0, y1);
	  k++;
	}
    }
  return TRUE;
}

gboolean
constrained_pixbuf_init (ConstrainedPixbuf *cp,
			 GdkPixbuf *src)
{
  GError *local_err = NULL;

  if (cp->pixbuf)
    g_object_unref (cp->pixbuf);
  cp->pixbuf = NULL;

  if (cp->scaled)
    g_object_unref (cp->scaled);
  cp->scaled = NULL;

  cp->pixbuf = gdk_pixbuf_copy (src);

  if (!cp->pixbuf)
    {
      g_free (cp);
      return FALSE;
    }
  
  cp->component_mask = COMPONENT_ALL; 

  if (!compute_hints (cp))
    return FALSE;

  return TRUE;
}

void
constrained_pixbuf_free (ConstrainedPixbuf *cp)
{
  if (cp->text)
    g_free (cp->text);
  if (cp->font)
    g_free (cp->font);
  
  if (cp->layout)
    g_object_unref (cp->layout);
  
  if (cp->pixbuf)
    g_object_unref (cp->pixbuf);
  if (cp->scaled)
    g_object_unref (cp->scaled);

  g_free (cp);
}

void
constrained_pixbuf_compute_layout (GtkWidget *widget,
				   ConstrainedPixbuf *cp,
				   int height, int width)
{
  char font_full [256];	    
  int pango_width, pango_height;
  PangoFontDescription *font_desc = NULL;
  int requested_height = height - (cp->border_top + cp->border_bottom);
  int requested_width = width - (cp->border_left + cp->border_right);
  int font_size = 3;
  gboolean fit = FALSE;
  gboolean fit_width = FALSE;
  
  PangoLayout *layout = gtk_widget_create_pango_layout (widget,	cp->text);
  
  while (!fit)
    {
      int pix_width, pix_height;
      font_size++;
      
      sprintf (font_full, "%s %d", cp->font, font_size);
      
      if (font_desc)
	pango_font_description_free(font_desc);

      if (font_size < 0)
	{
	  fit = TRUE;
	  font_size = 4;
	}

      font_desc = pango_font_description_from_string (font_full);
      pango_layout_set_font_description (layout, font_desc);
      
      pango_layout_get_size (layout, &pango_width, &pango_height);

      pix_height = PANGO_PIXELS (pango_height);
      pix_width = PANGO_PIXELS (pango_width);

      /* find optimal height */
      if (PANGO_PIXELS (pango_height) > requested_height ||
	  PANGO_PIXELS (pango_width) > requested_width)
	{
	  font_size--;
	  fit = TRUE;
	}
     }
  
  sprintf (font_full, "%s %d", cp->font, font_size);
  pango_font_description_free(font_desc);
  font_desc = pango_font_description_from_string (font_full);
  pango_layout_set_font_description (layout, font_desc);
  pango_layout_get_size (layout, &pango_width, &pango_height);

  cp->resulting_width = PANGO_PIXELS (pango_width);
  cp->center_x = (cp->resulting_width - PANGO_PIXELS (pango_width)) / 2;
  cp->center_y = (requested_height - PANGO_PIXELS (pango_height)) / 2;

  cp->layout = layout;
}

