/* GNOME INPUT METHOD SWITCHER
 * Copyright 2003 Sun Microsystems Inc.
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkdrawingarea.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtksignal.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <gnome.h>

#include "gnome-im-switcher.h"
#include "preference.h"
#include "language.h"
#include "widgets.h"
#include "utils.h"

#include <gconf/gconf-client.h>

static GConfClient *default_client = NULL;

static void gimlet_preference_set_font (GimletWindow *gimlet,
					const char *menu_font,
					const char *status_font);
static void gimlet_preference_set_color (GimletWindow *gimlet,
					 const char *foreground,
					 const char *background);

static void
appearance_update (GimletWindow *gimlet)
{
  gimlet_draw_icon (gimlet);

  panel_applet_gconf_set_bool (PANEL_APPLET (gimlet->applet),
			       "show_icon",
			       gimlet->showicon,
			       NULL);
  panel_applet_gconf_set_bool (PANEL_APPLET (gimlet->applet),
			       "show_text",
			       gimlet->showtext,
			       NULL);
  panel_applet_gconf_set_string (PANEL_APPLET (gimlet->applet),
				 "icon_name",
				 gimlet->base_iconname,
				 NULL);
  panel_applet_gconf_set_int (PANEL_APPLET (gimlet->applet),
			      "text_width",
			      gimlet->textwidth,
			      NULL);
  gimlet_update (gimlet);
}

static void
input_language_policy_update (GimletWindow *gimlet)
{
  panel_applet_gconf_set_int (PANEL_APPLET (gimlet->applet),
			      "input_language_policy",
			      gimlet->input_lang_policy,
			      NULL);
}

/* callbacks */
static void
on_text_only_toggled (GtkWidget *widget,
		      GtkWidget *dialog)
{
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");
  GtkWidget *w = g_object_get_data (G_OBJECT (dialog), "icon-picker");

  gimlet->showtext = TRUE;
  gimlet->showicon = FALSE;

  gtk_widget_set_sensitive (w, FALSE);

  appearance_update (gimlet);
}

static void
on_text_and_icon_toggled (GtkWidget *widget,
			  GtkWidget *dialog)
{
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");
  GtkWidget *w = g_object_get_data (G_OBJECT (dialog), "icon-picker");

  gimlet->showtext = TRUE;
  gimlet->showicon = TRUE;

  gtk_widget_set_sensitive (w, TRUE);

  appearance_update (gimlet);
}

static void
text_width_size_changed (GtkWidget *widget,
			 GtkWidget *dialog)
{
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");
  int val;

  val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));

  gimlet->textwidth = val;

  appearance_update (gimlet);
}

static void
icon_picker_changed (GtkWidget *icon_entry,
		     GimletWindow *gimlet)
{
  char *filename;

  filename = gnome_icon_entry_get_filename (GNOME_ICON_ENTRY (icon_entry));

  gimlet_change_base_icon (gimlet, filename);

  appearance_update (gimlet);

  g_free (filename);
}

static void
on_follow_application_toggled (GtkWidget *button,
			       GimletWindow *gimlet)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    {
      gimlet->input_lang_policy = FOLLOW_APPLICATION;
#ifdef MEANINGLESS_OPTION_PERHAPS
      gtk_widget_set_sensitive (gimlet->input_language_option_menu, FALSE);
#endif
    }

  input_language_policy_update (gimlet);
}

static void
on_follow_quick_access_menu_toggled (GtkWidget *button,
				     GimletWindow *gimlet)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    {
      gimlet->input_lang_policy = FOLLOW_QUICK_ACCESS_MENU;
#ifdef MEANINGLESS_OPTION_PERHAPS
      gtk_widget_set_sensitive (gimlet->input_language_option_menu, FALSE);
#endif
    }

  input_language_policy_update (gimlet);
}

#ifdef MEANINGLESS_OPTION_PERHAPS
static void
on_follow_current_locale_toggled (GtkWidget *button,
				  GimletWindow *gimlet)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    {
      gimlet->input_lang_policy = FOLLOW_CURRENT_LOCALE;
      gtk_widget_set_sensitive (gimlet->input_language_option_menu, FALSE);
    }

  input_language_policy_update (gimlet);
}

static void
on_follow_option_menu_toggled (GtkWidget *button,
			       GimletWindow *gimlet)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    {
      gimlet->input_lang_policy = FOLLOW_OPTION_MENU;
      gtk_widget_set_sensitive (gimlet->input_language_option_menu, TRUE);
    }

  input_language_policy_update (gimlet);
}
#endif

static void
update_color (GimletWindow *gimlet)
{
  gchar *text_color = NULL;
  gchar *background_color = NULL;

  if (gimlet->use_theme_colors)
    {
      text_color = panel_applet_gconf_get_string (gimlet->applet,
						  "text_color", NULL);
      background_color = panel_applet_gconf_get_string (gimlet->applet,
							"background_color",
							NULL);
    }
  gimlet_preference_set_color (gimlet, text_color, background_color);
  g_free (text_color);
  g_free (background_color);
}

static void
use_theme_colors_changed (GConfClient  *client,
			  guint         cnxn_id,
			  GConfEntry   *entry,
			  GimletWindow *gimlet)
{
  gboolean value;

  if (!entry->value || entry->value->type != GCONF_VALUE_BOOL)
    return;

  value = gconf_value_get_bool (entry->value);

  if (gimlet->use_theme_colors != (value != 0))
  {
      gimlet->use_theme_colors = (value != 0);
      update_color (gimlet);
  }
}

#define THEME_FONT_DIR "/desktop/gnome/interface"
#define THEME_FONT_KEY THEME_FONT_DIR "/font_name"

#define IM_KEY_DIR    "/desktop/gnome/input_methods"
#define ENABLE_IM_KEY IM_KEY_DIR "/support"
#define ENABLE_STATUS_KEY IM_KEY_DIR "/status"
#define ENABLE_LOOKUP_KEY IM_KEY_DIR "/lookup"
#define STATUS_PLACEMENT_KEY IM_KEY_DIR "/status_placement"

static const char *on_desktop_panel = "panel";
static const char *attach_to_application_frame = "application";

static const char *enabled = "enabled";
static const char *disabled = "diasbled";

static gchar*
gimlet_get_theme_font (void)
{
  return gconf_client_get_string (default_client, THEME_FONT_KEY, NULL);
}

static void
update_im_settings (GimletWindow *gimlet)
{
  gchar *str;

  str = NULL;
  str = gconf_client_get_string (default_client,
				 ENABLE_IM_KEY, NULL);
  if (str && strcmp (str, disabled) == 0)
    gimlet->im_enabled = FALSE;
  else
    gimlet->im_enabled = TRUE;
  g_free (str);

  str = NULL;
  str = gconf_client_get_string (default_client,
				 ENABLE_STATUS_KEY, NULL);
  if (str && strcmp (str, disabled) == 0)
    gimlet->status_enabled = FALSE;
  else
    gimlet->status_enabled = TRUE;
  g_free (str);

  str = NULL;
  str = gconf_client_get_string (default_client,
				 ENABLE_LOOKUP_KEY, NULL);
  if (str && strcmp (str, disabled) == 0)
    gimlet->lookup_enabled = FALSE;
  else
    gimlet->lookup_enabled = TRUE;
  g_free (str);

  str = NULL;
  str = gconf_client_get_string (default_client,
				 STATUS_PLACEMENT_KEY, NULL);
  if (str && (strcmp (str, attach_to_application_frame) == 0))
    gimlet->status_placement = ATTACH_TO_APP_FRAME;
  else
    gimlet->status_placement = ON_DESKTOP_PANEL;
  g_free (str);

}

static void
update_font (GimletWindow *gimlet)
{
  if (gimlet->use_theme_font)
    {
      gchar *font_name = NULL;
      font_name = gimlet_get_theme_font ();
      gimlet_preference_set_font (gimlet, font_name, font_name);
      g_free (font_name);
    }
  else
    {
      gchar *menu_font_name = NULL;
      gchar *status_font_name = NULL;
      menu_font_name = panel_applet_gconf_get_string (gimlet->applet,
						      "menu_font", NULL);
      status_font_name = panel_applet_gconf_get_string (gimlet->applet,
							"status_font", NULL);
      gimlet_preference_set_font (gimlet, menu_font_name, status_font_name);
      g_free (menu_font_name);
      g_free (status_font_name);
    }
}

static void
use_theme_font_changed (GConfClient  *client,
			guint         cnxn_id,
			GConfEntry   *entry,
			GimletWindow *gimlet)
{
  gboolean value;

  if (!entry->value || entry->value->type != GCONF_VALUE_BOOL)
    return;

  value = gconf_value_get_bool (entry->value);

  if (gimlet->use_theme_font != (value != 0))
  {
      gimlet->use_theme_font = (value != 0);
      update_font (gimlet);
  }
}

static void
im_setting_changed (GConfClient  *client,
		    guint         cnxn_id,
		    GConfEntry   *entry,
		    GimletWindow *gimlet)
{
  gchar *str = NULL;

  if (!entry->value || entry->value->type != GCONF_VALUE_STRING)
    return;

  if (strcmp (entry->key, ENABLE_IM_KEY) == 0)
    {
      str = g_strdup (gconf_value_get_string (entry->value));
      if (strcmp (str, disabled) == 0)
	gimlet->im_enabled = FALSE;
      else
	gimlet->im_enabled = TRUE;
    }
  else if (strcmp (entry->key, ENABLE_STATUS_KEY) == 0)
    {
      str = g_strdup (gconf_value_get_string (entry->value));
      if (strcmp (str, disabled) == 0)
	gimlet->status_enabled = FALSE;
      else
	gimlet->status_enabled = TRUE;
    }
  else if (strcmp (entry->key, ENABLE_LOOKUP_KEY) == 0)
    {
      str = g_strdup (gconf_value_get_string (entry->value));
      if (strcmp (str, disabled) == 0)
	gimlet->lookup_enabled = FALSE;
      else
	gimlet->lookup_enabled = TRUE;
    }
  else if (strcmp (entry->key, STATUS_PLACEMENT_KEY) == 0)
    {
      str = g_strdup (gconf_value_get_string (entry->value));
      if (strcmp (str, attach_to_application_frame) == 0)
	gimlet->status_placement = ATTACH_TO_APP_FRAME;
      else
	gimlet->status_placement = ON_DESKTOP_PANEL;
    }
  g_free (str);

  gimlet_window_show_or_hide (gimlet);
}

static void
default_font_changed (GConfClient  *client,
		      guint         cnxn_id,
		      GConfEntry   *entry,
		      GimletWindow *gimlet)
{
  gchar *menu_font_name = NULL;
  gchar *status_font_name = NULL;

  if (!entry->value || entry->value->type != GCONF_VALUE_STRING)
    return;

  if (strcmp (entry->key, THEME_FONT_KEY) == 0 && gimlet->use_theme_font)
    {
      menu_font_name = g_strdup (gconf_value_get_string (entry->value));
      status_font_name = g_strdup (gconf_value_get_string (entry->value));
    }
  else if (strstr (entry->key, "menu_font") && !gimlet->use_theme_font)
    {
      menu_font_name = g_strdup (gconf_value_get_string (entry->value));
      status_font_name = panel_applet_gconf_get_string (gimlet->applet,
							"status_font",
							NULL);
    }
  else if (strstr (entry->key, "status_font") && !gimlet->use_theme_font)
    {
      status_font_name = g_strdup (gconf_value_get_string (entry->value));
      menu_font_name = panel_applet_gconf_get_string (gimlet->applet,
						      "menu_font",
						      NULL);
    }

  gimlet_preference_set_font (gimlet, menu_font_name, status_font_name);
  g_free (menu_font_name);
  g_free (status_font_name);
}

static void
color_changed (GConfClient  *client,
	       guint         cnxn_id,
	       GConfEntry   *entry,
	       GimletWindow *gimlet)
{
  gchar *text_color = NULL;
  gchar *background_color = NULL;

  if (!entry->value || entry->value->type != GCONF_VALUE_STRING)
    return;

  if (strstr (entry->key, "text_color") && !gimlet->use_theme_colors)
    {
      text_color = g_strdup (gconf_value_get_string (entry->value));
      background_color = panel_applet_gconf_get_string (gimlet->applet,
							"background_color",
							NULL);
    }
  else if (strstr (entry->key, "background_color") && !gimlet->use_theme_colors)
    {
      background_color = g_strdup (gconf_value_get_string (entry->value));
      text_color = panel_applet_gconf_get_string (gimlet->applet,
						  "text_color",
						  NULL);
    }
  gimlet_preference_set_color (gimlet, text_color, background_color);
  g_free (text_color);
  g_free (background_color);
}


static void
show_icon_changed (GConfClient  *client,
		   guint         cnxn_id,
		   GConfEntry   *entry,
		   GimletWindow *gimlet)
{
  gboolean value;

  if (!entry->value || entry->value->type != GCONF_VALUE_BOOL)
    return;

  value = gconf_value_get_bool (entry->value);

  if (gimlet->showicon != (value != 0))
  {
      gimlet->showicon = (value != 0);
      appearance_update (gimlet);
  }
}

static void
show_text_changed (GConfClient  *client,
		   guint         cnxn_id,
		   GConfEntry   *entry,
		   GimletWindow *gimlet)
{
  gboolean value;

  if (!entry->value || entry->value->type != GCONF_VALUE_BOOL)
    return;

  value = gconf_value_get_bool (entry->value);

  if (gimlet->showtext != (value != 0))
  {
      gimlet->showtext = (value != 0);
      appearance_update (gimlet);
  }
}

static void
icon_name_changed (GConfClient  *client,
		   guint         cnxn_id,
		   GConfEntry   *entry,
		   GimletWindow *gimlet)
{
  gchar *value = NULL;

  if (!entry->value || entry->value->type != GCONF_VALUE_STRING)
    return;

  value = gconf_value_get_string (entry->value);

  if (strcmp (gimlet->base_iconname, value) != 0)
  {
      gimlet_change_base_icon (gimlet, value);
      appearance_update (gimlet);
  }
}

static void
text_width_changed (GConfClient  *client,
		    guint         cnxn_id,
		    GConfEntry   *entry,
		    GimletWindow *gimlet)
{
  int value;

  if (!entry->value || entry->value->type != GCONF_VALUE_INT)
    return;

  value = gconf_value_get_int (entry->value);

  if (gimlet->textwidth != value)
  {
      gimlet->textwidth = value;
      appearance_update (gimlet);
  }
}

static gboolean
change_font (const char *font_name, PangoFontDescription **font_desc)
{
  PangoFontDescription *desc;
  PangoFontDescription *tmp;

  desc = pango_font_description_from_string (font_name);

  if (desc)
    {
      /* Merge in case the new string isn't complete enough to
       * load a font
       */
      tmp = pango_font_description_copy (*font_desc);
      pango_font_description_merge (tmp, desc, TRUE);
      pango_font_description_free (desc);
      desc = tmp;
    }

  if (pango_font_description_equal (*font_desc, desc))
    pango_font_description_free (desc);
  else
    {
      pango_font_description_free (*font_desc);
      *font_desc = desc;
      return TRUE;
    }
  return FALSE;
}

static void
gimlet_preference_set_font (GimletWindow *gimlet,
			    const char *menu_font_name,
			    const char *status_font_name)
{
  /* menu font */
  (void)change_font (menu_font_name, &gimlet->menu_font);

  /* status font */
  if (change_font (status_font_name, &gimlet->status_font))
    gimlet_change_status_font (gimlet);
}

static void
gimlet_preference_set_color (GimletWindow *gimlet,
			     const char *text_color,
			     const char *background_color)
{
  if (text_color)
    gdk_color_parse (text_color, &gimlet->foreground);
  if (background_color)
    gdk_color_parse (background_color, &gimlet->background);

  gimlet_change_status_color (gimlet);
}

void
gimlet_preference_init (GimletWindow *gimlet)
{
  gchar *key;
  int int_value;
  gchar *font_name;

  panel_applet_add_preferences (gimlet->applet,
				"/schemas/apps/gimlet/preference",
				NULL);
  if (default_client == NULL)
    {
      default_client = gconf_client_get_default ();
      g_object_ref (G_OBJECT (default_client));
    }

  /* Colors */
  gimlet->use_theme_colors = panel_applet_gconf_get_bool (gimlet->applet,
							  "use_theme_colors", NULL);
  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "use_theme_colors");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)use_theme_colors_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "text_color");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)color_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "background_color");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)color_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  update_color (gimlet);

  /* Font */
  gimlet->use_theme_font = panel_applet_gconf_get_bool (gimlet->applet,
							"use_theme_font",
							NULL);
  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "use_theme_font");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)use_theme_font_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  gconf_client_notify_add (default_client, THEME_FONT_KEY,
			   (GConfClientNotifyFunc)default_font_changed,
			   gimlet, NULL, NULL);

  update_font (gimlet);

  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "menu_font");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)default_font_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "status_font");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)default_font_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  /* applet appearance */
  gimlet->showicon = panel_applet_gconf_get_bool (gimlet->applet,
						  "show_icon", NULL);
  gimlet->showtext = panel_applet_gconf_get_bool (gimlet->applet,
						  "show_text", NULL);
  gimlet->textwidth = panel_applet_gconf_get_int (gimlet->applet,
						  "text_width", NULL);
  gimlet->base_iconname = panel_applet_gconf_get_string (gimlet->applet,
							 "icon_name", NULL);

  if (!gimlet->showtext && !gimlet->showicon)
    gimlet->showtext = TRUE;
  if (gimlet->base_iconname == NULL)
    gimlet->base_iconname = g_strdup ("gnome-settings-im.png");

  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "show_icon");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)show_icon_changed,
			   gimlet, NULL, NULL);
  g_free (key);
  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "show_text");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)show_text_changed,
			   gimlet, NULL, NULL);
  g_free (key);
  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "icon_name");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)icon_name_changed,
			   gimlet, NULL, NULL);
  g_free (key);
  key = panel_applet_gconf_get_full_key (PANEL_APPLET (gimlet->applet),
					 "text_width");
  gconf_client_notify_add (default_client, key,
			   (GConfClientNotifyFunc)text_width_changed,
			   gimlet, NULL, NULL);
  g_free (key);

  /* Input Language */
  int_value = panel_applet_gconf_get_int (gimlet->applet,
					  "input_language_policy", NULL);
  gimlet->input_lang_policy = int_value;

  gimlet->conversion_on_start = panel_applet_gconf_get_bool (gimlet->applet,
							     "conversion_on_start", NULL);

  /* input method settings */
  update_im_settings (gimlet);
  gconf_client_notify_add (default_client, ENABLE_IM_KEY,
			   (GConfClientNotifyFunc)im_setting_changed,
			   gimlet, NULL, NULL);
  gconf_client_notify_add (default_client, ENABLE_STATUS_KEY,
			   (GConfClientNotifyFunc)im_setting_changed,
			   gimlet, NULL, NULL);
  gconf_client_notify_add (default_client, STATUS_PLACEMENT_KEY,
			   (GConfClientNotifyFunc)im_setting_changed,
			   gimlet, NULL, NULL);
}

static void
response_callback (GtkWidget *window,
                   int        id,
                   void      *data)
{
  if (id == GTK_RESPONSE_HELP)
    gimlet_util_show_help ("gnome-im-switcher-preferences", GTK_WINDOW (window));
  else
    {
      gtk_widget_destroy (GTK_WIDGET (window));
    }
}

static void
font_checkbutton_toggled (GtkToggleButton *button,
			  GtkWidget *dialog)
{
  GtkWidget *font_table = g_object_get_data (G_OBJECT (dialog),
					     "font-picker-table");
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");

  if (gtk_toggle_button_get_active (button))
    {
      gtk_widget_set_sensitive (font_table, FALSE);
      gimlet->use_theme_font = TRUE;
    }
  else
    {
      gtk_widget_set_sensitive (font_table, TRUE);
      gimlet->use_theme_font = FALSE;
    }

  panel_applet_gconf_set_bool (PANEL_APPLET (gimlet->applet),
			       "use_theme_font",
			       gimlet->use_theme_font,
			       NULL);
  return;
}

static void
colors_checkbutton_toggled (GtkToggleButton *button,
			    GtkWidget *dialog)
{
  GtkWidget *color_table = g_object_get_data (G_OBJECT (dialog),
					      "color-picker-table");
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");

  if (gtk_toggle_button_get_active (button))
    {
      gtk_widget_set_sensitive (color_table, FALSE);
      gimlet->use_theme_colors = TRUE;
    }
  else
    {
      gtk_widget_set_sensitive (color_table, TRUE);
      gimlet->use_theme_colors = FALSE;
    }

  panel_applet_gconf_set_bool (PANEL_APPLET (gimlet->applet),
			       "use_theme_colors",
			       gimlet->use_theme_colors,
			       NULL);
  return;
}

static void
conversion_on_start_checkbutton_toggled (GtkToggleButton *button,
					 GtkWidget *dialog)
{
  GimletWindow *gimlet = g_object_get_data (G_OBJECT (dialog), "gimlet");

  if (gtk_toggle_button_get_active (button))
    gimlet->conversion_on_start = TRUE;
  else
    gimlet->conversion_on_start = FALSE;

  panel_applet_gconf_set_bool (PANEL_APPLET (gimlet->applet),
			       "conversion_on_start",
			       gimlet->conversion_on_start,
			       NULL);
  return;
}

static gchar* 
gdk_color_to_string (GdkColor color)
{
  return g_strdup_printf ("#%04x%04x%04x",
			  color.red, 
			  color.green,
			  color.blue);
}

static void
menu_font_picker_font_set (GnomeFontPicker        *gfp,
			   const gchar            *font_name,
			   GimletWindow    	     *gimlet)
{
  g_return_if_fail (font_name != NULL);

  panel_applet_gconf_set_string (PANEL_APPLET (gimlet->applet),
				"menu_font",
				 font_name,
				 NULL);
}

static void
status_font_picker_font_set (GnomeFontPicker        *gfp,
			     const gchar            *font_name,
			     GimletWindow    	     *gimlet)
{
  g_return_if_fail (font_name != NULL);

  panel_applet_gconf_set_string (PANEL_APPLET (gimlet->applet),
				"status_font",
				 font_name,
				 NULL);
}

static void 
text_color_set (GnomeColorPicker      *cp, 
		guint                   r, 
		guint                   g,
		guint                   b,
		guint                   a,
		GimletWindow		*gimlet)
{
  GdkColor color;
  gchar *str_color = NULL;

  color.red = r;
  color.green = g;
  color.blue = b;

  str_color = gdk_color_to_string (color);
  g_return_if_fail (str_color != NULL);

  panel_applet_gconf_set_string (PANEL_APPLET (gimlet->applet),
				"text_color",
				 str_color,
				 NULL);
  g_free (str_color);
}

static void 
background_color_set (GnomeColorPicker      *cp, 
		      guint                   r, 
		      guint                   g,
		      guint                   b,
		      guint                   a,
		      GimletWindow	*gimlet)
{
  GdkColor color;
  gchar *str_color = NULL;

  color.red = r;
  color.green = g;
  color.blue = b;

  str_color = gdk_color_to_string (color);
  g_return_if_fail (str_color != NULL);

  panel_applet_gconf_set_string (PANEL_APPLET (gimlet->applet),
				"background_color",
				 str_color,
				 NULL);
  g_free (str_color);
}

#ifdef MEANINGLESS_OPTION_PERHAPS
static void
activate_cb (GtkWidget *menuitem, GimletWindow *gimlet)
{
  if (GTK_CHECK_MENU_ITEM (menuitem)->active)
    {
      gchar *iiim_lang;
      gchar *name;
      gchar *conversion_mode;
      GimletLanguage *language;

      iiim_lang = g_object_get_data (G_OBJECT (menuitem), "iiim-lang-name");

      if (gimlet->iiim_lang_in_option_menu)
	g_free (gimlet->iiim_lang_in_option_menu);
      gimlet->iiim_lang_in_option_menu = g_strdup (iiim_lang);
    }
}

static void
add_languages_in_option_menu (GimletWindow *gimlet)
{
  GtkWidget *menu;
  GtkWidget *menuitem;
  GSList *group = NULL;
  GSList *active_languages;
  GSList *tmp;

  menu = gtk_menu_new ();
  group = NULL;

  if (gimlet->quick_access_menu == NULL)
    gimlet_quick_access_menu_init (gimlet);

  active_languages = gimlet->quick_access_menu->active_languages;

  tmp = active_languages;
  while (tmp != NULL)
    {
      char *name;
      char *iiim_lang_name;
      name = gimlet_language_get_name (tmp->data);
      iiim_lang_name = gimlet_language_get_iiim_lang_name (tmp->data);

      menuitem = gtk_radio_menu_item_new_with_label (group, name);

      if ((gimlet->iiim_lang_in_option_menu == NULL && group == NULL) ||
          (gimlet->iiim_lang_in_option_menu &&
           strcmp (iiim_lang_name, gimlet->iiim_lang_in_option_menu) == 0))
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem),
					TRUE);
       
      group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (menuitem));

      g_object_set_data (G_OBJECT (menuitem), "iiim-lang-name",
			 (char *)iiim_lang_name);
      g_object_set_data (G_OBJECT (menuitem), "iiim-display-name",
			 (char *)name);
      g_signal_connect (menuitem, "activate",
			G_CALLBACK (activate_cb), gimlet);

      gtk_widget_show (menuitem);
      gtk_menu_shell_append (GTK_MENU_SHELL(menu), menuitem);

      tmp = tmp->next;
    }

  gtk_option_menu_set_menu (GTK_OPTION_MENU (gimlet->input_language_option_menu), menu);

}
#endif

void
preference_get_colors (GimletWindow *gimlet,
		       GdkColor         *foreground,
		       GdkColor         *background)
{
  gchar *text_color = NULL;
  gchar *background_color = NULL;
  text_color = panel_applet_gconf_get_string (gimlet->applet,
					      "text_color", NULL);
  background_color = panel_applet_gconf_get_string (gimlet->applet,
						    "background_color", NULL);
  gdk_color_parse (text_color, foreground);
  gdk_color_parse (background_color, background);

  g_free (text_color);
  g_free (background_color);
}

static void
colorpicker_set_if_changed (GtkWidget      *colorpicker,
                            const GdkColor *color)
{
  guint16 r, g, b;

  gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (colorpicker),
                              &r, &g, &b, NULL);

  if (r != color->red ||
      g != color->green ||
      b != color->blue)
    gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (colorpicker),
                                color->red,
                                color->green,
                                color->blue,
                                0xffff);
}

/* public */
GtkWidget*
gimlet_preference_dialog_new (GtkWindow *transient_parent,
			      GimletWindow *gimlet)
{
  GladeXML *xml;
  GtkWidget *w;
  GtkWidget *dialog;
  GtkTooltips *tooltips;
  gchar *default_font;
  GdkColor fg, bg;

  tooltips = gtk_tooltips_new();

  xml = gimlet_util_load_glade_file (GIMLET_GLADE_FILE,
				     "preference-dialog",
				     transient_parent);
  if (xml == NULL)
    return NULL;

  /* The dialog itself */
  dialog = glade_xml_get_widget (xml, "preference-dialog");
  g_object_set_data (G_OBJECT (dialog), "gimlet", gimlet);

  gimlet_util_set_unique_role (GTK_WINDOW (dialog), "gimlet-preferences");

  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK (response_callback),
                    NULL);

  /* font */
  w = glade_xml_get_widget (xml, "fonts_table");
  g_object_set_data (G_OBJECT (dialog),
                     "font-picker-table",
                     w);
  w = glade_xml_get_widget (xml, "default_font_checkbutton");
  g_object_set_data (G_OBJECT (dialog),
                     "default-font-checkbutton",
                     w);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				gimlet->use_theme_font);

  font_checkbutton_toggled (GTK_TOGGLE_BUTTON (w), dialog);
	
  g_signal_connect (G_OBJECT (w), "toggled", 
		    G_CALLBACK (font_checkbutton_toggled), 
		    dialog);

  w = glade_xml_get_widget (xml, "menu_fontpicker");
  g_signal_connect (G_OBJECT (w), "font_set",
		    G_CALLBACK (menu_font_picker_font_set),
		    gimlet);

  default_font = panel_applet_gconf_get_string (gimlet->applet,
						"menu_font", NULL);
  gnome_font_picker_set_font_name (GNOME_FONT_PICKER (w),
				   default_font);
  g_free (default_font);

  w = glade_xml_get_widget (xml, "status_fontpicker");
  g_signal_connect (G_OBJECT (w), "font_set",
		    G_CALLBACK (status_font_picker_font_set),
		    gimlet);

  default_font = panel_applet_gconf_get_string (gimlet->applet,
						"status_font", NULL);
  gnome_font_picker_set_font_name (GNOME_FONT_PICKER (w),
				   default_font);
  g_free (default_font);

  /* colors */
  w = glade_xml_get_widget (xml, "colors_table");
  g_object_set_data (G_OBJECT (dialog),
                     "color-picker-table",
                     w);
  w = glade_xml_get_widget (xml, "default_colors_checkbutton");
  g_object_set_data (G_OBJECT (dialog),
                     "default-colors-checkbutton",
                     w);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				gimlet->use_theme_colors);
    
  colors_checkbutton_toggled (GTK_TOGGLE_BUTTON (w), dialog);
  g_signal_connect (G_OBJECT (w), "toggled", 
		    G_CALLBACK (colors_checkbutton_toggled), 
		    dialog);

  preference_get_colors (gimlet, &fg, &bg);

  w = glade_xml_get_widget (xml, "text_colorpicker");
  g_signal_connect (G_OBJECT (w), "color_set",
		    G_CALLBACK (text_color_set),
		    gimlet);
  colorpicker_set_if_changed (w, &fg);

  w = glade_xml_get_widget (xml, "background_colorpicker");
  g_signal_connect (G_OBJECT (w), "color_set",
		    G_CALLBACK (background_color_set),
		    gimlet);
  colorpicker_set_if_changed (w, &bg);

  /* status apperance */
  w = glade_xml_get_widget (xml, "icon_picker");
  g_signal_connect (G_OBJECT (w), "changed",
		    G_CALLBACK (icon_picker_changed), gimlet);
  g_object_set_data (G_OBJECT (dialog), "icon-picker", w);
  gnome_icon_entry_set_filename (GNOME_ICON_ENTRY (w), gimlet->base_iconname);
  
  if (!gimlet->showicon)
    gtk_widget_set_sensitive (w, FALSE);

  w = glade_xml_get_widget (xml, "text_only");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_text_only_toggled), dialog);
  if (gimlet->showtext && !gimlet->showicon)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "text_on_icon");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_text_and_icon_toggled), dialog);
  if (gimlet->showtext && gimlet->showicon)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "text_width_spinbutton");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), gimlet->textwidth);
  g_signal_connect (G_OBJECT (w), "value_changed",
		    G_CALLBACK (text_width_size_changed), dialog);

  /* Input Language */

#ifdef MEANINGLESS_OPTION_PERHAPS
  w = glade_xml_get_widget (xml, "default-input-language-option");
  gimlet->input_language_option_menu = w;
  gtk_widget_set_sensitive (gimlet->input_language_option_menu, FALSE);

  add_languages_in_option_menu (gimlet);

  w = glade_xml_get_widget (xml, "follow-current-locale");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_follow_current_locale_toggled), gimlet);
  if (gimlet->input_lang_policy == FOLLOW_CURRENT_LOCALE)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "follow-option-menu");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_follow_option_menu_toggled), gimlet);
  if (gimlet->input_lang_policy == FOLLOW_OPTION_MENU)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

#endif

  w = glade_xml_get_widget (xml, "follow-application");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_follow_application_toggled), gimlet);
  if (gimlet->input_lang_policy == FOLLOW_APPLICATION)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "follow-quick-access-menu");
  g_signal_connect (G_OBJECT (w), "toggled",
		    G_CALLBACK (on_follow_quick_access_menu_toggled), gimlet);
  if (gimlet->input_lang_policy == FOLLOW_QUICK_ACCESS_MENU)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "conversion_on_start_checkbutton");
  g_object_set_data (G_OBJECT (dialog),
                     "conversion-on-start-checkbutton",
                     w);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				gimlet->conversion_on_start);

  conversion_on_start_checkbutton_toggled (GTK_TOGGLE_BUTTON (w), dialog);
	
  g_signal_connect (G_OBJECT (w), "toggled", 
		    G_CALLBACK (conversion_on_start_checkbutton_toggled), 
		    dialog);
  return dialog;
}
