/***************************************************************************
                        mesh.cpp  -  description                              
                           -------------------                                         
  begin                : Wed Dec 15 1999                                           
  copyright            : (C) 1999 by Jon Anderson                         
  email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "mesh.h"
#include "face.h"
#include "edge.h"
#include "vertex.h"
#include "uvcoord.h"
#include "material.h"
#include "texture.h"
#include "texturematerial.h"
#include <Util/util.h>
#include <objectdb.h>

#include <map>

typedef struct
{
   int f;
   int v;
   float tu;
   float tv;
}

Face_Point;

typedef struct
{
   int e;
   int v;
   float tu;
   float tv;
   Face_Point f1;
   Face_Point f2;
}

Edge_Point;



bool Mesh::draw_edges = true;
bool Mesh::cull_faces = true;
int Mesh::TYPE = Typed::getUID();

Mesh::Mesh( Entity *p ) : Object( p )
{
   faces.reserve( 250 );
   addType( TYPE );
}

Mesh::Mesh( Vector4 &p, Entity *parent ) : Object( p, parent )
{
   faces.reserve( 250 );
   addType( TYPE );
}

Mesh::Mesh( float x, float y, float z, Entity *parent ) : Object( x, y, z, parent )
{
   faces.reserve( 250 );
   addType( TYPE );
}

Mesh::~Mesh()
{}

void Mesh::drawFaces( )
{
   vector<SubObject *>::iterator it = faces.begin();

   while ( it != faces.end() )
   {
      ( *it ) -> draw();
      ++it;
   }
}

int Mesh::draw( int d_options = 0 )
{

   bool cull_enabled = glIsEnabled( GL_CULL_FACE );

   if ( cull_faces == true )
      glEnable( GL_CULL_FACE );
   else
      glDisable( GL_CULL_FACE );


   if ( draw_edges && ! SelectMode::is( Edge::TYPE ) )
   {

      int * polymode = new int;
      glGetIntegerv( GL_POLYGON_MODE, polymode );

      if ( *polymode == GL_FILL )
      {

         //draw filled and lighted;
         glEnable( GL_POLYGON_OFFSET_FILL );
         glPolygonOffset( 1.0, 1.0 );

         drawFaces();

         glDisable( GL_POLYGON_OFFSET_FILL );

         glPushAttrib( GL_LIGHTING_BIT | GL_TEXTURE_BIT | GL_COLOR_BUFFER_BIT );
         glDisable( GL_LIGHTING );
         glDisable( GL_TEXTURE_2D );
         glColor4f( 0, 0, 1, 1 );
         glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

         vector<SubObject *>::iterator it = faces.begin();

         while ( it != faces.end() )
         {
            // ( *it ) -> draw();
            // ++it;

            vector<int> *vlist = ( *it ) -> getVerts();
            vector<int> *uvlist = ( *it ) -> getUVs();
            GLenum glmode = GL_POLYGON;

            if ( vlist -> size() == 3 )
               glmode = GL_TRIANGLES;

            if ( vlist -> size() == 4 )
               glmode = GL_QUADS;

            float v[ 4 ];

            Vector4 uv;

            glBegin( glmode );

            vector<int>::iterator v_it = vlist -> begin();

            vector<int>::iterator uv_it = uvlist -> begin();

            while ( v_it != vlist -> end() )
            {

               getUVCoord( *uv_it ) ->getPosition( &uv );
               glTexCoord2fv( uv.v( v ) );
               getVertex( *v_it ) -> getPosition( &uv );
               glVertex3fv( uv.v( v ) ) ;
               ++v_it;
               ++uv_it;
            }

            glEnd();

            ++it;
         }

         glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
         glPopAttrib( );

      }

      else
      {
         drawFaces();
      }

      delete polymode;
   }

   else
   {
      if ( SelectMode::is( Edge::TYPE ) )
      {     //need to draw verts too?
         glEnable( GL_POLYGON_OFFSET_FILL );
         glPolygonOffset( 1.0, 1.0 );
         drawFaces();
         glDisable( GL_POLYGON_OFFSET_FILL );
         glPushAttrib( GL_LIGHTING_BIT | GL_TEXTURE_BIT | GL_COLOR_BUFFER_BIT );
         glDisable( GL_LIGHTING );
         glDisable( GL_TEXTURE_2D );

         glLineWidth( 2 );
         drawEdges( );
         glLineWidth( 1 );
         glPopAttrib();

      }

      else
      {
      	/* no special mode, so just draw as fast as we can.
      	 */
      	Vector4 pv;
      	Vector4 puv;
      	Vertex *vt;
      	UVCoord *uvt;
      	float t[4];
      	
      	vector<SubObject *>::iterator f_it = faces.begin();
      	vector<int> * vlist, *uvlist;
      	vector<int>::iterator v_it, uv_it;
      	while( f_it != faces.end() )
      	{
      		Face *f = (Face *)*f_it;
      		vlist = f -> getVerts();
      		uvlist = f -> getUVs();
      		
					v_it = vlist -> begin();
					uv_it = uvlist -> begin();
					glBegin ( GL_POLYGON );
					while( v_it != vlist -> end() && uv_it != uvlist -> end() )
					{
						vt = (Vertex *)verts[ *v_it ];		
						uvt = (UVCoord *)uvs[ *uv_it ];
						
						vt -> getPosition( &pv );
						uvt -> getPosition( &puv );
						
						glTexCoord2fv( puv.v( t ) );
						//glVertex3fv( puv.v( t ) );
						vt -> draw( VERTEX_NORMAL | VERTEX_COLOR | VERTEX_COORD );
						
						++v_it;
						++uv_it;
					}
					glEnd();
					
         //drawFaces();
         ++f_it;
				}
      }

   }

   if ( SelectMode::is( Vertex::TYPE ) )
   {     //need to draw verts too?

      drawVerts();
   }

   //revert to previous cull state.
   if ( cull_enabled )
      glEnable( GL_CULL_FACE );
   else
      glDisable( GL_CULL_FACE );

   return 0;
}


/**Merge a list of meshes.  The caller should delete the given meshes.
  *
  */

void Mesh::merge( vector<Object *> &objs )
{
   for ( int i = 0; i < ( int ) objs.size(); i++ )
   {
      int st_verts = verts.size();
      Mesh *m = ( Mesh * ) objs[ i ];

      //copy the verts over.
      int n = m->numVerts();

      Matrix44 mt;
      getCompleteMatrix( &mt );
      mt.Invert();

      for ( int j = 0; j < n; j++ )
      {
         Vertex *v = m->getVertex( j );
         Vector4 p;
         v->getTransformedPosition( &p );
				
         p = mt * p;

         createVertex( p );
      }

      //copy the faces over (this will create edges and uv coords)
      n = m->numFaces();

      for ( int j = 0; j < n; j++ )
      {
         Face *f = m->getFace( j );

         vector<int> f_vlist;
         vector<int> f_uvlist;

         f_vlist = * f->getVerts();
         f_uvlist = * f->getUVs();

         for ( int k = 0; k < ( int ) f_vlist.size(); k++ )
         {
            f_vlist[ k ] += st_verts;
         }

         Face * nf = createFace( f_vlist );

         //copy UV positions over.
         for ( int k = 0; k < ( int ) f_uvlist.size(); k++ )
         {
            Vector4 n_uv;
            m->getUVCoord( f_uvlist[ k ] ) ->getPosition( &n_uv );
            nf->setUVCoord( k, n_uv );
         }

      }

   }

   normalize();

}

/** collect all connected faces to each vertex, and
  * delete them.
  */
void Mesh::deleteVerts( vector<int> & p_verts )
{

   vector<int> flist;
   flist.reserve( p_verts.size() );

   for ( int i = 0; i < ( int ) p_verts.size(); i++ )
   {
      vector<int> *vflist = getVertex( p_verts[ i ] ) -> getFaces();

      for ( int j = 0; j < ( int ) vflist->size(); j++ )
      {
         flist.push_back( ( *vflist ) [ j ] );
      }

   }

   if ( flist.size() == 0 )
   {
      //no faces, so we can just delete the verts directly.
      for ( int i = 0; i < ( int ) p_verts.size(); i++ )
      {
         Vertex *v = getVertex( p_verts[ i ] );
         v -> detach( true );
      }

      cleanSubObjects();
      return ;
   }


   std::sort( flist.begin(), flist.end() );

   //remove any duplicates in the defunct queue..
   vector<int>::iterator dups = std::unique( flist.begin(), flist.end() );
   flist.erase( dups, flist.end() );

   deleteFaces( flist );

}

void Mesh::deleteEdges( vector<int> & p_edges )
{
   vector<int> flist;
   flist.reserve( p_edges.size() );

   for ( int i = 0; i < ( int ) p_edges.size(); i++ )
   {
      vector<int> *eflist = getEdge( p_edges[ i ] ) -> getFaces();

      for ( int j = 0; j < ( int ) eflist->size(); j++ )
      {
         flist.push_back( ( *eflist ) [ j ] );
      }

   }

   std::sort( flist.begin(), flist.end() );

   //remove any duplicates in the defunct queue..
   vector<int>::iterator dups = std::unique( flist.begin(), flist.end() );
   flist.erase( dups, flist.end() );

   deleteFaces( flist );

}

void Mesh::deleteFaces( vector<int> & p_faces )
{
   vector<int> f_verts;
   vector<int> f_edges;


   f_verts.reserve( p_faces.size() * 4 );
   f_edges.reserve( p_faces.size() * 4 );

   //loop through each face and detach it.
   for ( int i = 0; i < ( int ) p_faces.size(); i++ )
   {
      Face * f = getFace( p_faces[ i ] );

      //copy the edges and verts into a temporary storage.
      std::copy( f -> getVerts() -> begin(), f -> getVerts() -> end(), std::back_inserter( f_verts ) );
      std::copy( f -> getEdges() -> begin(), f -> getEdges() -> end(), std::back_inserter( f_edges ) );

      //detach and mark for removal.
      f -> detach( true );
   }

   cerr << "Marked faces for deletion." << endl;

   //for each vert in the face list, check if it no longer
   //has any faces.  If so, detach and remove it.
   for ( int i = 0; i < ( int ) f_verts.size(); i++ )
   {
      Vertex * v = getVertex( f_verts[ i ] );

      if ( v -> getFaces() -> size() == 0 )
         v -> detach( true );

   }

   cerr << "Marked Verts for deletion." << endl;

   //for each edge in the face list, check if it no longer
   //has any faces.  If so, detach and remove it.
   for ( int i = 0; i < ( int ) f_edges.size(); i++ )
   {
      Edge * e = getEdge( f_edges[ i ] );

      if ( e -> getFaces() -> size() == 0 )
         e -> detach( true );
   }

   cerr << "Marked edges for deletion." << endl;

   cleanSubObjects();
   normalize();
   // Util::dump( this );
}


/**Subdivides all faces in the mesh */
//KEEP TRACK OF UVS HERE LIKE IN SMOOTHING

void Mesh::subdivide()
{
   Mesh * m = new Mesh();
   m -> setPosition( pos );

   //copy verts over
   copyVertsPosTo( m );

   vector<Face_Point> fpts;
   vector<Edge_Point> epts;
   vector< vector<int> * > opts;

   int num_verts = verts.size();
   int num_faces = faces.size();
   int num_edges = edges.size();


   fpts.reserve( num_faces );
   epts.reserve( num_edges );
   opts.reserve( num_verts );


   //create the new face points.
   for ( int i = 0; i < ( int ) faces.size(); i++ )
   {
      Face_Point fp;
      Face *f = getFace( i );
      Vector4 c;
      c = f -> getCenter();
      Vertex *v = m -> createVertex( c );
      int nv = v -> getParentIndex();

      c = f -> getUVCenter();

      fp.f = i;
      fp.v = nv;
      fp.tu = c.x;
      fp.tv = c.y;

      fpts.push_back( fp );
   }

   //create the new edge points.
   for ( int i = 0; i < ( int ) edges.size(); i++ )
   {
      Edge_Point ep;
      Vector4 c;

      Edge *e = getEdge( i );
      c = e -> getCenter();

      Vertex *v = m -> createVertex( c );
      int nv = v -> getParentIndex();


      ep.e = i;
      ep.v = nv;

      epts.push_back( ep );
   }

   //capture the facepoints surrounding each vertex...
   for ( int i = 0; i < num_verts; i++ )
   {
      vector<int> *facepts = new vector<int>;
      vector<int> *vfaces = getVertex( i ) -> getFaces();
      *facepts = *vfaces;
      opts.push_back( facepts );
   }


   //subdivide all the faces...
   for ( int i = 0; i < num_faces; i++ )
   {
      Face *f = getFace( i );
      vector<int> *fverts = f->getVerts();

      for ( int j = 0; j < ( int ) fverts->size(); j++ )
      {
         int e1 = f->getEdgeBefore( ( *fverts ) [ j ] );
         int e2 = f->getEdgeAfter( ( *fverts ) [ j ] );

         //create a new face...
         m -> createFace( epts[ e1 ].v, ( *fverts ) [ j ], epts[ e2 ].v, fpts[ i ].v );

      }


   }

   //delete the face points vectors
   for ( int i = 0; i < ( int ) opts.size(); i++ )
   {
      opts[ i ] -> clear();

      delete opts[ i ];

   }

   opts.clear();

   m -> copyVertsTo( this );
   m -> copyEdgesTo( this );
   m -> copyFacesTo( this );
   m -> copyUVsTo( this );
   normalize();

   delete m;
   //  m -> move( 2, 2, 2 );

   //ObjectDB::getInstance() -> add( m );

}

void Mesh::copyFrom( Entity *rhs )
{
   *this = * static_cast<Mesh *>( rhs );
}


Mesh & Mesh::operator=( Mesh &rhs )
{
   //base copy.
   Object::operator=( rhs );

   return *this;
}

/**Turns the mesh inside out.  I.E. it reverses the winding of all the
  *faces in the mesh.
  */
void Mesh::reverse()
{

   vector<SubObject *>::iterator it = faces.begin();

   while ( it != faces.end() )
   {
      Face * f = static_cast<Face *>( *it );
      f -> reverse();
      ++it;
   }

   normalize();
}

Entity * Mesh::clone()
{
   Mesh * m = new Mesh();
   *m = *this;
   return m;
}

/**Converts all faces in the mesh into triangles.
  */
void Mesh::triangulate()
{
   FaceList _faces;

   int n = numFaces();

   for ( int i = 0; i < n; i++ )
   {
      Face *f = getFace( i );
      f->triangulate();
   }

   cleanSubObjects();
   normalize();

}

/**Converts all faces in the mesh into triangles.
  */
void Mesh::triangulateFaces( vector<int> &flist )
{
   FaceList _faces;

   int n = ( int ) flist.size();

   for ( int i = 0; i < n; i++ )
   {
      Face *f = getFace( flist[ i ] );
      f->triangulate();
   }

   cleanSubObjects();
   normalize();

}



int Mesh::findFace( Face *f )
{
   for ( int i = 0; i < ( int ) faces.size(); i++ )
   {
      if ( faces[ i ] == f )
         return i;
   }

   return -1;
}

/**Slices a mesh based on the verts passed in.  All faces whose verts are completely
  *within the list are recreated in a new mesh.
  */ 
/*
Object * Mesh::slice(vector<Vertex *> *vlist)
{
//find a list of faces whose verts are competely within the list.
Mesh * m = new Mesh(pos.x, pos.y, pos.z);
FaceList full_list;
FaceList partial_list;
VertexList edge_verts;
 
typedef map<int, int, less<int> > IntMap;
typedef IntMap::value_type IntValue;
 
IntMap vmap;
 
full_list.reserve(faces.size());
edge_verts.reserve(vlist->size());
 
for(int i=0; i<numFaces(); i++){
Face *f = getFace(i);
vector<int> *fv = f->getVerts();
bool completely_contained = true;
bool partial_contained = false;
for(int j=0; j<(int)fv->size(); j++){
 Vertex *v = getVertex((*fv)[j]);
 if(find(vlist->begin(), vlist->end(), v) == vlist->end()){
  completely_contained = false;
 }
 else {
  partial_contained = true;
 }
}
if(completely_contained) {
 full_list.push_back(f);
}
if(partial_contained && !completely_contained){
 partial_list.push_back(f);
}
  
}
//copy verts into new list;
for(int i = 0; i<(int)vlist->size(); i++){
Vertex *v = new Vertex(m);
*v = *((Vertex *)(*vlist)[i]);
vmap.insert(IntValue((*vlist)[i]->getParentIndex(), v->getParentIndex()));
}
 
//copy the faces into the new list;
for(int i = 0; i<(int)full_list.size(); i++){
Face *f = (Face *)full_list[i];
Face *nf = createFace( m );
*nf = *f;
 
}
 
//for all the partial faces, find the edge verts (those verts in the slice list
for(int i=0; i<(int)partial_list.size(); i++){
vector<int> *fv = partial_list[i]->getVerts();
for(int j=0; j<(int)fv->size(); j++){
 if(find(vlist->begin(), vlist->end(), verts[(*fv)[j]]) != vlist->end())
  edge_verts.push_back(getVertex((*fv)[j])); 
}                                         
}
 
sort(edge_verts.begin(), edge_verts.end());
 
//remove duplicates from the edge verts list
edge_verts.erase(unique(edge_verts.begin(), edge_verts.end()), edge_verts.end());
 
vector<Vertex *> remove_list;
remove_list.resize(vlist->size()-edge_verts.size());
 
sort(vlist->begin(), vlist->end());
 
//find the verts that should be removed.
set_difference(vlist->begin(), vlist->end(),
      edge_verts.begin(), edge_verts.end(),
      remove_list.begin());
      
 
//remove all fully contained faces
for(int i=0; i<(int)full_list.size(); i++){
Face *f = (Face *)full_list[i];
vector<int> *fv = f->getVerts();
for(int j=0; j<(int)fv->size(); j++){
 verts[(*fv)[j]]->removeFace(f->getParentIndex());
}
removeFace(f->getParentIndex());
}
//and all verts except the edge verts.
for(int i=0; i<(int)remove_list.size(); i++){
removeVert(remove_list[i]->getParentIndex()); 
}
 
m->center();
m->normalize(); 
 
Vector4 p1 = m->getPosition();
Vector4 p2 = getPosition();
 
Vector4 p3;
 
p3 = p2-p1;
p3.normalize();
m->move(p3.x, p3.y, p3.z);
 
return m;
 
}
*/

/**Connects a series of Edges/Vertices together.  It skips over any sections that don't share a face.
  * In the case of edges, all the edges are divided, and the midpoints are used for the Vertex list.
  */
void Mesh::connect( int type, vector<int> &list, bool loop )
{
   //if they are edges, divide them all up.
   vector<int> vlist;
   vlist.reserve( list.size() );

   if ( type == Edge::TYPE )
   {
      for ( int i = 0; i < ( int ) list.size(); i++ )
      {
         int v = getEdge( list[ i ] ) ->divideEdge();
         vlist.push_back( v );
      }
   }

   else
   {
      vlist = list;
   }

   //connect the verts as best we can.
   for ( int i = 0; i < ( int ) vlist.size() - 1; i++ )
   {
      Vertex *v1, *v2;

      v1 = getVertex( vlist[ i ] );
      v2 = getVertex( vlist[ i + 1 ] );

      vector<int> flist;
      flist = v1 -> getCommonFaces( v2 -> getParentIndex() );
      //divide the faces;
      //If they aren't diviable, it's just ignored.
      for ( int j = 0; j < ( int ) flist.size(); j++ )
      {
         getFace( flist[ j ] ) ->splitByVerts( v1 -> getParentIndex(), v2 -> getParentIndex() );

      }
   }


   //clean up any detached subobjects;
   cleanSubObjects();

   if ( loop && ( vlist.size() > 2 ) )
   {
      //attempt to connect the first and the last vertices
      vector<int> loop_verts;
      loop_verts.push_back( vlist[ 0 ] );
      loop_verts.push_back( vlist[ vlist.size() - 1 ] );

      connect( Vertex::TYPE, loop_verts, false );
   }
}

void Mesh::extrudeVerts( vector<int> &vlist, Vector4 &amt )
{
   amt.assign( .5, .5, .5, 1 );

   //for each vertex.
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      //connect all of it's edges.
      vector<int> connect_list;
      vector<int> ordered_edges;

      vector<int> elist;
      elist = * getVertex( vlist[ i ] ) -> getEdges();

      ordered_edges = getFaceOrderedEdges( elist );

      connect_list.reserve( elist.size() + 2 );

      for ( int j = 0; j < ( int ) ordered_edges.size(); j++ )
      {
         int v = getEdge( ( ordered_edges ) [ j ] ) ->divideEdge();
         connect_list.push_back( v );
      }

      connect_list.push_back( connect_list[ 0 ] );



      //connect the verts as best we can.
      for ( int j = 0; j < ( int ) connect_list.size() - 1; j++ )
      {
         Vertex *v1, *v2;

         v1 = getVertex( connect_list[ j ] );
         v2 = getVertex( connect_list[ j + 1 ] );

         vector<int> flist;
         flist = v1 -> getCommonFaces( v2 -> getParentIndex() );
         //divide the faces;
         //If they aren't diviable, it's just ignored.
         for ( int h = 0; h < ( int ) flist.size(); h++ )
            getFace( flist[ h ] ) ->splitByVerts( v1 -> getParentIndex(), v2 -> getParentIndex() );

      }

      //extrude the vert...
      Vector4 normal;

      normal = getVertex( vlist[ i ] ) ->getNormal();

      getVertex( vlist[ i ] ) ->move( normal.x * amt.x, normal.y * amt.y, normal.z * amt.z );
   }

   cleanSubObjects();

   normalize();

}

void Mesh::extrudeEdges( vector<int> &elist, Vector4 &amt )
{

   amt.assign( .5, .5, .5, .5 );

   //for each edge.
   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      //connect all of it's vert's edges.
      vector<int> elist1;
      vector<int> elist2;
      vector<int> evlist;
      vector<int> edge_list;

      vector<int> connect_list;
      vector<int> ordered_edges;

      //get the edges attached to this edge's verts.
      evlist = * getEdge(   elist[ i ]  ) -> getVerts();
      elist1 = * getVertex( evlist[ 0 ] ) -> getEdges();
      elist2 = * getVertex( evlist[ 1 ] ) -> getEdges();


      //copy into a single array.
      //elist1 -> reserve( elist1->size() + elist2->size() );

      for ( int t = 0; t < ( int ) elist2.size(); t++ )
      {
         elist1.push_back( elist2[ t ] );
      }

      int e = elist[ i ];

      //remove this edge from the list...
      vector<int>::iterator d = find( elist1.begin(), elist1.end(), e );

      while ( d != elist1.end() )
      {
         elist1.erase( d );
         d = find( elist1.begin(), elist1.end(), e );
      }

      //order the edges...
      ordered_edges = getFaceOrderedEdges( elist1 );

      connect_list.reserve( elist1.size() + 2 );

      //divide all the edges.
      for ( int j = 0; j < ( int ) ordered_edges.size(); j++ )
      {
         int v = getEdge( ( ordered_edges ) [ j ] ) ->divideEdge();
         connect_list.push_back( v );
      }

      //finish the loop.
      connect_list.push_back( connect_list[ 0 ] );

      //connect the verts as best we can.
      for ( int j = 0; j < ( int ) connect_list.size() - 1; j++ )
      {
         Vertex *v1, *v2;

         v1 = getVertex( connect_list[ j ] );
         v2 = getVertex( connect_list[ j + 1 ] );

         vector<int> flist;
         flist = v1 -> getCommonFaces( v2 -> getParentIndex() );

         //divide the faces;
         //If they aren't dividable, it's just ignored.
         for ( int h = 0; h < ( int ) flist.size(); h++ )
            getFace( flist[ h ] ) ->splitByVerts( v1 -> getParentIndex(), v2 -> getParentIndex() );

      }

      //extrude the edge...
      Vector4 normal;

      normal = getEdge( elist[ i ] ) ->getNormal();

      getEdge( elist[ i ] ) ->move( normal.x * amt.x, normal.y * amt.y, normal.z * amt.z );
 	
   }

	 cleanSubObjects();

   normalize();
}

void Mesh::extrudeFaces( vector<int> &flist, Vector4 &amt )
{
   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      getFace( flist[ i ] ) -> extrude( amt );

   }

   normalize();
}


/**Mirror the object by negating the verts in the appropriate axis */
void Mesh::mirror( int axis )
{
   Vector4 p;

   for ( int i = 0; i < ( int ) verts.size(); i++ )
   {

      getVertex( i ) ->getPosition( &p );

      switch ( axis )
      {
         case 0:
            getVertex( i ) ->setPosition( -p.x, p.y, p.z );
            break;
         case 1:
            getVertex( i ) ->setPosition( p.x, -p.y, p.z );
            break;
         case 2:
            getVertex( i ) ->setPosition( p.x, p.y, -p.z );
            break;
      }
   }

   //reverse the ordering of the faces
   for ( int i = 0; i < ( int ) faces.size(); i++ )
   {
      getFace( i ) ->reverse();
   }

   normalize();
}

/**Slices the given subobjects off into their own mesh.
  */
Mesh * Mesh::slice( int type, vector<int> &solist )
{
   vector<int> vlist;
   vector<int> flist;
   vector<int> elist;

   vlist.reserve( solist.size() );
   elist.reserve( solist.size() );
   flist.reserve( solist.size() );

   /* MODIFY THIS SO IT ONLY GETS THE VERTICES COMPLETELY CONTAINTED, NO EDGE VERTS */

   //get a list of all the vertices.
   if ( type != Vertex::TYPE )
   {
      for ( unsigned int i = 0; i < solist.size(); i++ )
      {
         SubObject *so = getSubObject( type, solist[ i ] );
         vector<int> * sovlist = so->getVerts();

         for ( unsigned int j = 0; j < sovlist->size(); j++ )
         {
            vlist.push_back( ( *sovlist ) [ j ] );
         }
      }
   }

   else
   {
      vlist = solist;
   }

   Util::removeDuplicates( vlist );

   // get the list of faces to remove
   for ( unsigned int i = 0; i < vlist.size(); i++ )
   {
      Vertex *v = getVertex( vlist[ i ] );
      int numFaces = v -> getNumFaces();

      for ( int j = 0; j < numFaces; j++ )
      {
         flist.push_back( v -> getFace( j ) );
      }
   }

   // get the list of edges to remove
   for ( unsigned int i = 0; i < vlist.size(); i++ )
   {
      Vertex *v = getVertex( vlist[ i ] );
      int numEdges = v -> getNumEdges();

      for ( int j = 0; j < numEdges; j++ )
      {
         elist.push_back( v -> getEdge( j ) );
      }
   }

   //remove all from this mesh.

   for ( unsigned int i = 0; i < vlist.size(); i++ )
      getVertex( vlist[ i ] ) ->detach();

   for ( unsigned int i = 0; i < elist.size(); i++ )
      getEdge( elist[ i ] ) ->detach();

   for ( unsigned int i = 0; i < flist.size(); i++ )
      getFace( flist[ i ] ) ->detach();

   cleanSubObjects();

   return 0;




}

/**Welds all the given verts into 1st one.*/
void Mesh::weldVerts( vector<int> &vlist )
{
   int v = vlist[ 0 ];

   for ( int i = 1; i < ( int ) vlist.size(); i++ )
   {
      getVertex( vlist[ i ] ) -> replace( v );
   }

   cleanSubObjects();
   cleanEdges();

}

void Mesh::weldFace( int a, int b )
{}


void Mesh::smooth( float tolerance )
{

   Mesh * m = new Mesh();

   m -> setPosition( pos );

   vector<Face_Point> fpts;
   vector<Edge_Point> epts;
   vector< vector<int> * > opts;

   int num_verts = verts.size();
   int num_faces = faces.size();
   int num_edges = edges.size();


   fpts.reserve( num_faces );
   epts.reserve( num_edges );
   opts.reserve( num_verts );


   copyVertsPosTo( m );

   //create the new face points.
   for ( int i = 0; i < ( int ) faces.size(); i++ )
   {
      Face_Point fp;
      Face *f = getFace( i );
      Vector4 c;
      c = f -> getCenter();
      Vertex *v = m -> createVertex( c );
      int nv = v -> getParentIndex();

      c = f -> getUVCenter();

      fp.f = i;
      fp.v = nv;
      fp.tu = c.x;
      fp.tv = c.y;

      fpts.push_back( fp );
   }

   //create the new edge points.
   Vector4 c;
   Vector4 vtmp;

   for ( int i = 0; i < ( int ) edges.size(); i++ )
   {
      Edge_Point ep;
      c.assign( 0, 0, 0, 0 );

      Edge *e = getEdge( i );
      c = e -> getCenter();

      if ( e->getNumFaces() >= 2 )
      {
         m -> getVertex( fpts[ e->getFace( 0 ) ].v ) -> getPosition( &vtmp );
         c += vtmp;
         m -> getVertex( fpts[ e->getFace( 1 ) ].v ) -> getPosition( &vtmp );
         c += vtmp;

         c.x /= 3;
         c.y /= 3;
         c.z /= 3;

         //grab the first UV's we find.


         ep.f1 = fpts[ e->getFace( 0 ) ];
         ep.f2 = fpts[ e->getFace( 1 ) ];
      }

      else
      {
         //if edge doesn't have 2 faces, make it
         //a hard edge.
         // ep.f1 = -1;
         // ep.f2 = -1;
      }

      Vertex *v = m -> createVertex( c );

      int nv = v -> getParentIndex();


      ep.e = i;
      ep.v = nv;

      epts.push_back( ep );
   }

   //capture the facepoints surrounding each vertex...
   for ( int i = 0; i < num_verts; i++ )
   {
      vector<int> *facepts = new vector<int>;
      vector<int> *vfaces = getVertex( i ) -> getFaces();
      *facepts = *vfaces;
      opts.push_back( facepts );
   }

   //reposition the old verts...

   for ( int i = 0; i < num_verts; i++ )
   {
      Vertex *v = m -> getVertex( i );
      int n = getVertex( i ) -> getNumEdges();

      Vector4 Q, R, S;
      Q.assign( 0, 0, 0, 0 );
      R.assign( 0, 0, 0, 0 );
      S.assign( 0, 0, 0, 0 );

      vector<int> *vfaces = opts[ i ];

      for ( int j = 0; j < ( int ) vfaces -> size(); j++ )
      {
         m -> getVertex( fpts[ ( *vfaces ) [ j ] ].v ) -> getPosition( &vtmp );
         Q += vtmp;
      }

      Q.x /= vfaces -> size();
      Q.y /= vfaces -> size();
      Q.z /= vfaces -> size();

      bool hard = false;
      vector<int> *vedges = getVertex( i ) -> getEdges();

      for ( int j = 0; j < ( int ) vedges -> size(); j++ )
      {
         //if the vert is connected to a 'hard' edge,
         //mark it as hard.
         hard = true;
         R += getEdge( ( *vedges ) [ j ] ) -> getCenter();
      }

      R.x /= n;
      R.y /= n;
      R.z /= n;


      v -> getPosition( &S );

      Q.x /= n;
      Q.y /= n;
      Q.z /= n;

      R.x = R.x * 2 / n;
      R.y = R.y * 2 / n;
      R.z = R.z * 2 / n;

      S.x = S.x * ( n - 3 ) / n;
      S.y = S.y * ( n - 3 ) / n;
      S.z = S.z * ( n - 3 ) / n;

      Vector4 new_pos( 0, 0, 0, 0 );

      new_pos = Q + R + S;
			new_pos.w = 1;
			
      //  if( !hard ) {
      v -> setPosition( new_pos );
      //   }

   }

   //subdivide all the faces...
   for ( int i = 0; i < num_faces; i++ )
   {
      Face *f = getFace( i );
      vector<int> *fverts = f->getVerts();

      for ( int j = 0; j < ( int ) fverts->size(); j++ )
      {
         int e1 = f->getEdgeBefore( ( *fverts ) [ j ] );
         int e2 = f->getEdgeAfter( ( *fverts ) [ j ] );

         //create a new face...
         m -> createFace( epts[ e1 ].v, ( *fverts ) [ j ], epts[ e2 ].v, fpts[ i ].v );

      }
   }

   //delete the face points vectors
   for ( int i = 0; i < ( int ) opts.size(); i++ )
   {
      opts[ i ] -> clear();

      delete opts[ i ];

   }

   opts.clear();

	 m -> copyVertsTo( this );
   m -> copyEdgesTo( this );
   m -> copyFacesTo( this );
   m -> copyUVsTo( this );


   normalize();

}

void Mesh::tighten( float tolerance )
{
   int n = verts.size();
   Vector4 ps[ n ];

   for ( int i = 0; i < n; i++ )
      ps[ i ] = getVertex( i ) ->getSmoothPosition( 100 );

   for ( int i = 0; i < n; i++ )
      verts[ i ] ->setPosition( ps[ i ].x, ps[ i ].y, ps[ i ].z );

   normalize();
}


void Mesh::divideEdges( vector<int> &e )
{
   for ( int i = 0; i < ( int ) e.size(); i++ )
   {
      getEdge( e[ i ] ) -> divideEdge();
   }

   cleanSubObjects();
   normalize();
}

void Mesh::reverseFaces( vector<int> &flist )
{
   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      int n = flist[ i ];
      Face *f = getFace( n );
      f -> reverse();
   }

   cleanSubObjects(); //not really needed here.
   normalize();
}

void Mesh::smoothFaces( vector<int> & )
{}

void Mesh::subdivideFaces( vector<int> & )
{}

void Mesh::bevelVerts( vector<int> &vlist, float amt )
{

   //for each vertex.
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      //connect all of it's edges.
      vector<int> connect_list;
      vector<int> ordered_edges;

      vector<int> elist;
      elist = * getVertex( vlist[ i ] ) -> getEdges();

      ordered_edges = getFaceOrderedEdges( elist );

      connect_list.reserve( elist.size() + 2 );

      for ( int j = 0; j < ( int ) ordered_edges.size(); j++ )
      {
         int v = getEdge( ( ordered_edges ) [ j ] ) ->divideEdge();
         connect_list.push_back( v );
      }

      connect_list.push_back( connect_list[ 0 ] );

      //connect the verts as best we can.
      for ( int j = 0; j < ( int ) connect_list.size() - 1; j++ )
      {
         Vertex *v1, *v2;

         v1 = getVertex( connect_list[ j ] );
         v2 = getVertex( connect_list[ j + 1 ] );

         vector<int> flist;
         flist = v1 -> getCommonFaces( v2 -> getParentIndex() );
         //divide the faces;
         //If they aren't dividable, it's just ignored.
         for ( int h = 0; h < ( int ) flist.size(); h++ )
            getFace( flist[ h ] ) ->splitByVerts( v1 -> getParentIndex(), v2 -> getParentIndex() );

      }

      connect_list.pop_back();

      Vertex *v1, *v2;
      v1 = getVertex( connect_list[ 0 ] );
      v2 = getVertex( connect_list[ 1 ] );

      vector<int> flist;
      flist = v1 -> getCommonFaces( v2 -> getParentIndex() );

      if ( getFace( flist[ 0 ] ) -> isOrientated( connect_list[ 0 ], connect_list[ 1 ] ) )
         std::reverse( connect_list.begin(), connect_list.end() );

      vector<int> sub;

      sub = * getVertex( vlist[ i ] ) -> getFaces();

      for ( int q = 0; q < ( int ) sub.size(); q++ )
      {
         getFace( sub[ q ] ) -> detach();
      }

      sub = * getVertex( vlist[ i ] ) -> getEdges();

      for ( int q = 0; q < ( int ) sub.size(); q++ )
      {
         getEdge( sub[ q ] ) -> detach();
      }

      createFace( connect_list );

   }

   cleanSubObjects();

   normalize();



}

void Mesh::bevelEdges( vector<int> &elist, float amt )
{


   //for each edge.
   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      //connect all of it's vert's edges.
      vector<int> elist1;
      vector<int> elist2;
      vector<int> evlist;
      vector<int> edge_list;

      vector<int> connect_list;
      vector<int> ordered_edges;

      //get the edges attached to this edge's verts.
      evlist = * getEdge( elist[ i ] ) -> getVerts();
			
      getEdge( elist[i] ) -> dump();

      elist1 = * getVertex( evlist[ 0 ] ) -> getEdges();
      elist2 = * getVertex( evlist[ 1 ] ) -> getEdges();


      //copy into a single array.
      //elist1 -> reserve( elist1->size() + elist2->size() );

      for ( int t = 0; t < ( int ) elist2.size(); t++ )
      {
         elist1.push_back( elist2[ t ] );
      }

      int e = elist[ i ];

      //remove this edge from the list...
      vector<int>::iterator d = find( elist1.begin(), elist1.end(), e );

      while ( d != elist1.end() )
      {
         elist1.erase( d );
         d = find( elist1.begin(), elist1.end(), e );
      }

      //order the edges...
      ordered_edges = getFaceOrderedEdges( elist1 );

      connect_list.reserve( elist1.size() + 2 );

      //divide all the edges.
      for ( int j = 0; j < ( int ) ordered_edges.size(); j++ )
      {
         int v = getEdge( ( ordered_edges ) [ j ] ) ->divideEdge();
         connect_list.push_back( v );
      }

      //finish the loop.
      connect_list.push_back( connect_list[ 0 ] );

      //connect the verts as best we can.
      for ( int j = 0; j < ( int ) connect_list.size() - 1; j++ )
      {
         Vertex *v1, *v2;

         v1 = getVertex( connect_list[ j ] );
         v2 = getVertex( connect_list[ j + 1 ] );

         vector<int> flist;
         flist = v1 -> getCommonFaces( v2 -> getParentIndex() );

         //divide the faces;
         //If they aren't dividable, it's just ignored.
         for ( int h = 0; h < ( int ) flist.size(); h++ )
            getFace( flist[ h ] ) ->splitByVerts( v1 -> getParentIndex(), v2 -> getParentIndex() );

      }

      //extrude the edge...
      connect_list.pop_back();

      Vertex *v1, *v2;

      v1 = getVertex( connect_list[ 0 ] );

      v2 = getVertex( connect_list[ 1 ] );

      vector<int> flist;

      flist = v1 -> getCommonFaces( v2 -> getParentIndex() );

      if ( getFace( flist[ 0 ] ) -> isOrientated( connect_list[ 0 ], connect_list[ 1 ] ) )
         std::reverse( connect_list.begin(), connect_list.end() );

      vector<int> everts;

      everts = * getEdge( elist[ i ] ) -> getVerts();

      for ( int t = 0; t < ( int ) everts.size(); t++ )
      {
         vector<int> sub;

         sub = * getVertex( everts[ t ] ) -> getFaces();

         for ( int q = 0; q < ( int ) sub.size(); q++ )
         {
            getFace( sub[ q ] ) -> detach();
         }

         sub = * getVertex( everts[ t ] ) -> getEdges();

         for ( int q = 0; q < ( int ) sub.size(); q++ )
         {
            getEdge( sub[ q ] ) -> detach();
         }
      }

      createFace( connect_list );


   }

   cleanSubObjects();

   normalize();

}

void Mesh::bevelFaces( vector<int> &flist, float amt )
{
   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      Vector4 normal;
      Face *f = getFace( flist[ i ] );

      f -> extrude();

      normal = f -> getNormal();
      f -> move ( .25 * normal.x, .25 * normal.y, .25 * normal.z );
      f -> scale ( .75, .75, .75 );
   }

   cleanSubObjects();
   normalize();

}


void Mesh::createNewFace( vector<int> &vlist )
{
   //create a new face with the given verts...

   Face * nf = createFace( vlist );
   Vector4 vtmp;
   //steal texture coordinates from neighbor faces
   //if there are some.
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      vector<int> *v_faces = getVertex( vlist[ i ] ) -> getFaces();

      if ( v_faces -> size() != 0 )
      {
         int fi = ( *v_faces ) [ 0 ];
         Face *f = getFace( fi );

         int f_uv = f -> findVert( vlist[ i ] );

         UVCoord *uv = f -> getUVPtr( f_uv );
         uv -> getPosition( &vtmp );
         nf -> setUVCoord( i, vtmp );
      }
   }


}

void Mesh::subdivideTriangles( vector<int> &flist )
{
   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      Face *f = getFace( flist[ i ] );
      f -> subdivideTriangles();
   }

   cleanSubObjects();
   normalize();


}

void Mesh::subdivideTriangles()
{
   int n = ( int ) faces.size();

   for ( int i = 0; i < n; i++ )
   {
      Face *f = getFace( i );
      f -> subdivideTriangles();
   }

   cleanSubObjects();
   normalize();


}

void Mesh::toggleEdgeHardness( vector<int> &elist )
{

   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      Edge *e = getEdge( elist[ i ] );
      e -> toggleHardness();
   }

}

void Mesh::setVertexColors( vector<int> &vlist, float r, float g, float b, float a )
{
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      Vertex *v = getVertex( vlist[ i ] );
      v -> setColor( r, g, b, a );
   }

}

void Mesh::resetVertexColors( vector<int> &vlist )
{
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      Vertex *v = getVertex( vlist[ i ] );
      v -> resetColor( );
   }
}


/** Explode this mesh into a single mesh for each
  * face.
  */
void Mesh::explode( vector<Mesh *> &meshes )
{
   Vector4 p;

   for ( int i = 0; i < numFaces(); i++ )
   {

      Mesh * m = new Mesh();

      Face *f = static_cast<Face *>( faces[ i ] );
      vector<int> *fverts = f->getVerts();

      vector<int> nverts;
      nverts.reserve( f -> getNumVerts() );

      p.assign( 0, 0, 0, 1 );

      for ( int j = 0; j < ( int ) fverts->size(); j++ )
      {

         getVertex( ( *fverts ) [ j ] ) ->getTransformedPosition( &p );
         Vertex *v = m -> createVertex( p );
         nverts.push_back( v -> getParentIndex() );

      }

      m -> createFace( nverts );
      m -> normalize();
      m -> center();
      meshes.push_back( m );

   }

}


inline Mesh * Mesh::sliceVerts( vector<int> &verts ) { return slice( Vertex::TYPE, verts );};

inline Mesh * Mesh::sliceFaces( vector<int> &verts ) { return slice( Face::TYPE, verts );};

inline Mesh * Mesh::sliceEdges( vector<int> &verts ) { return slice( Edge::TYPE, verts );};

inline void Mesh::connectVerts( vector<int> &v ) {connect( Vertex::TYPE, v, false );};

inline void Mesh::connectEdges( vector<int> &v ) {connect( Edge::TYPE, v, false );};

inline void Mesh::loopVerts( vector<int> &v ) {connect( Vertex::TYPE, v, true );};

inline void Mesh::loopEdges( vector<int> &v ) {connect( Edge::TYPE, v, true );};


