/*
 gui.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

gint gui_input_add(gint handle, GUIInputCondition condition,
                   GUIInputFunction function, gpointer data)
{
    return gdk_input_add(handle, condition, function, data);
}

void gui_input_remove(gint tag)
{
    gdk_input_remove(tag);
}

guint gui_timeout_add(guint32 interval, GUITimeoutFunction function, gpointer data)
{
    if (interval < 10)
	g_warning("gui_timeout_add(), interval = %d which is probably bug and eats all your cpu!", interval);
    return gtk_timeout_add(interval, function, data);
}

void gui_timeout_remove(gint tag)
{
    gtk_timeout_remove(tag);
}

/* Create dialog with label and close button */
void gui_dialog(gint type, gchar *labeltext, ...)
{
#ifdef HAVE_GNOME
    va_list args;
    gchar *str;

    va_start(args, labeltext);
    str = g_strdup_vprintf(labeltext, args);
    va_end(args);

    switch (type)
    {
        case DIALOG_OK:
            gnome_ok_dialog(str);
            break;
        case DIALOG_ERROR:
            gnome_error_dialog(str);
            break;
        case DIALOG_WARNING:
            gnome_warning_dialog(str);
            break;
    }
    g_free(str);
#else
    GtkWidget *dialog, *label;
    va_list args;
    gchar *str;

    g_return_if_fail(labeltext != NULL);

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    va_start(args, labeltext);
    str = g_strdup_vprintf(labeltext, args);
    va_end(args);

    label = gtk_label_new(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, TRUE, FALSE, 0);
    g_free(str);

    gtk_widget_show_all(dialog);
#endif
}

/* Create dialog with entry field */
GtkWidget *gui_entry_dialog(gchar *labeltext, gchar *defaultentry, gchar *history, GtkSignalFunc okfunc, gpointer data)
{
    GtkWidget *dialog, *hbox, *label, *entry, *gtkentry;

    g_return_val_if_fail(labeltext != NULL, NULL);
    g_return_val_if_fail(okfunc != NULL, NULL);

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, okfunc, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gtk_object_set_data(GTK_OBJECT(dialog), "data", data);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    label = gtk_label_new(labeltext);
    gtk_misc_set_alignment(GTK_MISC(label), 1, .5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gnome_entry_new(history);
    gtkentry = gnome_entry_gtk_entry(GNOME_ENTRY(entry));
    if (defaultentry != NULL) gtk_entry_set_text (GTK_ENTRY (gtkentry), defaultentry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    gtk_object_set_data(GTK_OBJECT(dialog), "entry", gtkentry);
    gtk_signal_connect_object(GTK_OBJECT (gtkentry), "activate",
                              okfunc, GTK_OBJECT(dialog));
    gtk_widget_grab_focus(gtkentry);

    gtk_widget_show_all(dialog);

    return dialog;
}

/* create text entry with label */
GtkWidget *gui_create_labelentry(GtkWidget *box, gchar *text, gchar *field, gboolean fill)
{
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *hbox;

    g_return_val_if_fail(box != NULL, NULL);
    g_return_val_if_fail(text != NULL, NULL);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(box), hbox, fill, fill, 0);

    label = gtk_label_new(text);
    gtk_misc_set_alignment(GTK_MISC(label), 1, .5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gtk_entry_new();
    if (field != NULL) gtk_entry_set_text (GTK_ENTRY (entry), field);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    return entry;
}

GtkWidget *gui_create_tablelabelentry(GtkWidget *table, gint xpos, gint ypos, gchar *text, gchar *field)
{
    GtkWidget *label, *entry;

    g_return_val_if_fail(table != NULL, NULL);
    g_return_val_if_fail(text != NULL, NULL);

    label = gtk_label_new(text);
    gtk_misc_set_alignment(GTK_MISC(label), 1, .5);
    gtk_table_attach(GTK_TABLE(table), label, xpos, xpos+1, ypos, ypos+1, GTK_FILL, 0, 0, 0);

    entry = gtk_entry_new();
    if (field != NULL) gtk_entry_set_text (GTK_ENTRY (entry), field);
    gtk_table_attach(GTK_TABLE(table), entry, xpos+1, xpos+2, ypos, ypos+1, GTK_FILL | GTK_EXPAND, 0, 0, 0);

    return entry;
}

static gboolean sig_gui_dialog(gchar *type, gchar *text)
{
    gint dtype, pos, lpos, last;

    /* split text every 60 chars */
    text = g_strdup(text); last = lpos = 0;
    for (pos = 0; text[pos] != '\0'; pos++, lpos++)
    {
	if (lpos > 60 && last != 0)
	{
	    text[last] = '\n';
	    lpos = pos-last-1; last = 0;
	}
	if (text[pos] == ' ')
	    last = pos;
    }

    if (g_strcasecmp(type, "warning") == 0)
        dtype = DIALOG_WARNING;
    else if (g_strcasecmp(type, "error") == 0)
        dtype = DIALOG_ERROR;
    else
        dtype = DIALOG_OK;

    gui_dialog(dtype, text);
    g_free(text);
    return FALSE;
}

gchar *gui_key2str(GdkEventKey *event)
{
    GString *cmd;
    gchar *ret;

    g_return_val_if_fail(event != NULL, NULL);

    /* get alt/ctrl/shift masks */
    cmd = g_string_new(NULL);
    if (event->state & GDK_SHIFT_MASK)
	g_string_prepend(cmd, "SHIFT-");
    if (event->state & GDK_MOD1_MASK)
	g_string_prepend(cmd, "ALT-");
    if (event->state & GDK_CONTROL_MASK)
	g_string_prepend(cmd, "CTRL-");

    /* get key name */
    if (event->keyval > 255)
	g_string_append(cmd, gdk_keyval_name(event->keyval));
    else
	g_string_sprintfa(cmd, "%c", toupper(event->keyval));

    ret = cmd->str;
    g_string_free(cmd, FALSE);
    return ret;
}

void gui_init(void)
{
    signal_add("gui dialog", (SIGNAL_FUNC) sig_gui_dialog);
    signal_add("gui exit", (SIGNAL_FUNC) gtk_main_quit);
}

void gui_deinit(void)
{
    signal_remove("gui dialog", (SIGNAL_FUNC) sig_gui_dialog);
    signal_remove("gui exit", (SIGNAL_FUNC) gtk_main_quit);
}
